/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */


package com.ubuntuone.android.files.service;

import java.io.File;
import java.net.URI;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;

import android.content.ContentResolver;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.ResultReceiver;
import android.provider.MediaStore.MediaColumns;

import com.ubuntuone.android.files.Preferences;
import com.ubuntuone.android.files.provider.MetaUtilities;
import com.ubuntuone.android.files.provider.MetaContract.Nodes;
import com.ubuntuone.android.files.provider.MetaContract.ResourceState;
import com.ubuntuone.android.files.receiver.NetworkReceiver;
import com.ubuntuone.android.files.service.MetaService.Status;
import com.ubuntuone.android.files.util.FileUtilities;
import com.ubuntuone.android.files.util.Log;
import com.ubuntuone.android.files.util.StorageInfo;

public final class UpDownServiceHelper {
	
	private static final String TAG = UpDownServiceHelper.class.getSimpleName();
	
	private static AtomicLong mRequestId;
	/** Maps upload resource paths to unique request ids. */
	private static HashMap<String, Long> mUploadRequests;
	/** Maps download resource paths to unique request ids. */
	private static HashMap<String, Long> mDownloadRequests;
	/** Thread to receive and dispatch {@link UpDownService} callbacks. */
	private static Handler mCallbackHandler;
	/** Result receiver dispatching to registered callbacks. */
	private static BinderCallback mBinderCallback;
	/** Registered REST requests result listeners. */
	private static List<ResultReceiver> mRegisteredCallbacks;
	
	static {
		// Time is just a unique seed only, this has nothing to do with time.
		mRequestId = new AtomicLong(System.currentTimeMillis());
		mUploadRequests = new HashMap<String, Long>();
		mDownloadRequests = new HashMap<String, Long>();
		// Run asynchronous callbacks to listeners on non-UI thread.
		final HandlerThread thread = new HandlerThread("UpDownCallback",
	            android.os.Process.THREAD_PRIORITY_BACKGROUND);
	    thread.start();
	    final Looper callbackLooper = thread.getLooper();
	    mCallbackHandler = new Handler(callbackLooper);
		// Set up binder callback.
		mBinderCallback = new BinderCallback(mCallbackHandler);
		mRegisteredCallbacks = new LinkedList<ResultReceiver>();
	}
	
	public static void registerCallback(final ResultReceiver callback) {
		synchronized (mRegisteredCallbacks) {
			if (!mRegisteredCallbacks.contains(callback)) {
				mRegisteredCallbacks.add(callback);
			}
		}
	}
	
	public static synchronized void unregisterCallback(
			final ResultReceiver callback) {
		synchronized (mRegisteredCallbacks) {
			mRegisteredCallbacks.remove(callback);
		}
	}
	
	public static boolean isUploadPending(final String resourcePath) {
		return mUploadRequests.containsKey(resourcePath);
	}
	
	public static boolean isDownloadPending(final String resourcePath) {
		return mDownloadRequests.containsKey(resourcePath);
	}
	
	public static void removePendingTransfers() {
		mUploadRequests.clear();
		mDownloadRequests.clear();
	}
	
	/**
	 * Before calling, first check if isDownloadPending().
	 * 
	 * @param context
	 * @param resourcePath
	 */
	public static void download(Context context, String resourcePath,
			boolean autoTransfer) {
		final long requestId = mRequestId.incrementAndGet();
		Log.d(TAG, String.format("download() %s (%d)", resourcePath, requestId));
		if (isDownloadPending(resourcePath)) {
			Log.i(TAG, "download already pending");
			return;
		}
		
		String kind = null;
		String name = null;
		String hash = null;
		long size = 0L;
		String contentPath = null;
		
		final String[] projection = new String[] { Nodes.NODE_RESOURCE_PATH,
				Nodes.NODE_KIND, Nodes.NODE_NAME, Nodes.NODE_HASH, Nodes.NODE_SIZE,
				Nodes.NODE_CONTENT_PATH };
		final Cursor result = MetaUtilities
				.getNodeCursorByResourcePath(resourcePath, projection);
		try {
			if (result.moveToFirst()) {
				kind = result.getString(result.getColumnIndex(Nodes.NODE_KIND));
				name = result.getString(result.getColumnIndex(Nodes.NODE_NAME));
				hash = result.getString(result.getColumnIndex(Nodes.NODE_HASH));
				size = result.getLong(result.getColumnIndex(Nodes.NODE_SIZE));
				contentPath = result.getString(
						result.getColumnIndex(Nodes.NODE_CONTENT_PATH));
			}
		} finally {
			if(result != null) {
				result.close();
			}
		}
		
		if (Nodes.KIND_FILE.equals(kind)) {
			// File download.
			if (!NetworkReceiver.isConnected() ||
					(autoTransfer & !NetworkReceiver.isConnectedPreferred(context))) {
				Log.d(TAG, "off-line or auto-download on non pref network, aborting");
				MetaUtilities.setState(resourcePath, ResourceState.STATE_GETTING_FAILED);
				MetaUtilities.notifyChange(Nodes.CONTENT_URI);
				return;
			}
			
			mDownloadRequests.put(resourcePath, requestId);
			MetaUtilities.setState(resourcePath, ResourceState.STATE_GETTING);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			
			final String data =
					FileUtilities.getFileSchemeFromResourcePath(resourcePath);
			
			final Intent intent = new Intent(UpDownService.ACTION_DOWNLOAD);
			intent.putExtra(UpDownService.EXTRA_CALLBACK, mBinderCallback);
			intent.putExtra(UpDownService.EXTRA_AUTO_TRANSFER, autoTransfer);
			intent.putExtra(UpDownService.EXTRA_RESOURCE_PATH, resourcePath);
			intent.putExtra(UpDownService.EXTRA_CONTENT_PATH, contentPath);
			intent.putExtra(UpDownService.EXTRA_METHOD, HttpGet.METHOD_NAME);
			intent.putExtra(UpDownService.EXTRA_NAME, name);
			intent.putExtra(UpDownService.EXTRA_HASH, hash);
			intent.putExtra(UpDownService.EXTRA_SIZE, size);
			intent.putExtra(UpDownService.EXTRA_DATA, data);
			context.startService(intent);
		} else {
			// Directory download.
			// Move this to UpDownService, in one throw.
			final String selection = Nodes.NODE_PARENT_PATH + "=? AND "
					+ Nodes.NODE_KIND + "=?";
			final String[] selectionArgs =
					new String[] { resourcePath, Nodes.KIND_FILE };
			
			final Cursor ch = context.getContentResolver()
					.query(Nodes.CONTENT_URI, projection,
							selection, selectionArgs, null);
			try {
				if (ch.moveToFirst()) {
					do {
						final String chResourcePath = ch.getString(
								ch.getColumnIndex(Nodes.NODE_RESOURCE_PATH));
						if (!isDownloadPending(chResourcePath)) {
							download(context, chResourcePath, autoTransfer);
						}
					} while (ch.moveToNext());
				}
			} finally {
				if(ch != null ) {
					ch.close();
				}
			}
		}
	}
	
	public static void upload(Context context, Uri uri,
			String parentResourcePath, boolean autoTransfer) {
		upload(context, uri, parentResourcePath, -1, autoTransfer);
	}
	
	private static class Meta {
		public String visibleName;
		public String canonicalPath;
		public String mimeType;
		public Long size;
	}

	public static void upload(Context context, Uri uri,
			String parentResourcePath, long dateAdded, boolean autoTransfer) {
		final long requestId = mRequestId.incrementAndGet();
		Log.d(TAG, String.format("upload() %s to %s (%d)",
				uri.toString(), parentResourcePath, requestId));

		Meta meta = getMeta(context, uri);
		if (meta.canonicalPath != null) {
			Log.i(TAG, String.format("name=%s, path=%s, size=%d",
					meta.visibleName, meta.canonicalPath, meta.size));
		} else {
			Log.d(TAG, "file meta not found for " + uri.toString());
			return;
		}
		
		// We assume here we either got the path from MediaColumns.DATA or
		// know the path from a File('file://...'). If this is a custom provider,
		// such as OpenIntents, this may be a problem.
		
		boolean needsScheme = !meta.canonicalPath.contains("://");
		String uriString;
		// Fixes prepending scheme, when scheme is already present.
		if (needsScheme) {
			uriString = ContentResolver.SCHEME_FILE + "://" + meta.canonicalPath;
		} else {
			uriString = meta.canonicalPath;
		}
		
		if (!MetaUtilities.isValidUriTarget(uriString)) {
			Log.w(TAG, "Unsupported upload request: " + uriString);
			return;
		}
		
		final String parentPathNode = MetaUtilities.getStringField(
				parentResourcePath, Nodes.NODE_CONTENT_PATH);
		
		String contentPath =
				String.format("%s/%s", parentPathNode, meta.visibleName);
		String resourcePath =
				String.format("%s/%s", parentResourcePath, meta.visibleName);
		
		int count = MetaUtilities.getCount(resourcePath);
		
		if (count == 0) {
			// Insert dummy entry representing upload.
			final ContentValues values = new ContentValues();
			values.put(Nodes.NODE_RESOURCE_PATH, resourcePath);
			values.put(Nodes.NODE_CONTENT_PATH, contentPath);
			values.put(Nodes.NODE_RESOURCE_STATE, ResourceState.STATE_POSTING);
			values.put(Nodes.NODE_KIND, Nodes.KIND_FILE);
			values.put(Nodes.NODE_PARENT_PATH, parentResourcePath);
			final String path = resourcePath.substring(1);
			values.put(Nodes.NODE_PATH, path);
			values.put(Nodes.NODE_NAME, meta.visibleName);
			values.put(Nodes.NODE_MIME, meta.mimeType);
			values.put(Nodes.NODE_SIZE, meta.size);
			Log.d(TAG, "saving data uri: " + uriString);
			values.put(Nodes.NODE_DATA, uriString);
			final Uri metaUri = context.getContentResolver()
					.insert(Nodes.CONTENT_URI, values);
			Log.d(TAG, "insterted dummy entry: " + metaUri.toString());
		} else {
			final String oldResourcePath = resourcePath;
			
			// Entry already present. Check if uploaded.
			if (Preferences.getBoolean(Preferences.AVOID_UPLOAD_DUPS_KEY, true)) {
				// Check if the file hasn't been already uploaded.
				final String cachedResource = MetaUtilities.isUploaded(uriString);
				if (cachedResource != null) {
					Log.i(TAG, "Not uploading, file already under " + cachedResource);
					MetaUtilities.setState(resourcePath, null);
					return;
				}
			}
			
			// Don't let an auto transfer overwrite a file that:
			// has same name && is not a failed upload
			final String state = MetaUtilities.getStringField(
					resourcePath, Nodes.NODE_RESOURCE_STATE);
			
			if (autoTransfer && state == null) {
				// Inject counter to find a new, unused resource path,
				// i.e. IMG001-3.jpg, IMG001-4.jpg
				String filename = null;
				for (int i = 1; count > 0; i++) {
					filename = FileUtilities.injectCounterIntoFilename(
							meta.visibleName, i);
					contentPath =
							String.format("%s/%s", parentPathNode, filename);
					resourcePath =
							String.format("%s/%s", parentResourcePath, filename);
					count = MetaUtilities.getCount(resourcePath);
				}
				Log.d(TAG, "autoTransfer renamed file to: " + filename);
				
				// Update cached resource path to reflect actual target resource path.
				if (!oldResourcePath.equals(resourcePath)) {
					MetaUtilities.updateStringField(
							oldResourcePath, Nodes.NODE_RESOURCE_PATH, resourcePath);
					MetaUtilities.updateStringField(
							oldResourcePath, Nodes.NODE_CONTENT_PATH, contentPath);
				}
			}
			MetaUtilities.setState(resourcePath, ResourceState.STATE_POSTING);
		}
		MetaUtilities.notifyChange(Nodes.CONTENT_URI);
		
		if (isUploadPending(resourcePath)) {
			Log.i(TAG, "upload already pending, not uploading");
			return;
		}
		Log.d(TAG, "scheduling upload " + requestId);
		mUploadRequests.put(resourcePath, requestId);
		
		if (dateAdded != -1) {
			// This is (ATM) last of auto-uploaded pictures, update timestamp.
			try {
				Log.d(TAG, String.format(
						"Saving last uploaded photo timestamp (%s): %d",
						uriString, dateAdded));
				StorageInfo.setLastUploadedPhotoTimestamp(dateAdded);
			} catch (StorageInfo.StorageNotAvailable e) {
				Log.e(TAG, "Saving last upload time.", e);
			}
		}
		
		if (autoTransfer & !NetworkReceiver.isConnectedPreferred(context)) {
			Log.d(TAG, "auto-upload on non preferred network, aborting");
			MetaUtilities.setState(resourcePath, ResourceState.STATE_POSTING_FAILED);
			MetaUtilities.notifyChange(Nodes.CONTENT_URI);
			return;
		}
		
		final Intent intent = new Intent(UpDownService.ACTION_UPLOAD);
		intent.putExtra(UpDownService.EXTRA_CALLBACK, mBinderCallback);
		intent.putExtra(UpDownService.EXTRA_AUTO_TRANSFER, autoTransfer);
		intent.putExtra(UpDownService.EXTRA_RESOURCE_PATH, resourcePath);
		intent.putExtra(UpDownService.EXTRA_CONTENT_PATH, contentPath);
		intent.putExtra(UpDownService.EXTRA_PARENT, parentResourcePath);
		intent.putExtra(UpDownService.EXTRA_METHOD, HttpPost.METHOD_NAME);
		intent.putExtra(UpDownService.EXTRA_NAME, meta.visibleName);
		intent.putExtra(UpDownService.EXTRA_SIZE, meta.size);
		intent.putExtra(UpDownService.EXTRA_MIME, meta.mimeType);
		intent.putExtra(UpDownService.EXTRA_DATA, uriString);
		context.startService(intent);
	}
	
	private static Meta getMeta(Context context, Uri uri) {
		Meta result = new Meta();
		try {
			final String scheme = uri.getScheme();
			if (ContentResolver.SCHEME_FILE.equals(scheme)) {
				final File f = new File(URI.create(uri.toString()));
				result.visibleName = f.getName();
				result.canonicalPath = f.getCanonicalPath();
				result.mimeType = FileUtilities.getMime(f.getName());
				result.size = f.length();
			} else if (ContentResolver.SCHEME_CONTENT.equals(scheme)) {
				final String[] projection = new String[] {
						MediaColumns.DISPLAY_NAME,
						MediaColumns.DATA,
						MediaColumns.MIME_TYPE,
						MediaColumns.SIZE
				};
				final Cursor c = context.getContentResolver()
						.query(uri, projection, null, null, null);
				try {
					if (c.moveToFirst()) {
						result.visibleName = c.getString(
								c.getColumnIndex(MediaColumns.DISPLAY_NAME));
						result.canonicalPath = c.getString(
								c.getColumnIndex(MediaColumns.DATA));
						result.mimeType = c.getString(
								c.getColumnIndex(MediaColumns.MIME_TYPE));
						result.size = c.getLong(
								c.getColumnIndex(MediaColumns.SIZE));
					}
				} finally {
					if(c != null ) {
						c.close();
					}
				}
			} else {
				Log.e(TAG, "unknown uri scheme: " + uri.toString());
			}
		} catch (Exception e) {
			Log.e(TAG, "This shouldn't happen.", e);
		}
		return result;
	}
	
	private static class BinderCallback extends ResultReceiver {

		public BinderCallback(Handler handler) {
			super(handler);
		}

		@Override
		protected void onReceiveResult(int resultCode, Bundle resultData) {
			switch (resultCode) {
			case Status.RUNNING:
				// Not used.
				break;
			case Status.PROGRESS:
				// Not used.
				break;
			case Status.ERROR:
				// XXX karni: Handle error?
				//$FALL-THROUGH$
			case Status.FINISHED:
				final String resourcePath = resultData
						.getString(UpDownService.EXTRA_RESOURCE_PATH);
				final String method = resultData
						.getString(UpDownService.EXTRA_METHOD);
				if (HttpPost.METHOD_NAME.equals(method)) {
					mUploadRequests.remove(resourcePath);
					Log.d(TAG, "removed upload from queue");
				} else if (HttpGet.METHOD_NAME.equals(method)) {
					mDownloadRequests.remove(resourcePath);
					Log.d(TAG, "removed download from queue");
				}
				int up = mUploadRequests.size();
				int down = mDownloadRequests.size();
				Log.d(TAG, String.format("pending up: %d down: %d", up, down));
				break;

			default:
				Log.w(TAG, "unhandled result code: " + resultCode);
				break;
			}
			
			synchronized (this) {
				for (ResultReceiver callback : mRegisteredCallbacks) {
					callback.send(resultCode, resultData);
				}
			}
		}
	}
	
	private UpDownServiceHelper() {
	}

}
