/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.util;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Environment;


public class ConfigUtilities {
	
	private static final String TAG = ConfigUtilities.class.getSimpleName();
	
	public static boolean isExternalStorageMounted() {
		return Environment.getExternalStorageState().equals(
				Environment.MEDIA_MOUNTED);
	}
	
	public static boolean isConnected_(Context context) {
		ConnectivityManager cm = (ConnectivityManager) 
				context.getSystemService(Context.CONNECTIVITY_SERVICE);
		boolean wifiConnected = cm.getNetworkInfo(
				ConnectivityManager.TYPE_WIFI).isConnected();
		boolean mobileConnected = cm.getNetworkInfo(
				ConnectivityManager.TYPE_MOBILE).isConnected();
		return wifiConnected || mobileConnected;
	}
	
	public static boolean isConnected(Context context, int networkType) {
		ConnectivityManager cm = (ConnectivityManager) 
				context.getSystemService(Context.CONNECTIVITY_SERVICE);
		NetworkInfo networkInfo = cm.getNetworkInfo(networkType);
		return networkInfo.isConnected();
	}
	
	public static int getAvailableNetworkType(Context context) {
		ConnectivityManager cm = (ConnectivityManager) 
				context.getSystemService(Context.CONNECTIVITY_SERVICE);
		NetworkInfo networkInfo;
		networkInfo = cm.getNetworkInfo(ConnectivityManager.TYPE_WIFI);
		if (NetworkInfo.State.CONNECTED.equals(networkInfo.getState()))
			return ConnectivityManager.TYPE_WIFI;
		networkInfo = cm.getNetworkInfo(ConnectivityManager.TYPE_MOBILE);
		if (NetworkInfo.State.CONNECTED.equals(networkInfo.getState()))
			return ConnectivityManager.TYPE_MOBILE;
		return -1;
	}
	
	public static boolean isBackgroundDataAllowed(Context ctx) {
		ConnectivityManager cm = (ConnectivityManager) 
				ctx.getSystemService(Context.CONNECTIVITY_SERVICE);
		return cm.getBackgroundDataSetting();
	}
	
	public static boolean isRoaming(Context ctx) {
		ConnectivityManager connManager = (ConnectivityManager)
				ctx.getSystemService(Context.CONNECTIVITY_SERVICE);
		final NetworkInfo mobile = connManager
				.getNetworkInfo(ConnectivityManager.TYPE_MOBILE);
		return mobile.isRoaming();
	}
	
	public static interface CanISync {
		public static final int CAN_SYNC		= 0;
		public static final int NO_NETWORK		= -1;
		public static final int NO_PREFERRED	= -2;
		public static final int NO_BACKGROUND	= -3;
		public static final int ROAMING			= -4;
	}
	
	@Deprecated
	public static int canSyncOnConnection(Context ctx) {
		boolean syncOnWifi = false;
//		Preferences.getBoolean(
//				Preferences.Options.SYNC_POLICY_WIFI, false);
		boolean wifiConnected = ConfigUtilities.isConnected(ctx,
				ConnectivityManager.TYPE_WIFI);
		
		boolean syncOnMobile = false;
//		Preferences.getBoolean(
//				Preferences.Options.SYNC_POLICY_MOBILE, false);
		boolean mobileConnected = ConfigUtilities.isConnected(ctx,
				ConnectivityManager.TYPE_MOBILE);
		boolean isRoaming = isRoaming(ctx);
		
		boolean syncInBackground = ConfigUtilities
				.isBackgroundDataAllowed(ctx);
		
		if (!wifiConnected && !mobileConnected) {
			Log.i(TAG, "can not sync, no network");
			return CanISync.NO_NETWORK;
		}
		
		if (!syncInBackground) {
			Log.i(TAG, "can not sync, background data disabled");
			return CanISync.NO_BACKGROUND;
		}
		
		int preferredNetwork = -1;
		
		if (mobileConnected && syncOnMobile)
			preferredNetwork = ConnectivityManager.TYPE_MOBILE;
		
		// Prever wifi over mobile.
		if (wifiConnected && syncOnWifi)
			preferredNetwork = ConnectivityManager.TYPE_WIFI;
		
		if (preferredNetwork == -1) {
			Log.i(TAG, "can not sync, no preferred network");
			return CanISync.NO_PREFERRED;
		}
		
		if (preferredNetwork == ConnectivityManager.TYPE_MOBILE && isRoaming) {
			Log.i(TAG, "can not sync, roaming!");
			return CanISync.ROAMING;
		}
		
		// We are connected and can sync on preferredNetwork.
		
		return CanISync.CAN_SYNC;
	}
	
}
