/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Michał Karnicki <michal.karnicki@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.util;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;
import java.util.logging.FileHandler;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import com.ubuntuone.android.files.activity.PreferencesActivity;

/**
 * This class uses the same class name and has same, most important method
 * signatures as the android.util.Log class, so it can be easily deployed
 * just by changing the import.<br />
 * <br />
 * By providing a {@link LogcatHandler} instance to a non-Android project
 * (such as ubuntuone-java-files-client, a Java project),
 * it can add that handler to it's regular {@link Logger} and use default logger
 * calls (such as {@link Logger#warning(String)}, {@link Logger#info(String)},
 * etc). This allows:<br />
 * - forwarding logger calls to logcat output<br />
 * - (optionally) collect logs in case of issues on user side
 */
public final class Log {
	private static String APP_TAG = "Log";
	
	public static int LOG_LEVEL = android.util.Log.DEBUG;

	private static final Logger logger;
	private static final Logger restLogger;
	
	private static FileHandler fileHandler;
	
	static {
		logger = Logger.getLogger("com.ubuntuone.u1f");
		logger.setUseParentHandlers(true);
		logger.setLevel(Level.INFO);
		// Forward rest logs to main app log.
		restLogger = Logger.getLogger("com.ubuntuone.rest");
		restLogger.setUseParentHandlers(false);
		restLogger.setParent(logger);
	}
	
	private Log() {
	}

	/**
	 * Application should set it's tag for it to properly appear in logcat.
	 * 
	 * @param tag
	 *            the application tag to set
	 */
	public static void setApplicationTag(String tag) {
		APP_TAG = tag;
	}

	/**
	 * Method to set logcat log level.
	 * 
	 * @param newLevel
	 *            the level of logcat logging to set
	 */
	public static void setLogcatLogLevel(int newLevel) {
		LOG_LEVEL = newLevel;
	}

	/**
	 * Method to set application {@link Logger} level.
	 * 
	 * @param verbose
	 *            the level of app logger to set
	 */
	public static void setLogLevel(Level verbose) {
		logger.setLevel(verbose);
	}
	
	public static boolean startLogging() {
		try {
			fileHandler = new FileHandler(PreferencesActivity.getLogFile().getPath(),
					/* limit in bytes */ 2*1024*2024, /* file count */ 1);
			if (fileHandler != null) {
				fileHandler.setFormatter(new LogFormatter());
				logger.addHandler(fileHandler);
				restLogger.setUseParentHandlers(true);
				restLogger.setLevel(Level.FINE);
				setLogcatLogLevel(android.util.Log.DEBUG);
				
				logger.info("Started collecting logs.");
				return true;
			}
		} catch (IOException e) {
			Log.e("Log", e.getMessage());
			e.printStackTrace();
		}
		return false;
	}
	
	public static void stopLogging() {
		if (fileHandler != null) {
			logger.info("Stopped collecting logs.");
			logger.removeHandler(fileHandler);
			
			fileHandler.flush();
			fileHandler.close();
			
			restLogger.setUseParentHandlers(false);
			restLogger.setLevel(Level.OFF);
			setLogcatLogLevel(android.util.Log.INFO);
		}
	}
	
	public static void wtf(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.SEVERE, logMessage);
		if (LOG_LEVEL <= android.util.Log.ASSERT)
			android.util.Log.e(APP_TAG, logMessage);
	}
	
	public static void wtf(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.SEVERE, logMessage);
		if (LOG_LEVEL <= android.util.Log.ASSERT)
			android.util.Log.e(APP_TAG, logMessage);
	}
		
	public static void e(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.SEVERE, logMessage);
		if (LOG_LEVEL <= android.util.Log.ERROR)
			android.util.Log.e(APP_TAG, logMessage);
	}
	
	public static void e(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.SEVERE, logMessage);
		if (LOG_LEVEL <= android.util.Log.ERROR)
			android.util.Log.e(APP_TAG, logMessage);
	}

	public static void w(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.WARNING, logMessage);
		if (LOG_LEVEL <= android.util.Log.WARN)
			android.util.Log.w(APP_TAG, logMessage);
	}
	
	public static void w(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.WARNING, logMessage);
		if (LOG_LEVEL <= android.util.Log.WARN)
			android.util.Log.w(APP_TAG, logMessage);
	}

	public static void i(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.INFO, logMessage);
		if (LOG_LEVEL <= android.util.Log.INFO)
			android.util.Log.i(APP_TAG, logMessage);
	}
	
	public static void i(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.INFO, logMessage);
		if (LOG_LEVEL <= android.util.Log.INFO)
			android.util.Log.i(APP_TAG, logMessage);
	}

	public static void d(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.FINE, logMessage);
		if (LOG_LEVEL <= android.util.Log.DEBUG)
			android.util.Log.d(APP_TAG, logMessage);
	}
	
	public static void d(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.FINE, logMessage);
		if (LOG_LEVEL <= android.util.Log.DEBUG)
			android.util.Log.d(APP_TAG, logMessage);
	}

	public static void v(final String tag, final String msg) {
		final String logMessage = format(tag, msg);
		logger.log(Level.FINER, logMessage);
		if (LOG_LEVEL <= android.util.Log.VERBOSE)
			android.util.Log.v(APP_TAG, logMessage);
	}
	
	public static void v(final String tag, final String msg, final Throwable tr) {
		final String logMessage = format(tag, msg, tr);
		logger.log(Level.FINER, logMessage);
		if (LOG_LEVEL <= android.util.Log.VERBOSE)
			android.util.Log.v(APP_TAG, logMessage);
	}
	
	/**
	 * Utility method to get the stack trace from a {@link Throwable}.
	 * 
	 * @param tr
	 *            the {@link Throwable} to get the stack trace of
	 * @return string representing stack trace
	 */
	public static String getStackTraceString(Throwable tr) {
		if (tr == null) {
			return "";
		}
		final StringWriter sw = new StringWriter();
		final PrintWriter pw = new PrintWriter(sw);
		tr.printStackTrace(pw);
		return sw.toString();
	}

	/**
	 * Utility method used to format tag and message.
	 * 
	 * @param tag
	 * @param msg
	 * @return formatted string containing tag and msg
	 */
	private static final String format(String tag, String msg) {
		return String.format("%s: %s", tag, msg);
	}
	
	/**
	 * Utility method used to format tag, message and {@link Throwable} stack trace.
	 * 
	 * @param tag
	 * @param msg
	 * @param tr
	 * @return formatted string containing tag, msg and tr stack trace
	 */
	private static final String format(String tag, String msg, Throwable tr) {
		return String.format("%s: %s\n%s", tag, msg, getStackTraceString(tr));
	}
	
	private static class LogFormatter extends Formatter {
		// format: <level>/<loggername> <sequence no.> <date-time> <message>
		@Override
		public String format(LogRecord r) {
			final String format = "%s/%s %d %s %s\r\n";
			final String date = DateFormat.getDateTimeInstance().format(new Date());
			
			return String.format(format,
					r.getLevel(),
					r.getLoggerName(),
					r.getSequenceNumber(),
					date,
					r.getMessage());
		}
	}
}
