/*
 * Ubuntu One Files - access Ubuntu One cloud storage on Android platform.
 * 
 * Copyright (C) 2011 Canonical Ltd.
 * Author: Chad MILLER <chad.miller@canonical.com>
 *   
 * This file is part of Ubuntu One Files.
 *  
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *  
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *  
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses 
 */

package com.ubuntuone.android.files.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import android.os.Environment;


/** Utility functions to query and store information about the SDCard "external
 * storage". */
public final class StorageInfo {
	private final static String TAG = StorageInfo.class.getSimpleName();
	private final static String STORAGE_CONFIGURED_FILENAME = "marked-as-configured.bin";
	private final static String STORAGE_SHOULD_SEND_PHOTOS_FILENAME = "should-send-photos.bin";
	private final static String LAST_PUSHED_PHOTO_TIMESTAMP_FILENAME = "photo-upload-timestamp.bin";
	private final static String IO_CHARSET = "US-ASCII";

	public static class StorageNotAvailable extends Exception {
		public static final long serialVersionUID = 0x1L;
		public StorageNotAvailable() { super(); }
		public StorageNotAvailable(String s) { super(s); }
	}

	private StorageInfo() {}

	private static File getDataDirectory()
			throws StorageNotAvailable {
		final File sdcardRoot = Environment.getExternalStorageDirectory();
		final File dataLocation = new File(sdcardRoot,
				"Android/data/com.ubuntuone.android.files/files/config");
		if (! dataLocation.isDirectory()) {
			Log.i(TAG, "Creating config directory for new storage.");
			dataLocation.delete(); // Doesn't throw IOException
			dataLocation.mkdirs();
		}
		if (! dataLocation.isDirectory()) {
			throw new StorageNotAvailable("Making data directory");
		}
		return dataLocation;
	}

	/** Has the SDCard been seen by this app before, and has the user
	 * acknowleged its settings? */
	public static boolean isAlreadyConfigured() throws StorageNotAvailable {
		final File source = new File(getDataDirectory(), STORAGE_CONFIGURED_FILENAME);
		Log.d(TAG, "Checking if storage seen before.");
		return source.exists();
	}

	/** Sets that the user has set some configuration on this SDCard
	 * previously. */
	public static void setHasConfiguredStorage() throws StorageNotAvailable {
		final File source = new File(getDataDirectory(), STORAGE_CONFIGURED_FILENAME);
		try {
			Log.d(TAG, "Marking storage as configured before.");
			source.createNewFile();
		} catch (IOException e) {
			Log.e(TAG, "Creating has-configured flag for device-storage.", e);
		}
	}

	public static boolean shouldImmediatelyUploadPhotos()
			throws StorageNotAvailable {
		final File source = new File(getDataDirectory(), STORAGE_SHOULD_SEND_PHOTOS_FILENAME);
		Log.d(TAG, "Checking if storage wants new photos uploaded.");
		return source.exists();
	}

	public static void setShouldImmediatelyUploadPhotos(boolean should)
			throws StorageNotAvailable {
		final File source = new File(getDataDirectory(), STORAGE_SHOULD_SEND_PHOTOS_FILENAME);
		Log.d(TAG, "Setting whether storage wants new photos uploaded. " + should);
		if (should) {
			try {
				source.createNewFile();
			} catch (IOException e) {
				Log.e(TAG, "Creating upload-immediately flag for device-storage.", e);
			}
		} else {
			source.delete();
		}
	}

	synchronized
	public static long getLastUploadedPhotoTimestamp() throws StorageNotAvailable {
		final File source = new File(getDataDirectory(), LAST_PUSHED_PHOTO_TIMESTAMP_FILENAME);
		Log.d(TAG, "Getting last known upload timestamp.");
		InputStream in = null;
		try {
			byte[] buffer = new byte[13]; // length of str(2**32)+3
			in = new BufferedInputStream(new FileInputStream(source));
			in.read(buffer, 0, 13);
			String value = new String(buffer, IO_CHARSET);
			if (value.equals("")) {
				return 0L; // Error value, if none
			}
			return parseLong(value); // FIXME Chad, this doesn't work Long.valueOf(value);
		} catch (FileNotFoundException e) {
			Log.e(TAG, "Reading from photo timestamp.", e);
			return 0L; // Error value, if none
		} catch (UnsupportedEncodingException e) {
			Log.e(TAG, "Reading from photo timestamp.", e);
			return 0L; // Error value, if none
		} catch (IOException e) {
			Log.e(TAG, "Reading from photo timestamp.");
			return 0L; // Error value, if none
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException e) {
					Log.e(TAG, "Reading from photo timestamp.", e);
					return 0L; // Error value, if none
				}
			}
		}
	}
	
	private static Long parseLong(String value) {
		Long result = Long.valueOf(0L);
		for (int i = 0; i < value.length(); i++) {
			char c = value.charAt(i);
			if ('0' <= c && c <= '9') {
				result = 10*result + (c - '0');
			} else {
				break;
			}
		}
		return result;
	}

	synchronized
	public static void setLastUploadedPhotoTimestamp(long timestamp)
			throws StorageNotAvailable {
		final File destination = new File(getDataDirectory(), LAST_PUSHED_PHOTO_TIMESTAMP_FILENAME);
		OutputStream out = null;
		Log.d(TAG, "Setting last known upload timestamp.");
		try {
			out = new BufferedOutputStream(new FileOutputStream(destination));
			out.write(Long.toString(timestamp).getBytes(IO_CHARSET));
		} catch (UnsupportedEncodingException e) {
			Log.e(TAG, "Reading from photo timestamp.", e);
			throw new StorageNotAvailable();
		} catch (IOException e) {
			Log.e(TAG, "Reading from photo timestamp.", e);
			throw new StorageNotAvailable();
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (IOException e) {
					Log.e(TAG, "Writing photo timestamp.", e);
					throw new StorageNotAvailable();
				}
			}
		}
	}

}
