package com.utyf.pmetro.map;

import android.util.Log;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.PriorityQueue;

/**
 * Created by Fedor on 17.03.2016.
 *
 * Weighted directed graph and shortest paths finding within it. Vertices of the graph are
 * consecutive integer numbers.
 */
class BaseGraph {
    private ArrayList<ArrayList<EdgeInfo>> edges;
    private int nVertices;
    private int startVertex;

    // temporal data
    private PriorityQueue<Integer> queue;

    // output data
    private int[] parents;
    private double[] distances;

    class EdgeInfo {
        public int toIdx;
        public double weight;

        public EdgeInfo(int toIdx, double weight) {
            this.toIdx = toIdx;
            this.weight = weight;
        }
    }

    public BaseGraph() {
        edges = new ArrayList<>();
        nVertices = 0;
    }

    public void addVertex() {
        edges.add(new ArrayList<EdgeInfo>());
        nVertices++;
    }

    public void addEdge(int from, int to, double weight) {
        edges.get(from).add(new EdgeInfo(to, weight));
    }

    // Compute shortest paths using Dijkstra's algorithm
    public void computeShortestPaths(int startVertex) {
        this.startVertex = startVertex;
        // Reuse allocated memory
        if (distances == null || distances.length < nVertices) {
            distances = new double[nVertices];
            parents = new int[nVertices];
            queue = new PriorityQueue<>(nVertices,
                    new Comparator<Integer>() {
                        @Override
                        public int compare(Integer lhs, Integer rhs) {
                            return Double.compare(distances[lhs], distances[rhs]);
                        }
                    });
        }

        Arrays.fill(distances, Double.POSITIVE_INFINITY);
        distances[startVertex] = 0;
        queue.offer(startVertex);
        // Contains value for each vertex, indicating if it has been already extracted from the queue
        boolean isVisited[] = new boolean[nVertices];

        while (!queue.isEmpty()) {
            int fromIdx = queue.poll();
            // This check increases queue size, but allows not to use "decrease key" operation
            if (isVisited[fromIdx])
                continue;
            isVisited[fromIdx] = true;
            for (EdgeInfo edgeInfo: edges.get(fromIdx)) {
                int toIdx = edgeInfo.toIdx;
                double newDistance = distances[fromIdx] + edgeInfo.weight;
                if (distances[toIdx] > newDistance) {
                    distances[toIdx] = newDistance;
                    parents[toIdx] = fromIdx;
                    queue.offer(toIdx);
                }
            }
        }
    }

    // Compute length of path from start vertex to end vertex. Shortest paths must be already
    // precomputed by calling computeShortestPaths.
    public double getPathLength(int v) {
        if (distances == null) {
            Log.e("getPathLength", "Shortest paths are not computed!");
            return Double.POSITIVE_INFINITY;
        }
        return distances[v];
    }

    // Get shortest path from start vertex to v. Shortest paths must be already precomputed by
    // calling computeShortestPaths.
    public ArrayList<Integer> getPath(int v) {
        ArrayList<Integer> path = new ArrayList<>();
        while (true) {
            path.add(v);
            if (v == startVertex)
                break;
            v = parents[v];
        }
        Collections.reverse(path);
        return path;
    }
}
