/*
 * Copyright © 2017 Michael von Glasow.
 * 
 * This file is part of Qz.
 *
 * Qz is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Qz is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qz.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.vonglasow.michael.qz.android.util;

import java.util.IllegalFormatException;

import com.vonglasow.michael.qz.R;

import android.content.Context;
import android.util.Log;
import eu.jacquet80.rds.app.oda.AlertC.Event;
import eu.jacquet80.rds.app.oda.AlertC.InformationBlock;
import eu.jacquet80.rds.app.oda.AlertC.Message;
import eu.jacquet80.rds.app.oda.tmc.SupplementaryInfo;
import eu.jacquet80.rds.app.oda.tmc.TMC;

/**
 * Methods for working with TMC messages.
 */
public class MessageHelper {
	private static final String TAG = "MessageHelper";

	private static final long SECURITY_MASK =               0x40000L;
	private static final long WARNING_ACCIDENT_MASK =           0x4L;
	private static final long WARNING_SLIPPERY_MASK =   0x200002000L;
	private static final long WARNING_WIND_MASK =       0x800010000L;
	private static final long WARNING_ROADWORKS_MASK =        0x400L;
	private static final long WARNING_DANGER_MASK =       0x1C01808L;
	private static final long WARNING_DELAY_MASK =     0x2000080000L;
	private static final long WARNING_QUEUE_MASK =       0x80000003L;
	private static final long RESTRICTION_CLOSED_MASK =       0x1F0L;
	private static final long RESTRICTION_SIZE_MASK =     0x2000000L;
	private static final long INFO_CANCELLATION_MASK = 0x4000100000L;
	private static final long INFO_TEMPERATURE_MASK =  0x1000004000L;
	private static final long INFO_VISIBILITY_MASK =         0x8000L;
	private static final long INFO_EVENT_MASK =             0x20000L;
	private static final long INFO_TIME_MASK =             0x200000L;
	private static final long INFO_CARPOOL_MASK =             0x200L;
	private static final long INFO_PARKING_MASK =         0xC000000L;
	// private static final long INFO_INFO_MASK = 0x570000000L;


	/**
	 * Message class types.
	 * 
	 * The message class determines the nature of the message, reflected in an icon displayed to
	 * the user. Each message has exactly one message class, determined by the update classes of
	 * its constituent TMC events and an order of precedence.
	 */
	public static enum MessageClass {
		SECURITY,
		WARNING_ACCIDENT,
		WARNING_SLIPPERY,
		WARNING_WIND,
		WARNING_ROADWORKS,
		WARNING_DANGER,
		WARNING_DELAY,
		WARNING_QUEUE,
		RESTRICTION_CLOSED,
		RESTRICTION_SIZE,
		INFO_CANCELLATION,
		INFO_TEMPERATURE,
		INFO_VISIBILITY,
		INFO_EVENT,
		INFO_TIME,
		INFO_CARPOOL,
		INFO_PARKING,
		INFO_INFO,
	}

	/**
	 * @brief Returns a descriptive string for the detailed location of the message.
	 * 
	 * The detailed location typically refers to a stretch of road or a point on a road, which is
	 * the best approximation of the location of the event.
	 * 
	 * @param message
	 * @return
	 */
	public static String getDetailedLocationName(Message message) {
		String primaryName = message.getPrimaryName();
		String primaryJunctionNumber = message.getPrimaryJunctionNumber();
		if ((primaryJunctionNumber != null) && (!primaryJunctionNumber.isEmpty())) {
			if (primaryName == null)
				primaryName = primaryJunctionNumber;
			else
				primaryName = primaryName + " (" + primaryJunctionNumber + ")";
		}
		String secondaryName = message.getSecondaryName();
		String secondaryJunctionNumber = message.getSecondaryJunctionNumber();
		if ((secondaryJunctionNumber != null) && (!secondaryJunctionNumber.isEmpty())) {
			if (secondaryName == null)
				secondaryName = secondaryJunctionNumber;
			else
				secondaryName = secondaryName + " (" + secondaryJunctionNumber + ")";
		}
		if (primaryName != null) {
			if (secondaryName == null)
				return primaryName;
			else
				return String.format("%s %s %s",
						secondaryName,
						message.isBidirectional ? "↔" : "→",
								primaryName);
		} else
			return null;
	}

	/**
	 * @brief Returns a user-friendly string for the events in the message.
	 * 
	 * @param message
	 * @return
	 */
	public static String getEventText(Context context, Message message) {
		StringBuilder eventBuilder = new StringBuilder();
		for(InformationBlock ib : message.getInformationBlocks()) {
			// TODO destination

			for (Event event : ib.getEvents()) {
				if ((eventBuilder.length() > 0) && (eventBuilder.charAt(eventBuilder.length() - 1) != '?'))
					eventBuilder.append(". ");
				String eventText = getSingleEventText(context, event);
				eventText = eventText.substring(0, 1).toUpperCase() + eventText.substring(1);
				eventBuilder.append(eventText);

				for (SupplementaryInfo si : event.getSupplementaryInfo())
					eventBuilder.append(", " + context.getString(
							context.getResources().getIdentifier("si_" + si.code, "string", context.getPackageName())));
			}

			int length = ib.length;
			if (length != -1)
				try {
					eventBuilder.append(" ").append(getFormattedLength(context, length));
				} catch (IllegalFormatException e) {
					Log.w(TAG, "IllegalFormatException while parsing route length string, skipping");
					e.printStackTrace();
				}

			int speed = ib.speed;
			if(speed != -1)
				try {
					eventBuilder.append(", ").append(getFormattedSpeedLimit(context, speed));
				} catch (IllegalFormatException e) {
					Log.w(TAG, "IllegalFormatException while parsing speed limit string, skipping");
					e.printStackTrace();
				}

			// TODO diversion route
		}
		if (eventBuilder.charAt(eventBuilder.length() - 1) != '?')
			eventBuilder.append(".");
		return eventBuilder.toString();
	}

	/**
	 * @brief Returns the message class for a message.
	 * 
	 * The message class is determined based on the update classes of the constituent events of the
	 * message. Conflicts in determining the message class are resolved by applying the following
	 * order of precedence:
	 * <ul>
	 * <li>Security</li>
	 * <li>Warning</li>
	 * <li>Restriction</li>
	 * <li>Info</li>
	 * </ul>
	 * @param message
	 * @return
	 */
	public static MessageClass getMessageClass(Message message) {
		long updateClasses = 0;
		for (int evtCode : message.getEvents())
			updateClasses |= 1L << (TMC.getEvent(evtCode).updateClass - 1);

		if ((updateClasses & SECURITY_MASK) != 0)
			return MessageClass.SECURITY;
		if ((updateClasses & WARNING_ACCIDENT_MASK) != 0)
			return MessageClass.WARNING_ACCIDENT;
		if ((updateClasses & WARNING_SLIPPERY_MASK) != 0)
			return MessageClass.WARNING_SLIPPERY;
		if ((updateClasses & WARNING_WIND_MASK) != 0)
			return MessageClass.WARNING_WIND;
		if ((updateClasses & WARNING_ROADWORKS_MASK) != 0)
			return MessageClass.WARNING_ROADWORKS;
		if ((updateClasses & WARNING_DANGER_MASK) != 0)
			return MessageClass.WARNING_DANGER;
		if ((updateClasses & WARNING_DELAY_MASK) != 0)
			return MessageClass.WARNING_DELAY;
		if ((updateClasses & WARNING_QUEUE_MASK) != 0)
			return MessageClass.WARNING_QUEUE;
		if ((updateClasses & RESTRICTION_CLOSED_MASK) != 0)
			return MessageClass.RESTRICTION_CLOSED;
		if ((updateClasses & RESTRICTION_SIZE_MASK) != 0)
			return MessageClass.RESTRICTION_SIZE;
		if ((updateClasses & INFO_CANCELLATION_MASK) != 0)
			return MessageClass.INFO_CANCELLATION;
		if ((updateClasses & INFO_TEMPERATURE_MASK) != 0)
			return MessageClass.INFO_TEMPERATURE;
		if ((updateClasses & INFO_VISIBILITY_MASK) != 0)
			return MessageClass.INFO_VISIBILITY;
		if ((updateClasses & INFO_EVENT_MASK) != 0)
			return MessageClass.INFO_EVENT;
		if ((updateClasses & INFO_TIME_MASK) != 0)
			return MessageClass.INFO_TIME;
		if ((updateClasses & INFO_CARPOOL_MASK) != 0)
			return MessageClass.INFO_CARPOOL;
		if ((updateClasses & INFO_PARKING_MASK) != 0)
			return MessageClass.INFO_PARKING;
		/* fallback */
		return MessageClass.INFO_INFO;
	}

	/**
	 * @brief Formats the route length as a string.
	 * 
	 * If the length argument exceeds 100, the "for more than 100 km" string will be returned.
	 * 
	 * @param context The context
	 * @param length The length of the route affected, in km
	 * 
	 * @return
	 */
	private static String getFormattedLength(Context context, int length) {
		if (length > 100)
			return context.getString(R.string.q_km_max);
		return context.getResources().getQuantityString(context.getResources().getIdentifier(
				"q_km", "plurals", context.getPackageName()), length, length);
	}

	/**
	 * @brief Formats an event quantifier as a string.
	 * 
	 * Quantifiers cannot be of type 0 or 1, as these are processed directly by
	 * {@link #getSingleEventText(Context, Event)}.
	 * 
	 * @param context The context
	 * @param event The event from which the quantifier will be taken
	 * 
	 * @return
	 */
	private static String getFormattedQuantifier(Context context, Event event) {
		int q = event.quantifier;
		// q == 0 is the highest value, i.e. 2^5 (types 0-5) or 2^8 (types 6-12)
		if (q == 0) {
			q = event.tmcEvent.quantifierType <= 5 ? 32 : 256;
		}

		int resid = 0;
		double val = -1;

		switch (event.tmcEvent.quantifierType) {
		/* 0 and 1 are handled separately, not here */
		case 2:
			/* visibility */
			resid = context.getResources().getIdentifier(
					"q_visibility_m", "plurals", context.getPackageName());
			q *= 10;
			break;

		case 3:
			/* probability */
			resid = context.getResources().getIdentifier(
					"q_probability", "plurals", context.getPackageName());
			q = q - 1 * 5;
			break;

		case 4:
			/* speed limit */
			resid = context.getResources().getIdentifier(
					"q_km_h", "plurals", context.getPackageName());
			q *= 5;
			break;

		case 5:
			/* duration */
			if (q <= 10) {
				resid = context.getResources().getIdentifier(
						"q_duration_min", "plurals", context.getPackageName());
				q *= 5;
			} else {
				resid = context.getResources().getIdentifier(
						"q_duration_h", "plurals", context.getPackageName());
				if (q <=22)
					q -= 10;
				else
					q = q - 20 * 6;
			}
			break;

		case 6:
			/* temperature */
			resid = context.getResources().getIdentifier(
					"q_temperature_celsius", "plurals", context.getPackageName());
			q -= 51;
			break;

		case 7:
			/* time */
			// TODO use current locale to format time
			return String.format("%02d:%02d",(q-1)/6, ((q-1)%6)*10);

		case 8:
			/* weight */
		case 9:
			/* dimension */
			val = q <= 100 ? q/10. : .5 * (q-80);
			if (event.tmcEvent.quantifierType == 8)
				resid = context.getResources().getIdentifier(
						"q_t", "plurals", context.getPackageName());
			else
				resid = context.getResources().getIdentifier(
						"q_m", "plurals", context.getPackageName());
			break;

		case 10:
			/* precipitation */
			resid = context.getResources().getIdentifier(
					"q_precipitation_mm", "plurals", context.getPackageName());
			break;

		case 11:
			/* frequency in MHz */
			resid = context.getResources().getIdentifier(
					"q_freq_mhz", "plurals", context.getPackageName());
			val = 87.5 + q / 10.;
			break;

		case 12:
			/* frequency in kHz */
			resid = context.getResources().getIdentifier(
					"q_freq_khz", "plurals", context.getPackageName());
			if (q < 16)
				q = q * 9 + 144;
			else
				q = q * 9 + 387;
			break;

		default:
			return "ILLEGAL";
		}

		if (val != -1)
			return context.getResources().getQuantityString(resid, (int) val, val);
		else
			return context.getResources().getQuantityString(resid, q, q);
	}

	/**
	 * @brief Format the speed limit as a string.
	 * 
	 * @param context The context
	 * @param speed The speed limit, in km/h
	 * 
	 * @return
	 */
	private static String getFormattedSpeedLimit(Context context, int speed) {
		return context.getResources().getQuantityString(context.getResources().getIdentifier(
				"q_maxspeed_km_h", "plurals", context.getPackageName()), speed, speed);
	}

	/**
	 * @brief Generates a formatted event description for a single event.
	 * 
	 * If the event has a quantifier, this method returns the quantifier string for the event
	 * with the quantifier parsed and inserted. Otherwise, the generic description is returned.
	 * 
	 * @param context The context
	 * @param event The event
	 * @return The message
	 */
	private static String getSingleEventText(Context context, Event event) {
		String text = "";
		if (event.quantifier != -1) {
			if (event.tmcEvent.quantifierType >= 2) {
				String q = getFormattedQuantifier(context, event);
				text = context.getString(context.getResources().getIdentifier(
						"event_" + event.tmcEvent.code + "_qn",
						"string", context.getPackageName()), q);
			} else {
				int q = (event.quantifier != 0) ? event.quantifier : 32;
				switch (event.tmcEvent.quantifierType) {
				case 0:
					if (q > 28)
						q = (q - 29) * 2 + 30;
					break;

				case 1:
					if (q > 14)
						q = (q - 12) * 50;
					else if (q > 4)
						q = (q - 4) * 10;
					break;
				}

				int resid = context.getResources().getIdentifier(
						"event_" + event.tmcEvent.code + "_qn", "plurals", context.getPackageName());
				if (resid != 0)
					text = context.getResources().getQuantityString(resid, q, q);
				else {
					resid = context.getResources().getIdentifier(
							"event_" + event.tmcEvent.code + "_qn", "string", context.getPackageName());
					if (resid != 0)
						text = context.getString(resid, Integer.toString(q));
				}
			}
		}
		if (text.isEmpty()) {
			text = context.getString(context.getResources().getIdentifier(
					"event_" + event.tmcEvent.code + "_q0", "string", context.getPackageName()));
		}
		return text;
	}
}
