
package com.webworxshop.swallowcatcher.downloads;

import android.util.Log;
import android.widget.Toast;
import android.os.Environment;
import android.os.Handler;
import android.os.Message;
import android.os.Bundle;
import android.net.wifi.WifiManager.WifiLock;
import android.os.PowerManager.WakeLock;
import android.content.Context;
import android.media.MediaScannerConnection;
import android.media.MediaScannerConnection.MediaScannerConnectionClient;
import android.net.Uri;

import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.HttpResponse;
import org.apache.http.HttpEntity;

import java.io.IOException;
import java.io.File;
import java.io.InputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.sql.SQLException;

import com.j256.ormlite.field.DatabaseField;
import com.j256.ormlite.table.DatabaseTable;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.android.apptools.OpenHelperManager;

import com.webworxshop.swallowcatcher.db.Episode;
import com.webworxshop.swallowcatcher.db.Subscription;
import com.webworxshop.swallowcatcher.db.DatabaseHelper;

@DatabaseTable(tableName = "downloads")
public class Download implements Runnable, MediaScannerConnection.MediaScannerConnectionClient
{
	public Download()
	{
	}

	public Download(DownloadService context, Episode ep, WifiLock wl, WakeLock pl, Handler handler)
	{
		this.context = context;
		this.episode = ep;
		this.wl = wl;
		this.pl = pl;
		this.handler = handler;
	}
	
	public void run()
	{
		DatabaseHelper helper = (DatabaseHelper)OpenHelperManager.getHelper(context);
	
		pl.acquire();
		wl.acquire();
	
		// open the output file
		FileOutputStream output = null;
		String state = Environment.getExternalStorageState();

		if(Environment.MEDIA_MOUNTED.equals(state))
		{
			try
			{
				// get the path
				Subscription sub = episode.getSubscription();
				Dao<Subscription, Object> sub_dao = helper.getSubscriptionDao();
				sub_dao.refresh(sub);
				
				// try to read the file and do import
				// The title must be added to the file name, for podcasts that have same-file-name items:
				file = getFile(sub.getName(), episode);
				File full_dir = new File(file.getParent());
				full_dir.mkdirs();
			
				output = new FileOutputStream(file, false);
			}
			catch(FileNotFoundException e)
			{
				Log.e("SwallowCatcher", "File not found!", e);
				// if this happens we get fail later, need to handle this better!
			}
			catch(SQLException e)
			{
				Log.e("SwallowCatcher", "Database error!", e);
				// if this happens we get fail later, need to handle this better!
			}
		}
		else
		{
			// handle this better
			Log.i("SwallowCatcher", "Can't write to external storage!");
			makeToast("Can't write to external storage!");
			return;
		}
			
		DefaultHttpClient client = new DefaultHttpClient();
		HttpGet method = new HttpGet(episode.getFileLink());
		
		try
    	{
    		HttpResponse response = client.execute(method);
    		HttpEntity entity = response.getEntity();
    		
    		length = entity.getContentLength();
    		InputStream input = entity.getContent();
    		
    		byte buffer[] = new byte[BUFFER_SIZE];
    		offset = 0;
    		
    		while(offset < length && !canceled)
    		{
    			// read chunk from input
    			int c = input.read(buffer, 0, BUFFER_SIZE);
    			
    			// write to output
    			output.write(buffer, 0, c);
    			
    			// update progress
    			offset += c;
    			progress = (int)(((double)offset / (double)length)*100);
    			if(progress > prev_progress)
    			{
    				Log.i("SwallowCatcher", "Downloading " + episode.getFileLink() + ": " + progress + "%");
    				
    				Message msg = handler.obtainMessage();
    				Bundle b = new Bundle();
    				b.putString("url", url);
    				b.putInt("progress", progress);
    				msg.setData(b);
    				msg.sendToTarget();
    				
    				prev_progress = progress;
    			}
    		}
    		input.close();
    		if (canceled)
    		{
    			Log.i("SwallowCatcher", "Download of " + episode.getFileLink() + " canceled!");
    			// TODO: remove file!
    		}
    		else
    		{
    			// mark episode as downloaded
    			try
    			{
					Dao<Episode, Object> ep_dao = helper.getEpisodeDao();
					episode.setStatus("downloaded");
					ep_dao.update(episode);
				}
				catch(SQLException e)
				{
					Log.e("SwallowCatcher", "Can't update episode status!", e);
				}
    		
    			// rescan media
    			scanner = new MediaScannerConnection(context, this);
    			scanner.connect();
    			finished = true;
    		}
    	}
    	catch(IOException e)
    	{
      		// handle error better than this
    		Log.e("SwallowCatcher", "Download error!", e);
    		makeToast("Download error! "+e.getMessage());
    	}
    	finally
    	{
    		wl.release();
    		pl.release();
    	}
	}
	
	/**
	 * A helper function to get a file object for a given podcast name and file name.
	 * @param podcastName
	 * @param episode 
	 * @return File ExternalStorageDirectory / Podcasts / podcastName / Title - filename
	 */
	public static File getFile(String podcastName, Episode episode) {
		String fileName = episode.getFileLink();
		String[] toks = fileName.split("/");
		fileName = episode.getTitle() +" - "+ toks[toks.length - 1];
		//Trim is necessary since a directory can't end with spaces.
		String path = makeSafe(podcastName.trim()) + "/" + makeSafe(fileName);

		// the path needs converting to make sure it's a valid path name
		Pattern p = Pattern.compile("[?><\\:*|^]");
		Matcher m = p.matcher(path);
		path = m.replaceAll("_");
		
		File dir = Environment.getExternalStorageDirectory(); 
		return new File(dir, "Podcasts/" + path);
	}
	
	/**
	 * Makes a filename safe by removing non-supported characters.
	 * @param filename
	 * @return filename with unsafe characters removed.
	 */
	private static String makeSafe(String filename) {
		StringBuilder fixedName = new StringBuilder();
		for (int c=0; c<filename.length(); c++) { // Make a valid name:
			if (VALIDCHARS.indexOf(filename.charAt(c))>-1) {
				fixedName.append(filename.charAt(c));
			}
		}
		return fixedName.toString();
	}
	
	/**
	 * Calls Toast thread-safe-ly, by running on UI thread.
	 * @param text
	 */
	private void makeToast(final String text) {
		context.progressUpdateHandler.post(new Runnable() {
			@Override
			public void run() {
				Toast.makeText(context, text, Toast.LENGTH_LONG).show();
			}
		});
	}
	
	public void onMediaScannerConnected()
	{
		Log.i("SwallowCatcher", "Scanning for: " + file.toString());
		scanner.scanFile(file.toString(), null);
	}
	
	public void onScanCompleted(String path, Uri uri)
	{
		Log.i("SwallowCatcher", "Finished scanning: " + path);
		scanner.disconnect();
	}
	
	public String toString()
	{
		return episode.toString() + " - " + episode.getSubscription().toString();
	}
	
	public int getProgress()
	{
		return progress;
	}
	
	public long getLength()
	{
		return length;
	}
	
	public long getOffset()
	{
		return offset;
	}
	
	public String getURL()
	{
		return episode.getFileLink();
	}
	
	public synchronized void cancel()
	{
		canceled = true;
	}
	
	public boolean getFinished()
	{
		return finished;
	}
	
	public boolean getCanceled()
	{
		return canceled;
	}
	
	private boolean finished = false;
	private boolean canceled = false;
	private DownloadService context;
	private String url, path;
	private WifiLock wl;
	private WakeLock pl;
	private Handler handler;
	private int progress = 0, prev_progress = -1;
	private File file = null;
	private MediaScannerConnection scanner;
	private static final int BUFFER_SIZE = 102400; // i.e. 100kB
	
	//Characters that are valid for file names:
	private final static String VALIDCHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 $%`-_@{}~!#().";
	
	public static final String COL_ID = "_id";
	public static final String COL_EPISODE = "episode";
	public static final String COL_LENGTH = "length";
	public static final String COL_OFFSET = "offset";
	
	@DatabaseField(generatedId = true, columnName=COL_ID)
	private int id;
	@DatabaseField(columnName=COL_EPISODE, foreign=true)
	private Episode episode;
	@DatabaseField(columnName=COL_LENGTH)
	private long length;
	@DatabaseField(columnName=COL_OFFSET)
	private long offset;
}

