package com.webworxshop.swallowcatcher.feeds;

import android.os.Handler;
import android.os.Message;
import android.os.Bundle;
import android.app.Activity;
import android.content.Context;
import android.util.Log;
import android.widget.Toast;
import android.os.Environment;
import android.net.wifi.WifiManager;
import android.net.wifi.WifiManager.WifiLock;
import android.os.PowerManager;
import android.os.PowerManager.WakeLock;

import java.io.IOException;
import java.io.InputStream;
import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.File;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.util.ArrayList;
import java.sql.SQLException;
import java.util.List;
import org.w3c.dom.*;
import org.w3c.tidy.Tidy;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import org.xml.sax.SAXException;

import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.HttpResponse;
import org.apache.http.HttpEntity;

import com.webworxshop.swallowcatcher.db.Subscription;
import com.webworxshop.swallowcatcher.db.Episode;
import com.webworxshop.swallowcatcher.db.DatabaseHelper;

import com.j256.ormlite.android.apptools.OpenHelperManager;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.stmt.QueryBuilder;
import com.j256.ormlite.stmt.PreparedQuery;

public class FeedDownloadThread extends Thread
{
	public FeedDownloadThread(Handler h, Activity a, Context c, Subscription s)
	{
		handler = h;
		activity = a;
		context = c;
		sub = s;
	}

	public void run()
	{
		WifiManager wm = (WifiManager)context.getSystemService(Context.WIFI_SERVICE);
		WifiLock wl = wm.createWifiLock(WifiManager.WIFI_MODE_FULL, "SwallowCatcher Feed Refresh Lock");
		
		PowerManager pm = (PowerManager)context.getSystemService(Context.POWER_SERVICE);
		WakeLock pl = pm.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "SwallowCatcher Feed Refresh Lock");
			
		try
		{	
			pl.acquire();
			wl.acquire();
			
			List<Subscription> subs;
			DatabaseHelper helper = (DatabaseHelper)OpenHelperManager.getHelper(context);
			Dao<Subscription, Object> sub_dao = helper.getSubscriptionDao();
			if(sub == null)
			{
				// get list of feeds from database
				subs = sub_dao.queryForAll();
			}
			else
			{
				subs = new ArrayList<Subscription>();
				subs.add(sub);
			}
			
			// calculate progress increment
			increment = 100/subs.size();
			
			// grab the feed urls
			DefaultHttpClient client = new DefaultHttpClient();
			for(int i = 0; i < subs.size(); i++)
			{
				HttpGet method = new HttpGet(subs.get(i).getURL());
				Log.i("SwallowCatcher", "Getting feed: " + subs.get(i).getURL());
				
				try
				{
					HttpResponse response = client.execute(method);
					HttpEntity entity = response.getEntity();
					
					if(entity != null)
					{
						Log.i("SwallowCatcher", "Fetching feed: " + subs.get(i).getURL());
						// do something with the response
						InputStream instream = entity.getContent();
						
						ArrayList<Episode> data;
						try
						{
							data = parseRSS(instream, subs.get(i), sub_dao);
							Dao<Episode, Object> ep_dao = helper.getEpisodeDao();
							boolean first = true;
							for(Episode e : data)
							{
								QueryBuilder<Episode, Object> qb = ep_dao.queryBuilder();
								qb.where().eq(Episode.COL_GUID, e.getGUID());
								PreparedQuery<Episode> p = qb.prepare();
								//Log.i("SwallowCatcher", "Query: " + p.getStatement());
								if(ep_dao.queryForFirst(p) == null)
								{
									if(subs.get(i).getNew() && !first)
									{
										e.setStatus("old");
									}
									else
									{
										e.setStatus("new");
									}
									//Log.i("SwallowCatcher", "Found New Episode: " + e);
									ep_dao.create(e);
								}
								first = false;
							}
							subs.get(i).setNew(false);
							sub_dao.update(subs.get(i));
						}
						catch(RuntimeException e)
						{
							Log.e("SwallowCatcher", "RSS parsing error!", e);
						}
						catch(SQLException e)
						{
							Log.e("SwallowCatcher", "Error inserting episode in database!", e);
						}
						instream.close();
						Log.i("SwallowCatcher", "Fetch complete");
					}
					else
					{
						// handle this!
						Log.i("SwallowCatcher", "Unable to fetch feed");
						makeToast("Unable to fetch feed");
					}
				}
				catch(IOException e)
				{
					Log.e("SwallowCatcher", "HTTP error!", e);
					makeToast("Podcast Update Error! "+e.getMessage());
				}
				
				Log.i("SwallowCatcher", "read feed");
				setProgress(increment);
			}
			OpenHelperManager.releaseHelper();//release was deprecated, renamed
			setProgress(100);
		}
		catch(SQLException e)
		{
			// TODO: handle this better
			Log.e("SwallowCatcher", "Cannot retrieve subscriptions!", e);
		}
		finally
		{
			wl.release();
    		pl.release();
    	}
	}
	
	protected void setProgress(int p)
	{
		Message msg = handler.obtainMessage();
		Bundle b = new Bundle();
		b.putInt("progress", p);
		msg.setData(b);
		handler.sendMessage(msg);
	}
	
	/**
	 * Calls Toast thread-safe-ly, by running on UI thread.
	 * @param text
	 */
	private void makeToast(final String text) {
		activity.runOnUiThread(new Runnable() {
			@Override
			public void run() {
				Toast.makeText(context, text, Toast.LENGTH_LONG).show();
			}
		});
	}
	
	protected ArrayList<Episode> parseRSS(InputStream input, Subscription sub, Dao<Subscription, Object> sub_dao)
	{
		try
		{
			Document dom;
			try
			{
				// try and parse with the standard parser first
				DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
				DocumentBuilder builder = factory.newDocumentBuilder();
				dom = builder.parse(input);
			}
			catch(SAXException ex)
			{
				// hmmm, bad xml. Fall back to using JTidy.
				// This is slower, but will adjust for bad XML
				Log.i("SwallowCatcher", "Problem with feed XML falling back to JTidy parser.");
				Tidy tidy = new Tidy();
				tidy.setXmlTags(true);
				dom = tidy.parseDOM(input, null);
			}
			
			Node channel = dom.getDocumentElement().getElementsByTagName("channel").item(0);
			NodeList items = channel.getChildNodes();
			
			ArrayList<Episode> data = new ArrayList<Episode>();
			for(int i = 0; i < items.getLength(); i++)
			{
				Node item = items.item(i);
				// update subscription details
				if(item.getNodeName().equals("title"))
				{
					sub.setName(item.getFirstChild().getNodeValue());
					continue;
				}
				else if(item.getNodeName().equals("description"))
				{
					// fix lo feed fail!
					if(item.getFirstChild() != null)
					{
						sub.setDescription(item.getFirstChild().getNodeValue());
					}
					continue;
				}
				else if(item.getNodeName().equals("itunes:image"))
				{
					NamedNodeMap attmap = item.getAttributes();
					downloadAlbumArt(sub, attmap.getNamedItem("href").getNodeValue());
					continue;
				}
				else if(!item.getNodeName().equals("item"))
				{
					continue;
				}
				
				Episode e = new Episode();
				e.setSubscription(sub);
				Log.i("SwallowCatcher", "subscription: " + sub.getID());
				
				NodeList children = item.getChildNodes();
				for(int j = 0; j < children.getLength(); j++)
				{
					Node child = children.item(j);
					if(child.getNodeName().equals("title"))
					{
						e.setTitle(child.getFirstChild().getNodeValue());
						Log.i("SwallowCatcher", "title: " + child.getFirstChild().getNodeValue());
					}
					else if(child.getNodeName().equals("guid"))
					{
						e.setGUID(child.getFirstChild().getNodeValue());
						Log.i("SwallowCatcher", "guid: " + child.getFirstChild().getNodeValue());
					}
					else if(child.getNodeName().equals("link"))
					{
						e.setLink(child.getFirstChild().getNodeValue());
						//Log.i("SwallowCatcher", "link: " + child.getFirstChild().getNodeValue());
					}
					else if(child.getNodeName().equals("description"))
					{
						e.setDescription(child.getTextContent());
					}
					else if(child.getNodeName().equals("pubDate"))
					{
						e.setDate(child.getFirstChild().getNodeValue());
						Log.i("SwallowCatcher", "date: " + child.getFirstChild().getNodeValue());
					}
					else if(child.getNodeName().equals("enclosure"))
					{
						NamedNodeMap attmap = child.getAttributes();
						e.setFileLink(attmap.getNamedItem("url").getNodeValue());
						//Log.i("SwallowCatcher", "file link: " + attmap.getNamedItem("url").getNodeValue());
					}
				}
				if(e.getGUID() == null)
				{
					Log.i("SwallowCatcher", "GUID was null, setting to link!");
					e.setGUID(e.getLink());
				}
				data.add(e);
			}
			sub_dao.update(sub);
			return data;
		}
		catch(Exception e)
		{
			throw new RuntimeException(e);
		}
	}
	
	// TODO: clean up this code and the code in Download.java into
	// a common file download utility class
	protected void downloadAlbumArt(Subscription sub, String url)
	{
		try
		{
			Log.i("SwallowCatcher", "Downloading cover art from: " + url);
			File file = getFile(sub.getName());
			File full_dir = new File(file.getParent());
			full_dir.mkdirs();
			
			DefaultHttpClient client = new DefaultHttpClient();
			HttpGet method = new HttpGet(url);
			HttpResponse response = client.execute(method);
			HttpEntity entity = response.getEntity();
			
			InputStream instream = entity.getContent();
			BufferedOutputStream outstream = new BufferedOutputStream(new FileOutputStream(file, false));
			
			byte buffer[] = new byte[BUFFER_SIZE];
			long offset = 0;
			long length = entity.getContentLength();
			
			while(offset < length)
			{
				int c = instream.read(buffer, 0, BUFFER_SIZE);
				outstream.write(buffer, 0, c);
				offset += c;
			}
			instream.close();
			outstream.close();
		}
		catch(FileNotFoundException e)
		{
			Log.e("SwallowCatcher", "File not found!", e);
		}
		catch(IOException e)
		{
			Log.e("SwallowCatcher", "Error downloading cover art.", e);
		}
	}
	
	/**
	 * A helper function to get a file object for a given podcast name and file name.
	 * @param podcastName
	 * @param episode 
	 * @return File ExternalStorageDirectory / Podcasts / podcastName / Title - filename
	 */
	public static File getFile(String podcastName) {
		//Trim is necessary since a directory can't end with spaces.
		String path = makeSafe(podcastName.trim()) + "/" + makeSafe("cover.jpg");

		// the path needs converting to make sure it's a valid path name
		Pattern p = Pattern.compile("[?><\\:*|^]");
		Matcher m = p.matcher(path);
		path = m.replaceAll("_");
		
		File dir = Environment.getExternalStorageDirectory(); 
		return new File(dir, "Podcasts/" + path);
	}
	
	/**
	 * Makes a filename safe by removing non-supported characters.
	 * @param filename
	 * @return filename with unsafe characters removed.
	 */
	private static String makeSafe(String filename) {
		StringBuilder fixedName = new StringBuilder();
		for (int c=0; c<filename.length(); c++) { // Make a valid name:
			if (VALIDCHARS.indexOf(filename.charAt(c))>-1) {
				fixedName.append(filename.charAt(c));
			}
		}
		return fixedName.toString();
	}

	private static final int BUFFER_SIZE = 102400;
	private final static String VALIDCHARS = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 $%`-_@{}~!#().";

	private int increment;
	private Handler handler;
	private Context context;
	private Activity activity;
	private Subscription sub = null;
}
