/*
 * Copyright (C) 2012 Xatik Group
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.xatik.app.droiddraw.client;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import android.app.WallpaperManager;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.net.Uri;
import android.os.Bundle;
import android.os.PowerManager;
import android.preference.PreferenceManager;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.Toast;

import com.actionbarsherlock.app.ActionBar;
import com.actionbarsherlock.app.SherlockActivity;
import com.actionbarsherlock.view.Menu;
import com.actionbarsherlock.view.MenuInflater;
import com.actionbarsherlock.view.MenuItem;
import com.xatik.app.droiddraw.client.databases.Command;
import com.xatik.app.droiddraw.client.databases.CommandsDbAdapter;
import com.xatik.app.droiddraw.client.databases.Script;
import com.xatik.app.droiddraw.client.databases.ScriptsDbAdapter;

public class DrawActivity extends SherlockActivity {
	public final static String SIS_BITMAP = "Bitmap";
	public final static String SIS_FIRSTDRAW = "FirstDraw";
	public final static String SIS_SHOWTURTLE = "ShowTurtle";
	public final static String SIS_CURRX = "CurrX";
	public final static String SIS_CURRY = "CurrY";
	public final static String SIS_ENDX = "EndX";
	public final static String SIS_ENDY = "EndY";
	public final static String SIS_ANGLE = "Angle";
	public final static String SIS_PAINTCOLOR = "PaintColor";
	public final static String SIS_PENDOWN = "PenDown";
	public final static String SIS_COUNTER = "Counter";
	public final static String SIS_RPTARRAYINDEX = "RepeatArrayIndex";
	public final static String SIS_RPTINDICES = "RepeatIndices";
	public final static String SIS_NUMREPEATS = "NumRepeats";
	public final static String SIS_NUMREPEATSSET = "NumRepeatsSet";
	
	private Script mScript;
	private ArrayList<Command> mCommands;
	private DrawView mDrawView;

	private boolean mFirstLoad = true;
	
	@Override
	protected void onCreate(Bundle savedInstanceState) {
		setTheme(R.style.Theme_Sherlock);
		super.onCreate(savedInstanceState);		
		
		ActionBar actionBar = getSupportActionBar();
        actionBar.setDisplayHomeAsUpEnabled(true);
		
        mScript = (savedInstanceState == null) ? null :
        	(Script) savedInstanceState.getSerializable(ScriptsDbAdapter.SCRIPT);
        
        Bundle extras = getIntent().getExtras();
        
        if (mScript == null) {
        	mScript = extras != null ? (Script) extras.getParcelable(ScriptsDbAdapter.SCRIPT) : null;
        }
        
        if (savedInstanceState == null) {
        	mCommands = null;
        } else {
        	mCommands = savedInstanceState.getParcelableArrayList(CommandsDbAdapter.DATABASE_TABLE);
        }
                
        if (mCommands == null) {
        	if (extras == null) {
        		mCommands = null;
        	} else {
        		mCommands = extras.getParcelableArrayList(CommandsDbAdapter.DATABASE_TABLE);
        	}
        }
        
        if (mScript != null) {
        	setTitle(mScript.get_name());
        }
        
        DisplayMetrics displayMetrics = new DisplayMetrics();
        getWindowManager().getDefaultDisplay().getMetrics(displayMetrics);
        int winWidth = Math.min(displayMetrics.widthPixels * 2, 2048);
        int winHeight = Math.min(displayMetrics.heightPixels * 2, 2048);
        
        boolean animate = PreferenceManager.getDefaultSharedPreferences(this).getBoolean(getResources().getString(R.string.pref_animate), true);
        
        PowerManager pm = (PowerManager) getSystemService(Context.POWER_SERVICE);
        
        if (savedInstanceState == null) {
        	mDrawView = new DrawView(this, mCommands, winWidth, winHeight, animate, pm);
        } else {
        	ArrayList<Boolean> temp = new ArrayList<Boolean>();
        	for (Boolean b : savedInstanceState.getBooleanArray(SIS_NUMREPEATSSET)) {
				temp.add(b);
			}
        	
        	mDrawView = new DrawView(this, mCommands, winWidth, winHeight, 
        			animate, (Bitmap) savedInstanceState.getParcelable(SIS_BITMAP),
        			savedInstanceState.getBoolean(SIS_FIRSTDRAW),
        			savedInstanceState.getBoolean(SIS_SHOWTURTLE),
        			savedInstanceState.getFloat(SIS_CURRX),
        			savedInstanceState.getFloat(SIS_CURRY),
        			savedInstanceState.getFloat(SIS_ENDX),
        			savedInstanceState.getFloat(SIS_ENDY),
        			savedInstanceState.getFloat(SIS_ANGLE),
        			savedInstanceState.getInt(SIS_PAINTCOLOR),
        			savedInstanceState.getBoolean(SIS_PENDOWN),
        			savedInstanceState.getInt(SIS_COUNTER),
        			savedInstanceState.getInt(SIS_RPTARRAYINDEX),
        			savedInstanceState.getIntegerArrayList(SIS_RPTINDICES),
        			savedInstanceState.getIntegerArrayList(SIS_NUMREPEATS),
        			temp, pm);
        }
        
        
        mDrawView.setDrawingCacheEnabled(true);
        mDrawView.setBackgroundColor(Color.BLACK);
    	//setContentView(mDrawView);
        
        final FrameLayout frame = new FrameLayout(this);
        frame.addView(mDrawView);
        
        final SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(this);
		boolean showTutorial = sp.getBoolean(getResources().getString(R.string.pref_draw_tut), true);
		
		final View tutView = LayoutInflater.from(getBaseContext()).inflate(R.layout.draw_tut, null);
		
        if (showTutorial) {
			frame.addView(tutView);			
		}
        
        setContentView(frame);
        
        if (showTutorial) {
        	// Set up the button event for the tutorial view
 			Button button = (Button) findViewById(R.id.completeButton);
 			
 			button.setOnClickListener(new View.OnClickListener() {
 	        	public void onClick(final View v) {
 	        		frame.removeView(tutView);
 	        		
 	        		Editor edit = sp.edit();
 	    			edit.putBoolean(getResources().getString(R.string.pref_draw_tut), false);
 	    			edit.commit();
 	            }
 			});
        }
	}
	
	@Override
	protected void onPause() {
		// TODO Auto-generated method stub
		super.onPause();
	}	

	@Override
	protected void onResume() {
		// TODO Auto-generated method stub
		super.onResume();
	}

	@Override
	protected void onSaveInstanceState(Bundle outState) {
		super.onSaveInstanceState(outState);
		outState.putParcelable(SIS_BITMAP, mDrawView.mScreenBitmap);
		outState.putBoolean(SIS_FIRSTDRAW, mDrawView.mFirstDraw);
		outState.putBoolean(SIS_SHOWTURTLE, mDrawView.mShowTurtle);
		outState.putFloat(SIS_CURRX, mDrawView.mCurrX);
		outState.putFloat(SIS_CURRY, mDrawView.mCurrY);
		outState.putFloat(SIS_ENDX, mDrawView.mEndX);
		outState.putFloat(SIS_ENDY, mDrawView.mEndY);
		outState.putFloat(SIS_ANGLE, mDrawView.mAngle);
		outState.putInt(SIS_PAINTCOLOR, mDrawView.mPaint.getColor());
		outState.putBoolean(SIS_PENDOWN, mDrawView.mPenDown);
		outState.putInt(SIS_COUNTER, mDrawView.mCounter);
		outState.putInt(SIS_RPTARRAYINDEX, mDrawView.mRepeatArrayIndex);
		outState.putIntegerArrayList(SIS_RPTINDICES, mDrawView.mRepeatIndices);
		outState.putIntegerArrayList(SIS_NUMREPEATS, mDrawView.mNumRepeats);
		boolean[] temp = new boolean[mDrawView.mNumRepeatsSet.size()];
		for (int i=0; i<mDrawView.mNumRepeatsSet.size(); i++) {
			temp[i] = mDrawView.mNumRepeatsSet.get(i);
		}
		outState.putBooleanArray(SIS_NUMREPEATSSET, temp);
	}
	
	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		MenuInflater inflater = getSupportMenuInflater();
        inflater.inflate(R.menu.draw, menu);
        return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
	        case android.R.id.home:
	        	setResult(RESULT_CANCELED);
	        	finish();
	            return true;
	        case R.id.menu_replay:
	        	DisplayMetrics displayMetrics = new DisplayMetrics();
	            getWindowManager().getDefaultDisplay().getMetrics(displayMetrics);
	            int winWidth = Math.min(displayMetrics.widthPixels * 2, 2048);
	            int winHeight = Math.min(displayMetrics.heightPixels * 2, 2048);
	            
	            PowerManager pm = (PowerManager) getSystemService(Context.POWER_SERVICE);
	            
	            mDrawView.cleanUp();
	            
	            boolean animate = PreferenceManager.getDefaultSharedPreferences(this).getBoolean(getResources().getString(R.string.pref_animate), true);
	            mDrawView = new DrawView(this, mCommands, winWidth, winHeight, animate, pm);
	            mDrawView.setDrawingCacheEnabled(true);
	            mDrawView.setBackgroundColor(Color.BLACK);
	        	setContentView(mDrawView);
	        	
	        	return true;
	        case R.id.menu_share:
	        	File file = takeScreenshot();
	        	
	        	if (file != null) {
		        	Intent share = new Intent(android.content.Intent.ACTION_SEND);
		        	share.setType("image/jpeg");
		        	share.putExtra(Intent.EXTRA_STREAM, Uri.fromFile(file));
		        	
		        	startActivity(Intent.createChooser(share, "Share via"));
	        	} else {
	        		Toast.makeText(this, "Could not share image", Toast.LENGTH_SHORT).show();
	        	}
	        	return true;
	        case R.id.menu_wallpaper:
	        	mDrawView.setDrawingCacheEnabled(true);
	    		mDrawView.buildDrawingCache(true);
	    		Bitmap bitmap = Bitmap.createBitmap(mDrawView.getDrawingCache());
	    		mDrawView.setDrawingCacheEnabled(false);
	    		
	    		WallpaperManager wallpaperManager = WallpaperManager.getInstance(this);
			try {
				wallpaperManager.setBitmap(bitmap);
				
				Toast.makeText(this, "Wallpaper Set", Toast.LENGTH_SHORT).show();
			} catch (IOException e) {
				e.printStackTrace();
			}
	    		
	    		return true;
	        case R.id.menu_animate:
	        	boolean menu_animate = false;
	        	if(item.isChecked()) {
	        		menu_animate = false;
	        	} else {
	        		menu_animate = true;
	        	}
	        	item.setChecked(menu_animate);
	        	
	        	SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(this);
	    		Editor edit = sp.edit();
	            edit.putBoolean(getResources().getString(R.string.pref_animate), menu_animate);
	    		edit.commit();
	        	return true;
	        default:
	        	Toast.makeText(this, "Got click: " + item.toString(), Toast.LENGTH_SHORT).show();
	            return true;
	            //return super.onOptionsItemSelected(item);
	    }
	}

	@Override
	public boolean onPrepareOptionsMenu(Menu menu) {
		super.onPrepareOptionsMenu(menu);
		
		if(mFirstLoad) {
			SharedPreferences sp = PreferenceManager.getDefaultSharedPreferences(this);
			
	        MenuItem animate = menu.findItem(R.id.menu_animate);
	        
	        boolean isAnimated = sp.getBoolean(getResources().getString(R.string.pref_animate), true);
	        animate.setChecked(isAnimated);
	        
	        mFirstLoad = false;
		}
		return true;
	}

	private File takeScreenshot() {
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy_MM_dd_HH_mm_ss");
		Date now = new Date();
		mDrawView.setDrawingCacheEnabled(true);
		mDrawView.buildDrawingCache(true);
		Bitmap bitmap = Bitmap.createBitmap(mDrawView.getDrawingCache());
		mDrawView.setDrawingCacheEnabled(false);
		String imageName = getExternalFilesDir(null).getPath() + "/" +
				mScript.get_name() + "_" + formatter.format(now) + getResources().getString(R.string.jpg);
		File file = new File(imageName);
		
		FileOutputStream fos;
		try {
			fos = new FileOutputStream(file);
			bitmap.compress(Bitmap.CompressFormat.JPEG, 100, fos);
			fos.close();
		} catch (FileNotFoundException e) {
			Log.e("DroidDrawShare", "File not found", e);
		} catch (IOException e) {
			Log.e("DroidDrawShare", "IOException", e);
		}
		
		return file;
	}
}
