/*
 *   Copyright 2013, 2014 Lukas Jirkovsky
 *
 *   This file is part of Počasí v krajích.
 *
 *   Počasí v krajích is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, version 3 of the License.
 *
 *   Počasí v krajích is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Počasí v krajích.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.jirkovsky.lukas.chmupocasi;

import android.app.Fragment;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;

import cz.jirkovsky.lukas.chmupocasi.forecast.DayForecast;
import cz.jirkovsky.lukas.chmupocasi.forecast.ForecastParser;
import cz.jirkovsky.lukas.chmupocasi.forecast.Region;

/**
 * Encapsulates downloading of forecast using an AsyncTask.
 */
@SuppressWarnings("WeakerAccess")
public class ForecastDownloader extends Fragment {
    // printf-like formatted URL with  a sing string (ie. %s) argument taking the region
    private final static String FORECAST_URL = "http://portal.chmi.cz/files/portal/docs/meteo/om/inform/p_%s.html";

    private Thread worker;
    private Handler handler;
    private DownloadForecastTask currentTask;
    private DownloadForecastTask waitingTask;
    private final Object taskLock = new Object();

    private int lastContentLength[];

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);

        lastContentLength = new int[Region.values().length];
    }

    public void setHandler(Handler handler) {
        this.handler = handler;
    }

    public synchronized void download(Region region, boolean checkChanged) {
        // prepare the task (and possibly execute it)
        synchronized (taskLock) {
            // if there is no running task, execute it immediately
            if (worker == null || !worker.isAlive()) {
                currentTask = new DownloadForecastTask(region, checkChanged);
                worker = new Thread(currentTask);
                worker.start();
            }
            // if there is a running task, add a new task only if it's different from the running task
            else if (currentTask.getRegion() != region) {
                waitingTask = new DownloadForecastTask(region, checkChanged);
            }
        }
    }

    class DownloadForecastTask implements Runnable {
        private final Region region;
        private final boolean checkChanged;
        private boolean error;
        private boolean changed;

        private DownloadForecastTask(Region region, boolean checkChanged) {
            this.region = region;
            this.checkChanged = checkChanged;
            // start with no error
            error = false;
            changed = false;
        }

        public Region getRegion() {
            return region;
        }

        @Override
        public void run() {
            final ArrayList<DayForecast> forecast = downloadUrl(String.format(FORECAST_URL, region.getLocation()));
            onPostExecute(forecast);
        }

        private void onPostExecute(ArrayList<DayForecast> s) {
            if (checkChanged && ! changed) {
                handler.sendMessage(Message.obtain(handler, 2, new MessageData(region, MessageData.TYPE.SUCCESS_UNCHANGED)));
            }
            // update the cache, if possible, otherwise add error
            else if (!error && s != null && ! s.isEmpty()) {
                // send success
                handler.sendMessage(Message.obtain(handler, 1, new MessageData(region, s)));
            }
            else {
                // send error
                handler.sendMessage(Message.obtain(handler, 3, new MessageData(region, MessageData.TYPE.ERROR)));
            }

            // execute the waiting task, if there is some
            synchronized (taskLock) {
                if (waitingTask != null) {
                    currentTask = waitingTask;
                    worker = new Thread(currentTask);
                    worker.start();
                    waitingTask = null;
                }
            }
        }

        private ArrayList<DayForecast> downloadUrl(String url) {
            InputStream is = null;
            HttpURLConnection conn = null;
            try {
                conn = (HttpURLConnection) (new URL(url)).openConnection();
                conn.setConnectTimeout(15000); // 15s
                conn.setReadTimeout(20000); // 20s
                conn.setRequestMethod("GET");
                conn.setDoInput(true);
                // needed fo getContentLength() to work
                conn.setRequestProperty("Accept-Encoding", "identity");
                // Starts the query
                conn.connect();

                if (checkChanged) {
                    int contentLength = conn.getContentLength();
                    if (contentLength != lastContentLength[region.ordinal()]) {
                        changed = true;
                        lastContentLength[region.ordinal()] = contentLength;
                    }
                    else {
                        changed = false;
                        return null;
                    }
                }

                // send the progress
                handler.sendMessage(Message.obtain(handler, 0, new MessageData(region, MessageData.TYPE.PROGRESS)));
                is = conn.getInputStream();

                return ForecastParser.parse(is);
            } catch (Exception e) {
                error = true;
            } finally {
                if (is != null) {
                    try {
                        is.close();
                    } catch (IOException e) {
                        // ignore exception here, as it's not problem
                    }
                }
                if (conn != null) {
                    conn.disconnect();
                }
            }
            return null;
        }
    }
}
