/*
 *   Copyright 2013, 2014 Lukas Jirkovsky
 *
 *   This file is part of Počasí v krajích.
 *
 *   Počasí v krajích is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, version 3 of the License.
 *
 *   Počasí v krajích is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Počasí v krajích.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.jirkovsky.lukas.chmupocasi.forecast;

import java.io.Serializable;

/**
 * A feature vector for the weather forecast.
 *
 * The values are based on information from:
 *  http://pocasi.chmi.cz/HK/metter.htm
 */
public class WeatherFeatures implements Serializable {
    private static final ParserTrie parserTrie;

    private double features[];
    private transient int featuresCount[];
    private transient double timeSpanQuantifier;
    private transient double lastQuantifier;
    private transient double quantifier;
    private transient int skip;

    private static enum WeatherFeature {
        // TODO: it would be nice to handle sun separately from the cloud cower in case that it is sunny in some places but cloudy elsewhere
        //SUN,
        CLOUD_COVER,
        RAIN,
        SHOWERS,
        STORM,
        FOG,
        SNOW
    }

    static {
        parserTrie = new ParserTrie();

        /***************/
        /* QUANTIFIERS */
        /***************/

        parserTrie.put("celé? území", new ParserTrie.FeatureParser() { // 70% – 100%
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier((features.quantifier < 1.0) ? 0.5*(features.quantifier + 0.85) : 0.85);
            }
        });
        parserTrie.put("většin? území", new ParserTrie.FeatureParser() { // 50% – 69%
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier((features.quantifier < 1.0) ? 0.5*(features.quantifier + 0.6) : 0.6);
            }
        });
        parserTrie.put("místy", new ParserTrie.FeatureParser() { // 30% – 49%
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier((features.quantifier < 1.0) ? 0.5*(features.quantifier + 0.4) : 0.4);
            }
        });
        parserTrie.put("ojediněl", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier((features.quantifier < 1.0) ? 0.5*(features.quantifier + 0.175) : 0.175);
            }
        });
        // this is not a quantifier for the affected area, but it is often used in forecast
        // quantifier 0.5 seems like a sane value
        parserTrie.put("občas", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier((features.quantifier < 1.0) ? 0.5*(features.quantifier + 0.5) : 0.5);
            }
        });

        /***************/
        /* TIME SPANS  */
        /***************/
        // The time span is used to reduce the quantifier in case the phenomenon is only for a limited time
        // eg. if the rain is only in the evening, it shouldn't be handled the same way as when it's raining all day
        parserTrie.put("ráno", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setTimeSpanQuantifier(0.75);
            }
        });
        parserTrie.put("odpoledne", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setTimeSpanQuantifier(0.85);
            }
        });
        parserTrie.put("večer", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setTimeSpanQuantifier(0.5);
            }
        });

        /****************/
        /* CONJUNCTIONS */
        /****************/
        // conjunctions keeps the quantifier
        // because all features resets the quantifier, they set it back the previous one
        ParserTrie.FeatureParser conjunctionParser = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.setQuantifier(features.lastQuantifier);
            }
        };
        parserTrie.put("nebo ", conjunctionParser);
        parserTrie.put("až ", conjunctionParser);

        /*********************/
        /* SUN & CLOUD COVER */
        /*********************/
        parserTrie.put("jasno", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, 0);
                features.resetQuantifier();
            }
        });
        parserTrie.put("skoro jasno", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, features.getQuantifier() * 0.1875);
                features.resetQuantifier();
            }
        });
        parserTrie.put("polojasno", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, features.getQuantifier() * 0.4375);
                features.resetQuantifier();
            }
        });
        parserTrie.put("oblačno", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, features.getQuantifier() * 0.6875);
                features.resetQuantifier();
            }
        });
        parserTrie.put("skoro zataženo", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, features.getQuantifier() * 0.875);
                features.resetQuantifier();
            }
        });
        ParserTrie.FeatureParser overcastParser = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.CLOUD_COVER, features.getQuantifier());
                features.resetQuantifier();
            }
        };
        parserTrie.put("zataženo", overcastParser);
        parserTrie.put("zataženo nízkou oblačností", overcastParser);

        ParserTrie.FeatureParser reduceCloudCoverParser = new ParserTrie.FeatureParser() {
            @Override
            void run(WeatherFeatures features) {
                // reduce the existing cloud cover a bit WITHOUT! averaging
                if (features.featuresCount[WeatherFeature.CLOUD_COVER.ordinal()] > 0) {
                    double val = - features.getQuantifier() * 0.35 *
                            features.features[WeatherFeature.CLOUD_COVER.ordinal()] / features.featuresCount[WeatherFeature.CLOUD_COVER.ordinal()];
                    features.addFeatureNoInc(WeatherFeature.CLOUD_COVER, val);
                }
                features.resetQuantifier();
            }
        };
        parserTrie.put("ubývání oblačn", reduceCloudCoverParser);
        parserTrie.put("protrhávání oblačn", reduceCloudCoverParser);

        ParserTrie.FeatureParser reduceRainParser = new ParserTrie.FeatureParser() {
            @Override
            void run(WeatherFeatures features) {
                // reduce the existing cloud cover a bit WITHOUT! averaging
                if (features.featuresCount[WeatherFeature.RAIN.ordinal()] > 0) {
                    double val = - features.getQuantifier() * 0.3 *
                            features.features[WeatherFeature.RAIN.ordinal()] / features.featuresCount[WeatherFeature.RAIN.ordinal()];
                    features.addFeatureNoInc(WeatherFeature.RAIN, val);
                }
                if (features.featuresCount[WeatherFeature.SHOWERS.ordinal()] > 0) {
                    double val = - features.getQuantifier() * 0.3 *
                            features.features[WeatherFeature.SHOWERS.ordinal()] / features.featuresCount[WeatherFeature.SHOWERS.ordinal()];
                    features.addFeatureNoInc(WeatherFeature.SHOWERS, val);
                }
                features.resetQuantifier();
            }
        };
        parserTrie.put("slábnutí sráž", reduceRainParser);
        parserTrie.put("ubývání sráž", reduceRainParser);
        parserTrie.put("ustávání sráž", reduceRainParser);

        /****************************/
        /* PARSER THAT DOES NOTHING */
        /****************************/
        // it just skips the word and decrements features.skip

        ParserTrie.FeatureParser ignoreParser = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                // ignore
                features.resetQuantifier();
            }
        };

        /**********************/
        /*    RAIN & SNOW     */
        /**********************/
        ParserTrie.FeatureParser rainParser = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.RAIN, features.getQuantifier());
                features.resetQuantifier();
            }
        };
        parserTrie.put("déšť", rainParser);
        parserTrie.put("dešt", rainParser);
        parserTrie.put("dešť", rainParser);

        parserTrie.put("přeháňk", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.SHOWERS, features.getQuantifier());
                features.resetQuantifier();
            }
        });

        // TODO
        parserTrie.put("mrholení", ignoreParser);

        parserTrie.put("bouřk", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.STORM, features.getQuantifier());
                features.resetQuantifier();
            }
        });
        // usually something along the way "v bouřkách vítr přechodně zesílí", so ignore
        parserTrie.put("bouřkách", ignoreParser);

        ParserTrie.FeatureParser snowParser = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.SNOW, features.getQuantifier());
                features.resetQuantifier();
            }
        };
        parserTrie.put("sníh", snowParser);
        parserTrie.put("sněh", snowParser);
        parserTrie.put("sněž", snowParser);

        ParserTrie.FeatureParser snowParser2 = new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.SNOW, features.getQuantifier() * 0.75);
                features.resetQuantifier();
            }
        };

        parserTrie.put("sněhové přeháňky", snowParser2);
        parserTrie.put("sněhovým? přeháňk", snowParser2);

        parserTrie.put("smíšené", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.SNOW, features.getQuantifier() * 0.5);
                features.addFeature(WeatherFeature.RAIN, features.getQuantifier() * 0.5);
                features.resetQuantifier();
            }
        });

        parserTrie.put("sněhové jazyky", ignoreParser);
        parserTrie.put("sněhových jazyků", ignoreParser);

        parserTrie.put("mlh", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.addFeature(WeatherFeature.FOG, features.getQuantifier());
                features.resetQuantifier();
            }
        });

        /****************/
        /*    OTHER     */
        /****************/
        // skip the feature if it's used in a conditional statement
        parserTrie.put("při ", new ParserTrie.FeatureParser() {
            @Override
            public void run(WeatherFeatures features) {
                features.skip = 1;
                features.resetQuantifier();
            }
        });

        // special case for Prague, because it may contain
        //  "Předpokládané množství srážek/nový sníh"
        // which should not be matched as snow
        parserTrie.put("srážek/nový sníh", ignoreParser);
    }

    /**
     * Constructs feature vector for the weather forecast.
     */
    public WeatherFeatures(String forecast) {
        int pos = 0;
        features = new double[WeatherFeature.values().length];
        featuresCount = new int[WeatherFeature.values().length];
        timeSpanQuantifier = 1.0;
        lastQuantifier = 1.0;
        quantifier = 1.0;
        skip = 0;

        while (pos < forecast.length()) {
            ParserTrie.Value value = parserTrie.get(forecast, pos);
            if (value != null) {
                // skip the already parsed part of the string. The -1 is needed in case the keyword ends with ' '
                pos = consume(forecast, value.end - 1);
                if (--skip >= 0) {
                    timeSpanQuantifier = 1.0;
                    lastQuantifier = 1.0;
                    quantifier = 1.0;
                } else {
                    value.parser.run(this);
                }
            }
            else {
                // skip to the next word
                int tmp = forecast.indexOf(' ', pos);
                if (tmp >= 0)
                    pos = tmp + 1;
                else
                    break;
            }

            // check if we have crossed the sentence boundary
            // it is pos -2 because "pos - 2" is the last character, "pos - 1" is space and "pos" is the next character
            if (pos > 2 && forecast.charAt(pos - 2) == '.') {
                timeSpanQuantifier = 1.0;
                lastQuantifier = 1.0;
                quantifier = 1.0;
                skip = 0;
            }
        }

        // average everything as needed
        for (int i = 0; i < WeatherFeature.values().length; ++i) {
            if (featuresCount[i] > 0)
                features[i] = features[i] / featuresCount[i];
            else
                features[i] = 0.0;
        }
    }

    public double getCloudCover() {
        return features[WeatherFeature.CLOUD_COVER.ordinal()];
    }

    public double getRain() {
        return features[WeatherFeature.RAIN.ordinal()];
    }

    public double getShowers() {
        return features[WeatherFeature.SHOWERS.ordinal()];
    }

    public double getStorm() {
        return features[WeatherFeature.STORM.ordinal()];
    }

    public double getSnow() {
        return features[WeatherFeature.SNOW.ordinal()];
    }

    private void addFeature(WeatherFeature feature, double value) {
        addFeatureNoInc(feature, value);
        featuresCount[feature.ordinal()]++;
    }

    /// adds a new feature without incrementing its count
    private void addFeatureNoInc(WeatherFeature feature, double value) {
        features[feature.ordinal()] = features[feature.ordinal()] + value;
    }

    private void setQuantifier(double value) {
        lastQuantifier = value;
        quantifier = value;
    }

    private void setTimeSpanQuantifier(double value) {
        timeSpanQuantifier = value;
    }

    private void resetQuantifier() {
        lastQuantifier = quantifier;
        quantifier = 1.0;
    }

    // returns the quantifier with the time span applied
    public double getQuantifier() {
        return timeSpanQuantifier * quantifier;
    }

    /**
     * Consume string until next space (including the space)
     * @param string string to process
     * @param pos position where the process starts
     * @return position of the next word in a string or
     *         string.length if there is no other word.
     */
    private static int consume(String string, int pos) {
        int tmp = string.indexOf(' ', pos);
        return (tmp >= 0) ? tmp + 1 : string.length();
    }
}
