var __assign = (this && this.__assign) || Object.assign || function(t) {
    for (var s, i = 1, n = arguments.length; i < n; i++) {
        s = arguments[i];
        for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
            t[p] = s[p];
    }
    return t;
};
import graphqlAnywhere from 'graphql-anywhere';
import { assign, isEqual, getDefaultValues, getQueryDefinition, isJsonValue, isIdValue, toIdValue, getStoreKeyName, } from 'apollo-utilities';
/**
 * The key which the cache id for a given value is stored in the result object. This key is private
 * and should not be used by Apollo client users.
 *
 * Uses a symbol if available in the environment.
 *
 * @private
 */
export var ID_KEY = typeof Symbol !== 'undefined' ? Symbol('id') : '@@id';
/**
 * Resolves the result of a query solely from the store (i.e. never hits the server).
 *
 * @param {Store} store The {@link NormalizedCache} used by Apollo for the `data` portion of the
 * store.
 *
 * @param {DocumentNode} query The query document to resolve from the data available in the store.
 *
 * @param {Object} [variables] A map from the name of a variable to its value. These variables can
 * be referenced by the query document.
 *
 * @param {any} previousResult The previous result returned by this function for the same query.
 * If nothing in the store changed since that previous result then values from the previous result
 * will be returned to preserve referential equality.
 */
export function readQueryFromStore(options) {
    var optsPatch = { returnPartialData: false };
    return diffQueryAgainstStore(__assign({}, options, optsPatch)).result;
}
var readStoreResolver = function (fieldName, idValue, args, context, _a) {
    var resultKey = _a.resultKey, directives = _a.directives;
    assertIdValue(idValue);
    var objId = idValue.id;
    var obj = context.store.get(objId);
    var storeKeyName = fieldName;
    if (args || directives) {
        // We happen to know here that getStoreKeyName returns its first
        // argument unmodified if there are no args or directives, so we can
        // avoid calling the function at all in that case, as a small but
        // important optimization to this frequently executed code.
        storeKeyName = getStoreKeyName(storeKeyName, args, directives);
    }
    var fieldValue = void 0;
    if (obj) {
        fieldValue = obj[storeKeyName];
        if (typeof fieldValue === 'undefined' &&
            context.cacheRedirects &&
            (obj.__typename || objId === 'ROOT_QUERY')) {
            var typename = obj.__typename || 'Query';
            // Look for the type in the custom resolver map
            var type = context.cacheRedirects[typename];
            if (type) {
                // Look for the field in the custom resolver map
                var resolver = type[fieldName];
                if (resolver) {
                    fieldValue = resolver(obj, args, {
                        getCacheKey: function (storeObj) {
                            return toIdValue({
                                id: context.dataIdFromObject(storeObj),
                                typename: storeObj.__typename,
                            });
                        },
                    });
                }
            }
        }
    }
    if (typeof fieldValue === 'undefined') {
        if (!context.returnPartialData) {
            throw new Error("Can't find field " + storeKeyName + " on object (" + objId + ") " + JSON.stringify(obj, null, 2) + ".");
        }
        context.hasMissingField = true;
        return fieldValue;
    }
    // if this is an object scalar, it must be a json blob and we have to unescape it
    if (isJsonValue(fieldValue)) {
        // If the JSON blob is the same now as in the previous result, return the previous result to
        // maintain referential equality.
        //
        // `isEqual` will first perform a referential equality check (with `===`) in case the JSON
        // value has not changed in the store, and then a deep equality check if that fails in case a
        // new JSON object was returned by the API but that object may still be the same.
        if (idValue.previousResult &&
            isEqual(idValue.previousResult[resultKey], fieldValue.json)) {
            return idValue.previousResult[resultKey];
        }
        return fieldValue.json;
    }
    // If we had a previous result, try adding that previous result value for this field to our field
    // value. This will create a new value without mutating the old one.
    if (idValue.previousResult) {
        fieldValue = addPreviousResultToIdValues(fieldValue, idValue.previousResult[resultKey]);
    }
    return fieldValue;
};
/**
 * Given a store and a query, return as much of the result as possible and
 * identify if any data was missing from the store.
 * @param  {DocumentNode} query A parsed GraphQL query document
 * @param  {Store} store The Apollo Client store object
 * @param  {any} previousResult The previous result returned by this function for the same query
 * @return {result: Object, complete: [boolean]}
 */
export function diffQueryAgainstStore(_a) {
    var store = _a.store, query = _a.query, variables = _a.variables, previousResult = _a.previousResult, _b = _a.returnPartialData, returnPartialData = _b === void 0 ? true : _b, _c = _a.rootId, rootId = _c === void 0 ? 'ROOT_QUERY' : _c, fragmentMatcherFunction = _a.fragmentMatcherFunction, config = _a.config;
    // Throw the right validation error by trying to find a query in the document
    var queryDefinition = getQueryDefinition(query);
    variables = assign({}, getDefaultValues(queryDefinition), variables);
    var context = {
        // Global settings
        store: store,
        returnPartialData: returnPartialData,
        dataIdFromObject: (config && config.dataIdFromObject) || null,
        cacheRedirects: (config && config.cacheRedirects) || {},
        // Flag set during execution
        hasMissingField: false,
    };
    var rootIdValue = {
        type: 'id',
        id: rootId,
        previousResult: previousResult,
    };
    var result = graphqlAnywhere(readStoreResolver, query, rootIdValue, context, variables, {
        fragmentMatcher: fragmentMatcherFunction,
        resultMapper: resultMapper,
    });
    return {
        result: result,
        complete: !context.hasMissingField,
    };
}
export function assertIdValue(idValue) {
    if (!isIdValue(idValue)) {
        throw new Error("Encountered a sub-selection on the query, but the store doesn't have an object reference. This should never happen during normal use unless you have custom code that is directly manipulating the store; please file an issue.");
    }
}
/**
 * Adds a previous result value to id values in a nested array. For a single id value and a single
 * previous result then the previous value is added directly.
 *
 * For arrays we put all of the ids from the previous result array in a map and add them to id
 * values with the same id.
 *
 * This function does not mutate. Instead it returns new instances of modified values.
 *
 * @private
 */
function addPreviousResultToIdValues(value, previousResult) {
    // If the value is an `IdValue`, add the previous result to it whether or not that
    // `previousResult` is undefined.
    //
    // If the value is an array, recurse over each item trying to add the `previousResult` for that
    // item.
    if (isIdValue(value)) {
        return __assign({}, value, { previousResult: previousResult });
    }
    else if (Array.isArray(value)) {
        var idToPreviousResult_1 = new Map();
        // If the previous result was an array, we want to build up our map of ids to previous results
        // using the private `ID_KEY` property that is added in `resultMapper`.
        if (Array.isArray(previousResult)) {
            previousResult.forEach(function (item) {
                // item can be null
                if (item && item[ID_KEY]) {
                    idToPreviousResult_1.set(item[ID_KEY], item);
                    // idToPreviousResult[item[ID_KEY]] = item;
                }
            });
        }
        // For every value we want to add the previous result.
        return value.map(function (item, i) {
            // By default the previous result for this item will be in the same array position as this
            // item.
            var itemPreviousResult = previousResult && previousResult[i];
            // If the item is an id value, we should check to see if there is a previous result for this
            // specific id. If there is, that will be the value for `itemPreviousResult`.
            if (isIdValue(item)) {
                itemPreviousResult =
                    idToPreviousResult_1.get(item.id) || itemPreviousResult;
            }
            return addPreviousResultToIdValues(item, itemPreviousResult);
        });
    }
    // Return the value, nothing changed.
    return value;
}
/**
 * Maps a result from `graphql-anywhere` to a final result value.
 *
 * If the result and the previous result from the `idValue` pass a shallow equality test, we just
 * return the `previousResult` to maintain referential equality.
 *
 * We also add a private id property to the result that we can use later on.
 *
 * @private
 */
function resultMapper(resultFields, idValue) {
    // If we had a previous result, we may be able to return that and preserve referential equality
    if (idValue.previousResult) {
        var currentResultKeys_1 = Object.keys(resultFields);
        var sameAsPreviousResult = 
        // Confirm that we have the same keys in both the current result and the previous result.
        Object.keys(idValue.previousResult).every(function (key) { return currentResultKeys_1.indexOf(key) > -1; }) &&
            // Perform a shallow comparison of the result fields with the previous result. If all of
            // the shallow fields are referentially equal to the fields of the previous result we can
            // just return the previous result.
            //
            // While we do a shallow comparison of objects, but we do a deep comparison of arrays.
            currentResultKeys_1.every(function (key) {
                return areNestedArrayItemsStrictlyEqual(resultFields[key], idValue.previousResult[key]);
            });
        if (sameAsPreviousResult) {
            return idValue.previousResult;
        }
    }
    Object.defineProperty(resultFields, ID_KEY, {
        enumerable: false,
        configurable: true,
        writable: false,
        value: idValue.id,
    });
    return resultFields;
}
/**
 * Compare all the items to see if they are all referentially equal in two arrays no matter how
 * deeply nested the arrays are.
 *
 * @private
 */
function areNestedArrayItemsStrictlyEqual(a, b) {
    // If `a` and `b` are referentially equal, return true.
    if (a === b) {
        return true;
    }
    // If either `a` or `b` are not an array or not of the same length return false. `a` and `b` are
    // known to not be equal here, we checked above.
    if (!Array.isArray(a) || !Array.isArray(b) || a.length !== b.length) {
        return false;
    }
    // Otherwise let us compare all of the array items (which are potentially nested arrays!) to see
    // if they are equal.
    return a.every(function (item, i) { return areNestedArrayItemsStrictlyEqual(item, b[i]); });
}
//# sourceMappingURL=readFromStore.js.map