import DOMMatrixReadOnly, {initDOMMatrixReadOnly} from './DOMMatrixReadOnly'
import {
    multiplyAndApply,
    rotateAxisAngleArray,
} from './utilities'

initDOMMatrixReadOnly()

var DOMMatrix = (function (DOMMatrixReadOnly) {
    function DOMMatrix(arg) {
        const numArgs = arguments.length
        if (numArgs === 0) {
            DOMMatrixReadOnly.call(this, [1, 0, 0, 1, 0, 0])
        }
        else if (numArgs === 1) {
            if (typeof arg == 'string') {
                throw new Error('CSS transformList arg not yet implemented.')
                // TODO validate that syntax of transformList matches transform-list (http://www.w3.org/TR/css-transforms-1/#typedef-transform-list).
            }
            else if (arg instanceof DOMMatrix) {
                DOMMatrixReadOnly.call(this, arg._matrix)
            }
            else if (arg instanceof Float32Array || arg instanceof Float64Array || arg instanceof Array) {
                DOMMatrixReadOnly.call(this, arg)
            }
        }
        else {
            throw new Error('Wrong number of arguments to DOMMatrix constructor.')
        }
    }

    if ( DOMMatrixReadOnly ) DOMMatrix.__proto__ = DOMMatrixReadOnly;
    DOMMatrix.prototype = Object.create( DOMMatrixReadOnly && DOMMatrixReadOnly.prototype );
    DOMMatrix.prototype.constructor = DOMMatrix;

    var prototypeAccessors = { a: {},b: {},c: {},d: {},e: {},f: {},m11: {},m12: {},m13: {},m14: {},m21: {},m22: {},m23: {},m24: {},m31: {},m32: {},m33: {},m34: {},m41: {},m42: {},m43: {},m44: {} };

    // Mutable transform methods
    DOMMatrix.prototype.multiplySelf = function multiplySelf (other) {
        if (!(other instanceof DOMMatrix))
            { throw new Error('The argument to multiplySelf must be an instance of DOMMatrix') }

        // TODO: avoid creating a new array, just apply values directly.
        multiplyAndApply(this, other, this)

        if (!other.is2D) { this._is2D = false }

        return this
    };

    DOMMatrix.prototype.preMultiplySelf = function preMultiplySelf (other) {
        if (!(other instanceof DOMMatrix))
            { throw new Error('The argument to multiplySelf must be an instance of DOMMatrix') }

        // TODO: avoid creating a new array, just apply values directly.
        multiplyAndApply(other, this, this)

        if (!other.is2D) { this._is2D = false }

        return this
    };

    DOMMatrix.prototype.translateSelf = function translateSelf (tx, ty, tz) {
        if ( tz === void 0 ) tz = 0;

        // TODO: check args are numbers

        if (arguments.length === 1)
            { throw new Error('The first two arguments (X and Y translation values) are required (the third, Z translation, is optional).') }

        // http://www.w3.org/TR/2012/WD-css3-transforms-20120911/#Translate3dDefined
        const translationMatrix = new DOMMatrix([
            // column-major:
            1, 0, 0, 0,
            0, 1, 0, 0,
            0, 0, 1, 0,
            tx,ty,tz,1 ])

        this.multiplySelf(translationMatrix)

        if (tz != 0) {
            this._is2D = false
        }

        return this
    };

    DOMMatrix.prototype.scaleSelf = function scaleSelf (scale, originX, originY) {
        if ( originX === void 0 ) originX = 0;
        if ( originY === void 0 ) originY = 0;

        this.translateSelf(originX, originY)

        this.multiplySelf(new DOMMatrix([
            // 2D:
            /*a*/scale, /*b*/0,
            /*c*/0,     /*d*/scale,
            /*e*/0,     /*f*/0 ]))

        this.translateSelf(-originX, -originY)
        return this
    };

    DOMMatrix.prototype.scale3dSelf = function scale3dSelf (scale, originX, originY, originZ) {
        if ( originX === void 0 ) originX = 0;
        if ( originY === void 0 ) originY = 0;
        if ( originZ === void 0 ) originZ = 0;

        this.translateSelf(originX, originY, originZ)

        this.multiplySelf(new DOMMatrix([
            // 3D
            scale, 0,     0,     0,
            0,     scale, 0,     0,
            0,     0,     scale, 0,
            0,     0,     0,     1 ]))

        this.translateSelf(-originX, -originY, -originZ)
        return this
    };

    DOMMatrix.prototype.scaleNonUniformSelf = function scaleNonUniformSelf (scaleX, scaleY, scaleZ, originX, originY, originZ) {
        if ( scaleY === void 0 ) scaleY = 1;
        if ( scaleZ === void 0 ) scaleZ = 1;
        if ( originX === void 0 ) originX = 0;
        if ( originY === void 0 ) originY = 0;
        if ( originZ === void 0 ) originZ = 0;

        this.translateSelf(originX, originY, originZ)

        this.multiplySelf(new DOMMatrix([
            // 3D
            scaleX, 0,      0,      0,
            0,      scaleY, 0,      0,
            0,      0,      scaleZ, 0,
            0,      0,      0,      1 ]))

        this.translateSelf(-originX, -originY, -originZ)

        if (scaleZ !== 1 || originZ !== 0) { this._is2D = false }

        return this
    };

    DOMMatrix.prototype.rotateSelf = function rotateSelf (angle, originX, originY) {
        if ( originX === void 0 ) originX = 0;
        if ( originY === void 0 ) originY = 0;

        this.translateSelf(originX, originY)

        // axis of rotation
        var ref = [0,0,1];
        var x = ref[0];
        var y = ref[1];
        var z = ref[2]; // We're rotating around the Z axis.

        this.rotateAxisAngleSelf(x, y, z, angle)

        this.translateSelf(-originX, -originY)
        return this
    };

    // TODO
    DOMMatrix.prototype.rotateFromVectorSelf = function rotateFromVectorSelf (x, y) {
        throw new Error('rotateFromVectorSelf is not implemented yet.')
    };

    DOMMatrix.prototype.rotateAxisAngleSelf = function rotateAxisAngleSelf (x, y, z, angle) {
        const rotationMatrix = new DOMMatrix(rotateAxisAngleArray(x,y,z,angle))
        this.multiplySelf(rotationMatrix)
        return this
    };

    DOMMatrix.prototype.skewXSelf = function skewXSelf (sx) {
        throw new Error('skewXSelf is not implemented yet.')
    };

    DOMMatrix.prototype.skewYSelf = function skewYSelf (sy) {
        throw new Error('skewYSelf is not implemented yet.')
    };

    DOMMatrix.prototype.invertSelf = function invertSelf () {
        throw new Error('invertSelf is not implemented yet.')
    };

    DOMMatrix.prototype.setMatrixValue = function setMatrixValue (/*DOMString*/ transformList) {
        throw new Error('setMatrixValue is not implemented yet.')
    };

    prototypeAccessors.a.get = function () { return this.m11 };
    prototypeAccessors.b.get = function () { return this.m12 };
    prototypeAccessors.c.get = function () { return this.m21 };
    prototypeAccessors.d.get = function () { return this.m22 };
    prototypeAccessors.e.get = function () { return this.m41 };
    prototypeAccessors.f.get = function () { return this.m42 };

    prototypeAccessors.m11.get = function () { return this._matrix[0]  };
    prototypeAccessors.m12.get = function () { return this._matrix[4]  };
    prototypeAccessors.m13.get = function () { return this._matrix[8]  };
    prototypeAccessors.m14.get = function () { return this._matrix[12] };

    prototypeAccessors.m21.get = function () { return this._matrix[1]  };
    prototypeAccessors.m22.get = function () { return this._matrix[5]  };
    prototypeAccessors.m23.get = function () { return this._matrix[9]  };
    prototypeAccessors.m24.get = function () { return this._matrix[13] };

    prototypeAccessors.m31.get = function () { return this._matrix[2]  };
    prototypeAccessors.m32.get = function () { return this._matrix[6]  };
    prototypeAccessors.m33.get = function () { return this._matrix[10] };
    prototypeAccessors.m34.get = function () { return this._matrix[14] };

    prototypeAccessors.m41.get = function () { return this._matrix[3]  };
    prototypeAccessors.m42.get = function () { return this._matrix[7]  };
    prototypeAccessors.m43.get = function () { return this._matrix[11] };
    prototypeAccessors.m44.get = function () { return this._matrix[15] };

    prototypeAccessors.a.set = function (value) { this.m11 = value };
    prototypeAccessors.b.set = function (value) { this.m12 = value };
    prototypeAccessors.c.set = function (value) { this.m21 = value };
    prototypeAccessors.d.set = function (value) { this.m22 = value };
    prototypeAccessors.e.set = function (value) { this.m41 = value };
    prototypeAccessors.f.set = function (value) { this.m42 = value };

    prototypeAccessors.m11.set = function (value) { this._matrix[0]  = value };
    prototypeAccessors.m12.set = function (value) { this._matrix[4]  = value };
    prototypeAccessors.m13.set = function (value) { this._matrix[8]  = value };
    prototypeAccessors.m14.set = function (value) { this._matrix[12] = value };

    prototypeAccessors.m21.set = function (value) { this._matrix[1]  = value };
    prototypeAccessors.m22.set = function (value) { this._matrix[5]  = value };
    prototypeAccessors.m23.set = function (value) { this._matrix[9]  = value };
    prototypeAccessors.m24.set = function (value) { this._matrix[13] = value };

    prototypeAccessors.m31.set = function (value) { this._matrix[2]  = value };
    prototypeAccessors.m32.set = function (value) { this._matrix[6]  = value };
    prototypeAccessors.m33.set = function (value) { this._matrix[10] = value };
    prototypeAccessors.m34.set = function (value) { this._matrix[14] = value };

    prototypeAccessors.m41.set = function (value) { this._matrix[3]  = value };
    prototypeAccessors.m42.set = function (value) { this._matrix[7]  = value };
    prototypeAccessors.m43.set = function (value) { this._matrix[11] = value };
    prototypeAccessors.m44.set = function (value) { this._matrix[15] = value };

    Object.defineProperties( DOMMatrix.prototype, prototypeAccessors );

    return DOMMatrix;
}(DOMMatrixReadOnly));

export default DOMMatrix;

//# sourceMappingURL=data:application/json;charset=utf-8;base64,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