/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * @format
 *
 */

"use strict";
var _extends =
  Object.assign ||
  function(target) {
    for (var i = 1; i < arguments.length; i++) {
      var source = arguments[i];
      for (var key in source) {
        if (Object.prototype.hasOwnProperty.call(source, key)) {
          target[key] = source[key];
        }
      }
    }
    return target;
  };
function _asyncToGenerator(fn) {
  return function() {
    var gen = fn.apply(this, arguments);
    return new Promise(function(resolve, reject) {
      function step(key, arg) {
        try {
          var info = gen[key](arg);
          var value = info.value;
        } catch (error) {
          reject(error);
          return;
        }
        if (info.done) {
          resolve(value);
        } else {
          return Promise.resolve(value).then(
            function(value) {
              step("next", value);
            },
            function(err) {
              step("throw", err);
            }
          );
        }
      }
      return step("next");
    });
  };
}

const GraphNotFoundError = require("./IncrementalBundler/GraphNotFoundError");
const IncrementalBundler = require("./IncrementalBundler");
const RevisionNotFoundError = require("./IncrementalBundler/RevisionNotFoundError");

const debounceAsyncQueue = require("./lib/debounceAsyncQueue");
const formatBundlingError = require("./lib/formatBundlingError");
const getGraphId = require("./lib/getGraphId");
const hmrJSBundle = require("./DeltaBundler/Serializers/hmrJSBundle");
const nullthrows = require("nullthrows");
const parseOptionsFromUrl = require("./lib/parseOptionsFromUrl");
const splitBundleOptions = require("./lib/splitBundleOptions");
const url = require("url");
var _require = require("metro-core"),
  _require$Logger = _require.Logger;
const createActionStartEntry = _require$Logger.createActionStartEntry,
  createActionEndEntry = _require$Logger.createActionEndEntry,
  log = _require$Logger.log;

/**
 * The HmrServer (Hot Module Reloading) implements a lightweight interface
 * to communicate easily to the logic in the React Native repository (which
 * is the one that handles the Web Socket connections).
 *
 * This interface allows the HmrServer to hook its own logic to WS clients
 * getting connected, disconnected or having errors (through the
 * `onClientConnect`, `onClientDisconnect` and `onClientError` methods).
 */
class HmrServer {
  constructor(bundler, createModuleId, config) {
    this._config = config;
    this._bundler = bundler;
    this._createModuleId = createModuleId;
  }

  onClientConnect(clientUrl, sendFn) {
    var _this = this;
    return _asyncToGenerator(function*() {
      const send = function(message) {
        sendFn(JSON.stringify(message));
      };

      const urlObj = nullthrows(url.parse(clientUrl, true));
      const query = nullthrows(urlObj.query);

      let revPromise;
      if (query.bundleEntry != null) {
        // TODO(T34760695): Deprecate
        urlObj.pathname = query.bundleEntry.replace(/\.js$/, ".bundle");
        delete query.bundleEntry;
        var _parseOptionsFromUrl = parseOptionsFromUrl(
          url.format(urlObj),
          _this._config.projectRoot,
          new Set(_this._config.resolver.platforms)
        );
        const options = _parseOptionsFromUrl.options;
        var _splitBundleOptions = splitBundleOptions(options);
        const entryFile = _splitBundleOptions.entryFile,
          transformOptions = _splitBundleOptions.transformOptions;

        const graphId = getGraphId(entryFile, transformOptions);
        revPromise = _this._bundler.getRevisionByGraphId(graphId);

        if (!revPromise) {
          send({
            type: "error",
            body: formatBundlingError(new GraphNotFoundError(graphId))
          });

          return null;
        }
      } else {
        const revisionId = query.revisionId;
        revPromise = _this._bundler.getRevision(revisionId);

        if (!revPromise) {
          send({
            type: "error",
            body: formatBundlingError(new RevisionNotFoundError(revisionId))
          });

          return null;
        }
      }
      var _ref = yield revPromise;
      const graph = _ref.graph,
        id = _ref.id;

      const client = {
        send,
        // Listen to file changes.
        unlisten: function() {
          return unlisten();
        },
        revisionId: id
      };

      const unlisten = _this._bundler
        .getDeltaBundler()
        .listen(
          graph,
          debounceAsyncQueue(_this._handleFileChange.bind(_this, client), 50)
        );

      return client;
    })();
  }

  onClientError(client, e) {
    this._config.reporter.update({
      type: "hmr_client_error",
      error: e
    });

    this.onClientDisconnect(client);
  }

  onClientDisconnect(client) {
    client.unlisten();
  }

  _handleFileChange(client) {
    var _this2 = this;
    return _asyncToGenerator(function*() {
      const processingHmrChange = log(
        createActionStartEntry({ action_name: "Processing HMR change" })
      );

      client.send({ type: "update-start" });
      const message = yield _this2._prepareMessage(client);
      client.send(message);
      client.send({ type: "update-done" });

      log(
        _extends({}, createActionEndEntry(processingHmrChange), {
          outdated_modules:
            message.type === "update" ? message.body.delta.length : undefined
        })
      );
    })();
  }

  _prepareMessage(client) {
    var _this3 = this;
    return _asyncToGenerator(function*() {
      try {
        const revPromise = _this3._bundler.getRevision(client.revisionId);

        if (!revPromise) {
          return {
            type: "error",
            body: formatBundlingError(
              new RevisionNotFoundError(client.revisionId)
            )
          };
        }
        var _ref2 = yield _this3._bundler.updateGraph(yield revPromise, false);
        const revision = _ref2.revision,
          delta = _ref2.delta;

        client.revisionId = revision.id;

        return {
          type: "update",
          body: {
            id: revision.id,
            delta: hmrJSBundle(delta, revision.graph, {
              createModuleId: _this3._createModuleId,
              projectRoot: _this3._config.projectRoot
            })
          }
        };
      } catch (error) {
        const formattedError = formatBundlingError(error);

        _this3._config.reporter.update({ type: "bundling_error", error });

        return { type: "error", body: formattedError };
      }
    })();
  }
}

module.exports = HmrServer;
