/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *  strict-local
 * @format
 */

"use strict";

// Debounces calls with the given delay, and queues the next call while the
// previous one hasn't completed so that no two calls can execute concurrently.
function _asyncToGenerator(fn) {
  return function() {
    var gen = fn.apply(this, arguments);
    return new Promise(function(resolve, reject) {
      function step(key, arg) {
        try {
          var info = gen[key](arg);
          var value = info.value;
        } catch (error) {
          reject(error);
          return;
        }
        if (info.done) {
          resolve(value);
        } else {
          return Promise.resolve(value).then(
            function(value) {
              step("next", value);
            },
            function(err) {
              step("throw", err);
            }
          );
        }
      }
      return step("next");
    });
  };
}
function debounceAsyncQueue(fn, delay) {
  let execute = (() => {
    var _ref = _asyncToGenerator(function*() {
      const currentCallbacks = callbacks;
      callbacks = [];
      executing = true;
      const res = yield fn();
      currentCallbacks.forEach(function(c) {
        return c(res);
      });
      executing = false;
      if (callbacks.length > 0) {
        yield execute();
      }
    });
    return function execute() {
      return _ref.apply(this, arguments);
    };
  })();
  let timeout;
  let waiting = false;
  let executing = false;
  let callbacks = [];

  return () =>
    new Promise((resolve, reject) => {
      callbacks.push(resolve);

      if (!executing) {
        if (waiting) {
          clearTimeout(timeout);
        } else {
          waiting = true;
        }
        timeout = setTimeout(
          _asyncToGenerator(function*() {
            waiting = false;
            yield execute();
          }),
          delay
        );
      }
    });
}

module.exports = debounceAsyncQueue;
