import { IGQLType, IGQLField, ISDL } from '../model';
/**
 * Parses a datamodel given as DSL
 * to an internal representation, convenient for
 * working with.
 */
export default abstract class Parser {
    /**
     * Shorthand to parse the datamodel, given an SDL string.
     * @param schemaString The datamodel as SDL string.
     * @returns A list of types found in the datamodel.
     */
    parseFromSchemaString(schemaString: string): ISDL;
    /**
     * Parses the datamodel from a graphql-js schema.
     * @param schema The graphql-js schema, representing the datamodel.
     * @returns A list of types found in the datamodel.
     */
    parseFromSchema(schema: any): ISDL;
    /**
     * Checks if the given field is an ID field
     * @param field
     */
    protected abstract isIdField(field: any): boolean;
    /**
     * Checks if the given field is read-only.
     * If the field is an ID field, this method is not called and
     * read-only is assumed.
     * @param field
     */
    protected abstract isReadOnly(field: any): boolean;
    /**
     * Finds a directive on a field or type by name.
     * @param fieldOrType
     * @param name
     */
    protected getDirectiveByName(fieldOrType: any, name: string): any;
    /**
     * Checks if a directive on a given field or type ecists
     * @param fieldOrType
     * @param name
     */
    protected hasDirective(fieldOrType: any, name: string): boolean;
    /**
     * Checks if the given field is unique.
     * @param field
     */
    protected isUniqe(field: any): boolean;
    /**
     * Gets a fields default value. If no default
     * value is given, returns null.
     * @param field
     */
    protected getDefaultValue(field: any): any;
    /**
     * Gets a fields relation name. If no relation
     * exists, returns null.
     * @param field
     */
    protected getRelationName(field: any): string | null;
    /**
     * Parses a model field, respects all
     * known directives.
     * @param field
     */
    protected parseField(field: any): IGQLField;
    /**
     * Checks if the given type is an embedded type.
     * @param type
     */
    protected abstract isEmbedded(type: any): boolean;
    /**
     * Parases an object type.
     * @param type
     */
    protected parseObjectType(type: any): IGQLType;
    /**
     * Parses all object types in the schema.
     * @param schema
     */
    protected parseObjectTypes(schema: any): IGQLType[];
    /**
     * Parses all enum types in the schema.
     * @param schema
     */
    protected parseEnumTypes(schema: any): IGQLType[];
    /**
     * Resolves and connects all realtion fields found
     * in the given type list.
     * @param types
     */
    protected resolveRelations(types: IGQLType[]): void;
    /**
     * Traverses an AST branch and finds the next type.
     * This will skip modifiers like NonNullType or ListType.
     * @param type
     */
    protected parseType(type: any): any;
    /**
     * Traverses an AST branch and returns the modifier
     * of the type: Either ListType or NonNullType.
     * @param type
     * @param acc
     */
    protected parseKind(type: any, acc: any): any;
}
