"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
Object.defineProperty(exports, "__esModule", { value: true });
var directives_1 = require("../directives");
var scalar_1 = require("../scalar");
var indent = '  ';
var comment = '#';
var Renderer = /** @class */ (function () {
    function Renderer() {
    }
    Renderer.prototype.render = function (input) {
        var _this = this;
        // Sort alphabetically. Enums last. 
        var sortedTypes = input.types.slice().sort(function (a, b) {
            if (a.isEnum === b.isEnum) {
                return a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1;
            }
            else if (b.isEnum) {
                return -1;
            }
            else {
                return 1;
            }
        });
        return sortedTypes.map(function (t) {
            if (t.isEnum) {
                return _this.renderEnum(t);
            }
            else {
                return _this.renderType(t);
            }
        }).join("\n\n");
    };
    Renderer.prototype.renderType = function (type) {
        var _this = this;
        var typedirectives = type.directives || [];
        // TODO Move direction magic to superclass
        if (type.isEmbedded) {
            typedirectives.push({ name: directives_1.DirectiveKeys.isEmbedded, arguments: {} });
        }
        var renderedDirectives = this.renderDirectives(typedirectives);
        var sortedFields = type.fields.slice().sort(function (a, b) { return a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1; });
        var renderedFields = sortedFields.map(function (x) { return _this.renderField(x); });
        var renderedTypeName = renderedDirectives.length > 0 ?
            "type " + type.name + " " + renderedDirectives :
            "type " + type.name;
        var _a = this.renderComments(type, ''), renderedComments = _a.renderedComments, hasError = _a.hasError;
        var allFieldsHaveError = type.fields.every(function (x) { return x.comments !== undefined && x.comments.some(function (c) { return c.isError; }); });
        var commentPrefix = allFieldsHaveError ? comment + " " : '';
        if (renderedComments.length > 0) {
            return renderedComments + "\n" + commentPrefix + renderedTypeName + " {\n" + renderedFields.join('\n') + "\n" + commentPrefix + "}";
        }
        else {
            return "" + commentPrefix + renderedTypeName + " {\n" + renderedFields.join('\n') + "\n" + commentPrefix + "}";
        }
    };
    Renderer.prototype.renderComments = function (type, spacing) {
        var renderedComments = type.comments !== undefined ? type.comments.map(function (x) { return "" + spacing + comment + " " + x.text; }).join('\n') : [];
        var hasError = type.comments !== undefined ? type.comments.some(function (x) { return x.isError; }) : false;
        return { renderedComments: renderedComments, hasError: hasError };
    };
    Renderer.prototype.renderField = function (field) {
        var fieldDirectives = field.directives || [];
        // TODO Move direction magic to superclass
        if (field.defaultValue !== null) {
            fieldDirectives.push({ name: directives_1.DirectiveKeys.default, arguments: { value: this.renderValue(field.type, field.defaultValue) } });
        }
        if (field.isUnique) {
            fieldDirectives.push({ name: directives_1.DirectiveKeys.isUnique, arguments: {} });
        }
        if (field.relationName !== null) {
            fieldDirectives.push({ name: directives_1.DirectiveKeys.relation, arguments: { name: field.relationName } });
        }
        if (field.isId) {
            fieldDirectives.push({ name: directives_1.DirectiveKeys.isId, arguments: {} });
        }
        var renderedDirectives = this.renderDirectives(fieldDirectives);
        var type = this.extractTypeIdentifier(field.type);
        if (field.isList) {
            // Lists are always required in Prisma
            type = "[" + type + "!]!";
        }
        else if (field.isRequired) {
            type = type + "!";
        }
        var renderedField = renderedDirectives.length > 0 ?
            field.name + ": " + type + " " + renderedDirectives :
            field.name + ": " + type;
        var _a = this.renderComments(field, indent), renderedComments = _a.renderedComments, hasError = _a.hasError;
        if (renderedComments.length > 0) {
            if (hasError) {
                return renderedComments + "\n" + indent + comment + " " + renderedField;
            }
            else {
                return renderedComments + "\n" + indent + renderedField;
            }
        }
        else {
            return "" + indent + renderedField;
        }
    };
    Renderer.prototype.renderEnum = function (type) {
        var values = [];
        for (var _i = 0, _a = type.fields; _i < _a.length; _i++) {
            var field = _a[_i];
            if (field.defaultValue !== null) {
                values.push("" + indent + field.name + " = " + this.renderValue(field.type, field.defaultValue));
            }
            else {
                values.push("" + indent + field.name);
            }
        }
        return "enum " + type.name + " {\n" + values.join('\n') + "\n}";
    };
    Renderer.prototype.renderDirectives = function (directives) {
        var _this = this;
        var sortedDirectives = directives.slice().sort(function (a, b) { return a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1; });
        return this.mergeDirectives(sortedDirectives).map(function (x) { return _this.renderDirective(x); }).join(" ");
    };
    Renderer.prototype.renderDirective = function (directive) {
        var args = [];
        for (var _i = 0, _a = Object.keys(directive.arguments); _i < _a.length; _i++) {
            var arg = _a[_i];
            // TODO: We don't know the type. Is this a problem?
            args.push(arg + ": " + this.renderValue('', directive.arguments[arg]));
        }
        if (args.length > 0) {
            return "@" + directive.name + "(" + args.join(', ') + ")";
        }
        else {
            return "@" + directive.name;
        }
    };
    /**
     * Merges directives by summarizing arguments of
     * directives with equal name.
     */
    Renderer.prototype.mergeDirectives = function (directives) {
        // Group by name
        var grouped = directives.reduce(function (r, a) {
            r[a.name] = r[a.name] || [];
            r[a.name].push(a);
            return r;
        }, {});
        var merged = [];
        // Merge with same name
        for (var _i = 0, _a = Object.keys(grouped); _i < _a.length; _i++) {
            var name_1 = _a[_i];
            merged.push({
                name: name_1,
                arguments: grouped[name_1].reduce(function (r, a) {
                    return __assign({}, a.arguments, r.arguments);
                }, {})
            });
        }
        return merged;
    };
    Renderer.prototype.extractTypeIdentifier = function (type) {
        if (typeof type === 'string') {
            return type;
        }
        else {
            return type.name;
        }
    };
    Renderer.prototype.renderValue = function (type, value) {
        var strType = this.extractTypeIdentifier(type);
        if (strType === scalar_1.TypeIdentifiers.string ||
            strType === scalar_1.TypeIdentifiers.json ||
            strType === scalar_1.TypeIdentifiers.dateTime) {
            return "\"" + value + "\"";
        }
        else {
            return value;
        }
    };
    return Renderer;
}());
exports.default = Renderer;
//# sourceMappingURL=renderer.js.map