"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var type_1 = require("graphql/type");
var gqlAssert_1 = require("./gqlAssert");
var AstTools = /** @class */ (function () {
    function AstTools() {
    }
    /**
     * Checks if two schemas are equal.
     *
     * A schema A is considered equal to schema B if:
     *  * A has exactly the same types as B.
     *  * For field, the type, name and arguments match exactly.
     * @param original
     * @param toCheck
     */
    AstTools.assertTypesEqual = function (original, toCheck, modelName) {
        var typeMapA = original.getTypeMap();
        var typeMapB = toCheck.getTypeMap();
        var errors = AstTools.allTypesFromAInB(typeMapA, typeMapB, true) + AstTools.allTypesFromAInB(typeMapB, typeMapA, false);
        if (errors > 0) {
            console.error(errors + " type checking errors. Type equality check for model " + modelName + " failed.");
            gqlAssert_1.default.raise(errors + " type checking errors. Type equality check for model " + modelName + " failed.");
        }
    };
    AstTools.allTypesFromAInB = function (typeMapA, typeMapB, aIsOriginal) {
        var errors = 0;
        for (var _i = 0, _a = Object.keys(typeMapA); _i < _a.length; _i++) {
            var typeName = _a[_i];
            var bType = typeMapB[typeName];
            if (bType === undefined && typeName === 'Json') {
                continue; // TODO: We are generating Json only if it is used. Sync Prisma and generation
            }
            if (bType === undefined) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("Type " + typeName + " is missing.");
                }
                else {
                    console.error("Type " + typeName + " is extra.");
                }
                continue; // Deeper check has no meaning. 
            }
            var aType = typeMapA[typeName];
            if (aType.constructor.name !== bType.constructor.name) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("Type " + typeName + " should be " + aType.constructor.name + ", was " + bType.constructor.name + ".");
                }
                else {
                    console.error("Type " + typeName + " should be " + typeof (bType.constructor.name) + ", was " + aType.constructor.name + ".");
                }
                continue; // Deeper check has no meaning. 
            }
            if (type_1.isEnumType(aType)) {
                errors = errors + AstTools.allEnumFieldsFromAInB(aType, bType, aIsOriginal);
            }
            else if (type_1.isObjectType(aType) || type_1.isInputObjectType(aType) || type_1.isInterfaceType(aType)) {
                errors = errors + AstTools.allFieldsFromAInB(aType, bType, aIsOriginal);
            }
            else if (type_1.isScalarType(aType)) {
                errors = errors + AstTools.scalarTypeEquals(aType, bType, aIsOriginal);
            }
            else {
                gqlAssert_1.default.raise("Unknown type for object " + aType.name + ": " + aType.constructor.name);
            }
        }
        return errors;
    };
    AstTools.scalarTypeEquals = function (typeA, typeB, aIsOriginal) {
        if (typeA.name !== typeB.name) {
            if (aIsOriginal) {
                console.error("Scalar type " + typeB.name + " is missing..");
            }
            else {
                console.error("Scalar type " + typeA.name + " is extra.");
            }
            return 1;
        }
        return 0;
    };
    AstTools.allEnumFieldsFromAInB = function (typeA, typeB, aIsOriginal) {
        var aVals = typeA.getValues();
        var bVals = typeB.getValues();
        if (aVals.length === 0 || bVals.length === 0) {
            console.error("There should never be empty enums: " + typeA.name);
            return 1;
        }
        var errors = 0;
        for (var _i = 0, aVals_1 = aVals; _i < aVals_1.length; _i++) {
            var aValue = aVals_1[_i];
            var bValue = typeB.getValue(aValue.name);
            if (bValue === undefined) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("-- Type " + typeB.name + " is missing value " + aValue.name + ".");
                }
                else {
                    console.error("-- Type " + typeA.name + " has extra value " + aValue.name + ".");
                }
            }
        }
        return errors;
    };
    AstTools.allFieldsFromAInB = function (typeA, typeB, aIsOriginal) {
        var aFields = typeA.getFields();
        var bFields = typeB.getFields();
        if (aFields.length === 0 || bFields.length === 0) {
            console.error("There should never be empty types: " + typeA.name);
            return 1;
        }
        var errors = 0;
        for (var _i = 0, _a = Object.keys(aFields); _i < _a.length; _i++) {
            var fieldName = _a[_i];
            var bField = bFields[fieldName];
            if (bField === undefined) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("-- Type " + typeB.name + " is missing field " + fieldName + ".");
                }
                else {
                    console.error("-- Type " + typeA.name + " has extra field " + fieldName + ".");
                }
                continue; // Deeper check has no meaning. 
            }
            var fieldA = aFields[fieldName];
            var fieldB = bFields[fieldName];
            if (fieldA.type.toString() !== fieldB.type.toString()) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("---- Field " + typeB.name + "." + fieldName + " has incorrect type: " + fieldA.type + " expected, " + fieldB.type + " given.");
                }
                else {
                    console.error("---- Field " + typeA.name + "." + fieldName + " has incorrect type: " + fieldB.type + " expected, " + fieldA.type + " given.");
                }
            }
            errors = errors + AstTools.allArgsFromAInB(fieldA, fieldB, typeA, aIsOriginal);
        }
        return errors;
    };
    AstTools.allArgsFromAInB = function (fieldA, fieldB, type, aIsOriginal) {
        if (fieldA.args === null || fieldA.args === undefined) {
            return 0;
        }
        var errors = 0;
        if (fieldB.args === null || fieldB.args === undefined) {
            errors = errors + 1;
            if (aIsOriginal) {
                console.error("------ Field " + type.name + "." + fieldB.name + " does not have any arguments.");
            }
            else {
                console.error("------ Field " + type.name + "." + fieldA.name + " should not have any arguments.");
            }
        }
        var argsA = Object.keys(fieldA.args);
        var argsB = Object.keys(fieldB.args);
        if (argsA.length !== argsB.length) {
            if (aIsOriginal) {
                console.error("------ Field " + type.name + "." + fieldB.name + " has missmatching argument counts: " + argsA.length + " expected, " + argsB.length + " found.");
            }
            else {
                console.error("------ Field " + type.name + "." + fieldB.name + " has missmatching argument counts: " + argsB.length + " expected, " + argsA.length + " found.");
            }
            return errors; // Deeper check is not useful. 
        }
        for (var i = 0; i < argsA.length; i++) {
            if (argsA[i] !== argsB[i]) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("-------- Field " + type.name + "." + fieldB.name + ", argument " + argsB[i] + " has incorrect name: " + argsA[i] + " expected.");
                }
                else {
                    console.error("-------- Field " + type.name + "." + fieldA.name + ", argument " + argsA[i] + " has incorrect na,e: " + argsA[i] + " expected.");
                }
            }
            var argA = fieldA.args[argsA[i]];
            var argB = fieldB.args[argsA[i]];
            if (String(argA.type) !== String(argB.type)) {
                errors = errors + 1;
                if (aIsOriginal) {
                    console.error("-------- Field " + type.name + "." + fieldB.name + ", argument " + argsB[i] + " has incorrect type: " + argA.type + " expected, " + argB.type + " given.");
                }
                else {
                    console.error("-------- Field " + type.name + "." + fieldA.name + ", argument " + argsA[i] + " has incorrect type: " + argB.type + " expected, " + argA.type + " given.");
                }
            }
        }
        return errors;
    };
    return AstTools;
}());
exports.default = AstTools;
//# sourceMappingURL=astTools.js.map