"use strict";
// Transform SVG PathData
// http://www.w3.org/TR/SVG/paths.html#PathDataBNF
var __extends = (this && this.__extends) || (function () {
    var extendStatics = Object.setPrototypeOf ||
        ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
        function (d, b) { for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p]; };
    return function (d, b) {
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
var stream_1 = require("stream");
var mathUtils_js_1 = require("./mathUtils.js");
var SVGPathData_1 = require("./SVGPathData");
var SVGPathDataTransformer = (function (_super) {
    __extends(SVGPathDataTransformer, _super);
    function SVGPathDataTransformer(transformFunction) {
        var _this = _super.call(this, { objectMode: true }) || this;
        // Transform function needed
        if ("function" !== typeof transformFunction) {
            throw new Error("Please provide a transform callback to receive commands.");
        }
        _this._transformer = transformFunction;
        return _this;
    }
    SVGPathDataTransformer.prototype._transform = function (commands, encoding, done) {
        if (!(commands instanceof Array)) {
            commands = [commands];
        }
        for (var _i = 0, commands_1 = commands; _i < commands_1.length; _i++) {
            var command = commands_1[_i];
            this.push(this._transformer(command));
        }
        done();
    };
    return SVGPathDataTransformer;
}(stream_1.Transform));
exports.SVGPathDataTransformer = SVGPathDataTransformer;
(function (SVGPathDataTransformer) {
    // Predefined transforming functions
    // Rounds commands values
    function ROUND(roundVal) {
        if (roundVal === void 0) { roundVal = 1e13; }
        mathUtils_js_1.assertNumbers(roundVal);
        function rf(val) { return Math.round(val * roundVal) / roundVal; }
        return function round(command) {
            if ("undefined" !== typeof command.x1) {
                command.x1 = rf(command.x1);
            }
            if ("undefined" !== typeof command.y1) {
                command.y1 = rf(command.y1);
            }
            if ("undefined" !== typeof command.x2) {
                command.x2 = rf(command.x2);
            }
            if ("undefined" !== typeof command.y2) {
                command.y2 = rf(command.y2);
            }
            if ("undefined" !== typeof command.x) {
                command.x = rf(command.x);
            }
            if ("undefined" !== typeof command.y) {
                command.y = rf(command.y);
            }
            return command;
        };
    }
    SVGPathDataTransformer.ROUND = ROUND;
    // Relative to absolute commands
    function TO_ABS() {
        return INFO(function (command, prevX, prevY) {
            if (command.relative) {
                // x1/y1 values
                if ("undefined" !== typeof command.x1) {
                    command.x1 += prevX;
                }
                if ("undefined" !== typeof command.y1) {
                    command.y1 += prevY;
                }
                // x2/y2 values
                if ("undefined" !== typeof command.x2) {
                    command.x2 += prevX;
                }
                if ("undefined" !== typeof command.y2) {
                    command.y2 += prevY;
                }
                // Finally x/y values
                if ("undefined" !== typeof command.x) {
                    command.x += prevX;
                }
                if ("undefined" !== typeof command.y) {
                    command.y += prevY;
                }
                command.relative = false;
            }
            return command;
        });
    }
    SVGPathDataTransformer.TO_ABS = TO_ABS;
    // Absolute to relative commands
    function TO_REL() {
        return INFO(function (command, prevX, prevY) {
            if (!command.relative) {
                // x1/y1 values
                if ("undefined" !== typeof command.x1) {
                    command.x1 -= prevX;
                }
                if ("undefined" !== typeof command.y1) {
                    command.y1 -= prevY;
                }
                // x2/y2 values
                if ("undefined" !== typeof command.x2) {
                    command.x2 -= prevX;
                }
                if ("undefined" !== typeof command.y2) {
                    command.y2 -= prevY;
                }
                // Finally x/y values
                if ("undefined" !== typeof command.x) {
                    command.x -= prevX;
                }
                if ("undefined" !== typeof command.y) {
                    command.y -= prevY;
                }
                command.relative = true;
            }
            return command;
        });
    }
    SVGPathDataTransformer.TO_REL = TO_REL;
    // Convert H, V, Z and A with rX = 0 to L
    function NORMALIZE_HVZ(normalizeZ, normalizeH, normalizeV) {
        if (normalizeZ === void 0) { normalizeZ = true; }
        if (normalizeH === void 0) { normalizeH = true; }
        if (normalizeV === void 0) { normalizeV = true; }
        return INFO(function (command, prevX, prevY, pathStartX, pathStartY) {
            if (isNaN(pathStartX) && !(command.type & SVGPathData_1.SVGPathData.MOVE_TO)) {
                throw new Error("path must start with moveto");
            }
            if (normalizeH && command.type & SVGPathData_1.SVGPathData.HORIZ_LINE_TO) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                command.y = command.relative ? 0 : prevY;
            }
            if (normalizeV && command.type & SVGPathData_1.SVGPathData.VERT_LINE_TO) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                command.x = command.relative ? 0 : prevX;
            }
            if (normalizeZ && command.type & SVGPathData_1.SVGPathData.CLOSE_PATH) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                command.x = command.relative ? pathStartX - prevX : pathStartX;
                command.y = command.relative ? pathStartY - prevY : pathStartY;
            }
            if (command.type & SVGPathData_1.SVGPathData.ARC && (0 === command.rX || 0 === command.rY)) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                delete command.rX;
                delete command.rY;
                delete command.xRot;
                delete command.lArcFlag;
                delete command.sweepFlag;
            }
            return command;
        });
    }
    SVGPathDataTransformer.NORMALIZE_HVZ = NORMALIZE_HVZ;
    /*
     * Transforms smooth curves and quads to normal curves and quads (SsTt to CcQq)
     */
    function NORMALIZE_ST() {
        var prevCurveC2X = NaN;
        var prevCurveC2Y = NaN;
        var prevQuadCX = NaN;
        var prevQuadCY = NaN;
        return INFO(function (command, prevX, prevY) {
            if (command.type & SVGPathData_1.SVGPathData.SMOOTH_CURVE_TO) {
                command.type = SVGPathData_1.SVGPathData.CURVE_TO;
                prevCurveC2X = isNaN(prevCurveC2X) ? prevX : prevCurveC2X;
                prevCurveC2Y = isNaN(prevCurveC2Y) ? prevY : prevCurveC2Y;
                command.x1 = command.relative ? prevX - prevCurveC2X : 2 * prevX - prevCurveC2X;
                command.y1 = command.relative ? prevY - prevCurveC2Y : 2 * prevY - prevCurveC2Y;
            }
            if (command.type & SVGPathData_1.SVGPathData.CURVE_TO) {
                prevCurveC2X = command.relative ? prevX + command.x2 : command.x2;
                prevCurveC2Y = command.relative ? prevY + command.y2 : command.y2;
            }
            else {
                prevCurveC2X = NaN;
                prevCurveC2Y = NaN;
            }
            if (command.type & SVGPathData_1.SVGPathData.SMOOTH_QUAD_TO) {
                command.type = SVGPathData_1.SVGPathData.QUAD_TO;
                prevQuadCX = isNaN(prevQuadCX) ? prevX : prevQuadCX;
                prevQuadCY = isNaN(prevQuadCY) ? prevY : prevQuadCY;
                command.x1 = command.relative ? prevX - prevQuadCX : 2 * prevX - prevQuadCX;
                command.y1 = command.relative ? prevY - prevQuadCY : 2 * prevY - prevQuadCY;
            }
            if (command.type & SVGPathData_1.SVGPathData.QUAD_TO) {
                prevQuadCX = command.relative ? prevX + command.x1 : command.x1;
                prevQuadCY = command.relative ? prevY + command.y1 : command.y1;
            }
            else {
                prevQuadCX = NaN;
                prevQuadCY = NaN;
            }
            return command;
        });
    }
    SVGPathDataTransformer.NORMALIZE_ST = NORMALIZE_ST;
    /*
     * A quadratic bézier curve can be represented by a cubic bézier curve which has
     * the same end points as the quadratic and both control points in place of the
     * quadratic"s one.
     *
     * This transformer replaces QqTt commands with Cc commands respectively.
     * This is useful for reading path data into a system which only has a
     * representation for cubic curves.
     */
    function QT_TO_C() {
        var prevQuadX1 = NaN;
        var prevQuadY1 = NaN;
        return INFO(function (command, prevX, prevY) {
            if (command.type & SVGPathData_1.SVGPathData.SMOOTH_QUAD_TO) {
                command.type = SVGPathData_1.SVGPathData.QUAD_TO;
                prevQuadX1 = isNaN(prevQuadX1) ? prevX : prevQuadX1;
                prevQuadY1 = isNaN(prevQuadY1) ? prevY : prevQuadY1;
                command.x1 = command.relative ? prevX - prevQuadX1 : 2 * prevX - prevQuadX1;
                command.y1 = command.relative ? prevY - prevQuadY1 : 2 * prevY - prevQuadY1;
            }
            if (command.type & SVGPathData_1.SVGPathData.QUAD_TO) {
                prevQuadX1 = command.relative ? prevX + command.x1 : command.x1;
                prevQuadY1 = command.relative ? prevY + command.y1 : command.y1;
                var x1 = command.x1;
                var y1 = command.y1;
                command.type = SVGPathData_1.SVGPathData.CURVE_TO;
                command.x1 = ((command.relative ? 0 : prevX) + x1 * 2) / 3;
                command.y1 = ((command.relative ? 0 : prevY) + y1 * 2) / 3;
                command.x2 = (command.x + x1 * 2) / 3;
                command.y2 = (command.y + y1 * 2) / 3;
            }
            else {
                prevQuadX1 = NaN;
                prevQuadY1 = NaN;
            }
            return command;
        });
    }
    SVGPathDataTransformer.QT_TO_C = QT_TO_C;
    function INFO(f) {
        var prevXAbs = 0;
        var prevYAbs = 0;
        var pathStartXAbs = NaN;
        var pathStartYAbs = NaN;
        return function transform(command) {
            if (isNaN(pathStartXAbs) && !(command.type & SVGPathData_1.SVGPathData.MOVE_TO)) {
                throw new Error("path must start with moveto");
            }
            var result = f(command, prevXAbs, prevYAbs, pathStartXAbs, pathStartYAbs);
            if (command.type & SVGPathData_1.SVGPathData.CLOSE_PATH) {
                prevXAbs = pathStartXAbs;
                prevYAbs = pathStartYAbs;
            }
            if ("undefined" !== typeof command.x) {
                prevXAbs = (command.relative ? prevXAbs + command.x : command.x);
            }
            if ("undefined" !== typeof command.y) {
                prevYAbs = (command.relative ? prevYAbs + command.y : command.y);
            }
            if (command.type & SVGPathData_1.SVGPathData.MOVE_TO) {
                pathStartXAbs = prevXAbs;
                pathStartYAbs = prevYAbs;
            }
            return result;
        };
    }
    SVGPathDataTransformer.INFO = INFO;
    /*
     * remove 0-length segments
     */
    function SANITIZE(EPS) {
        if (EPS === void 0) { EPS = 0; }
        mathUtils_js_1.assertNumbers(EPS);
        var prevCurveC2X = NaN;
        var prevCurveC2Y = NaN;
        var prevQuadCX = NaN;
        var prevQuadCY = NaN;
        return INFO(function (command, prevX, prevY, pathStartX, pathStartY) {
            var abs = Math.abs;
            var skip = false;
            var x1Rel = 0;
            var y1Rel = 0;
            if (command.type & SVGPathData_1.SVGPathData.SMOOTH_CURVE_TO) {
                x1Rel = isNaN(prevCurveC2X) ? 0 : prevX - prevCurveC2X;
                y1Rel = isNaN(prevCurveC2Y) ? 0 : prevY - prevCurveC2Y;
            }
            if (command.type & (SVGPathData_1.SVGPathData.CURVE_TO | SVGPathData_1.SVGPathData.SMOOTH_CURVE_TO)) {
                prevCurveC2X = command.relative ? prevX + command.x2 : command.x2;
                prevCurveC2Y = command.relative ? prevY + command.y2 : command.y2;
            }
            else {
                prevCurveC2X = NaN;
                prevCurveC2Y = NaN;
            }
            if (command.type & SVGPathData_1.SVGPathData.SMOOTH_QUAD_TO) {
                prevQuadCX = isNaN(prevQuadCX) ? prevX : 2 * prevX - prevQuadCX;
                prevQuadCY = isNaN(prevQuadCY) ? prevY : 2 * prevY - prevQuadCY;
            }
            else if (command.type & SVGPathData_1.SVGPathData.QUAD_TO) {
                prevQuadCX = command.relative ? prevX + command.x1 : command.x1;
                prevQuadCY = command.relative ? prevY + command.y1 : command.y2;
            }
            else {
                prevQuadCX = NaN;
                prevQuadCY = NaN;
            }
            if (command.type & SVGPathData_1.SVGPathData.LINE_COMMANDS ||
                command.type & SVGPathData_1.SVGPathData.ARC && (0 === command.rX || 0 === command.rY || !command.lArcFlag) ||
                command.type & SVGPathData_1.SVGPathData.CURVE_TO || command.type & SVGPathData_1.SVGPathData.SMOOTH_CURVE_TO ||
                command.type & SVGPathData_1.SVGPathData.QUAD_TO || command.type & SVGPathData_1.SVGPathData.SMOOTH_QUAD_TO) {
                var xRel = "undefined" === typeof command.x ? 0 :
                    (command.relative ? command.x : command.x - prevX);
                var yRel = "undefined" === typeof command.y ? 0 :
                    (command.relative ? command.y : command.y - prevY);
                x1Rel = !isNaN(prevQuadCX) ? prevQuadCX - prevX :
                    "undefined" === typeof command.x1 ? x1Rel :
                        command.relative ? command.x :
                            command.x1 - prevX;
                y1Rel = !isNaN(prevQuadCY) ? prevQuadCY - prevY :
                    "undefined" === typeof command.y1 ? y1Rel :
                        command.relative ? command.y :
                            command.y1 - prevY;
                var x2Rel = "undefined" === typeof command.x2 ? 0 :
                    (command.relative ? command.x : command.x2 - prevX);
                var y2Rel = "undefined" === typeof command.y2 ? 0 :
                    (command.relative ? command.y : command.y2 - prevY);
                if (abs(xRel) <= EPS && abs(yRel) <= EPS &&
                    abs(x1Rel) <= EPS && abs(y1Rel) <= EPS &&
                    abs(x2Rel) <= EPS && abs(y2Rel) <= EPS) {
                    skip = true;
                }
            }
            if (command.type & SVGPathData_1.SVGPathData.CLOSE_PATH) {
                if (abs(prevX - pathStartX) <= EPS && abs(prevY - pathStartY) <= EPS) {
                    skip = true;
                }
            }
            return skip ? [] : command;
        });
    }
    SVGPathDataTransformer.SANITIZE = SANITIZE;
    // SVG Transforms : http://www.w3.org/TR/SVGTiny12/coords.html#TransformList
    // Matrix : http://apike.ca/prog_svg_transform.html
    // a c e
    // b d f
    function MATRIX(a, b, c, d, e, f) {
        mathUtils_js_1.assertNumbers(a, b, c, d, e, f);
        return INFO(function (command, prevX, prevY, pathStartX) {
            var origX1 = command.x1;
            var origX2 = command.x2;
            // if isNaN(pathStartX), then this is the first command, which is ALWAYS an
            // absolute MOVE_TO, regardless what the relative flag says
            var comRel = command.relative && !isNaN(pathStartX);
            var x = "undefined" !== typeof command.x ? command.x : (comRel ? 0 : prevX);
            var y = "undefined" !== typeof command.y ? command.y : (comRel ? 0 : prevY);
            if (command.type & SVGPathData_1.SVGPathData.HORIZ_LINE_TO && 0 !== b) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                command.y = command.relative ? 0 : prevY;
            }
            if (command.type & SVGPathData_1.SVGPathData.VERT_LINE_TO && 0 !== c) {
                command.type = SVGPathData_1.SVGPathData.LINE_TO;
                command.x = command.relative ? 0 : prevX;
            }
            if ("undefined" !== typeof command.x) {
                command.x = (command.x * a) + (y * c) + (comRel ? 0 : e);
            }
            if ("undefined" !== typeof command.y) {
                command.y = (x * b) + command.y * d + (comRel ? 0 : f);
            }
            if ("undefined" !== typeof command.x1) {
                command.x1 = command.x1 * a + command.y1 * c + (comRel ? 0 : e);
            }
            if ("undefined" !== typeof command.y1) {
                command.y1 = origX1 * b + command.y1 * d + (comRel ? 0 : f);
            }
            if ("undefined" !== typeof command.x2) {
                command.x2 = command.x2 * a + command.y2 * c + (comRel ? 0 : e);
            }
            if ("undefined" !== typeof command.y2) {
                command.y2 = origX2 * b + command.y2 * d + (comRel ? 0 : f);
            }
            function sqr(x) { return x * x; }
            var det = a * d - b * c;
            if ("undefined" !== typeof command.xRot) {
                // Skip if this is a pure translation
                if (1 !== a || 0 !== b || 0 !== c || 1 !== d) {
                    // Special case for singular matrix
                    if (0 === det) {
                        // In the singular case, the arc is compressed to a line. The actual geometric image of the original
                        // curve under this transform possibly extends beyond the starting and/or ending points of the segment, but
                        // for simplicity we ignore this detail and just replace this command with a single line segment.
                        delete command.rX;
                        delete command.rY;
                        delete command.xRot;
                        delete command.lArcFlag;
                        delete command.sweepFlag;
                        command.type = SVGPathData_1.SVGPathData.LINE_TO;
                    }
                    else {
                        // Convert to radians
                        var xRot = command.xRot * Math.PI / 180;
                        // Convert rotated ellipse to general conic form
                        // x0^2/rX^2 + y0^2/rY^2 - 1 = 0
                        // x0 = x*cos(xRot) + y*sin(xRot)
                        // y0 = -x*sin(xRot) + y*cos(xRot)
                        // --> A*x^2 + B*x*y + C*y^2 - 1 = 0, where
                        var sinRot = Math.sin(xRot);
                        var cosRot = Math.cos(xRot);
                        var xCurve = 1 / sqr(command.rX);
                        var yCurve = 1 / sqr(command.rY);
                        var A = sqr(cosRot) * xCurve + sqr(sinRot) * yCurve;
                        var B = 2 * sinRot * cosRot * (xCurve - yCurve);
                        var C = sqr(sinRot) * xCurve + sqr(cosRot) * yCurve;
                        // Apply matrix to A*x^2 + B*x*y + C*y^2 - 1 = 0
                        // x1 = a*x + c*y
                        // y1 = b*x + d*y
                        //      (we can ignore e and f, since pure translations don"t affect the shape of the ellipse)
                        // --> A1*x1^2 + B1*x1*y1 + C1*y1^2 - det^2 = 0, where
                        var A1 = A * d * d - B * b * d + C * b * b;
                        var B1 = B * (a * d + b * c) - 2 * (A * c * d + C * a * b);
                        var C1 = A * c * c - B * a * c + C * a * a;
                        // Unapply newXRot to get back to axis-aligned ellipse equation
                        // x1 = x2*cos(newXRot) - y2*sin(newXRot)
                        // y1 = x2*sin(newXRot) + y2*cos(newXRot)
                        // A1*x1^2 + B1*x1*y1 + C1*y1^2 - det^2 =
                        //   x2^2*(A1*cos(newXRot)^2 + B1*sin(newXRot)*cos(newXRot) + C1*sin(newXRot)^2)
                        //   + x2*y2*(2*(C1 - A1)*sin(newXRot)*cos(newXRot) + B1*(cos(newXRot)^2 - sin(newXRot)^2))
                        //   + y2^2*(A1*sin(newXRot)^2 - B1*sin(newXRot)*cos(newXRot) + C1*cos(newXRot)^2)
                        //   (which must have the same zeroes as)
                        // x2^2/newRX^2 + y2^2/newRY^2 - 1
                        //   (so we have)
                        // 2*(C1 - A1)*sin(newXRot)*cos(newXRot) + B1*(cos(newXRot)^2 - sin(newXRot)^2) = 0
                        // (A1 - C1)*sin(2*newXRot) = B1*cos(2*newXRot)
                        // 2*newXRot = atan2(B1, A1 - C1)
                        var newXRot = ((Math.atan2(B1, A1 - C1) + Math.PI) % Math.PI) / 2;
                        // For any integer n, (atan2(B1, A1 - C1) + n*pi)/2 is a solution to the above; incrementing n just swaps
                        // the x and y radii computed below (since that"s what rotating an ellipse by pi/2 does).  Choosing the
                        // rotation between 0 and pi/2 eliminates the ambiguity and leads to more predictable output.
                        // Finally, we get newRX and newRY from the same-zeroes relationship that gave us newXRot
                        var newSinRot = Math.sin(newXRot);
                        var newCosRot = Math.cos(newXRot);
                        command.rX = Math.abs(det) /
                            Math.sqrt(A1 * sqr(newCosRot) + B1 * newSinRot * newCosRot + C1 * sqr(newSinRot));
                        command.rY = Math.abs(det) /
                            Math.sqrt(A1 * sqr(newSinRot) - B1 * newSinRot * newCosRot + C1 * sqr(newCosRot));
                        command.xRot = newXRot * 180 / Math.PI;
                    }
                }
            }
            // sweepFlag needs to be inverted when mirroring shapes
            // see http://www.itk.ilstu.edu/faculty/javila/SVG/SVG_drawing1/elliptical_curve.htm
            // m 65,10 a 50,25 0 1 0 50,25
            // M 65,60 A 50,25 0 1 1 115,35
            if ("undefined" !== typeof command.sweepFlag && 0 > det) {
                command.sweepFlag = +!command.sweepFlag;
            }
            return command;
        });
    }
    SVGPathDataTransformer.MATRIX = MATRIX;
    function ROTATE(a, x, y) {
        if (x === void 0) { x = 0; }
        if (y === void 0) { y = 0; }
        mathUtils_js_1.assertNumbers(a, x, y);
        var sin = Math.sin(a);
        var cos = Math.cos(a);
        return MATRIX(cos, sin, -sin, cos, x - x * cos + y * sin, y - x * sin - y * cos);
    }
    SVGPathDataTransformer.ROTATE = ROTATE;
    function TRANSLATE(dX, dY) {
        if (dY === void 0) { dY = 0; }
        mathUtils_js_1.assertNumbers(dX, dY);
        return MATRIX(1, 0, 0, 1, dX, dY);
    }
    SVGPathDataTransformer.TRANSLATE = TRANSLATE;
    function SCALE(dX, dY) {
        if (dY === void 0) { dY = dX; }
        mathUtils_js_1.assertNumbers(dX, dY);
        return MATRIX(dX, 0, 0, dY, 0, 0);
    }
    SVGPathDataTransformer.SCALE = SCALE;
    function SKEW_X(a) {
        mathUtils_js_1.assertNumbers(a);
        return MATRIX(1, 0, Math.atan(a), 1, 0, 0);
    }
    SVGPathDataTransformer.SKEW_X = SKEW_X;
    function SKEW_Y(a) {
        mathUtils_js_1.assertNumbers(a);
        return MATRIX(1, Math.atan(a), 0, 1, 0, 0);
    }
    SVGPathDataTransformer.SKEW_Y = SKEW_Y;
    function X_AXIS_SYMMETRY(xOffset) {
        if (xOffset === void 0) { xOffset = 0; }
        mathUtils_js_1.assertNumbers(xOffset);
        return MATRIX(-1, 0, 0, 1, xOffset, 0);
    }
    SVGPathDataTransformer.X_AXIS_SYMMETRY = X_AXIS_SYMMETRY;
    function Y_AXIS_SYMMETRY(yOffset) {
        if (yOffset === void 0) { yOffset = 0; }
        mathUtils_js_1.assertNumbers(yOffset);
        return MATRIX(1, 0, 0, -1, 0, yOffset);
    }
    SVGPathDataTransformer.Y_AXIS_SYMMETRY = Y_AXIS_SYMMETRY;
    // Convert arc commands to curve commands
    function A_TO_C() {
        return INFO(function (command, prevX, prevY) {
            if (SVGPathData_1.SVGPathData.ARC === command.type) {
                return mathUtils_js_1.a2c(command, command.relative ? 0 : prevX, command.relative ? 0 : prevY);
            }
            return command;
        });
    }
    SVGPathDataTransformer.A_TO_C = A_TO_C;
    // @see annotateArcCommand
    function ANNOTATE_ARCS() {
        return INFO(function (c, x1, y1) {
            if (c.relative) {
                x1 = 0;
                y1 = 0;
            }
            if (SVGPathData_1.SVGPathData.ARC === c.type) {
                mathUtils_js_1.annotateArcCommand(c, x1, y1);
            }
            return c;
        });
    }
    SVGPathDataTransformer.ANNOTATE_ARCS = ANNOTATE_ARCS;
    function CLONE() {
        return function (c) {
            var result = {};
            // tslint:disable-next-line
            for (var key in c) {
                result[key] = c[key];
            }
            return result;
        };
    }
    SVGPathDataTransformer.CLONE = CLONE;
    // @see annotateArcCommand
    function CALCULATE_BOUNDS() {
        var clone = CLONE();
        var toAbs = TO_ABS();
        var qtToC = QT_TO_C();
        var normST = NORMALIZE_ST();
        var f = INFO(function (command, prevXAbs, prevYAbs) {
            var c = normST(qtToC(toAbs(clone(command))));
            function fixX(absX) {
                if (absX > f.maxX) {
                    f.maxX = absX;
                }
                if (absX < f.minX) {
                    f.minX = absX;
                }
            }
            function fixY(absY) {
                if (absY > f.maxY) {
                    f.maxY = absY;
                }
                if (absY < f.minY) {
                    f.minY = absY;
                }
            }
            if (c.type & SVGPathData_1.SVGPathData.DRAWING_COMMANDS) {
                fixX(prevXAbs);
                fixY(prevYAbs);
            }
            if (c.type & SVGPathData_1.SVGPathData.HORIZ_LINE_TO) {
                fixX(c.x);
            }
            if (c.type & SVGPathData_1.SVGPathData.VERT_LINE_TO) {
                fixY(c.y);
            }
            if (c.type & SVGPathData_1.SVGPathData.LINE_TO) {
                fixX(c.x);
                fixY(c.y);
            }
            if (c.type & SVGPathData_1.SVGPathData.CURVE_TO) {
                // add start and end points
                fixX(c.x);
                fixY(c.y);
                var xDerivRoots = mathUtils_js_1.bezierRoot(prevXAbs, c.x1, c.x2, c.x);
                for (var _i = 0, xDerivRoots_1 = xDerivRoots; _i < xDerivRoots_1.length; _i++) {
                    var derivRoot = xDerivRoots_1[_i];
                    if (0 < derivRoot && 1 > derivRoot) {
                        fixX(mathUtils_js_1.bezierAt(prevXAbs, c.x1, c.x2, c.x, derivRoot));
                    }
                }
                var yDerivRoots = mathUtils_js_1.bezierRoot(prevYAbs, c.y1, c.y2, c.y);
                for (var _a = 0, yDerivRoots_1 = yDerivRoots; _a < yDerivRoots_1.length; _a++) {
                    var derivRoot = yDerivRoots_1[_a];
                    if (0 < derivRoot && 1 > derivRoot) {
                        fixY(mathUtils_js_1.bezierAt(prevYAbs, c.y1, c.y2, c.y, derivRoot));
                    }
                }
            }
            if (c.type & SVGPathData_1.SVGPathData.ARC) {
                // add start and end points
                fixX(c.x);
                fixY(c.y);
                mathUtils_js_1.annotateArcCommand(c, prevXAbs, prevYAbs);
                // p = cos(phi) * xv + sin(phi) * yv
                // dp = -sin(phi) * xv + cos(phi) * yv = 0
                var xRotRad = c.xRot / 180 * Math.PI;
                // points on ellipse for phi = 0° and phi = 90°
                var x0 = Math.cos(xRotRad) * c.rX;
                var y0 = Math.sin(xRotRad) * c.rX;
                var x90 = -Math.sin(xRotRad) * c.rY;
                var y90 = Math.cos(xRotRad) * c.rY;
                // annotateArcCommand returns phi1 and phi2 such that -180° < phi1 < 180° and phi2 is smaller or greater
                // depending on the sweep flag. Calculate phiMin, phiMax such that -180° < phiMin < 180° and phiMin < phiMax
                var _b = c.phi1 < c.phi2 ?
                    [c.phi1, c.phi2] :
                    (-180 > c.phi2 ? [c.phi2 + 360, c.phi1 + 360] : [c.phi2, c.phi1]), phiMin_1 = _b[0], phiMax = _b[1];
                var normalizeXiEta = function (_a) {
                    var xi = _a[0], eta = _a[1];
                    var phiRad = Math.atan2(eta, xi);
                    var phi = phiRad * 180 / Math.PI;
                    return phi < phiMin_1 ? phi + 360 : phi;
                };
                // xi = cos(phi), eta = sin(phi)
                var xDerivRoots = mathUtils_js_1.intersectionUnitCircleLine(x90, -x0, 0).map(normalizeXiEta);
                for (var _c = 0, xDerivRoots_2 = xDerivRoots; _c < xDerivRoots_2.length; _c++) {
                    var derivRoot = xDerivRoots_2[_c];
                    if (derivRoot > phiMin_1 && derivRoot < phiMax) {
                        fixX(mathUtils_js_1.arcAt(c.cX, x0, x90, derivRoot));
                    }
                }
                var yDerivRoots = mathUtils_js_1.intersectionUnitCircleLine(y90, -y0, 0).map(normalizeXiEta);
                for (var _d = 0, yDerivRoots_2 = yDerivRoots; _d < yDerivRoots_2.length; _d++) {
                    var derivRoot = yDerivRoots_2[_d];
                    if (derivRoot > phiMin_1 && derivRoot < phiMax) {
                        fixY(mathUtils_js_1.arcAt(c.cY, y0, y90, derivRoot));
                    }
                }
            }
            return command;
        });
        f.minX = Infinity;
        f.maxX = -Infinity;
        f.minY = Infinity;
        f.maxY = -Infinity;
        return f;
    }
    SVGPathDataTransformer.CALCULATE_BOUNDS = CALCULATE_BOUNDS;
})(SVGPathDataTransformer = exports.SVGPathDataTransformer || (exports.SVGPathDataTransformer = {}));
exports.SVGPathDataTransformer = SVGPathDataTransformer;
//# sourceMappingURL=SVGPathDataTransformer.js.map