"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var SVGPathData_1 = require("./SVGPathData");
function rotate(_a, rad) {
    var x = _a[0], y = _a[1];
    return [
        x * Math.cos(rad) - y * Math.sin(rad),
        x * Math.sin(rad) + y * Math.cos(rad),
    ];
}
exports.rotate = rotate;
var DEBUG_CHECK_NUMBERS = true;
function assertNumbers() {
    var numbers = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        numbers[_i] = arguments[_i];
    }
    if (DEBUG_CHECK_NUMBERS) {
        for (var i = 0; i < numbers.length; i++) {
            if ("number" !== typeof numbers[i]) {
                throw new Error("assertNumbers arguments[" + i + "] is not a number. " + typeof numbers[i] + " == typeof " + numbers[i]);
            }
        }
    }
    return true;
}
exports.assertNumbers = assertNumbers;
var PI = Math.PI;
/**
 * https://www.w3.org/TR/SVG/implnote.html#ArcImplementationNotes
 * Fixes rX and rY.
 * Ensures lArcFlag and sweepFlag are 0 or 1
 * Adds center coordinates: command.cX, command.cY (relative or absolute, depending on command.relative)
 * Adds start and end arc parameters (in degrees): command.phi1, command.phi2; phi1 < phi2 iff. c.sweepFlag == true
 */
function annotateArcCommand(c, x1, y1) {
    c.lArcFlag = (0 === c.lArcFlag) ? 0 : 1;
    c.sweepFlag = (0 === c.sweepFlag) ? 0 : 1;
    // tslint:disable-next-line
    var rX = c.rX, rY = c.rY, x = c.x, y = c.y;
    rX = Math.abs(c.rX);
    rY = Math.abs(c.rY);
    var _a = rotate([(x1 - x) / 2, (y1 - y) / 2], -c.xRot / 180 * PI), x1_ = _a[0], y1_ = _a[1];
    var testValue = Math.pow(x1_, 2) / Math.pow(rX, 2) + Math.pow(y1_, 2) / Math.pow(rY, 2);
    if (1 < testValue) {
        rX *= Math.sqrt(testValue);
        rY *= Math.sqrt(testValue);
    }
    c.rX = rX;
    c.rY = rY;
    var c_ScaleTemp = (Math.pow(rX, 2) * Math.pow(y1_, 2) + Math.pow(rY, 2) * Math.pow(x1_, 2));
    var c_Scale = (c.lArcFlag !== c.sweepFlag ? 1 : -1) *
        Math.sqrt(Math.max(0, (Math.pow(rX, 2) * Math.pow(rY, 2) - c_ScaleTemp) / c_ScaleTemp));
    var cx_ = rX * y1_ / rY * c_Scale;
    var cy_ = -rY * x1_ / rX * c_Scale;
    var cRot = rotate([cx_, cy_], c.xRot / 180 * PI);
    c.cX = cRot[0] + (x1 + x) / 2;
    c.cY = cRot[1] + (y1 + y) / 2;
    c.phi1 = Math.atan2((y1_ - cy_) / rY, (x1_ - cx_) / rX);
    c.phi2 = Math.atan2((-y1_ - cy_) / rY, (-x1_ - cx_) / rX);
    if (0 === c.sweepFlag && c.phi2 > c.phi1) {
        c.phi2 -= 2 * PI;
    }
    if (1 === c.sweepFlag && c.phi2 < c.phi1) {
        c.phi2 += 2 * PI;
    }
    c.phi1 *= 180 / PI;
    c.phi2 *= 180 / PI;
}
exports.annotateArcCommand = annotateArcCommand;
/**
 * Solves a quadratic system of equations of the form
 *      a * x + b * y = c
 *      x² + y² = 1
 * This can be understood as the intersection of the unit circle with a line.
 *      => y = (c - a x) / b
 *      => x² + (c - a x)² / b² = 1
 *      => x² b² + c² - 2 c a x + a² x² = b²
 *      => (a² + b²) x² - 2 a c x + (c² - b²) = 0
 */
function intersectionUnitCircleLine(a, b, c) {
    assertNumbers(a, b, c);
    // cf. pqFormula
    var termSqr = a * a + b * b - c * c;
    if (0 > termSqr) {
        return [];
    }
    else if (0 === termSqr) {
        return [
            [
                (a * c) / (a * a + b * b),
                (b * c) / (a * a + b * b)
            ]
        ];
    }
    var term = Math.sqrt(termSqr);
    return [
        [
            (a * c + b * term) / (a * a + b * b),
            (b * c - a * term) / (a * a + b * b)
        ],
        [
            (a * c - b * term) / (a * a + b * b),
            (b * c + a * term) / (a * a + b * b)
        ]
    ];
}
exports.intersectionUnitCircleLine = intersectionUnitCircleLine;
exports.DEG = Math.PI / 180;
function lerp(a, b, t) {
    return (1 - t) * a + t * b;
}
exports.lerp = lerp;
function arcAt(c, x1, x2, phiDeg) {
    return c + Math.cos(phiDeg / 180 * PI) * x1 + Math.sin(phiDeg / 180 * PI) * x2;
}
exports.arcAt = arcAt;
function bezierRoot(x0, x1, x2, x3) {
    var EPS = 1e-6;
    var x01 = x1 - x0;
    var x12 = x2 - x1;
    var x23 = x3 - x2;
    var a = 3 * x01 + 3 * x23 - 6 * x12;
    var b = (x12 - x01) * 6;
    var c = 3 * x01;
    // solve a * t² + b * t + c = 0
    if (Math.abs(a) < EPS) {
        // equivalent to b * t + c =>
        return [-c / b];
    }
    return pqFormula(b / a, c / a, EPS);
}
exports.bezierRoot = bezierRoot;
function bezierAt(x0, x1, x2, x3, t) {
    // console.log(x0, y0, x1, y1, x2, y2, x3, y3, t)
    var s = 1 - t;
    var c0 = s * s * s;
    var c1 = 3 * s * s * t;
    var c2 = 3 * s * t * t;
    var c3 = t * t * t;
    return x0 * c0 + x1 * c1 + x2 * c2 + x3 * c3;
}
exports.bezierAt = bezierAt;
function pqFormula(p, q, PRECISION) {
    if (PRECISION === void 0) { PRECISION = 1e-6; }
    // 4 times the discriminant:in
    var discriminantX4 = p * p / 4 - q;
    if (discriminantX4 < -PRECISION) {
        return [];
    }
    else if (discriminantX4 <= PRECISION) {
        return [-p / 2];
    }
    var root = Math.sqrt(discriminantX4);
    return [-(p / 2) - root, -(p / 2) + root];
}
function a2c(arc, x0, y0) {
    if (!arc.cX) {
        annotateArcCommand(arc, x0, y0);
    }
    var phiMin = Math.min(arc.phi1, arc.phi2), phiMax = Math.max(arc.phi1, arc.phi2), deltaPhi = phiMax - phiMin;
    var partCount = Math.ceil(deltaPhi / 90);
    var result = new Array(partCount);
    var prevX = x0, prevY = y0;
    for (var i = 0; i < partCount; i++) {
        var phiStart = lerp(arc.phi1, arc.phi2, i / partCount);
        var phiEnd = lerp(arc.phi1, arc.phi2, (i + 1) / partCount);
        var deltaPhi_1 = phiEnd - phiStart;
        var f = 4 / 3 * Math.tan(deltaPhi_1 * exports.DEG / 4);
        // x1/y1, x2/y2 and x/y coordinates on the unit circle for phiStart/phiEnd
        var _a = [
            Math.cos(phiStart * exports.DEG) - f * Math.sin(phiStart * exports.DEG),
            Math.sin(phiStart * exports.DEG) + f * Math.cos(phiStart * exports.DEG)
        ], x1 = _a[0], y1 = _a[1];
        var _b = [Math.cos(phiEnd * exports.DEG), Math.sin(phiEnd * exports.DEG)], x = _b[0], y = _b[1];
        var _c = [x + f * Math.sin(phiEnd * exports.DEG), y - f * Math.cos(phiEnd * exports.DEG)], x2 = _c[0], y2 = _c[1];
        result[i] = { relative: arc.relative, type: SVGPathData_1.SVGPathData.CURVE_TO };
        var transform = function (x, y) {
            var _a = rotate([x * arc.rX, y * arc.rY], arc.xRot), xTemp = _a[0], yTemp = _a[1];
            return [arc.cX + xTemp, arc.cY + yTemp];
        };
        _d = transform(x1, y1), result[i].x1 = _d[0], result[i].y1 = _d[1];
        _e = transform(x2, y2), result[i].x2 = _e[0], result[i].y2 = _e[1];
        _f = transform(x, y), result[i].x = _f[0], result[i].y = _f[1];
        if (arc.relative) {
            result[i].x1 -= prevX;
            result[i].y1 -= prevY;
            result[i].x2 -= prevX;
            result[i].y2 -= prevY;
            result[i].x -= prevX;
            result[i].y -= prevY;
        }
        _g = [result[i].x, result[i].y], prevX = _g[0], prevY = _g[1];
    }
    return result;
    var _d, _e, _f, _g;
}
exports.a2c = a2c;
//# sourceMappingURL=mathUtils.js.map