/**
	MoTAC - digital board for TAC board game
    Copyright (C) 2013-2014  Carsten Karbach
    
    Contact by mail carstenkarbach@gmx.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

package de.karbach.tac.ui.fragments;

import android.annotation.TargetApi;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.graphics.drawable.BitmapDrawable;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.app.Fragment;
import android.support.v4.app.FragmentTransaction;
import android.text.method.LinkMovementMethod;
import android.view.GestureDetector;
import android.view.HapticFeedbackConstants;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.View.OnLongClickListener;
import android.view.ViewGroup;
import android.widget.ImageButton;
import android.widget.PopupMenu;
import android.widget.TextView;
import de.karbach.tac.Preferences;
import de.karbach.tac.R;
import de.karbach.tac.core.ListManager;
import de.karbach.tac.ui.Board;
import de.karbach.tac.ui.BoardControl;
import de.karbach.tac.ui.BoardScale;
import de.karbach.tac.ui.EnDisImageButton;

/**
 * Digital TAC board for a local board.
 * 
 * @author Carsten Karbach
 *
 */
public class LocalBoard extends Fragment {

	private GestureDetector gdt;
	
	private ScaleGestureDetector scaleDetector; 
	
	protected BoardControl controller;
	
	private BoardScale scaler;
	
	protected Board board;
	
	/**
	 * Percent factor for scaling dialog icons
	 */
	private static final double dialogIconPercent = 0.2;
	
	/**
	 * The root view generated by this fragment.
	 * Needed to reference in functions called within the onCreateView.
	 */
	protected View rootView;
	
	/**
	 * Stores the layout ID of the layout, which has to be loaded
	 */
	protected int layoutId;
	
	/**
	 * Stores persistently, which buttons were already used once
	 */
	protected ListManager buttonsUsed;
	
	public LocalBoard(){
		layoutId = R.layout.portraitboard;
	}
	
	/**
	 * 
	 * @return true, if orientation of the fragment is landscape. False otherwise
	 */
	public boolean isOrientationLandscape(){
		return getResources().getConfiguration().orientation == Configuration.ORIENTATION_LANDSCAPE;
	}
	
	@Override
	public View onCreateView(LayoutInflater inflater, ViewGroup container,
			Bundle savedInstanceState) {
		
		if( isOrientationLandscape()){
			layoutId = R.layout.landscapeboard;
		}
		
		this.setHasOptionsMenu(true);
		
		rootView = inflater.inflate(layoutId, container, false);
		
		CardGridFragment cardgridFrag = null;
		
		if( isOrientationLandscape()){
			//Add cardgrid fragment into container
			cardgridFrag = new CardGridFragment();
			FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
			transaction.replace(R.id.cardgrid_container, cardgridFrag);
			transaction.commit();
		}
		
		board = (Board)rootView.findViewById(R.id.tac);
        initController();
        if(cardgridFrag != null){
        	controller.registerCardGridFragment(cardgridFrag);
        }
        gdt = new GestureDetector(board.getContext(), controller);
        scaler = new BoardScale(controller);
        scaleDetector = new ScaleGestureDetector(board.getContext(), scaler);
        
        initButtonActions();
		
		return rootView;
	}

    @Override
    public void onResume() {
        super.onResume();
        if(controller != null){
            controller.makeAndShowBoardImage(true);
        }
    }

    /* (non-Javadoc)
         * @see android.support.v4.app.Fragment#onActivityCreated(android.os.Bundle)
         */
	@Override
	public void onActivityCreated(Bundle savedInstanceState) {
        super.onActivityCreated(savedInstanceState);
		
		//Check here again as now the view can actually be accessed
		this.controller.checkButtonStates();
	}
	
	/**
	 * Function needs to be called for every motion event.
	 * This seems to be possible only from the activities onTouch function.
	 * 
	 * @param v
	 * @param event
	 * @return true, if the event is consumed
	 */
	public boolean onTouchEvent(View v, MotionEvent event) {
		if(gdt != null ){
			if(! buttonsUsed.wasItemUsed("board") && event.getAction() == MotionEvent.ACTION_UP){
				this.showButtonHelp("board");
				buttonsUsed.usedItem("board");
			}
			scaleDetector.onTouchEvent(event);
			gdt.onTouchEvent(event);
			controller.onTouch(event);
		}
		return false;
	}
	
    /**
     * Initialise the controller. Works as factory method for the controller.
     */
    protected void initController(){
    	controller = new BoardControl(board, this);
    }
    
    /**
     * Scales a given drawable identified by resource ID to the size specified
     * by the percentScale. 
     * @param drawableId the ID of the drawable, which should be resized
     * @param percentScale between 0 and 1, the smaller of height and width multiplied by this factor is the new height or width of the scaled image
     * @return bitmap resized to the reuired size
     */
    private Bitmap getScaledBitmap(int drawableId, double percentScale){
    	Bitmap bitmapOrg = BitmapFactory.decodeResource(getResources(), drawableId);
    	int width = bitmapOrg.getWidth();
    	int height = bitmapOrg.getHeight();

        View rootView = getView();
        int viewWidth = rootView==null?1:rootView.getWidth();
        int viewHeight = rootView==null?1:rootView.getHeight();

    	double widthFactor = viewWidth/(double) width;
    	double heightFactor = viewHeight/(double) height;
    	double factor = widthFactor;
    	if(heightFactor < widthFactor){
    		factor = heightFactor;
    	}
    	
    	int newWidth = (int)Math.round( percentScale*factor*width );
    	int newHeight = (int)Math.round( percentScale*factor*height );

    	 // calculate the scale - in this case = 0.4f
    	float scaleWidth = ((float) newWidth) / width;
    	float scaleHeight = ((float) newHeight) / height;

    	// create matrix for the manipulation
    	Matrix matrix = new Matrix();

    	// resize the bit map
    	matrix.postScale(scaleWidth, scaleHeight);

    	// recreate the new Bitmap
    	Bitmap resizedBitmap = Bitmap.createBitmap(bitmapOrg, 0, 0, 
    	                              width, height, matrix, true); 

    	return resizedBitmap;
    }
    
    /**
     * Holds last dialog shown
     */
    private Dialog helpdialog;
    
    /**
     * 
     * @return true, if a help dialog is currently shown, false otherwise
     */
    public boolean isShowingDialog(){
    	if(helpdialog != null && helpdialog.isShowing()){
    		return true;
    	}
    	else{
    		return false;
    	}
    }
    
    /**
     * Show multiple help dialogs in a row.
     * Show the first one, then allow to show the next by clicking on the next button.
     * 
     * @param buttonNames the names of buttons to show help dialogs for
     */
    protected void showButtonHelp(String[] buttonNames){
    	
    	if(buttonNames == null || buttonNames.length<=0){
    		return;
    	}
    	
    	String buttonName = buttonNames[0];
    	
    	int messageId = -1;
    	int drawableId = -1;
    	int titleId = -1;
    	if(buttonName.equals("overview")){
    		messageId = R.string.overview_help;
    		titleId = R.string.menuoverviewstring;
    		drawableId = R.drawable.tac;
    	}
    	if(buttonName.equals("left")){
    		messageId = R.string.backhelp;
    		titleId = R.string.menuleftstring;
    		drawableId = R.drawable.button_back;
    	}
    	else if(buttonName.equals("color")){
    		messageId = R.string.colorhelp;
    		titleId = R.string.menuswitchstring;
    		drawableId = R.drawable.button_color;
    	}
    	else if(buttonName.equals("right")){
    		messageId = R.string.nexthelp;
    		titleId = R.string.menurightstring;
    		drawableId = R.drawable.button_next;
    	}
    	else if(buttonName.equals("move")){
    		messageId = R.string.movehelp;
    		titleId = R.string.menumovestring;
    		drawableId = R.drawable.button_move;
    	}
    	else if(buttonName.equals("zoomin")){
    		messageId = R.string.zoominhelp;
    		titleId = R.string.menuplusstring;
    		drawableId = R.drawable.button_zoomin;
    	}
    	else if(buttonName.equals("zoomout")){
    		messageId = R.string.zoomouthelp;
    		titleId = R.string.menuminusstring;
    		drawableId = R.drawable.button_zoomout;
    	}
    	else if(buttonName.equals("tac")){
    		messageId = R.string.restarthelp;
    		titleId = R.string.menurestartstring;
    		drawableId = R.drawable.button_restart;
    	}
    	else if(buttonName.equals("board")){
    		messageId = R.string.boardhelp;
    		titleId = R.string.menuboardstring;
    		drawableId = R.drawable.board;
    	}
    	else if(buttonName.equals("license")){
    		messageId = R.string.license_short;
    		titleId = R.string.menu_license_string;
    		drawableId = R.drawable.tac;
    	}
    	else if(buttonName.equals("licenselong")){
    		messageId = R.string.license_long;
    		titleId = R.string.menu_license_string;
    		drawableId = R.drawable.tac;
    	}
    	
    	if(messageId != -1){
    		AlertDialog.Builder builder = new AlertDialog.Builder(getActivity());
	        builder.setMessage(messageId);
	        
	        android.content.DialogInterface.OnClickListener nextClickListener;
	        int negString = R.string.ok;
			//Show rest of help dialogs
	        if(buttonNames.length > 1){
				final String[] rest = new String[buttonNames.length-1];
                System.arraycopy(buttonNames, 1, rest, 0, rest.length);
	        	nextClickListener = new android.content.DialogInterface.OnClickListener() {
					
					@Override
					public void onClick(DialogInterface dialog, int which) {
						showButtonHelp(rest);
					}
				};
				builder.setPositiveButton(R.string.next_string, nextClickListener);
				negString = R.string.close_string;
	        }
	        
	        builder.setNegativeButton(negString, null);
	        builder.setIcon( new BitmapDrawable(getResources(), getScaledBitmap(drawableId, dialogIconPercent)) );
	        builder.setTitle(titleId);
	        // Create the AlertDialog object and return it
	        Dialog dia = builder.create();
	        
	        helpdialog = dia;
	        
	        dia.show();
	        
	        View messageView=dia.findViewById(android.R.id.message);
	        
	        if(messageView instanceof TextView){
	        	((TextView)messageView).setMovementMethod(LinkMovementMethod.getInstance());
	        }
    	}
    	
    }
    
    /**
     * Show a help dialog for the given button
     * @param buttonName the name of the button for which help dialog must be created
     */
    protected void showButtonHelp(String buttonName){
    	showButtonHelp(new String[]{buttonName});
    }
    
    /**
     * Call this function, when a long press action took place to show help
     */
    protected void longPressVibrate(){
    	View view = this.getView();
    	if(view == null){
    		return;
    	}
    	view.performHapticFeedback(HapticFeedbackConstants.LONG_PRESS);
    }
    
    /**
     * Define the actions of all the buttons around the board.
     */
    protected void initButtonActions(){
    	
    	buttonsUsed = new ListManager(getActivity());

    	//Button for info and settings
    	final ImageButton helpbutton = (ImageButton) rootView.findViewById(R.id.menuHelpButton);
    	helpbutton.setOnClickListener(new OnClickListener() {
			
    		@TargetApi(Build.VERSION_CODES.HONEYCOMB)
			@Override
			public void onClick(View v) {
				
				if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
					//Show sequence of help dialogs
			        showButtonHelp(new String[]{"overview","board", "move", "color", "left", "right", "zoomin", "zoomout", "tac"});
			        return;
			    }

				PopupMenu popup = new PopupMenu(getActivity(), helpbutton);  
	            //Inflating the Popup using xml file  
	            popup.getMenuInflater().inflate(R.menu.help, popup.getMenu());  

	            //registering popup with OnMenuItemClickListener  
	            popup.setOnMenuItemClickListener(new PopupMenu.OnMenuItemClickListener() {  
	             public boolean onMenuItemClick(MenuItem item) {  
	            	 return onOptionsItemSelected(item); 
	             }  
	            });  

	            popup.show();//showing popup menu 
				
			}
		});
    	
    	//Button for info and settings
    	final ImageButton menubutton = (ImageButton) rootView.findViewById(R.id.menuButton);
    	menubutton.setOnClickListener(new OnClickListener() {
			
			@TargetApi(Build.VERSION_CODES.HONEYCOMB)
			@Override
			public void onClick(View v) {
				
				if (Build.VERSION.SDK_INT < Build.VERSION_CODES.HONEYCOMB) {
			        getActivity().openOptionsMenu();
			        return;
			    }

				PopupMenu popup = new PopupMenu(getActivity(), menubutton);  
	            //Inflating the Popup using xml file  
	            popup.getMenuInflater().inflate(R.menu.activity_tac, popup.getMenu());  

	            //registering popup with OnMenuItemClickListener  
	            popup.setOnMenuItemClickListener(new PopupMenu.OnMenuItemClickListener() {  
	             public boolean onMenuItemClick(MenuItem item) {  
	            	 return onOptionsItemSelected(item); 
	             }  
	            });  

	            popup.show();//showing popup menu 
				
			}
		});
    	
    	ImageButton leftButton = (ImageButton)rootView.findViewById(R.id.leftButton);
    	leftButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("left");

				return false;
			}
		});
    	leftButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				buttonsUsed.usedItem("left");
				controller.stepLeft();
			}
		});
    	
    	ImageButton rightButton = (ImageButton)rootView.findViewById(R.id.rightButton);
    	rightButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("right");

				return false;
			}
		});
    	rightButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				buttonsUsed.usedItem("right");
				controller.stepRight();
			}
		});
    	
    	ImageButton tacButton = (ImageButton)rootView.findViewById(R.id.tacButton);
    	tacButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("tac");

				return false;
			}
		});
    	tacButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				buttonsUsed.usedItem("tac");
				controller.restart();
			}
		});
    	
    	EnDisImageButton moveButton = (EnDisImageButton)rootView.findViewById(R.id.moveButton);
    	moveButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("move");

				return false;
			}
		});
    	moveButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				if(! buttonsUsed.wasItemUsed("move")){
					showButtonHelp("move");
				}
				buttonsUsed.usedItem("move");
				controller.switchMoveMode();
			}
		});
    	
    	ImageButton plusButton = (ImageButton)rootView.findViewById(R.id.plusButton);
    	plusButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("zoomin");

				return false;
			}
		});
    	plusButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				buttonsUsed.usedItem("zoomin");
				controller.zoomIn();
			}
		});
    	
    	ImageButton minusButton = (ImageButton)rootView.findViewById(R.id.minusButton);
    	minusButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("zoomout");

				return false;
			}
		});
    	minusButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				buttonsUsed.usedItem("zoomout");
				controller.zoomOut();
			}
		});
    	
    	ImageButton colorButton = (ImageButton)rootView.findViewById(R.id.switchButton);
    	colorButton.setOnLongClickListener(new OnLongClickListener() {
			
			@Override
			public boolean onLongClick(View v) {
				longPressVibrate();
				showButtonHelp("color");

				return false;
			}
		});
    	colorButton.setOnClickListener(new OnClickListener() {
			@Override
			public void onClick(View v) {
				if(! buttonsUsed.wasItemUsed("color")){
					showButtonHelp("color");
				}
				buttonsUsed.usedItem("color");
				controller.switchColorSwitchMode();
			}
		});
    	
    	controller.checkButtonStates();
    }
    
    public void onStop(){
    	super.onStop();
    	controller.stopActivity();
    }

    
    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
    	super.onCreateOptionsMenu(menu, inflater);
        inflater.inflate(R.menu.activity_tac, menu);
    }
    
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
       super.onOptionsItemSelected(item);

    	// Handle item selection
        switch (item.getItemId()) {
        	case R.id.menu_license:
        	showButtonHelp(new String[]{"license", "licenselong"});
        	return true;
        	case R.id.menu_settings:
            	Intent intent = new Intent(this.getActivity(), Preferences.class);
            	startActivity(intent);
                return true;
            case R.id.menu_info:
            	FragmentTransaction transaction = getActivity().getSupportFragmentManager().beginTransaction();
            	AboutFragment about = new AboutFragment();
            	
            	transaction.replace(R.id.fragment_container, about);
            	transaction.addToBackStack("about");
            	transaction.setTransition(FragmentTransaction.TRANSIT_FRAGMENT_FADE);
            	transaction.commit();
            	
                return true;
            case R.id.help_overview:
            	showButtonHelp("overview");
            	return true;
            case R.id.help_board:
            	showButtonHelp("board");
            	return true;
            case R.id.help_color:
            	showButtonHelp("color");
            	return true;
            case R.id.help_left:
            	showButtonHelp("left");
            	return true;
            case R.id.help_right:
            	showButtonHelp("right");
            	return true;
            case R.id.help_restart:
            	showButtonHelp("tac");
            	return true;
            case R.id.help_move:
            	showButtonHelp("move");
            	return true;
            case R.id.help_zoomin:
            	showButtonHelp("zoomin");
            	return true;
            case R.id.help_zoomout:
            	showButtonHelp("zoomout");
            	return true;
            case R.id.menu_make_image:
                this.controller.makeAndShowBoardImage(false);
                return true;
            case R.id.menu_show_exports:
                this.controller.showExports();
                return true;
            default:
                return super.onOptionsItemSelected(item);
        }
    }
    
}
