package de.onyxbits.droidentify;

/*
 * Copyright (C) 2014 Jared Rummler <jared.rummler@gmail.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.util.Locale;

import android.os.Build;
import android.text.TextUtils;

/**
 * Helper class to get the consumer friendly device name.</p>
 * 
 * {@code String device = Devices.getDeviceName();}</p>
 * 
 * The above code will get the device name as specified in the supported device
 * list that is maintained by Google.</p>
 * 
 * See: https://support.google.com/googleplay/answer/1727131</p>
 * 
 * @author Jared Rummler <jared.rummler@gmail.com>
 */
public class Devices {

	// @formatter:off

	/**
	 * This array is using the list of supported devices on Google Play.</p>
	 * 
	 * Any device that does not have both the retail name and marketing name is
	 * not included.</p>
	 * 
	 * See the following URL for the PDF that this list was derived from:
	 * https://support.google.com/googleplay/answer/1727131</p>
	 * 
	 * Each String uses a vertical line to separate the manufacturer, name,
	 * device, and model:</p>
	 * 
	 * "Retail Branding|Marketing Name|Device|Model"</p>
	 * 
	 * Last updated: 2/5/2015</p>
	 */
	private static final String[] SUPPORTED_DEVICES = {
			"7Eleven|IN265|IN265|IN265",
			"A.O.I. ELECTRONICS FACTORY|A.O.I.|TR10CS1_11|TR10CS1",
			"AMTC|AMTC MD1001 Tablet|MD1001|MD1001",
			"AMTC|AMTC MD7008|MD7008A|MD7008",
			"AMTC|AMTC MD7081|MD7081A|MD7081",
			"AOC|A2272PWHT|AOC_WW|G2SMNT",
			"AOC|AOC-A2272PW4T|AOC_WW|G3SMNTA22",
			"AOC|AOC-A2272PW4T|AOC_WW|G3SMNTA24",
			"AOC|AOC-A2472PW4T|G3SMNTA24|G3SMNTA24",
			"AV|AV10|RCT6203W46|AV10",
			"AV|AV7|RCT6773W22|AV7",
			"AZBox|AZDroid|AZDroid|AZDroid",
			"AZPEN|AZPEN A720|wing-inet|A720",
			"AZPEN|AZPEN A920|wing-ibt|A920",
			"Abocom|AboCom A08SM|A08|A08S",
			"Acer|A1-713|acer_aprilia|A1-713",
			"Acer|A1-810|mango|A1-810",
			"Acer|A1-811|mango|A1-811",
			"Acer|A1-830|ducati|A1-830",
			"Acer|A100|vangogh|A100",
			"Acer|A110|a110|A110",
			"Acer|A211|picasso_e2|A211",
			"Acer|A3-A10|zara|A3-A10",
			"Acer|A3-A11|zara|A3-A11",
			"Acer|A3-A20|acer_harley|A3-A20",
			"Acer|A3-A20FHD|acer_harleyfhd|A3-A20FHD",
			"Acer|A500|picasso|A500",
			"Acer|A511|picasso_m|A511",
			"Acer|A700|picasso_mf|A700",
			"Acer|A700/A701|picasso_mf|A700",
			"Acer|AK330s|C8n|AK330S",
			"Acer|Aspire A3|aa3-600|AA3-600",
			"Acer|B1-710|B1-710|B1-710",
			"Acer|B1-711|B1-711|B1-711",
			"Acer|B1-720|b1-720|b1-720",
			"Acer|B1-721|b1-721|b1-721",
			"Acer|B1-730|vespatn|B1-730",
			"Acer|B1-730HD|vespa|B1-730HD",
			"Acer|B1-A71|B1-A71|B1-A71",
			"Acer|CloudMobile S500|a9|S500",
			"Acer|DA220HQL|DA220HQL|DA220HQL",
			"Acer|DA221HQL|da1|DA221HQL",
			"Acer|DA222HQL|da2|DA222HQL",
			"Acer|DA222HQLA|da2|DA222HQLA",
			"Acer|DA222HQLA|da222hqla|DA222HQLA",
			"Acer|DA223HQL|da3|DA223HQL",
			"Acer|DA241HL|DA241HL|DA241HL",
			"Acer|DA245HQL|da245hql|DA245HQL",
			"Acer|E380|acer_e3n|E380",
			"Acer|G1-715|G1-715|G1-715",
			"Acer|I110|I1|I110",
			"Acer|Icona One 7|vespa|B1-730HD",
			"Acer|Iconia One 7|G1-725|G1-725",
			"Acer|Iconia One 7|vespa|B1-730HD",
			"Acer|Iconia One 7|vespa2|B1-750",
			"Acer|Iconia One 7|vespatn|B1-730",
			"Acer|Iconia One 8|vespa8|A1-850",
			"Acer|Iconia One 8|vespa8|B1-810",
			"Acer|Iconia Tab 10|acer_harley|A3-A20",
			"Acer|Iconia Tab 10|acer_harleyfhd|A3-A20FHD",
			"Acer|Iconia Tab 7|acer_aprilia|A1-713",
			"Acer|Iconia Tab 7|acer_apriliahd|A1-713HD",
			"Acer|Iconia Tab 8|ducati2fhd|A1-840FHD",
			"Acer|Iconia Tab 8|ducati2hd|A1-840",
			"Acer|Iconia Tab 8|ducati2hd3g|A1-841",
			"Acer|Iconia Tab A100 (VanGogh)|vangogh|A100",
			"Acer|Iconia Tab A100 (VanGogh)|vangogh|A101",
			"Acer|Iconia TalkTab 7|acer_a1_724|A1-724",
			"Acer|Liquid|a1|Acer Liquid",
			"Acer|Liquid|a1|Acer S100",
			"Acer|Liquid|a1|Liquid",
			"Acer|Liquid C1|I1|I110",
			"Acer|Liquid E2|C11|V370",
			"Acer|Liquid E3|acer_e3|E380",
			"Acer|Liquid E3|acer_e3n|E380",
			"Acer|Liquid E3S|acer_ZXR|Z130",
			"Acer|Liquid E600|e600|E600",
			"Acer|Liquid E700|acer_e39|E39",
			"Acer|Liquid Jade|acer_S55|S55",
			"Acer|Liquid Jade S|acer_S56|S56",
			"Acer|Liquid S1|a10|S510",
			"Acer|Liquid S2|a12|S520",
			"Acer|Liquid S3|s3|S53",
			"Acer|Liquid X1|s3|S53",
			"Acer|Liquid Z200|acer_z200|Z200",
			"Acer|Liquid Z205|acer_z205|Z205",
			"Acer|Liquid Z205|acer_z205p|Z205",
			"Acer|Liquid Z205|acer_z205p|Z205P",
			"Acer|Liquid Z3|ZX|Z130",
			"Acer|Liquid Z3|acer_ZXR|Z130",
			"Acer|Liquid Z4|acer_Z6|Z160",
			"Acer|Liquid Z410|acer_z410|Z410",
			"Acer|Liquid Z5|acer_ZXL|Z150",
			"Acer|Liquid Z500|acer_Z500|Z500",
			"Acer|M3-2200|da2|M3-2200",
			"Acer|N3-2200|da2|N3-2200",
			"Acer|Picasso|picasso|A500",
			"Acer|Picasso|picasso|A501",
			"Acer|Picasso|ventana|A500",
			"Acer|Picasso|ventana|G100W",
			"Acer|Picasso|ventana|TPA60W",
			"Acer|TA2|ta2|TA272HUL",
			"Acer|TA272HUL|ta2|TA272HUL",
			"Acer|Tab 7|td070va1|TD070VA1",
			"Acer|Z110|Z1|Z110",
			"Acer|Z120|Z2|Z120",
			"Allfine|Allfine M721|86S|M721",
			"Allfine|Allfine PC1038Q|fine10_Joy|PC1038Q",
			"Alphascan|SAT2242 WorkTab|ADP_WW|G2SMNT",
			"Amplify|TR10CD3|TR10CD3_1|TR10CD3",
			"Amplify|TR10CD3|TR10CD3_2|TR10CD3",
			"Anydata|Aquaris I8|bq_Aquaris|bq Aquaris",
			"Anydata|Auchan MID7317CP Tablet|MID7317CP|MID7317CP",
			"Anydata|COBY MID7055|MID7055|MID7055",
			"Anydata|Carrefour CT710|M755ND|CT710",
			"Anydata|Carrefour CT720 / Emdoor EM63 Tablet|EM63|EM63",
			"Anydata|Coby MID1065|MID1065|MID1065",
			"Anydata|Coby MID7065|MID7065|MID7065",
			"Anydata|Coby MID8065|MID8065|MID8065",
			"Anydata|D2-721G|D2-721G|D2-721G",
			"Anydata|D2-727G|D2-727|D2-727",
			"Anydata|DOPO GMS-718 Tablet / Discovery|DT088|GS-718",
			"Anydata|DOTPAD DP3D8 / Gadmei|E8-3D|DP3D8",
			"Anydata|Digix TAB-840_G|TAB-840_G|TAB-840_G",
			"Anydata|Eviant MT8000|PDM829MD|MT8000",
			"Anydata|Garmin nuvi3590|scorpio|nuvi 3590",
			"Anydata|Garmin nuvi3595|scorpio|nuvi 3595",
			"Anydata|Grundig GR-TB10S Tablet|GR-TB10S|GR-TB10S",
			"Anydata|HCL ME TABLET PC U2|M712MC|U2",
			"Anydata|HKC P771A|P771A_WIFI|P771A",
			"Anydata|HKC P774A|P774A|P774A",
			"Anydata|HKC P776A|P776A|P776A",
			"Anydata|HKC P778A|HKCP778A|P778A",
			"Anydata|HKC P886A|P886A|P886A",
			"Anydata|HS_7DTB14|HS_7DTB14|HS_7DTB14",
			"Anydata|Insignia NS-13T001 Tablet|oracle|NS-13T001",
			"Anydata|Jiateng JT1241|JT1241|JT1241",
			"Anydata|Lazer MD7305 Tablet / AMTC|MD7305|MD7305",
			"Anydata|Le Pan S|oracle|Le Pan S",
			"Anydata|LePanII|LePanII_wifi|LePanII",
			"Anydata|Leader I10A-LE|I10A-LE|I10A-LE",
			"Anydata|Mach_Speed Trio G2 Tablet|Trio_Stealth_G2|G2",
			"Anydata|Marquis Tablet|marquis_tablet|Marquis_MP977",
			"Anydata|Matsunichi M97|oracle|M97",
			"Anydata|Monster M7 Tablet|MONSTERM7|M7",
			"Anydata|Nextbook NX007HD Tablet|M7000ND|NX007HD",
			"Anydata|Nextbook NX008HD8G Tablet|M8000ND|NX008HD8G",
			"Anydata|Nextbook NX008HI Tablet / Carrefour CT810|M909NP|NX008HI",
			"Anydata|Nextbook Next7D12 Tablet|M757ND|Next7D12",
			"Anydata|Nextbook Next7P12|M727MC|Next7P12",
			"Anydata|Pendo PNDPP48GP|PNDPP48GP|PNDPP48GP",
			"Anydata|Philips W336|Crane|Philips W336",
			"Anydata|Philips W536|Philips_WG-MANTO-RU_B|Philips W536",
			"Anydata|Philips W626|sangfei73_gb|W626",
			"Anydata|Philips W632|robot|Philips W632",
			"Anydata|Philips W832|Philips_WG-ROVER-RU_A|Philips W832",
			"Anydata|Proscan PLT7223G|PLT7223G|PLT7223G",
			"Anydata|Proscan PLT7777|AMLMID710K|PLT7777",
			"Anydata|Proscan PLT8223G|PLT8223G|PLT8223G",
			"Anydata|RCA RCT6078W2|AMLEM62|RCT6078W2",
			"Anydata|RCA RCT6078W2|EM62|RCT6078W2",
			"Anydata|TecToy TT-2500|AML757ND|TECTOYTT2500",
			"Anydata|Visual Land Prestige 7D|PRO7D|PRO7D",
			"Anydata|Vivitar Camelio Tablet|PI070H04CA|Camelio Family tablet",
			"Anydata|Vivitar XO Tablet|PI070H08XO|XO Learning tablet",
			"Anydata|Zeki TBDG773|TBDG773|TBDG773",
			"Anydata|ematic EGP008|EGP008|EGP008",
			"Anydata|ematic EGP010|EGP010|EGP010",
			"Anydata|ematic EGS004|EGS004|EGS004",
			"Anydata|ematic EGS102|EGS102|EGS102",
			"Anydata|essentielb ST8003/FT8001 Tablet|M805ND|Smart\\'TAB 8003",
			"Anydata|iCraig CMP748|CMP748|CMP748",
			"Anydata|iCraig CMP749|CMP749|CMP749",
			"Apex|TM785CH|tm785ch|TM785CH",
			"Archos|101 Childpad|A101CHP|ARCHOS 101 CHILDPAD",
			"Archos|101 Cobalt|AC101CO|Archos 101 Cobalt",
			"Archos|101 G9|A101|ARCHOS 101G9",
			"Archos|101 Neon|A101NE|Archos 101 Neon",
			"Archos|101 Platinum|A101PL|ARCHOS 101 PLATINUM",
			"Archos|101 Titanium|A101TI|ARCHOS 101 Titanium",
			"Archos|101 XS|A101XS|ARCHOS 101G10",
			"Archos|101 Xenon|a101xe|Archos 101 Xenon",
			"Archos|101XS2|AC101XS2|ARCHOS 101 XS 2",
			"Archos|101b XS2|ac101bxs2|ARCHOS 101b XS2",
			"Archos|40 Titanium|a40ti|Archos 40 Titanium",
			"Archos|40b Titanium|a40btisr|Archos 40b Titanium Surround",
			"Archos|40c Titanium|ac40cti|Archos 40c Titanium",
			"Archos|45 Helium 4G|a45he|Archos 45 Helium 4G",
			"Archos|45 Platinum|msm8625|Archos 45 Platinum",
			"Archos|45 Titanium|a45ti|Archos 45 Titanium",
			"Archos|45c Platinum|ac45cpl|Archos 45c Platinum",
			"Archos|50 Helium 4G|a50he|Archos 50 Helium 4G",
			"Archos|50 Neon|a50ne|Archos 50 Neon",
			"Archos|50 Oxygen|a50ox|Archos 50 Oxygen",
			"Archos|50 Platinum|msm8625|Archos 50 Platinum",
			"Archos|50 Titanium|a50ti|Archos 50 Titanium",
			"Archos|50b Platinum|ac50bpl|Archos 50b Platinum",
			"Archos|53 Platinum|msm8625|Archos 53 Platinum",
			"Archos|53 Titanium|a53ti|Archos 53 Titanium",
			"Archos|70 Cobalt|AC70CO|ARCHOS 70 Cobalt",
			"Archos|70 Neon|ac70ne|Archos 70 Neon",
			"Archos|70 Titanium|A70TI|ARCHOS 70 Titanium",
			"Archos|70 Xenon|a70xe|Archos 70 Xenon",
			"Archos|70b Titanium|A70BTI|ARCHOS 70b TITANIUM",
			"Archos|79 Neon|a79ne|Archos 79 Neon",
			"Archos|79 Platinium|AC79PL|ARCHOS 79 Platinum",
			"Archos|79 Xenon|a79xe|Archos 79 Xenon",
			"Archos|80 Carbon|AC80CA|ARCHOS 80 Carbon",
			"Archos|80 Childpad|A80CHP|ARCHOS 80 CHILDPAD",
			"Archos|80 Cobalt|A80CO|ARCHOS 80 COBALT",
			"Archos|80 G9|A80|ARCHOS 80G9",
			"Archos|80 Platinum|A80PL|ARCHOS 80 Platinum",
			"Archos|80 Titanium|A80TI|ARCHOS 80 TITANIUM",
			"Archos|80 XS|A80XSK|ARCHOS 80XSK",
			"Archos|80 Xenon|A80XE|Archos 80 Xenon",
			"Archos|80b Platinum|A80BPL|ARCHOS 80b PLATINUM",
			"Archos|90 Neon|a90ne|Archos 90 Neon",
			"Archos|90b Neon|ac90bne|Archos 90b Neon",
			"Archos|97 Carbon|A97C|ARCHOS 97 CARBON",
			"Archos|97 Cobalt|AC97CO|Archos 97 Cobalt",
			"Archos|97 Platinum|A97PL|ARCHOS 97 Platinum",
			"Archos|97 Titanium HD|A97TIHD|ARCHOS 97 TITANIUMHD",
			"Archos|97 Xenon|A97XE|ARCHOS 97 XENON",
			"Archos|97b Platinum|AC97BPL|ARCHOS 97b PLATINUM",
			"Archos|97b Titanium|A97BTI|ARCHOS 97B TITANIUM",
			"Archos|ARCHOS 101 Oxygen|ac101ox|Archos 101 Oxygen",
			"Archos|ARCHOS 90 Copper|ac90cv|Archos 90 Copper",
			"Archos|Archos 101 Copper|ac101cv|Archos 101 Copper",
			"Archos|Archos 101 Xenon|ac101cxe|Archos 101c Xenon",
			"Archos|Archos 101b Neon|ac101bne|Archos 101b Neon",
			"Archos|Archos 101b Platinum|ac101bpl|Archos 101b Platinium",
			"Archos|Archos 101c Neon|ac101cne|Archos 101c Neon",
			"Archos|Archos 59 Titanium|ac59ti|Archos 59 Titanium",
			"Archos|Archos 64 Xenon|ac64xe|Archos 64 Xenon",
			"Archos|Archos 70 Copper|ac70cv|Archos 70 Copper",
			"Archos|Archos 70b Cobalt|a70bco|Archos 70b Cobalt",
			"Archos|Archos 70b Xenon|ac70bxe|Archos 70b Xenon",
			"Archos|Archos 70c Cobalt|ac70cco|Archos 70c Cobalt",
			"Archos|Archos 70it 2|A70it2|ARCHOS 70it2",
			"Archos|Archos 70it 2|A70it2|ARCHOS 70it2G8",
			"Archos|Archos 79 Cobalt|ac79co|Archos 79 Cobalt",
			"Archos|Archos 80 Helium|ac80he|Archos 80 Helium 4G",
			"Archos|Archos 80b xenon|ac80bxe|Archos 80b Xenon",
			"Archos|Archos ArcBook|a101db|Archos 101 DB",
			"Archos|Auchan QiLive 45|ql45|Qilive 45",
			"Archos|Auchan QiLive 50|ql50|Qilive 50",
			"Archos|Auchan Qilive 40|ql40|Qilive 40",
			"Archos|Auchan Qilive8|QiLive8|QiLive 8",
			"Archos|Auchan Qilive8QC|QiLive8QC|QiLive 8QC",
			"Archos|Auchan Qilive97|QiLive97|QiLive 97",
			"Archos|Auchan Qilive97R|QiLive97R|QiLive 97R",
			"Archos|Auchan Selecline 10|SELECLINE10|Selecline 10",
			"Archos|Bush 7.0 MyTablet|ac70pl|Bush 7.0 MyTablet",
			"Archos|Bush 8.0 MyTablet|ac80cpl|Bush 8.0 MyTablet",
			"Archos|Bush MyTablet 7|ac70bu|BUSH 7.0 TABLET",
			"Archos|Bush Mytablet 2|a80bu|Bush Mytablet 2",
			"Archos|ChefPad|A97CFP|Archos Chefpad",
			"Archos|FamilyPad 2|A133FP2|ARCHOS FAMILYPAD 2",
			"Archos|GamePad|A70GP|ARCHOS GAMEPAD",
			"Archos|GamePad 2|A70GP2|ARCHOS GAMEPAD2",
			"Archos|KUNO 4+|kuno4p|KUNO 4+",
			"Archos|Kuno|KUNO4|KUNO4",
			"Archos|My Tablet 101|ac101bu|BUSH 10.1 TABLET",
			"Archos|QiLive 101|aqilive101|QILIVE 101",
			"Archos|Qilive 53|msm8625|Qilive 53",
			"Archos|Qilive 7|qilive7|Qilive 7",
			"Archos|Qilive 97R2|qilive97r2|Qilive 97R-2",
			"Archos|Qilive79|Qilive79|Qilive 79",
			"Archos|Qilive7V2|ql70v2|Qilive 70v2",
			"Archos|Quechua Phone 5|A50RG11|Quechua Phone 5",
			"Archos|Quechua Tablet 8|A80RG11|A80RG11",
			"Archos|Smart Home Tablet|hometablet|Archos Smart Home Tablet",
			"Archos|TV Connect|LUDO|ARCHOS LUDOG10",
			"Astak|NEOS|Wenu|NEOS",
			"Astro Queo|A712|A712|A712",
			"AstroQueo|polaris_wifionly|polaris-wifionly|A912",
			"Asus|ASUS ETBW11AA|ETBW11AA|ETBW11AA",
			"Asus|ASUS ETBW11AA|TF101G|ETBW11AA",
			"Asus|ASUS Fonepad|K00G|K00G",
			"Asus|ASUS Fonepad 7|K01F|K01F",
			"Asus|ASUS Fonepad 7 LTE|K00Y|K00Y",
			"Asus|ASUS Fonepad 8|K016_2|K016",
			"Asus|ASUS Fonepad 8|K016_3|K016",
			"Asus|ASUS Fonepad 8|K016_4|K016",
			"Asus|ASUS Fonepad 8(FE380CG)|K016_1|K016",
			"Asus|ASUS Fonepad ME371MG|ME371MG|ME371MG",
			"Asus|ASUS MeMO Pad 7 (ME572CL)|K00R|K00R",
			"Asus|ASUS MeMO Pad FHD 10|ME302KL|ME302KL",
			"Asus|ASUS MeMO Pad HD7 Dual SIM|ASUS-K00S|ASUS K00S",
			"Asus|ASUS MeMO Pad Smart 10|ME301T|ME301T",
			"Asus|ASUS MeMo Pad ME302C|ME302C|ME302C",
			"Asus|ASUS Pad|K010|K010",
			"Asus|ASUS Trans AiO P1801|P1801-T|ASUS Tablet P1801-T",
			"Asus|ASUS TransBook Trio|TX201LA|TX201LA",
			"Asus|ASUS Transformer AiO P1801|P1801-T|ASUS Tablet P1801-T",
			"Asus|ASUS Transformer AiO P1802|P1802-T|ASUS Tablet P1802-T",
			"Asus|ASUS Transformer Book Trio|TX201LA|TX201LA",
			"Asus|ASUS Transformer Pad|K010|K010",
			"Asus|ASUS Transformer Pad|K010_1|K010",
			"Asus|ASUS Transformer Pad Infinity|TF700KL|ASUS Transformer Pad TF700KL",
			"Asus|ASUS Transformer Pad TF300TG|TF300TG|ASUS Transformer Pad TF300TG",
			"Asus|ASUS Transformer Pad TF300TL|TF300TL|ASUS Transformer Pad TF300TL",
			"Asus|ASUS Transformer Pad TF502T|TF502T|ASUS Transformer Pad TF502T",
			"Asus|ASUS Transformer Pad TF600T|TF600T|ASUS Transformer Pad TF600T",
			"Asus|ASUS ZenWatch|anthias|ASUS ZenWatch",
			"Asus|Asus Fonepad Note 6|K00G|K00G",
			"Asus|Asus MeMo Pad 7|K012_2|K012_2",
			"Asus|Asus Pad TF303CL|K014|K014",
			"Asus|Asus Transformer Pad|K018|K018",
			"Asus|Asus Transformer Pad TF303CL|K014|K014",
			"Asus|Cube|asus_google_cube|asus_google_cube",
			"Asus|Eee Pad|EeePad|Transformer TF101",
			"Asus|Eee Pad|EeePad|Transformer TF101G",
			"Asus|Eee Pad Slider|SL101|Slider SL101",
			"Asus|Eee Pad TF101|TF101|TF101",
			"Asus|Eee Pad TF101|TF101|Transformer TF101",
			"Asus|Eee Pad TF101-WiMAX|TF101-WiMAX|TF101-WiMAX",
			"Asus|Eee Pad Transformer|TF101|Transformer TF101",
			"Asus|Eee Pad Transformer Prime|TF201|TF201",
			"Asus|Eee Pad Transformer Prime|TF201|Transformer Prime TF201",
			"Asus|Eee Pad Transformer Prime|TF201|Transformer TF201",
			"Asus|EeePad Slider SL101|SL101|Slider SL101",
			"Asus|Fonepad 7|K00Z|K00Z",
			"Asus|Fonepad 7|K012|K012",
			"Asus|Fonepad 7|K01N_1|K01N",
			"Asus|Fonepad 7|K01N_2|K01N",
			"Asus|Fonepad 7 (FE375CG)|K019_1|K019",
			"Asus|Fonepad 7 (FE375CXG)|K019_3|K019",
			"Asus|Fonepad 7 (FE375CXG)|K019_4|K019",
			"Asus|Fonepad 7 LTE|K01Q|K01Q",
			"Asus|Fonepad 7(FE375CXG)|K019_2|K019",
			"Asus|Fonepad HD7|K00E|K00E",
			"Asus|K00F|K00F|K00F",
			"Asus|K00L|K00L|K00L",
			"Asus|ME173X|ME173X|ME173X",
			"Asus|ME302KL|ME302KL|ME302KL",
			"Asus|MeMO PAD|me172v|ME172V",
			"Asus|MeMO Pad 7|K013|K013",
			"Asus|MeMO Pad 7|K013C|K013C",
			"Asus|MeMO Pad 7|K013_1|K013",
			"Asus|MeMO Pad 7|K017|K017",
			"Asus|MeMO Pad 7|K01A|K01A",
			"Asus|MeMO Pad 7 LTE|K00X|ASUS MeMO Pad 7",
			"Asus|MeMO Pad 7 ME572C|K007|K007",
			"Asus|MeMO Pad 8|K011|K011",
			"Asus|MeMO Pad 8|K011_1|K011",
			"Asus|MeMO Pad ME103K|K01E_1|K01E",
			"Asus|MeMO Pad ME103K|K01E_2|K01E",
			"Asus|MeMO pad 8 AST21|K015|AST21",
			"Asus|MeMO pad 8 ME581C|K01H|K01H",
			"Asus|MeMO pad 8 ME581CL|K015|K015",
			"Asus|MeMo Pad HD 7|K00U|K00U",
			"Asus|Memo Pad 7|K01U_1|K01U",
			"Asus|Memo Pad 7|K01U_2|K01U",
			"Asus|Memo Pad HD7 Dual SIM|ASUS-K00S|ASUS K00S",
			"Asus|Nexus 7 (2012)|grouper|Nexus 7",
			"Asus|Nexus 7 (2012)|tilapia|Nexus 7",
			"Asus|Nexus 7 (2013)|deb|Nexus 7",
			"Asus|Nexus 7 (2013)|flo|Nexus 7",
			"Asus|Nexus Player|fugu|Nexus Player",
			"Asus|Nuvifone|a50|Garmin-Asus A50",
			"Asus|PadFone|PadFone|PadFone",
			"Asus|PadFone 2|A68|PadFone 2",
			"Asus|PadFone E|ASUS-T008|PadFone T008",
			"Asus|PadFone Infinity|A80|PadFone Infinity",
			"Asus|PadFone Infinity|ASUS-A80|PadFone Infinity",
			"Asus|PadFone Infinity Lite|ASUS-A80|PadFone Infinity",
			"Asus|PadFone Mini|ASUS-T00C|PadFone T00C",
			"Asus|PadFone S|ASUS_T00N|ASUS_T00N",
			"Asus|PadFone T004|ASUS-A86|PadFone T004",
			"Asus|PadFone Xmini|ASUS-T00S|ASUS PadFone X mini",
			"Asus|PadFone mini|ASUS-T00C|PadFone T00C",
			"Asus|PadFone mini|ASUS_T00E|ASUS_T00E",
			"Asus|PadFone-T004|ASUS-A86|PadFone T004",
			"Asus|Padfone S|ASUS_T00N|ASUS_T00N",
			"Asus|Padfone X|ASUS-T00D|ASUS PadFone X",
			"Asus|Padfone mini|ASUS-T00S|ASUS_T00T",
			"Asus|T101TA|T10xTA|T10xTA",
			"Asus|TF700T|TF700T|ASUS Pad TF700T",
			"Asus|TF700T|TF700T|ASUS Transformer Pad TF700T",
			"Asus|TX201LAF|TX201LAF|TX201LAF",
			"Asus|Trans Pad Infinity|K00C|K00C",
			"Asus|Transformer 10(TF103CE)|K010E|K010E",
			"Asus|Transformer Pad|K00C|K00C",
			"Asus|Transformer Pad|K010_3|K010",
			"Asus|Transformer Pad Infinity|K00C|K00C",
			"Asus|Transformer Pad TF303K|K01B|K01B",
			"Asus|Transformer TF101G|TF101G|Transformer TF101G",
			"Asus|ZenFone 4|ASUS_T00I|ASUS_T00I",
			"Asus|ZenFone 4|ASUS_T00I|ASUS_T00I-D",
			"Asus|ZenFone 4|ASUS_T00Q|ASUS_T00Q",
			"Asus|ZenFone 5|ASUS_T00F|ASUS_T00F",
			"Asus|ZenFone 5|ASUS_T00F1|ASUS_T00F",
			"Asus|ZenFone 5|ASUS_T00J|ASUS_T00J",
			"Asus|ZenFone 5|ASUS_T00J1|ASUS_T00J",
			"Asus|ZenFone 5|ASUS_T00K|ASUS_T00K",
			"Asus|ZenFone 5 LTE|ASUS_T00P|ASUS_T00P",
			"Asus|ZenFone 6|ASUS_T00G|ASUS_T00G",
			"Asus|ZenFone 6|ASUS_Z002|ASUS_Z002",
			"Asus|ZenFone C|ASUS_Z007|ASUS_Z007",
			"AtGames-zooti|ZOOTI PAD ZT-701|ZT-701|ZT-701",
			"Auchan|Selecline X35T|X35T|X35T",
			"Audiovox|Audiovox T752 Tablet|AD7L|T752",
			"Audiovox|Audiovox T852 Tablet|MID30801|T852",
			"Avoca|STB7012|STB7012|STB7012",
			"Avoca|STB7013|AVOCA|STB7013",
			"Avoca|STB8098|STB8098|STB8098",
			"Avoca|STB9097|STB9097|STB9097",
			"BYD|DynaVox T10|T10|T10",
			"BYD|FarEastone Smart 502|Smart502|Smart 502",
			"BYD|INHON G3|G3|G3",
			"BYD|K-Touch E815|E815|E815",
			"BYD|M601|M601|M601",
			"BYD|Moii E996+|E996plus|moii E996+",
			"BYD|PT452E|PT452E|Digma Linx 4.5 PT452E",
			"BYD|Prestigio PAP5430|PAP5430|PAP5430",
			"BYD|SI4301|SI4301|SI4301",
			"BYD|SOLO S450|S450|S450",
			"BYD|SP355AWG|msm7627a_sku3|SP355AWG",
			"BYD|Smart 401|tianyu72_wet_jb3|Smart 401",
			"BYD|Smart 402|hawaii_garnet_c_w68tk|Smart402",
			"BYD|Solo S350|S350|S350",
			"BYD|WISKY W032I|W032I|W032I",
			"BYD|X910|X910|XOLO X910",
			"BYD|Zippers|Zippers|Vexia Zippers",
			"Barnes and Noble|NOOK\\xc2\\xae HD|hummingbird|BNTV400",
			"Barnes and Noble|NOOK\\xc2\\xae HD+|ovation|BNTV600",
			"Beeline|Beeline Tab|Beeline_Tab|Beeline Tab",
			"Beeline|Beeline Tab 2|Tab_2|Tab_2",
			"BenQ|A3|A3|A3",
			"BenQ|Agora 4G|Agora_4G|Agora 4G",
			"BenQ|BenQ A3|A3|A3",
			"BenQ|BenQ A3c|BenQ_A3c|BenQ A3c",
			"BenQ|BenQ B50|B50|B50",
			"BenQ|BenQ B502|B502|B502",
			"BenQ|BenQ F3|F3|F3",
			"BenQ|BenQ F4|benq_a3s|BenQ A3s",
			"BenQ|BenQ F5|BENQ_F5|BenQ F5",
			"BenQ|BenQ F5|F5|BenQ F5",
			"BenQ|BenQ F5_06|F5|F5_06",
			"BenQ|BenQ F5_13|F5|F5_13",
			"BenQ|BenQ F5_15|F5|F5_15",
			"BenQ|BenQ F5_16|F5|F5_16",
			"BenQ|BenQ F5_19|F5|F5_19",
			"BenQ|BenQ T3|T3_17A|BenQ T3",
			"BenQ|BenQ T3_08|T3_17A|BenQ T3",
			"BenQ|JM-250|JM-250|JM-250",
			"BenQ|Kogan 4G+|Agora_4G_PLUS|Agora 4G+",
			"BenQ|PT2200|PT2200|PT2200",
			"BenQ|T3|T3|BenQ T3",
			"Best Buy|MG103|K010|K010",
			"Bigben|BB8252|Bigben-TAB|BB8252",
			"Bigben|GAMETAB-ONE|GAMETAB-ONE|GAMETAB-ONE",
			"Binatone|SMART64|SMART64|SMART64",
			"Binatone|Smart 63|SMART63|SMART63",
			"Binatone|Smart 66|SMART66|SMART66",
			"Bitmore|Bitmore Tab870|Tab870|Tab870",
			"Bitmore|GTAB700|GTAB700|NID_7010",
			"Bitmore|GTAB900|GTAB900|S952",
			"Bitmore|LUXYA MID704DC Tablet / Bitmore Tab770|MY7317P|Tab770",
			"Blaupunkt|Endeavour101|Blaupunkt|Endeavour101",
			"Bluedot|BLUEDOT BNT-700K|BNT-700K|BNT-700K",
			"Booken|Cybook Tablet|cybtt10_bk|Cybook-Tablet",
			"Borqs|Falcon|falcon|falcon",
			"Bouygues Telecom|Bbox Miami|HMB4213H|BouygtelTV",
			"Bouygues Telecom|Bouygues Telecom Bs 451|Bs_451|Bs 451",
			"Brightstar|M8046IU|M8047IU|M8047IU",
			"British Telecom|HomeSmartphone d800|d800|BT Home SmartPhone S II",
			"Bunbungame|MiSS|MiSS|MiSS",
			"Bungbungame|KALOS|FG6Q_N|KALOS",
			"Bush|MyTablet 79|ac79bu|BUSH 7.85 TABLET",
			"C&M|Smart TV II|stb_catv_cnm|SX930C-CC",
			"CAT|CAT B15|TOUGH|B15",
			"CAT|CAT B15Q|B15Q|B15Q",
			"CHN TELECOM|CHN-UCAN MJ|lefhd|CHN-UCAN MJ",
			"CJ HelloVision|Hello TV Smart|SX930C_CJ|SX930C_CJ",
			"CMCC|M811|m811|M811",
			"CMCC|M811|m811hk|M811",
			"COMIO|AD7003|AD7003|AD7003",
			"Camelus|Camelus L10|TR-10HBT|L10",
			"Camelus|Camelus L7|TR-7U|L7",
			"Candor|PHS-601|phs601|PHS-601",
			"Carrefour|CT1000|Carrefour|CT1000",
			"Carrefour|Carrefour CT1020W|M1010FP|CT1020W",
			"Carrefour|Carrefour CT820|M878LP|CT820",
			"Casper|TA80CA2|TA80CA2_1|TA80CA2",
			"Casper|VIA_T41|VIA_T41|VIA_T41",
			"Cat|B15Q|CatB15Q|B15Q",
			"Cat|S50|CatS50|S50",
			"Cellon|A27|C8680|A27",
			"Cellon|A5801|C8680|A5801",
			"Cellon|Alpha Ice|M8047|Alpha Ice",
			"Cellon|C3668|C3668|C3668",
			"Cellon|CI|Dream-C1|CI",
			"Cellon|CJ-1984|C8690|C8690",
			"Cellon|CJ-1984|C8690|CJ-1984",
			"Cellon|CJ-1984|C8690|ELITE 4.7 HD",
			"Cellon|EI|Dream-E1|EI",
			"Cellon|EIII|Dream-E3|EIII",
			"Cellon|EK-8680|C8680|EK-8680",
			"Cellon|Explosion|C8690|Explosion",
			"Cellon|HW-W900|C8690|HW-W900",
			"Cellon|HWW820|C8680|HW-W820",
			"Cellon|Houston|C8680|Houston",
			"Cellon|ICON|ICON|ICON",
			"Cellon|IQ 1.1|IQ1-1|i-mobile IQ1-1",
			"Cellon|Linx PS474S|C8690|Linx PS474S",
			"Cellon|MTC SMART Run|SMART_Run|MTC SMART Run",
			"Cellon|Mach|C8680|Mach",
			"Cellon|Newman N2|C8690|Newman N2",
			"Cellon|Q|M8047XT|Q",
			"Cellon|S21|S21|Q-Smart S21",
			"Cellon|SI|Dream-S1|SI",
			"Cellon|SM55|C8660|C8660",
			"Cellon|SM55|C8660|SM55",
			"Cellon|Starmobile KNIGHT|KNIGHT|Starmobile KNIGHT",
			"Cellon|VEON_C8680|C8680|VEON_C8680",
			"Cellon|i-STYLE7|c8669SA|i-mobile i-STYLE 7",
			"Cellon|i-STYLE7A|c8669SA|i-mobile i-STYLE 7A",
			"Cellon|i-mobile IQ X|M8047SA|i-mobile IQ X",
			"Cellon|i-mobile IQ X|M8047SA|i-mobile IQ XA",
			"Cellon|i-mobile IQ X2A|M8050SA|i-mobile IQ X2",
			"Cellon|i-mobile IQ X2A|M8050SA|i-mobile IQ X2A",
			"Cellon|iris pro 30|pro30|iris pro 30",
			"Cellon|starTIM1|starTIM1|starTIM1",
			"Cherry Mobile|Cherry W900 LTE|al7|W900 LTE",
			"Cisco|Cisco CP-DX650|CP-DX650|CP-DX650",
			"Cisco|Cisco Desktop Collaboration Experience DX70|CP-DX70|CP-DX70",
			"Cisco|Cisco Desktop Collaboration Experience DX80|CP-DX80|CP-DX80",
			"Clementoni|Clempad 4.4 Basic Special|Clempad2_special|Clempad2_special",
			"Clementoni|Clempad 4.4 Plus|Clempad2_plus|Clempad2_plus",
			"Clementoni|Clempad 4.4 XL|Clempad2_XL|Clempad2_XL",
			"Clementoni|My first Clempad 4.4 Basic Special|MFC2_Special|MFC2_Special",
			"Clementoni|My first Clempad 4.4 Plus|MFC2_Plus|MFC2_Plus",
			"ClickN Kids|ClickN Kids|g04e|CK07T",
			"Compal|TANK 4.5|ka09|TANK 4.5",
			"Condor|C1+|C1Plus|C1+",
			"Condor|C_5|u970_5|C-6",
			"Condor|HS-U939|u939e_6|HS-U939",
			"Coolpad|5860E|CP5860E|5860E",
			"Coolpad|801EM|801EM|801EM",
			"Coolpad|8970L|8970L|Coolpad 8970L",
			"Coolpad|Bs501|cp8870u|Bs 501",
			"Coolpad|CM001|CoolpadCM001|Coolpad CM001",
			"Coolpad|CP5108|msm7627a_a8_5108_new|Coolpad 5108",
			"Coolpad|CP5210A|msm7627a_d7_5210a|Coolpad 5210A",
			"Coolpad|CP5310|cp5310|Coolpad 5310",
			"Coolpad|CP5660S|5560S|5560S",
			"Coolpad|CP5952|Coolpad5952|Coolpad 5952",
			"Coolpad|CP8021|Coolpad8021|Coolpad 8021",
			"Coolpad|CP8089|CP8089|Coolpad 8089",
			"Coolpad|CP8089Q|CP8089Q|Coolpad 8089Q",
			"Coolpad|CP8295M|8295M|Coolpad8295M",
			"Coolpad|CP8735|8735|Coolpad 8735",
			"Coolpad|CP8750|8750|Coolpad8750",
			"Coolpad|CP9190L|Coolpad9190L|Coolpad 9190L",
			"Coolpad|CP9250L|Coolpad9250L|Coolpad 9250L",
			"Coolpad|CP9970|9970|Coolpad 9970",
			"Coolpad|CPA520|CoolpadA520|Coolpad A520",
			"Coolpad|CPT1|CoolpadT1|Coolpad T1",
			"Coolpad|Coolmini|Coolmini|Coolmini",
			"Coolpad|Coolpad 3700A|3700A|CP3700A",
			"Coolpad|Coolpad 5109|Coolpad5109|Coolpad 5109",
			"Coolpad|Coolpad 5200|Coolpad5200|Coolpad 5200",
			"Coolpad|Coolpad 5216D|5216D|Coolpad 5216D",
			"Coolpad|Coolpad 5217|msm8610|Coolpad 5217",
			"Coolpad|Coolpad 5218S|msm8x25q_a5y_5218s|Coolpad 5218S",
			"Coolpad|Coolpad 5219|5219|Coolpad 5219",
			"Coolpad|Coolpad 5219_C00|5219_C00|Coolpad 5219_C00",
			"Coolpad|Coolpad 5891|Coolpad5891|Coolpad 5891",
			"Coolpad|Coolpad 5891S|Coolpad5891S|Coolpad 5891S",
			"Coolpad|Coolpad 5951|Coolpad5951|Coolpad 5951",
			"Coolpad|Coolpad 7061|CP7061|Coolpad 7061",
			"Coolpad|Coolpad 7230S|Coolpad7230S|Coolpad 7230S",
			"Coolpad|Coolpad 7232|CP7232|Coolpad 7232",
			"Coolpad|Coolpad 7251|Coolpad7251|Coolpad 7251",
			"Coolpad|Coolpad 7270|Coolpad7270|Coolpad 7270",
			"Coolpad|Coolpad 7270I|7270I|Idea ULTRA Pro",
			"Coolpad|Coolpad 7275|CP7275|Coolpad 7275",
			"Coolpad|Coolpad 7295A|Coolpad7295A|Coolpad 7295+",
			"Coolpad|Coolpad 7295A|Coolpad7295A|Coolpad 7295A",
			"Coolpad|Coolpad 7296|Coolpad7296|Coolpad 7296",
			"Coolpad|Coolpad 7620L|Coolpad7620L|Coolpad 7620L",
			"Coolpad|Coolpad 8079|Coolpad8079|Coolpad 8079",
			"Coolpad|Coolpad 8122|Coolpad8122|Coolpad 8122",
			"Coolpad|Coolpad 8198W|Coolpad8198W|Coolpad 8198W",
			"Coolpad|Coolpad 8297D|Coolpad8297D|Coolpad 8297D",
			"Coolpad|Coolpad 8670|Coolpad8670|Coolpad 8670",
			"Coolpad|Coolpad 8702|Coolpad8702|Coolpad 8702",
			"Coolpad|Coolpad 8729|Coolpad8729|Coolpad 8729",
			"Coolpad|Coolpad 8730L|Coolpad8730L|Coolpad 8730L",
			"Coolpad|Coolpad 8732|Coolpad8732|Coolpad 8732",
			"Coolpad|Coolpad 8736|8736|Coolpad 8736",
			"Coolpad|Coolpad 8971|8971|Coolpad 8971",
			"Coolpad|Coolpad 9150W|Coolpad9150W|Coolpad 9150W",
			"Coolpad|Coolpad 9190_T00|Coolpad9190_T00|Coolpad 9190_T00",
			"Coolpad|Coolpad 9970L|9970L|Coolpad 9970L",
			"Coolpad|Coolpad 9976A|Coolpad9976A|Coolpad 9976A",
			"Coolpad|Coolpad 9976T|Coolpad9976T|Coolpad 9976T",
			"Coolpad|Coolpad Flo|7560T|Coolpad Flo",
			"Coolpad|Coolpad5218D|msm7627a_a5y_5218d|Coolpad_5218D",
			"Coolpad|Coolpad5315|msm8610_s10_cp5315|Coolpad 5315",
			"Coolpad|Coolpad5872|Coolpad5872|Coolpad 5872",
			"Coolpad|Coolpad5879T|Coolpad5879T|Coolpad 5879T",
			"Coolpad|Coolpad5891Q|Coolpad5891Q|Coolpad 5891Q",
			"Coolpad|Coolpad5892|Coolpad5892|Coolpad 5892",
			"Coolpad|Coolpad5950|Coolpad5950|Coolpad 5950",
			"Coolpad|Coolpad5950T|Coolpad5950T|Coolpad 5950T",
			"Coolpad|Coolpad7270_W00|Coolpad7270_W00|Coolpad 7270_W00",
			"Coolpad|Coolpad7295C|Coolpad7295C|Coolpad 7295C",
			"Coolpad|Coolpad7295C_C00|Coolpad7295C_C00|Coolpad 7295C_C00",
			"Coolpad|Coolpad7295I|Coolpad7295I|Spice Mi-515",
			"Coolpad|Coolpad7295T|Coolpad7295T|Coolpad 7295T",
			"Coolpad|Coolpad7296S|Coolpad7296S|Coolpad 7296S",
			"Coolpad|Coolpad7298A|Coolpad7298A|Coolpad 7298A",
			"Coolpad|Coolpad7298D|Coolpad7298D|Coolpad 7298D",
			"Coolpad|Coolpad7320|Coolpad7320|Coolpad 7320",
			"Coolpad|Coolpad7920|7920|Coolpad7920",
			"Coolpad|Coolpad801ES|801ES|801ES",
			"Coolpad|Coolpad8085Q|Coolpad8085Q|Coolpad8085Q",
			"Coolpad|Coolpad8198T|8198T|Coolpad8198T",
			"Coolpad|Coolpad8297|Coolpad8297|Coolpad 8297",
			"Coolpad|Coolpad8297W|Coolpad8297W|Coolpad 8297W",
			"Coolpad|Coolpad8705|Coolpad8705|Coolpad 8705",
			"Coolpad|Coolpad8720L|Coolpad8720L|Coolpad 8720L",
			"Coolpad|Coolpad8908|8908|Coolpad8908",
			"Coolpad|Forward_EVOLVE|Forward_EVOLVE|Forward_EVOLVE",
			"Coolpad|Idea ULTRA|Coolpad7295S|Idea ULTRA",
			"Coolpad|Karbonn Titanium S5 Plus|Karbonn|Karbonn Titanium S5 Plus",
			"Coolpad|MEDION P5001|MEDION|MEDION P5001",
			"Coolpad|MEDION P5001|P5001|MEDION P5001",
			"Coolpad|Mtag 353|msm7627a_ea92_mts|Mtag 353",
			"Coolpad|STARADDICT III|cp8861u|STARADDICT III",
			"Coolpad|Spice Mi-496|Coolpad7268I|Spice Mi-496",
			"Coolpad|Vodafone Smart 4G|cp8860u|Vodafone Smart 4G",
			"Coolpad|VodafoneSmart4|VodafoneSmart4|Vodafone Smart 4",
			"Coolpad|VodafoneSmart4turbo|VodafoneSmart4turbo|Vodafone Smart 4 turbo",
			"Coolpad|cp7236|msm8610_w7_cp7236|Coolpad 7236",
			"Crosscall|ODYSSEY|ODYSSEY3G|ODYSSEY",
			"DEA Factory|KTAB|KTAB|KTAB",
			"DIFRNCE|DIT702201|DIT702201|NID_7010",
			"Danew|DANEW Dslide971DC|Dslide971DC|Dslide971DC",
			"Danew|DSlide 710|Dslide710|Dslide710",
			"Danew|Danew DSlide 750|Danew_DSlide750|DSlide750",
			"Danew|Danew Dslide 1013|DSlide1013|DSlide 1013",
			"Dell|Cloud Connect|capri_wyse|CS-1A13",
			"Dell|Streak|streak|001DL",
			"Dell|Streak|streak|Dell M01M",
			"Dell|Streak|streak|Dell Streak",
			"Dell|Venue 7|LW|Venue 7 3741",
			"Dell|Venue 7|Venue7|Venue 7 3740",
			"Dell|Venue 7|Venue7|Venue7 3740",
			"Dell|Venue 7|Venue7|Venue7 3740 LTE",
			"Dell|Venue 7|thunderbird|Venue 7 3730",
			"Dell|Venue 7|thunderbird|Venue 7 HSPA+",
			"Dell|Venue 8|BB|Venue 8 7840",
			"Dell|Venue 8|Venue8|Venue 8 3840",
			"Dell|Venue 8|Venue8|Venue8 3840",
			"Dell|Venue 8|Venue8|Venue8 3840 LTE",
			"Dell|Venue 8|yellowtail|Venue 8 3830",
			"Dell|Venue 8|yellowtail|Venue 8 HSPA+",
			"DigiLand|DL1010Q|DL1010Q|DL1010Q",
			"DigiLand|DL700D|DL700D|DL700D",
			"DigiLand|DL701Q|DL701Q|DL701Q",
			"DigiLand|DL785D|DL785D|PLT7803G",
			"Digilite|TA70CA1|TA70CA1|TA70CA1",
			"Digilite|TA70CA2|TA70CA2_1|TA70CA2",
			"Digilite|TA70CA3|TA70CA3_1|TA70CA3",
			"Digilite|TR10CD1|TR10CD1_5|TR10CD1",
			"Digilite|TR10CS1|TR10CS1_3|TR10CS1",
			"Diginnos|Diginnos DG-D07S/GP|DG-D07SGP|DG-D07SGP",
			"Digital2|Digital2 Deluxe|nuclear-D2_Deluxe|Digital2-Deluxe",
			"Digital2|Digital2 Platinum|D2-963G|Digital2 Platinum",
			"Digital2|Digital2 Plus|Digital2Plus|Digital2 Plus",
			"DigitalStream|DigitalStream DWA1015D Tablet|DWA1015D|DWA1015D",
			"Digitron|i-Buddie|TR10CD1_10|TR10CD1",
			"Digitron|i_Buddie|TR10CS1_12|TR10CS1",
			"Digix|TAB-740_G|TAB-740_G|TAB-740_G",
			"Ditigron|i-Buddie|TR10CD1_11|TR10CD1",
			"Dopo|DOPO EM63 Tablet|EM63|EM63",
			"Dopo|DOPO GS-918|GS_918|GS-918",
			"Dopo|DP7856K|DP7856K|DP7856K",
			"Dopo|DPA23D|polaris-wifionly|DPA23D",
			"Dopo|DPM1081|RCT6203W46|DPM1081",
			"Dopo|DPM7827|RCT6773W22|DPM7827",
			"Doro|Doro Liberto 820 Mini|820Mini|Doro Liberto 820 Mini",
			"Doro|Liberto 810|Liberto810|Doro Liberto 810",
			"Doro|Liberto 810|Liberto810|Doro Liberto 810-orange",
			"Doro|Liberto 810|Liberto810_S12|Doro Liberto 810",
			"Doro|Liberto 820|Liberto820|Doro Liberto 820",
			"Doro|Liberto 820 Mini|820Mini|Doro Liberto 820 Mini",
			"Dreamax|DMX-ST7A|DMX-ST7A|DMX-ST7A",
			"EBN|NEBUPOS3|UY3A|NEBUPOS3",
			"ECS|RealPad MA7BX2|RealPad_MA7BX2_1|RealPad MA7BX2",
			"ECS|TC69CA2|TC69CA2_1|TC69CA2",
			"ECS|TR10CS1|TR10CS1_8|TR10CS1",
			"ECS|TR10RS1|TR10RS1_2|TR10RS1",
			"EG936D|Hisense|EG936D|HS-EG936D",
			"EMTEC|EMTEC F400|f400|f400",
			"ENTITY|GDENTMY7317|GDENTMY7317|GDENTMY7317",
			"ENVIZEN DIGITAL|ENVIZEN DIGITAL V917G|GK-MID9022|V917G",
			"ENVIZEN DIGITAL|V100MD|V100MD|V100MD",
			"Edcon|Platiunm vp.1|VerssedVP1|Verssed VP1",
			"Edcon|Platiunm vp.2|VerssedVP2|Verssed VP2",
			"Ematic|EGD078|EGD078|EGD078",
			"Ematic|EGD170|EGD170|Ematic EGD170",
			"Ematic|EGD172|EGD172|EGD172",
			"Ematic|EGD213|EGD213|EGD213",
			"Ematic|EGQ337|EGQ337|EGQ337",
			"Ematic|EGQ780|EGQ780|EGQ780",
			"Ematic|EMATICEGS109|EGS109|EGS109",
			"Ematic|Ematic EGD103|EGD103|EGD103",
			"Ematic|Ematic EGQ307|EGQ307|EGQ307",
			"Ematic|Ematic EGQ327M|EGQ327M|EGQ327M",
			"Ematic|Ematic EGS017|EGS017|EGS017",
			"Ematic|Ematic ETH102|ETH102|ETH102",
			"Ematic|FunTab 2|FunTab2|FunTab2",
			"Ematic|Sprout Channel Cubby|Cubby|SproutChannelCubby",
			"Emdoor|EM_I8180|em_i8180|em_i8180",
			"Emerson|Emerson EM749/748|EM749_748|EM749_748",
			"Enspert|CINK KING|s9081|CINK KING",
			"Enspert|CINK KING|s9081|Cynus T2",
			"Enspert|CINK KING|s9081|IMD501",
			"Enspert|CINK KING|s9081|freebit PandA",
			"Enspert|CINK SLIM|s8073|CINK SLIM",
			"Enspert|CINK SLIM|s8073|Fly_IQ442",
			"Enspert|CINK SLIM|s8073|WIKO-CINK SLIM",
			"Enspert|Carrefour CT1010|Carrefour|CT1010",
			"Enspert|Cynus F3|s7511|Cynus F3",
			"Enspert|DARKSIDE|s9301|Cinema",
			"Enspert|DARKSIDE|s9301|DARKSIDE",
			"Enspert|Lazer X40E|X40E|X40E",
			"Enspert|MG|MGA930|A930",
			"Enspert|STAIRWAY|s9111|STAIRWAY",
			"Enspert|WIKO CINK PEAX|s9091|CINK PEAX",
			"Enspert|WIKO CINK PEAX|s9091|EverClassic",
			"Enspert|WIKO CINK+|s7505|CINKPLUS",
			"Enspert|WIKO SUBLIM|s9070|SUBLIM",
			"Enspert|Wiko CINK FIVE|s9201|CINK FIVE",
			"Enspert|Wiko CINK FIVE|s9201|Cynus T5",
			"Enspert|orion|orion|NGM Orion",
			"Enspert|vanitysmart|vanitysmart|NGM Vanity Smart",
			"EssentielB|EssentielB Smart\\'TAB 7800|fiber-785q6|Smart\\'Tab_7800",
			"EssentielB|EssentielB Smart\\'TAB 9001|fiber-smart|Smart\\'TAB 9001",
			"Etuline|ETL-S4521|u939e_2|ETL-S4521",
			"Etuline|ETL-S5042|u970_4|ETL-S5042",
			"EveryPad2|Lenovo|A3500F|EveryPad2",
			"Explay|4Game|4Game|4Game",
			"Explay|ATV|ATV|ATV",
			"Explay|Air|Air|Air",
			"Explay|Atom|Atom|Atom",
			"Explay|Bit|Bit|Bit",
			"Explay|Cosmic|Cosmic|Cosmic",
			"Explay|Craft|Craft|Craft",
			"Explay|Easy|Easy|Easy",
			"Explay|Favorite|Favorite|Favorite",
			"Explay|Fog|Fog|Fog",
			"Explay|Fresh|Fresh|Fresh",
			"Explay|Gravity|T4728|Gravity",
			"Explay|Hit|Hit|Hit",
			"Explay|Indigo|Indigo|Indigo",
			"Explay|Joy TV|Joy_TV|Joy_TV",
			"Explay|Light|Light|Light",
			"Explay|M1 Plus|Fog|M1_Plus",
			"Explay|Onyx|Onyx|Onyx",
			"Explay|Phantom|Phantom|Phantom",
			"Explay|Prime|Prime|Prime",
			"Explay|Pulsar|Pulsar|Pulsar",
			"Explay|Rio Play|RioPlay|RioPlay",
			"Explay|Shine|T5246|Shine",
			"Explay|Tornado|Tornado|Tornado",
			"Explay|Tornado 3G|Tornado3G|Tornado3G",
			"Explay|Vega|Vega|Vega",
			"FBC|M24IS810|MX24|TDA02",
			"FBC|M24IS820|fst01ms|M24IS820",
			"FaceVsion|FaceVsion OC1020A|OC1020A|FV-FG6",
			"FarEasTone|Smart 403|msm8226|Smart 403",
			"FarEasTone|Smart 505|Smart505|Smart 505",
			"FarEasTone|Smart501|Smart501|Smart501",
			"FarEasTone|Smart503|MC2|Smart503",
			"Fly|Fly ERA Nano 7|Fly_IQ4407|IQ4407",
			"Fly|Fly EVO Chic 1|IQ4405_Quad|Fly IQ4405_Quad",
			"Fly|Fly Ego Art 2|Fly_Ego_Art_2|Fly Ego Art 2",
			"Fly|Fly Era Energy 1|IQ4502_Quad|IQ4502 Quad",
			"Fly|Fly Era Life 2|IQ456|IQ456",
			"Fly|Fly Era Nano 6|Era_Nano_6|IQ4406",
			"Fly|Fly Era Nano 8|IQ4400|IQ4400",
			"Fly|Fly Era Nano 9|IQ436i|IQ436i",
			"Fly|Fly Era Style 3|IQ4415_Quad|IQ4415 Quad",
			"Fly|Fly Evo Chic 2|IQ459|IQ459",
			"Fly|Fly Evo Chic 2|IQ459|IQ459 Quad",
			"Fly|Fly Evo Energy4|IQ4501|Evo Energy4",
			"Fly|Fly Evo Tech 3|IQ4414_Quad|IQ4414 Quad",
			"Fly|Fly IQ434|IQ434|IQ434",
			"Fly|Fly IQ4490i|IQ4490i|IQ4490i",
			"Fly|Fly IQ4503|IQ4503|FLY IQ4503",
			"Fly|Fly IQ4503 Quad|IQ4503_Quad|IQ4503 Quad",
			"Fly|Fly IQ4504 Quad|IQ4504_Quad|IQ4504 Quad",
			"Fly|Fly IQ4505|IQ4505|Fly IQ4505",
			"Fly|Fly IQ4505 Quad|IQ4505_Quad|IQ4505 Quad",
			"Fly|Fly IQ4512 Quad|IQ4512_Quad|IQ4512 Quad",
			"Fly|Fly IQ4513|IQ4513|IQ4513",
			"Fly|Fly IQ4513 Octa|IQ4513_Octa|IQ4513 Octa",
			"Fly|Fly IQ4514 Quad|IQ4514_Quad|Fly IQ4514",
			"Fly|Fly IQ4514 Quad|IQ4514_Quad|Fly IQ4514 AF",
			"Fly|Fly IQ4514 Quad|IQ4514_Quad|IQ4514 Quad",
			"Fly|Fly IQ4515 Quad|IQ4515_Quad|IQ4515 Quad",
			"Fly|Fly IQ4516 Octa|IQ4516_Octa|IQ4516",
			"Fly|Fly IQ4516 Octa|IQ4516_Octa|IQ4516 Octa",
			"Fly|Fly IQ458 Quad Evo Tech 2|IQ458|IQ458 Quad",
			"Fly|Fly IQ4602|Fly_IQ4602|Fly IQ4602",
			"Fly|Fly IQ4602|IQ4602_Quad|IQ4602 Quad",
			"Fly|Fly Iris|FLY_IQ4400|FLY_IQ4400_AF",
			"Fly|Fly Phoenix 2|IQ4410i|Phoenix 2",
			"Fly|Flylife Connect 10.1|Flylife_Connect_101_3G_2|Flylife Connect 10.1 3G 2",
			"Fly|Flylife Connect 7 3G 2|Flylife7|Flylife Connect 7 3G 2",
			"Fly|Flylife Connect 7.85 3G 2|Flylife|Flylife Connect 7.85 3G 2",
			"Fly|IQ4401|IQ4401|IQ4401",
			"Fly|IQ4409 Quad|IQ4409_Quad|IQ4409 Quad",
			"Fly|IQ4413 Quad|IQ4413_Quad|IQ4413_Quad",
			"Fly|IQ4415|IQ4415|IQ4415",
			"Fly|IQ4416|IQ4416|IQ4416",
			"Fly|IQ4417 Quad|IQ4417_Quad|IQ4417 Quad",
			"Fly|IQ4418|IQ4418|Fly IQ4418 AF",
			"Fly|IQ4418|IQ4418|IQ4418",
			"Fly|IQ4601|IQ4601|IQ4601",
			"Fly|Tornado One|IQ4511_Octa|FLY IQ4511",
			"Fly|Tornado One|IQ4511_Octa|IQ4511 Octa",
			"Fly|Tornado One|IQ4511_Octa|IQ4511 TR",
			"Foxconn|A5|A5|A5",
			"Foxconn|CSL Spice MI700|mi700|CSL_Spice_MI700",
			"Foxconn|CSL Spice MI700|mi700|Spice_MI700",
			"Foxconn|Casper_VIA_A6108|SI1|Casper_VIA_A6108",
			"Foxconn|Commtiva N700|n700|Commtiva-N700",
			"Foxconn|Dolphin 70e Black|dblack|Dolphin 70e Black",
			"Foxconn|InFocus IN260|IN260|IN260",
			"Foxconn|MUSN COUPLE|couple|COUPLE",
			"Foxconn|SH530U|sh530u|SH530U",
			"Foxconn|SH930W|FCN|SH930W",
			"Foxconn|SHARP  SH837W|S1U|SH837W",
			"Foxconn|SHARP SH630E|SAE|SHP-SH630E",
			"Foxconn|SHARP SH631W|S2U|SH631W",
			"Foxconn|SHARP SH837W|S1U|SH837W",
			"Foxconn|ViewPhone3|viewphone3|ViewPhone3",
			"Foxconn|XOLO|AZ510|X900",
			"Foxconn|XOLO_X1000|SI1|XOLO_X1000",
			"Foxconn|vizio VP700|S1U|VP700",
			"Foxconn|vizio VP800|FCN|VP800",
			"Fujitsu|ARROWS A SoftBank 101F|SBM101F|101F",
			"Fujitsu|ARROWS A SoftBank 201F|SBM201F|201F",
			"Fujitsu|ARROWS A SoftBank 202F|SBM202F|202F",
			"Fujitsu|ARROWS A SoftBank 301F|SBM301F|301F",
			"Fujitsu|ARROWS ES IS12F|FJI12|IS12F",
			"Fujitsu|ARROWS Kiss F-03D|f11sky|F-03D",
			"Fujitsu|ARROWS Kiss F-03E|F03E|F-03E",
			"Fujitsu|ARROWS M305|M305|M305",
			"Fujitsu|ARROWS Me F-11D|F11D|F-11D",
			"Fujitsu|ARROWS NX F-06E|F06E|F-06E",
			"Fujitsu|ARROWS S EM01F|EM01F|EM01F",
			"Fujitsu|ARROWS Tab F-05E|F05E|F-05E",
			"Fujitsu|ARROWS Tab FJT21|FJT21_jp_kdi|FJT21",
			"Fujitsu|ARROWS Tab LTE F-01D|blaze_tablet|F-01D",
			"Fujitsu|ARROWS Tab Wi-Fi FAR70B|FAR70B|FAR70B",
			"Fujitsu|ARROWS Tab Wi-Fi FAR75A/70A|blaze_tablet|FAR7",
			"Fujitsu|ARROWS V F-04E|F04E|F-04E",
			"Fujitsu|ARROWS X F-02E|F02E|F-02E",
			"Fujitsu|ARROWS X F-10D|F10D|F-10D",
			"Fujitsu|ARROWS X LTE F-05D|blaze|F-05D",
			"Fujitsu|ARROWS Z FJL22|FJL22_jp_kdi|FJL22",
			"Fujitsu|ARROWS Z ISW11F|FJI11|ISW11F",
			"Fujitsu|ARROWS Z ISW13F|FJI13|ISW13F",
			"Fujitsu|ARROWS ef FJL21|FJL21|FJL21",
			"Fujitsu|ARROWS mu F-07D|f11apo|F-074",
			"Fujitsu|ARROWS mu F-07D|f11apo|F-07D",
			"Fujitsu|ARROWS mu F-07D|f11apo|FJJB091",
			"Fujitsu|Disney Mobile on docomo F-07E|F07E|F-07E",
			"Fujitsu|Disney Mobile on docomo F-08D|blaze|F-08D",
			"Fujitsu|F-01F|F01F|F-01F",
			"Fujitsu|F-02F|F02F|F-02F",
			"Fujitsu|F-02G|F02G|F-02G",
			"Fujitsu|F-03F|F03F|F-03F",
			"Fujitsu|F-03G|F03G|F-03G",
			"Fujitsu|F-04F|F04F|F-04F",
			"Fujitsu|F-05F|F05F|F-05F",
			"Fujitsu|F-06F|F06F|F-06F",
			"Fujitsu|F-09D ANTEPRIMA|F09D|F-09D",
			"Fujitsu|F-09E|F09E|F-09E",
			"Fujitsu|F-12C|f11eif|F-12C",
			"Fujitsu|FARTM9334P1|FARTM9334|FARTM9334P1",
			"Fujitsu|FARTM933KZ|FARTM933KZ|FARTM933KZ",
			"Fujitsu|FARTMB611Y|FARTMB611Y|FARTMB611Y",
			"Fujitsu|M01|M01|M01",
			"Fujitsu|M363/P|UY1|M363/P",
			"Fujitsu|REGZA Phone IS11T|TSI11|IS11T",
			"Fujitsu|REGZA Phone T-01C|tg03|IS04",
			"Fujitsu|REGZA Phone T-01C|tg03|T-01C",
			"Fujitsu|REGZA Phone T-01D|blaze|T-01D",
			"Fujitsu|REGZA Phone T-02D|T02D|T-02D",
			"Fujitsu|Raku-Raku SMART PHONE F-12D|F12D|F-12D",
			"Fujitsu|Raku-Raku SMART PHONE2 F-08E|F08E|F-08E",
			"Fujitsu|STYLISTIC M350/CA2|M350|M350",
			"Fujitsu|STYLISTIC M532|mx532|M532",
			"Fujitsu|STYLISTIC M702|M702|M702",
			"Fujitsu|STYLISTIC MH350|MH350|MH350",
			"Fujitsu|STYLISTIC S01|S01|S01-orange",
			"Fujitsu|SZJ-JS101|SZJ101|SZJ-JS101",
			"G53|TA71CA3|TA71CA3_1|TA71CA3",
			"GD Classbook|TA80CA1|TA80CA1_2|TA80CA1",
			"GENERAL MOBILE|General Mobile E-tab 4|mehmet|e-tab4",
			"GOCLEVER|QUANTUM_700S/QUANTUM 700S|QUANTUM_700S|QUANTUM_700S",
			"GOCLEVER|TQ700/QUANTUM 700/TAB|TQ700|TQ700",
			"GSmart|Arty A3|Arty_A3|GSmart Arty A3",
			"GSmart|GSmart Aku A1|Aku_A1|GSmart Aku A1",
			"GSmart|GSmart Alto A2|Alto_A2|GSmart Alto A2",
			"GSmart|GSmart GS202+|GSmart_GS202|GSmart GS202+",
			"GSmart|GSmart GSmart SX1|GBC_bravo|GSmart SX1",
			"GSmart|GSmart Guru G1|Guru_G1|GSmart Guru G1",
			"GSmart|GSmart Maya M1|MayaM1|GSmart Maya M1",
			"GSmart|GSmart Maya M1 v2|Maya_M1_v2|GSmart Maya M1 v2",
			"GSmart|GSmart Rey R3|Rey_R3|GSmart Rey R3",
			"GSmart|GSmart Rio R1|Rio_R1|GSmart Rio R1",
			"GSmart|GSmart Roma R2|Roma_R2|GSmart Roma R2",
			"GSmart|GSmart SX1|bravo|GSmart SX1",
			"GSmart|GSmart Sierra S1|GSmart_Sierra_S1|GSmart Sierra S1",
			"GSmart|GSmart T4|GSmart_T4|GSmart T4",
			"GSmart|GSmart Tuku T2|Tuku_T2|GSmart Tuku T2",
			"GSmart|GX2|GBC_ares|GX2",
			"GSmart|Mika M2|Mika_M2|GSmart Mika M2",
			"GSmart|Mika M3|Mika_M3|GSmart Mika M3",
			"GSmart|Saga_S3|Saga_S3|GSmart Saga S3",
			"GSmart|Simba SX1|GBC_bravo|Simba SX1",
			"GSmart|Simba SX1|bravo|Simba SX1",
			"Galapad|Galapad S6|GALAPAD_S6|GALAPAD_S6",
			"Garmin|Monterra|monterra|Monterra",
			"Garmin|Nuvi 3595|scorpio|nuvi 3595",
			"Garmin|Nuvifone|a50|Garmin-Asus A50",
			"Garmin|Nuvifone|a50|Garminfone",
			"Garmin|Nuvifone|a50|nuvifone A50",
			"Garmin|Nuvifone|a50|n\\xc3\\xbcvifone A50",
			"Garmin|nuvi 3590|scorpio|nuvi 3590",
			"Garmin|nuvi 3592|scorpio|nuvi 3592",
			"Garmin|nuvi 3595|scorpio|nuvi 3595",
			"General Mobile|Discovery Air|Discovery_Air|Discovery Air",
			"GeneralMobile|E-Tab10|blue_eye|E-Tab10",
			"Geniatech|Enjoy TV Quad-Core Box|ATV581|Enjoy TV Quad-Core Box",
			"Getac|Getac Z710|z710|Getac Z710",
			"Gigabyte|9788|9788|9788",
			"Gigabyte|Boston 4G|boston_4g|Boston 4G",
			"Gigabyte|E9771|E9771|E9771",
			"Gigabyte|GSmart G1310|G1310|GSmart G1310",
			"Gigabyte|GSmart G1315|G1315|GSmart G1315",
			"Gigabyte|GSmart G1317|G1317D|GSmart G1317D",
			"Gigabyte|GSmart G1342|msm7627a|GSmart G1342",
			"Gigabyte|GSmart G1345|msm7627_sku2|GSmart G1345",
			"Gigabyte|GSmart G1355|G1355|GSmart G1355",
			"Gigabyte|GSmart G1362|G1362|GSmart G1362",
			"Gigabyte|GSmart GS202|GS202|GSmart GS202",
			"Gigabyte|Gigabyte SB506|SB506|SB506",
			"Gigabyte|Gigabyte SB510 / IBT-102|SB510|SB510",
			"Gigabyte|Hipstreet 9DTB7 / Lazer MY9308P|AMLMY9308P|9DTB7",
			"Gigabyte|Hipstreet 9DTB7 / Lazer MY9308P|AMLMY9308P|MY9308P",
			"Gigabyte|P102g|P102g|P102G",
			"Gigabyte|PAPILIO G1|inhon77_cidals|INHON PAPILIO G1",
			"Gigabyte|PROSCAN PLT1066 / MAG MAGPAD / TEAC TEACTAB / DOPO GS-1008|MID1005_K|GS-1008",
			"Gigabyte|PROSCAN PLT1066 / MAG MAGPAD / TEAC TEACTAB / DOPO GS-1008|MID1005_K|MAGPAD",
			"Gigabyte|PROSCAN PLT1066 / MAG MAGPAD / TEAC TEACTAB / DOPO GS-1008|MID1005_K|PLT1066",
			"Gigabyte|PROSCAN PLT1066 / MAG MAGPAD / TEAC TEACTAB / DOPO GS-1008|MID1005_K|TEACTAB",
			"Gigabyte|SAF3011|SAF3011|SAF3011",
			"Gigabyte|TB100|TB100|Gigabyte TB100",
			"Gigabyte|TB100|TB100|TB100",
			"Gigabyte|TECNO P3S|P3S|TECNO P3S",
			"Gigabyte|TM105|TM105|TM105",
			"Gigabyte|TM105A|TM105A|Aero1021/P1021HCBA4C1VXX",
			"Gigabyte|TM105A|TM105A|Glacier",
			"Gigabyte|TM105A|TM105A|Qrypton1010",
			"Gigabyte|TM105A|TM105A|TM105A",
			"Gigabyte|TM75A|TM75A|GSmart7Tab",
			"Gigabyte|TM75A|TM75A|Qrypton7",
			"Gigabyte|TM75A|TM75A|TM75A",
			"Gigabyte|TM75A|TM75A|Voyager",
			"Gigabyte|TM75A-V|TM75A-V|TM75A-V",
			"Gigabyte|W5510|W5510|W5510",
			"Gigabyte|W6360|W6360|W6360",
			"Gigabyte|mexico|Lanix|iliumPAD",
			"Gigaset|Gigaset QV1030|FG6Q|Gigaset QV1030",
			"Gigaset|Gigaset QV830|UY8|Gigaset QV830",
			"Gigaset|Gigaset QV831|UY8|Gigaset QV831",
			"Gigaset|Maxwell 10|maxwell_10|80-1",
			"Gigaset|SL930|lion_s|SL930",
			"Gionee|M2|M2|M2",
			"Gionee|M3|M3|M3",
			"Gionee|P4S|P4S|P4S",
			"Gionee|P6|P6|P6",
			"Gionee|S5.1|S5_1|S5.1",
			"Gionee|S5.5|S5_5|S5.5",
			"Gionee|V4S|V4S|V4S",
			"Gionee|V5|V5|V5",
			"Google|Galaxy S4 Google Play edition|jgedlte|GT-I9505G",
			"Google|Project Tango Tablet Development Kit|yellowstone|Project Tango Tablet Development Kit",
			"Google|Project Tango Tablet Development Kit|yellowstone|Yellowstone",
			"Gradiente|iphone C600|GBC_bravo|iphone C600",
			"Gradiente|iphone C600|bravo|iphone C600",
			"Grundig|GTB1030|TA10CA2|TA10CA2",
			"Grundig|TC69CA2|GTB801|GTB 801",
			"HANNSpree|HANNSpree HSG1279|HANNSpad|HSG1279",
			"HIsense|HS-EG916|EG916|HS-EG916",
			"HKC|HKC A79 Tablet|EM62|A79",
			"HKT|eye3|UY3-PCW|eye3",
			"HP|10|spruce|HP 10",
			"HP|10 Plus|torsa|HP 10 Plus",
			"HP|7 G2|redwood|HP 7 G2",
			"HP|7 Plus|Ilex|HP 7 Plus",
			"HP|7 Plus G2|deschutes|HP 7 Plus G2",
			"HP|7 Tablet|Mesquite|HP 7",
			"HP|7 Voice Tab|klondike|HP 7 VoiceTab",
			"HP|7 VoiceTab|yukon|HP 7 VoiceTab",
			"HP|7.1|Holly|HP 7.1",
			"HP|8|Fir|HP 8",
			"HP|8 G2|maple|HP 8 G2",
			"HP|Bonsai 10 HD|bonsai10|HP Slate 10 HD",
			"HP|HP Slate 7|pine|HP Slate 7",
			"HP|HP SlateBook 10 x2 PC|3645|HP SlateBook 10 x2 PC",
			"HP|Pro Slate 12|dane|HP Pro Slate 12",
			"HP|Pro Slate 8|malamute|HP Pro Slate 8",
			"HP|Slate 10 HD|bonsai10|HP Slate 10 HD",
			"HP|Slate 10 Plus|linkplus|HP Slate 10 Plus",
			"HP|Slate 17|franky|HP Slate 17",
			"HP|Slate 21 Pro|ranger|Slate 21 Pro",
			"HP|Slate 6 Voice Tab|pomegranate|HP Slate 6 Voice Tab",
			"HP|Slate 6 Voice Tab II|avocado|HP Slate 6 Voice Tab II",
			"HP|Slate 6 VoiceTab Plus|mekong|HP Slate 6 VoiceTab Plus",
			"HP|Slate 7 Beats Special Edition|fairfax|HP Slate 7 Beats Special Edition",
			"HP|Slate 7 Beats Special Edition|fairfax|HP Slate7 Beats Special Edition",
			"HP|Slate 7 Extreme|olive|HP Slate7 Extreme",
			"HP|Slate 7 HD|bonsai10|HP Slate 10 HD",
			"HP|Slate 7 HD|bonsai7|HP Slate 7 HD",
			"HP|Slate 7 Voice Tab|almond|HP Slate 7 Voice Tab",
			"HP|Slate 7 VoiceTab Ultra|charm|HP Slate 7 VoiceTab Ultra",
			"HP|Slate 8 Plus|vogue|HP Slate 8 Plus",
			"HP|Slate 8 Pro|dogwood|HP Slate 8 Pro",
			"HP|Slate21|phobos|Slate 21",
			"HP|Slate7 Plus|birch|HP Slate 7 Plus",
			"HP|Slate8 Pro|fig|HP Slate 8 Pro",
			"HP|SlateBook 14|200a|HP SlateBook 14 PC",
			"HP|Tablet 10|balsa|HP_10_Tablet",
			"HP|Voice Tab 7|almond|HP Slate 7 Voice Tab",
			"HTC|\\tADR6325|lexikon|ADR6325",
			"HTC|ADR6330VW|blissc|ADR6330VW",
			"HTC|ADR6425LVW|vigor|ADR6425LVW",
			"HTC|AT&T HTC One X+|evitareul|HTC EVARE_UL",
			"HTC|AT&T HTC One X+|evitareul|HTC One X+",
			"HTC|Aria|liberty|HTC Aria",
			"HTC|Aria|liberty|HTC Aria A6380",
			"HTC|Aria|liberty|HTC Gratia A6380",
			"HTC|Aria|liberty|HTC Liberty",
			"HTC|Butterfly S|dlxpul|HTC Butterfly s",
			"HTC|Butterfly S|dlxpul|HTC_Butterfly_s_901s",
			"HTC|Chacha|chacha|HTC ChaCha A810b",
			"HTC|Chacha|chacha|HTC ChaCha A810e",
			"HTC|Chacha|chacha|HTC ChaChaCha A810e",
			"HTC|Chacha|chacha|HTC Status",
			"HTC|Desire|bravo|HTC Desire",
			"HTC|Desire|bravo|X06HT",
			"HTC|Desire|bravoc|PB99400",
			"HTC|Desire 320|htc_v01_u|HTC 0PF11",
			"HTC|Desire 320|htc_v01_u|HTC Desire 320",
			"HTC|Desire 510|htc_a11ul8x26|HTC Desire 510",
			"HTC|Desire 516 dual sim|htc_v2_dcg|HTC Desire 516 dual sim",
			"HTC|Desire 516 dual sim|htc_v2_dug|HTC Desire 516 dual sim",
			"HTC|Desire 516 dual sim|htc_v2_dug|HTC V2",
			"HTC|Desire 620 dual sim|htc_a31dtul|HTC_D620u",
			"HTC|Desire 820|htc_a51ul|HTC 0PFJ4",
			"HTC|Desire 820|htc_a51ul|HTC Desire 820",
			"HTC|Desire D820mini|htc_a31dtul|HTC D820mt",
			"HTC|Desire D820mini|htc_a31dtul|HTC D820mu",
			"HTC|Desire HD|ace|001HT",
			"HTC|Desire HD|ace|Desire HD",
			"HTC|Desire HD|ace|HTC Desire HD A9191",
			"HTC|Desire HD|ace|Inspire HD",
			"HTC|Desire L by HTC|cp2u|Desire L by HTC",
			"HTC|Desire S|saga|HTC Desire S",
			"HTC|Desire VC|primodd|HTC Desire VC",
			"HTC|Desire VC|primodd|HTC Desire VC T328d",
			"HTC|Desire VC|primodd|HTC PRO_DD",
			"HTC|Droid DNA|dlx|HTC6435LRA",
			"HTC|Droid DNA|dlx|HTC6435LVW",
			"HTC|Droid Eris|desirec|Eris",
			"HTC|Droid Eris|desirec|Pulse",
			"HTC|Droid Incredible|inc|ADR6300",
			"HTC|EVO 3D|shooter|HTCEVOV4G",
			"HTC|EVO 3D|shooter|PG86100",
			"HTC|EVO LTE 4G|jewel|EVO",
			"HTC|EVO Shift 4G|speedy|PG06100",
			"HTC|Evo 4G|supersonic|PC36100",
			"HTC|Flyer|flyer|HTC Flyer",
			"HTC|Flyer|flyer|HTC Flyer P510e",
			"HTC|Flyer|flyer|HTC Flyer P511e",
			"HTC|Flyer|flyer|HTC Flyer P512",
			"HTC|Flyer|flyer|HTC_Flyer_P512_NA",
			"HTC|G1|dream|HTC Dream",
			"HTC|G2|vision|HTC Vision",
			"HTC|G2|vision|T-Mobile G2",
			"HTC|HTC 0P9C8|htc_a5dwgl|HTC 0P9C8",
			"HTC|HTC 0P9C8|htc_a5dwgl|HTC Desire 816 dual sim",
			"HTC|HTC 5060 dual sim|z4dug|HTC Desire 500 dual sim",
			"HTC|HTC 601e|m4|HTC 601e",
			"HTC|HTC 606w|cp3dug|HTC 606w",
			"HTC|HTC 606w|cp3dug|HTC Desire 600",
			"HTC|HTC 606w|cp3dug|HTC Desire 600 dual sim",
			"HTC|HTC 606w|cp3dug|HTC PO49120",
			"HTC|HTC 608t|cp3dtg|HTC 608t",
			"HTC|HTC 609d|cp3dcg|HTC 609d",
			"HTC|HTC 6160|zaradug|HTC 6160",
			"HTC|HTC 619d|zarawl|HTC 619d",
			"HTC|HTC 7060|cp5dug|HTC_7060",
			"HTC|HTC 710C|htc_a5chl|710C",
			"HTC|HTC 802d|m7cdwg|HTC 802d",
			"HTC|HTC 803e|t6ul|HTC 803e",
			"HTC|HTC 8060|t6dug|HTC 8060",
			"HTC|HTC 8088|t6tl|HTC 8088",
			"HTC|HTC 809d|t6dwg|HTC 809d",
			"HTC|HTC 8160|t6uhl|HTC 8160",
			"HTC|HTC 9060|dlxpul|HTC 901e",
			"HTC|HTC 9060|m7|HTC 801e",
			"HTC|HTC 9060|m7cdug|HTC 802w",
			"HTC|HTC 9088|dlpdtu|HTC 9088",
			"HTC|HTC 919d|dlpdwg|HTC 919d",
			"HTC|HTC Butterfly|dlxu|HTC Butterfly",
			"HTC|HTC Butterfly|dlxu|HTC DLX_U",
			"HTC|HTC Butterfly|dlxu|HTC X920e",
			"HTC|HTC Butterfly|dlxub1|HTC Butterfly",
			"HTC|HTC Butterfly|dlxub1|HTC DLXUB1",
			"HTC|HTC Butterfly 2|htc_b2ul|HTC Butterfly 2",
			"HTC|HTC Butterfly 2|htc_b2ul|HTC_B810x",
			"HTC|HTC Butterfly s 9060|dlpdug|HTC 9060",
			"HTC|HTC D316d|htc_v2_c|HTC D316d",
			"HTC|HTC D610t|htc_a3tl|HTC D610t",
			"HTC|HTC D816d|htc_a5dwg|HTC D816d",
			"HTC|HTC D816d|htc_a5dwg|HTC_D816d",
			"HTC|HTC D816e|htc_a5dugl|HTC D816e",
			"HTC|HTC D816h|htc_a5mgp_dug|HTC D816h",
			"HTC|HTC D816t|htc_a5tl|HTC D816t",
			"HTC|HTC D816v|htc_a5dwgl|HTC D816v",
			"HTC|HTC D816w|htc_a5dug|HTC D816w",
			"HTC|HTC D820 Mini|htc_a31dtul|HTC D820mt",
			"HTC|HTC D820t|htc_a51dtul|HTC D820t",
			"HTC|HTC D820u|htc_a51dtul|HTC D820u",
			"HTC|HTC DROID DNA|dlx|HTC6435LRA",
			"HTC|HTC DROID Incredible 4G LTE|fireball|ADR6410LRA",
			"HTC|HTC DROID Incredible 4G LTE|fireball|ADR6410LVW",
			"HTC|HTC DROID Incredible 4G LTE|fireball|ADR6410OM",
			"HTC|HTC Desire 200|gtou|HTC Desire 200",
			"HTC|HTC Desire 200|gtou|HTC_Desire_200",
			"HTC|HTC Desire 210 dual sim|htc_v0_dug|HTC Desire 210 dual sim",
			"HTC|HTC Desire 300|g3u|HTC 301e",
			"HTC|HTC Desire 300|g3u|HTC Desire 300",
			"HTC|HTC Desire 300|g3u|HTC_0P6A1",
			"HTC|HTC Desire 300|g3u|HTC_Desire_300",
			"HTC|HTC Desire 310|htc_v1_dug|HTC D310w",
			"HTC|HTC Desire 310|htc_v1_dug|HTC Desire 310 dual sim",
			"HTC|HTC Desire 310|htc_v1_u|HTC Desire 310",
			"HTC|HTC Desire 310|htc_v1_u|HTC_D310n",
			"HTC|HTC Desire 310|htc_v1_u|HTC_V1",
			"HTC|HTC Desire 320|htc_v01_u|HTC 0PF120",
			"HTC|HTC Desire 320|htc_v01_u|HTC Desire 320",
			"HTC|HTC Desire 500|z4u|HTC Desire 500",
			"HTC|HTC Desire 500|z4u|HTC_Desire_500",
			"HTC|HTC Desire 500 dual sim|z4dug|HTC 5060",
			"HTC|HTC Desire 500 dual sim|z4dug|HTC Desire 500 dual sim",
			"HTC|HTC Desire 501|htc_csnu|HTC Desire 501",
			"HTC|HTC Desire 501|htc_csnu|HTC_603h",
			"HTC|HTC Desire 501 dual sim|csndug|HTC Desire 501 dual sim",
			"HTC|HTC Desire 5088|z4td|HTC 5088",
			"HTC|HTC Desire 510|htc_a11chl|0PCV1",
			"HTC|HTC Desire 510|htc_a11ul|HTC 0PCV20",
			"HTC|HTC Desire 510|htc_a11ul|HTC Desire 510",
			"HTC|HTC Desire 510|htc_a11ul|HTC_0PCV2",
			"HTC|HTC Desire 516|htc_v2_dcg|HTC D516d",
			"HTC|HTC Desire 516|htc_v2_dtg|HTC D516t",
			"HTC|HTC Desire 516|htc_v2_dug|HTC C2",
			"HTC|HTC Desire 516|htc_v2_dug|HTC D516w",
			"HTC|HTC Desire 516|htc_v2_dug|HTC V2",
			"HTC|HTC Desire 516 dual sim|htc_v2_dug|HTC Desire 516 dual sim",
			"HTC|HTC Desire 526G+ dual sim|htc_v02_dug|HTC 0PL41",
			"HTC|HTC Desire 526G+ dual sim|htc_v02_dug|HTC 0PL4100",
			"HTC|HTC Desire 526G+ dual sim|htc_v02_dug|HTC Desire 526GPLUS dual sim",
			"HTC|HTC Desire 526G+ dual sim|htc_v02_dug|HTC_D526h",
			"HTC|HTC Desire 600|cp3dug|HTC Desire 600",
			"HTC|HTC Desire 600 Dual SIM|cp3dug|HTC Desire 600 dual sim",
			"HTC|HTC Desire 600c Dual SIM|cp3dcg|HTC 609d",
			"HTC|HTC Desire 600c Dual SIM|cp3dcg|HTC Desire 600c dual sim",
			"HTC|HTC Desire 601|zara|HTC Desire 601",
			"HTC|HTC Desire 601|zara|HTC_0P4E2",
			"HTC|HTC Desire 601|zaracl|HTC Desire 601",
			"HTC|HTC Desire 601|zaracl|HTC0P4E1",
			"HTC|HTC Desire 601 dual sim|zaradug|HTC Desire 601 dual sim",
			"HTC|HTC Desire 606w|cp3dug|HTC 606w",
			"HTC|HTC Desire 609d|cp3dcg|HTC 609d",
			"HTC|HTC Desire 610|htc_a3qhdul|HTC Desire 610",
			"HTC|HTC Desire 610|htc_a3qhdul|HTC_0P9O2",
			"HTC|HTC Desire 610|htc_a3qhdul|HTC_D610x",
			"HTC|HTC Desire 610|htc_a3ul|HTC Desire 610",
			"HTC|HTC Desire 616 dual sim|htc_v3_dug|HTC D616w",
			"HTC|HTC Desire 616 dual sim|htc_v3_dug|HTC Desire 616 dual sim",
			"HTC|HTC Desire 616 dual sim|htc_v3_dug|HTC V3",
			"HTC|HTC Desire 620|htc_a31ul|HTC 0PE64",
			"HTC|HTC Desire 620|htc_a31ul|HTC Desire 620",
			"HTC|HTC Desire 620G dual sim|htc_a31mg_dug|HTC 0PE65",
			"HTC|HTC Desire 620G dual sim|htc_a31mg_dug|HTC Desire 620G dual sim",
			"HTC|HTC Desire 620G dual sim|htc_a31mg_dug|HTC_D620h",
			"HTC|HTC Desire 700 dual sim|cp5dug|HTC Desire 700 dual sim",
			"HTC|HTC Desire 700 dual sim|cp5dwg|HTC Desire 700 dual sim",
			"HTC|HTC Desire 700 dual sim|cp5dwg|HTC_709d",
			"HTC|HTC Desire 7060|cp5dug|HTC_7060",
			"HTC|HTC Desire 7088|cp5dtu|HTC 7088",
			"HTC|HTC Desire 709d|cp5dwg|HTC 709d",
			"HTC|HTC Desire 816|htc_a5chl|HTC Desire 816",
			"HTC|HTC Desire 816|htc_a5ul|HTC Desire 816",
			"HTC|HTC Desire 816|htc_a5ul|HTC_0P9C2",
			"HTC|HTC Desire 816|htc_a5ul|HTC_D816x",
			"HTC|HTC Desire 816 dual sim|htc_a5dug|HTC Desire 816 dual sim",
			"HTC|HTC Desire 816 dual sim|htc_a5dwg|HTC Desire 816 dual sim",
			"HTC|HTC Desire 816G dual|htc_a5mgp_dug|HTC Desire 816G dual sim",
			"HTC|HTC Desire 816G dual SIM|htc_a5mgp_dug|HTC Desire 816G dual sim",
			"HTC|HTC Desire 816G dual sim|htc_a5mg_dug|HTC Desire 816G dual sim",
			"HTC|HTC Desire 816G dual sim|htc_a5mgp_dug|HTC Desire 816G dual sim",
			"HTC|HTC Desire 820 dual sim|htc_a51dtul|HTC Desire 820 dual sim",
			"HTC|HTC Desire 820s|htc_a50ml_dtul|HTC D820ts",
			"HTC|HTC Desire 820s|htc_a50ml_dtul|HTC D820us",
			"HTC|HTC Desire 820s|htc_a50ml_dtul|HTC Desire 820s dual sim",
			"HTC|HTC Desire C|golfc|HTC Desire C",
			"HTC|HTC Desire C|golfu|HTC Desire C",
			"HTC|HTC Desire EYE|htc_eyetuhl|HTC 0PFH2",
			"HTC|HTC Desire EYE|htc_eyetuhl|HTC Desire EYE",
			"HTC|HTC Desire EYE|htc_eyetuhl|HTC_M910x",
			"HTC|HTC Desire EYE|htc_eyeul|HTC 0PFH11",
			"HTC|HTC Desire EYE|htc_eyeul|HTC Desire EYE",
			"HTC|HTC Desire Eye|htc_eyeul_att|HTC Desire Eye",
			"HTC|HTC Desire L dual sim|cp2dug|HTC Desire L dual sim",
			"HTC|HTC Desire P|magniu|HTC Desire P",
			"HTC|HTC Desire SV|magnids|HTC Desire SV",
			"HTC|HTC Desire V|primods|HTC Desire Q",
			"HTC|HTC Desire V|primods|HTC Desire U",
			"HTC|HTC Desire V|primods|HTC Desire U dual sim",
			"HTC|HTC Desire V|primods|HTC Desire V",
			"HTC|HTC Desire V|primods|HTC PROMIN_U",
			"HTC|HTC Desire V|primods|HTC PRO_DS",
			"HTC|HTC Desire V|primods|HTC T327w",
			"HTC|HTC Desire X|protou|HTC Desire X",
			"HTC|HTC Desire X|protou|HTC POO_U",
			"HTC|HTC Desire X dual sim|protodug|HTC Desire X dual sim",
			"HTC|HTC Desire XC dual sim|protodcg|HTC Desire XC dual sim",
			"HTC|HTC EVO 3D ISW12HT|shooterk|ISW12HT",
			"HTC|HTC EVO 3D X515m|shooteru|HTC EVO 3D X515a",
			"HTC|HTC EVO 3D X515m|shooteru|HTC EVO 3D X515m",
			"HTC|HTC EVO 3D X515m|shooteru|HTC Inspire 3D",
			"HTC|HTC EVO 4G LTE|jewel|EVO",
			"HTC|HTC Explorer A310e|pico|HTC Explorer",
			"HTC|HTC Explorer A310e|pico|HTC Explorer A310b",
			"HTC|HTC Explorer A310e|pico|HTC Explorer A310e",
			"HTC|HTC Hero|hero|ERA G2 Touch",
			"HTC|HTC Hero|hero|HTC Hero",
			"HTC|HTC Hero|hero|T-Mobile G2 Touch",
			"HTC|HTC Hero|hero|T-Mobile_G2_Touch",
			"HTC|HTC Hero|hero|dopod A6288",
			"HTC|HTC J Butterfly|dlxj|HTL21",
			"HTC|HTC J One|m7wlj|HTL22",
			"HTC|HTC J Z321e|valentewxc9|HTC J Z321e",
			"HTC|HTC M8t|htc_m8tl|HTC M8t",
			"HTC|HTC ONE S|ville|HTC One S",
			"HTC|HTC ONE SV|k2u|HTC K2_U",
			"HTC|HTC ONE SV|k2u|HTC One SV",
			"HTC|HTC ONE SV|k2ul|HTC One SV",
			"HTC|HTC One|m7|HTC 801e",
			"HTC|HTC One|m7|HTC One",
			"HTC|HTC One|m7|HTC One 801e",
			"HTC|HTC One|m7|HTC_PN071",
			"HTC|HTC One|m7cdtu|HTC 802t",
			"HTC|HTC One|m7cdtu|HTC 802t 16GB",
			"HTC|HTC One|m7cdug|HTC 802w",
			"HTC|HTC One|m7cdug|HTC One dual sim",
			"HTC|HTC One|m7cdwg|HTC 802d",
			"HTC|HTC One|m7cdwg|HTC One dual 802d",
			"HTC|HTC One|m7cdwg|HTC One dual sim",
			"HTC|HTC One|m7wls|HTC One",
			"HTC|HTC One|m7wls|HTCONE",
			"HTC|HTC One|m7wlv|HTC One",
			"HTC|HTC One|m7wlv|HTC6500LVW",
			"HTC|HTC One (E8)|htc_mecdwg|HTC M8Sd",
			"HTC|HTC One (E8)|htc_mectl|HTC M8St",
			"HTC|HTC One (E8)|htc_mectl|HTC One_E8",
			"HTC|HTC One (E8)|htc_mecul|HTC One_E8",
			"HTC|HTC One (E8)|htc_mecul|HTC_M8Sx",
			"HTC|HTC One (E8)|htc_mecwhl|0PAJ5",
			"HTC|HTC One (E8) dual sim|htc_mecdug|HTC One_E8 dual sim",
			"HTC|HTC One (E8) dual sim|htc_mecdwg|HTC 0PAJ4",
			"HTC|HTC One (E8) dual sim|htc_mecdwg|HTC One_E8 dual sim",
			"HTC|HTC One (M8)|htc_m8|HTC M8w",
			"HTC|HTC One (M8)|htc_m8|HTC One_M8",
			"HTC|HTC One (M8)|htc_m8|HTC_0P6B",
			"HTC|HTC One (M8)|htc_m8|HTC_0P6B6",
			"HTC|HTC One (M8)|htc_m8|HTC_M8x",
			"HTC|HTC One (M8)|htc_m8dug|HTC One_M8 dual sim",
			"HTC|HTC One (M8)|htc_m8dwg|HTC M8d",
			"HTC|HTC One (M8)|htc_m8whl|831C",
			"HTC|HTC One (M8)|htc_m8wl|HTC One_M8",
			"HTC|HTC One (M8)|htc_m8wl|HTC6525LVW",
			"HTC|HTC One 801e|m7|HTC One 801e",
			"HTC|HTC One 801e|m7|HTC One 801s",
			"HTC|HTC One Dual 802d|m7cdwg|HTC One dual 802d",
			"HTC|HTC One Dual Sim|m7cdwg|HTC One dual sim",
			"HTC|HTC One E8 dual|htc_mecdwg|HTC_M8Sd",
			"HTC|HTC One Google Play edition|m7|HTC One",
			"HTC|HTC One M8 eye 4G LTE\\xe8\\x81\\x94\\xe9\\x80\\x9a\\xe7\\x89\\x88|htc_melsuhl|HTC M8Ew",
			"HTC|HTC One S|ville|HTC One S",
			"HTC|HTC One S|ville|HTC VLE_U",
			"HTC|HTC One S|villec2|HTC One S",
			"HTC|HTC One S Special Edition|villeplus|HTC One S Special Edition",
			"HTC|HTC One SC|cp2dcg|HTC One SC",
			"HTC|HTC One SC|cp2dcg|HTC One SC T528d",
			"HTC|HTC One SV|k2cl|C525c",
			"HTC|HTC One SV|k2plccl|HTC One SV",
			"HTC|HTC One SV|k2ul|HTC K2_UL",
			"HTC|HTC One SV|k2ul|HTC One SV",
			"HTC|HTC One SV BLK|k2plccl|HTC One SV",
			"HTC|HTC One SV BLK|k2plccl|HTC One SV BLK",
			"HTC|HTC One V|primoc|HTC One V",
			"HTC|HTC One V|primou|HTC ONE V",
			"HTC|HTC One V|primou|HTC One V",
			"HTC|HTC One VX|totemc2|HTC One VX",
			"HTC|HTC One X|endeavoru|HTC One X",
			"HTC|HTC One X|evita|HTC One X",
			"HTC|HTC One X+|enrc2b|HTC One X+",
			"HTC|HTC One X+|evitareul|HTC One X+",
			"HTC|HTC One XL|evita|HTC One X",
			"HTC|HTC One XL|evita|HTC One XL",
			"HTC|HTC One XL|evita|HTC_One_XL",
			"HTC|HTC One XL|evitautl|HTC EVA_UTL",
			"HTC|HTC One max|t6ul|HTC One max",
			"HTC|HTC One max|t6ul|HTC_One_max",
			"HTC|HTC One max|t6whl|HTC0P3P7",
			"HTC|HTC One mini|htc_m4|HTC One mini",
			"HTC|HTC One mini|m4|HTC One mini",
			"HTC|HTC One mini|m4|HTC_One_mini_601e",
			"HTC|HTC One mini|m4|HTC_PO582",
			"HTC|HTC One mini 2|htc_memul|HTC One mini 2",
			"HTC|HTC One mini 2|htc_memul|HTC_M8MINx",
			"HTC|HTC One mini 2|htc_memul|HTC_One_mini_2",
			"HTC|HTC One mini 601E|m4|HTC_One_mini_601e",
			"HTC|HTC One mini2|htc_memul|HTC One mini 2",
			"HTC|HTC One remix|htc_memwl|HTC6515LVW",
			"HTC|HTC One_E8|htc_mecul_emea|HTC One_E8",
			"HTC|HTC Onex X|endeavoru|HTC One X",
			"HTC|HTC PO091|csndug|HTC PO091",
			"HTC|HTC Rhyme S510b|bliss|HTC Rhyme S510b",
			"HTC|HTC S720e|endeavoru|HTC S720e",
			"HTC|HTC Salsa C510e|icong|HTC Salsa C510b",
			"HTC|HTC Salsa C510e|icong|HTC Salsa C510e",
			"HTC|HTC Sensation XL with Beats Audio X315e|runnymede|HTC Sensation XL with Beats Audio X315b",
			"HTC|HTC Sensation XL with Beats Audio X315e|runnymede|HTC Sensation XL with Beats Audio X315e",
			"HTC|HTC Sensation XL with Beats Audio X315e|runnymede|HTC-X315E",
			"HTC|HTC Sensation XL with Beats Audio X315e|runnymede|Sensation XL with Beats Audio",
			"HTC|HTC T329d|protodcg|HTC T329d",
			"HTC|HTC T528w|cp2dug|HTC Desire L dual sim",
			"HTC|HTC Velocity 4G|holiday|HTC PH39100",
			"HTC|HTC Velocity 4G|holiday|HTC Raider X710e",
			"HTC|HTC Velocity 4G|holiday|HTC Velocity 4G",
			"HTC|HTC Velocity 4G|holiday|HTC Velocity 4G X710s",
			"HTC|HTC Velocity 4G|holiday|HTC-X710a",
			"HTC|HTC Wildfire S A515c|marvelc|HTC Wildfire S A515c",
			"HTC|HTC Wildfire S A515c|marvelc|HTC-PG762",
			"HTC|HTC Wildfire S A515c|marvelc|USCCADR6230US",
			"HTC|HTC desire 608t|cp3dtg|HTC 608t",
			"HTC|HTC first|mystul|HTC first",
			"HTC|HTC6435LVW|dlx|HTC6435LVW",
			"HTC|HTC909d|dlpdwg|HTC 919d",
			"HTC|HTCDesire612VZW|htc_a3qhdcl|HTC331ZLVW",
			"HTC|HTCDesire612VZW|htc_a3qhdcl|HTC331ZLVWPP",
			"HTC|HTCEVODesign4G|kingdom|HTC Acquire",
			"HTC|HTCEVODesign4G|kingdom|HTC EVO Design C715e",
			"HTC|HTCEVODesign4G|kingdom|HTC Hero S",
			"HTC|HTCEVODesign4G|kingdom|HTC Kingdom",
			"HTC|HTCEVODesign4G|kingdom|HTCEVODesign4G",
			"HTC|HTCEVODesign4G|kingdom|PH44100",
			"HTC|HTCOneMaxVZW|t6wl|HTC6600LVW",
			"HTC|HTC_0P3P5|t6ul|HTC_0P3P5",
			"HTC|HTC_0P4E2|zara|HTC_0P4E2",
			"HTC|HTC_A510c|marvelc|HTC_A510c",
			"HTC|HTC_Amaze_4G|ruby|HTC Amaze 4G",
			"HTC|HTC_Amaze_4G|ruby|HTC Ruby",
			"HTC|HTC_Amaze_4G|ruby|HTC_Amaze_4G",
			"HTC|HTC_D820f|htc_a51tuhl|HTC_D820f",
			"HTC|HTC_D820u|htc_a51dtul|HTC_D820u",
			"HTC|HTL23|htc_b2wlj|HTL23",
			"HTC|Hero|heroc|HERO200",
			"HTC|ISW13HT|valentewx|ISW13HT",
			"HTC|Incredible 2|vivow|ADR6350",
			"HTC|Incredible 2|vivow|HTC IncredibleS S710d",
			"HTC|Incredible S|vivo|HTC Incredible S",
			"HTC|Incredible S|vivo|HTC_S710E",
			"HTC|KDDI Infobar A02|imnj|HTX21",
			"HTC|Legend|legend|HTC Legend",
			"HTC|Nexus 9|flounder|Nexus 9",
			"HTC|Nexus One|passion|Nexus One",
			"HTC|One XL|evita|HTC One XL",
			"HTC|Puccini|puccinilte|HTC PG09410",
			"HTC|Puccini|puccinilte|HTC-P715a",
			"HTC|Sensation 4G|pyramid|HTC Sensation",
			"HTC|Sensation 4G|pyramid|HTC Sensation 4G",
			"HTC|Sensation 4G|pyramid|HTC Sensation XE with Beats Audio",
			"HTC|Sensation 4G|pyramid|HTC Sensation XE with Beats Audio Z715a",
			"HTC|Sensation 4G|pyramid|HTC Sensation XE with Beats Audio Z715e",
			"HTC|Sensation 4G|pyramid|HTC Sensation Z710a",
			"HTC|Sensation 4G|pyramid|HTC Sensation Z710e",
			"HTC|Sensation 4G|pyramid|HTC-Z710a",
			"HTC|Thunderbolt|mecha|ADR6400L",
			"HTC|Thunderbolt|mecha|HTC Mecha",
			"HTC|Wildfire|buzz|HTC Wildfire",
			"HTC|Wildfire CDMA|bee|HTC Bee",
			"HTC|Wildfire CDMA|bee|HTC Wildfire",
			"HTC|Wildfire S|marvel|HTC Wildfire S",
			"HTC|Wildfire S|marvel|HTC Wildfire S A510b",
			"HTC|Wildfire S|marvel|HTC Wildfire S A510e",
			"HTC|Wildfire S|marvel|HTC-A510a",
			"HTC|hTC Desire 601|zara|HTC Desire 601",
			"HTC|htc desire 400 dual sim|cp2dug|HTC Desire 400 dual sim",
			"HTC|htc desire 610|htc_a3qhdul|HTC Desire 610",
			"HTC|htc desire x|protou|HTC Desire X",
			"HTC|myTouch 3G|sapphire|HTC Magic",
			"HTC|myTouch 3G|sapphire|T-Mobile myTouch 3G",
			"HTC|myTouch 3G Slide|espresso|T-Mobile myTouch 3G Slide",
			"HTC|myTouch 4G|glacier|HTC Glacier",
			"HTC|myTouch 4G|glacier|HTC Panache",
			"HTC|myTouch 4G Slide|doubleshot|myTouch_4G_Slide",
			"Haier|A-TT00|A-TT00|A-TT00",
			"Haier|AD6B1H|AD6B1H|Smartfren Andromax AD6B1H",
			"Haier|AT7003|AT7003|AT7003",
			"Haier|Beeline Smart 3|Beeline_Smart3|Beeline Smart 3",
			"Haier|CDP7TAB4C8|CDP7TAB4C8|CDP7TAB4C8",
			"Haier|CETAB10KML9|CETAB10KML9|CETAB10KML9",
			"Haier|CETAB7ML9|CETAB7ML9|CETAB7ML9",
			"Haier|CETAB9ML9|CETAB9ML9|CETAB9ML9",
			"Haier|CMP_765|CMP_765|CMP_765",
			"Haier|CT1030|CT1030|CT1030",
			"Haier|CT301_W818|msm7627a|CT301_W818",
			"Haier|CT825|fiber-a31stm|CT825",
			"Haier|CYCLONE_ODYSSEY|CYCLONE_ODYSSEY|CYCLONE_ODYSSEY",
			"Haier|D2-721|D2-721G|D2-721G",
			"Haier|D2-751G|d2PAD|D2-751G",
			"Haier|D2-927G|D2-927G|D2-927G",
			"Haier|D2-961G|D2-961G|D2-961G",
			"Haier|GF88|I7A-LE|I7A-LE",
			"Haier|HG-9041|HG-9041|HG-9041",
			"Haier|HS-10DTB4|HS-10DTB4|HS-10DTB4",
			"Haier|HS-7DTB25|HS-7DTB25|HS-7DTB25",
			"Haier|HS-7DTB29-8GB|HS-7DTB29-8GB|HS-7DTB29-8GB",
			"Haier|HS-9DTB37|HS_9DTB37|HS_9DTB37",
			"Haier|HT-I860|HT-I860|HT-I860",
			"Haier|Haier|Homesurf744|Binatone Homesurf744 / Hubble Smart7 (Homesurf744)",
			"Haier|Haier|PAD69H|PAD69H",
			"Haier|Haier|PAD_D85|pad_d85",
			"Haier|HomeSurf742_AT7003|HomeSurf742_AT7003|HomeSurf742_AT7003",
			"Haier|Homesurf844|Homesurf844|Binatone Homesurf844 / Hubble Smart8 (Homesurf844)",
			"Haier|INGO-TAB|INGO-TAB|INGO-TAB",
			"Haier|Konnect_350|Konnect_350|Konnect_350",
			"Haier|LE39M7000CF|LE39M7000CF|LE39M7000CF",
			"Haier|LE48M7000CF|LE48M7000CF|LE48M7000CF",
			"Haier|LE55M7000CF|LE55M7000CF|LE55M7000CF",
			"Haier|Logicom S9782|Logicom-S9782|Logicom-S9782",
			"Haier|MD210|MD210|MD210",
			"Haier|MEGAFON MS3A|Megafon|MS3A",
			"Haier|MT-700|MT-700|MT-700",
			"Haier|NEO10-1|NEO10-1|NEO10-1",
			"Haier|NEO7-1|NEO7-1|NEO7-1",
			"Haier|NEO7-2|NEO7-2|NEO7-2",
			"Haier|NEO8-1|NEO8-1|NEO8-1",
			"Haier|P2|P2|P2",
			"Haier|PAD1021|PAD1021|PAD1021",
			"Haier|PAD1042|PAD1042|PAD1042",
			"Haier|PAD712|PAD712|PAD712",
			"Haier|PAD722|PAD722|PAD722",
			"Haier|PAD841|PAD841|PAD841",
			"Haier|PADCT1010W|PADCT1010W|PADCT1010W",
			"Haier|PAD_1042|PAD_1042|PAD_1042",
			"Haier|PNDPP410GP|PNDPP410GP|PNDPP410GP",
			"Haier|PNDPP44Q7GPBLK|PNDPP44Q7GPBLK|PNDPP44Q7GPBLK",
			"Haier|PNDPP47GP|PNDPP47GP|PNDPP47GP",
			"Haier|SP-5100|SP-5100|SP-5100",
			"Haier|Skill|Skill|Skill",
			"Haier|Smartfren Andromax AD681H|AD681H|Smartfren Andromax AD681H",
			"Haier|Smartfren Andromax AD9A1H|AD9A1H|Smartfren Andromax AD9A1H",
			"Haier|StarQ_Q5002|StarQ_Q5002|Q5002",
			"Haier|T785B|T785B|T785B",
			"Haier|TAB-700|TAB_700|TAB-700",
			"Haier|TAB-700|TAB_700|TAB700MPG",
			"Haier|W717|W717|W717",
			"Haier|W757|W757|W757",
			"Haier|W860|W860|W860",
			"Harris|Harris 12131-1000|msm8660_cougar|Harris 12131-1000",
			"Hipstreet|7DTB25|7DTB25|7DTB25",
			"Hipstreet|HS-10DTB12|Hipstreet|HS-10DTB12",
			"Hipstreet|HS-10DTB12A|RCT6203W46|HS_10DTB12A",
			"Hipstreet|HS-7DTB27|HS-7DTB27|HS-7DTB27",
			"Hipstreet|HS-7DTB35|RCT6773W22|HS_7DTB35",
			"Hipstreet|HS-7DTB39|HS-7DTB39|HS-7DTB39",
			"Hipstreet|HS-9DTB7A|HS-9DTB7A|HS-9DTB7A",
			"Hipstreet|Hipstreet HS-10DTB8|HS-10DTB8|HS-10DTB8",
			"Hisense|55H6SG|Vision|Vision",
			"Hisense|Andromax-c|AD686G|Andromax-c",
			"Hisense|C-4|C_4|C-4",
			"Hisense|CMB405|L690AE|CMB405",
			"Hisense|CMB501|u970_carrefour|CMB501",
			"Hisense|CMB510|L696|CMB510",
			"Hisense|E100T|E100T|Hisense E100T",
			"Hisense|E200T|E200T|HS-E200T",
			"Hisense|E2070|mt8312|E2070",
			"Hisense|E2371CH|rk3026|E2371CH",
			"Hisense|E260T|E260T|HS-E260T",
			"Hisense|E260U|E260U|Hisense E260U",
			"Hisense|E270BSA|E270BSA|E270BSA",
			"Hisense|E600MH02|E600MH02|HS-E600M",
			"Hisense|E601M|E601M|E601M",
			"Hisense|E860|E860|E860",
			"Hisense|EG680|eg980ae|EG680",
			"Hisense|EG68AE|EG668AE|LT668",
			"Hisense|EG929|EG929_TW|EG929",
			"Hisense|EG950|EG950TW|EG950",
			"Hisense|EG970|EG970TW|EG970",
			"Hisense|EG978|EG978|HS-EG978",
			"Hisense|EG98|EG98|EG98",
			"Hisense|EG98|EG98TW|EG98",
			"Hisense|EG981|EG981|HS-EG981",
			"Hisense|F270BW|F270BW|F270BW",
			"Hisense|F5070|mt8389|F5070",
			"Hisense|F5180|f5180|F5180",
			"Hisense|F5281CH|rk3288|F5281CH",
			"Hisense|G1|G1|G1",
			"Hisense|G2-3GP|G2-3GP|G2-3GP",
			"Hisense|G610M|G610M|Hisense G610M",
			"Hisense|H800T|H800T|HS-H800T",
			"Hisense|HS-E600M|E600M|HS-E600M",
			"Hisense|HS-E620M|E620M|HS-E620M",
			"Hisense|HS-E917|E917|HS-E917",
			"Hisense|HS-E936|E936|HS-E936",
			"Hisense|HS-E968|E968|HS-E968",
			"Hisense|HS-EG929|EG929|msm8625",
			"Hisense|HS-EG939|EG939|HS-EG939",
			"Hisense|HS-EG958|EG958|HS-EG958",
			"Hisense|HS-EG966|EG966|HS-EG966",
			"Hisense|HS-EG971|EG971|HS-EG971",
			"Hisense|HS-EG980|eg980|HS-EG980",
			"Hisense|HS-EG98C|EG98C|HS-EG98C",
			"Hisense|HS-I630M|msm8226|HS-I630M",
			"Hisense|HS-I630T|msm8226|HS-I630T",
			"Hisense|HS-T9|T9|HS-T9",
			"Hisense|HS-T926|t926|HS-T926",
			"Hisense|HS-T928|t928|HS-T928",
			"Hisense|HS-T939|HS-T939|HS-T939",
			"Hisense|HS-T959|t959|HS-T959",
			"Hisense|HS-T959S|t959s|HS-T959S",
			"Hisense|HS-T966|HS-T966|HS-T966",
			"Hisense|HS-T967|HS-T967|HS-T967",
			"Hisense|HS-T968|t968|HS-T968",
			"Hisense|HS-T968S|t968s|HS-T968S",
			"Hisense|HS-T970|t970|HS-T970",
			"Hisense|HS-T978|t978|HS-T978",
			"Hisense|HS-T980|t980|HS-T980",
			"Hisense|HS-U820|U820B|HS-U820",
			"Hisense|HS-U9|U9|HS-U9",
			"Hisense|HS-U912|U912A_1|HS-U912",
			"Hisense|HS-U912|U912B|HS-U912",
			"Hisense|HS-U912|U912C|HS-U912",
			"Hisense|HS-U929|u929de|HS-U929",
			"Hisense|HS-U929|u929de_3|HS-U929",
			"Hisense|HS-U936|U936|HS-U936",
			"Hisense|HS-U939|u939|HS-U939",
			"Hisense|HS-U939|u939e_1|HS-U939",
			"Hisense|HS-U939|u939e_3|HS-U939",
			"Hisense|HS-U939|u939e_4|HS-U939",
			"Hisense|HS-U939|u939e_5|HS-U939",
			"Hisense|HS-U950|U950_5|HS-U950",
			"Hisense|HS-U966|u966|HS-U966",
			"Hisense|HS-U966|u966ae|HS-U966",
			"Hisense|HS-U970|u970e_6|HS-U970",
			"Hisense|HS-U970|u970s_export|HS-U970",
			"Hisense|HS-U971|U971AE|HS-U971",
			"Hisense|HS-U978|u978|HS-U978",
			"Hisense|HS-U98|U98|U98",
			"Hisense|HS-U98|U98_FR|HS-U98",
			"Hisense|HS-U980|u980|HS-U980",
			"Hisense|HS-U980|u980be|HS-U980",
			"Hisense|HS-U980|u980be_1|HS-U980",
			"Hisense|HS-U980|u980be_2|HS-U980",
			"Hisense|HS-U980|u980ce_1|HS-U980",
			"Hisense|HS-U980|u980ce_3|HS-U980",
			"Hisense|HS-U980|u980ce_4|HS-U980",
			"Hisense|HS-U980|u980ce_5|HS-U980",
			"Hisense|HS-X5T|X5T|HS-X5T",
			"Hisense|HS-X68T|x68t|HS-X68T",
			"Hisense|HS-X6T|X6T|HS-X6T",
			"Hisense|HS-X8C|x8c|HS-X8C",
			"Hisense|HS-X8T|X8T|HS-X8T",
			"Hisense|HS-X8U|x8u|HS-X8U",
			"Hisense|Hisense E360M|e360m|Hisense E360M",
			"Hisense|Hisense E613M|E613M|Hisense E613M",
			"Hisense|Hisense E620M|E602M|Hisense E602M",
			"Hisense|Hisense E621T|E621T|Hisense E621T",
			"Hisense|Hisense EG606|EG606_TW|EG606",
			"Hisense|Hisense I631M|i631m|Hisense I631M",
			"Hisense|Hisense I632M|I632M|Hisense I632M",
			"Hisense|Hisense I632T|I632T|Hisense I632T",
			"Hisense|Hisense I639M|I639M|Hisense I639M",
			"Hisense|Hisense I639T|I639T|Hisense I639T",
			"Hisense|Hisense I639T|I639TA1|Hisense I639T",
			"Hisense|Hisense TV|hisense_gx1200v|hisense_gx1200v",
			"Hisense|I300T|I300T|Hisense I300T",
			"Hisense|I630U|I630U|Hisense I630U",
			"Hisense|K220_FHD|mst628|K220_FHD",
			"Hisense|L691|l691|HS-L691",
			"Hisense|L691|l691e_2|HS-L691",
			"Hisense|LED32G180|hisenseLED32G180|LED32G180",
			"Hisense|LED32K220|mst628|LED32K220",
			"Hisense|LED32K280J3D|mt5880|LED32K280J3D",
			"Hisense|LED32K360|mt5880|LED32K360",
			"Hisense|LED32K360J|mt5880|LED32K360J",
			"Hisense|LED32K360J3D|mt5880|LED32K360J3D",
			"Hisense|LED32K360X3D|mt5880|LED32K360X3D",
			"Hisense|LED32K370|mt5880|LED32K370",
			"Hisense|LED32K600J|mt5880|LED32K600J",
			"Hisense|LED32K600X3D|mt5880|LED32K600X3D",
			"Hisense|LED32K610X3D|mt5880|LED32K610X3D",
			"Hisense|LED32L288|mt5880|LED32L288",
			"Hisense|LED39K280J3D|mt5880|LED39K280J3D",
			"Hisense|LED39K360J|mt5880|LED39K360J",
			"Hisense|LED39K360X3D|mt5880|LED39K360X3D",
			"Hisense|LED39K600X3D|mt5880|LED39K600X3D",
			"Hisense|LED39K600X3DU|mst901|LED39K600X3DU",
			"Hisense|LED39K610X3D|mt5880|LED39K610X3D",
			"Hisense|LED39K680X3DU|mt5327|LED39K680X3DU",
			"Hisense|LED40K260X3D|mt5880|LED40K260X3D",
			"Hisense|LED40K360J|mt5880|LED40K360J",
			"Hisense|LED40K360X3D|mt5880|LED40K360X3D",
			"Hisense|LED40K370|mt5880|LED40K370",
			"Hisense|LED40K681X3DU|msd6a918|LED40K681X3DU",
			"Hisense|LED40L288|mt5880|LED40L288",
			"Hisense|LED42G200|hisenseLED42G200|LED42G200",
			"Hisense|LED42K280J3D|mt5880|LED42K280J3D",
			"Hisense|LED42K330X3D|mt5880|LED42K330X3D",
			"Hisense|LED42K360J|mt5880|LED42K360J",
			"Hisense|LED42K360X3D|mt5880|LED42K360X3D",
			"Hisense|LED42K370|mt5880|LED42K370",
			"Hisense|LED42K380U|msd6a918|LED42K380U",
			"Hisense|LED42K560X3D|hisenseLED42K560X3D|LED42K560X3D",
			"Hisense|LED42K600A3D|mt5880|LED42K600A3D",
			"Hisense|LED42K600X3D|mt5880|LED42K600X3D",
			"Hisense|LED42K610J3DP|mt5880|LED42K610J3DP",
			"Hisense|LED42K610X3D|mt5880|LED42K610X3D",
			"Hisense|LED42K660X3D|hisenseLED42K660X3D|LED42K660X3D",
			"Hisense|LED42K680X3DU|mt5327|LED42K680X3DU",
			"Hisense|LED42L288|mt5880|LED42L288",
			"Hisense|LED46K260X3D|mt5880|LED46K260X3D",
			"Hisense|LED46K280J3D|mt5880|LED46K280J3D",
			"Hisense|LED46K330X3D|mt5880|LED46K330X3D",
			"Hisense|LED46K360J|mt5880|LED46K360J",
			"Hisense|LED46K360X3D|mt5880|LED46K360X3D",
			"Hisense|LED46K660X3D|hisense6a801|LED46K660X3D",
			"Hisense|LED47K560J3D|hisenseLED47K560J3D|LED47K560J3D",
			"Hisense|LED47K600X3D|mt5880|LED47K600X3D",
			"Hisense|LED47K610J3DP|mt5880|LED47K610J3DP",
			"Hisense|LED48K360X3D|mt5880|LED48K360X3D",
			"Hisense|LED48K370|mt5880|LED48K370",
			"Hisense|LED48K380U|msd6a918|LED48K380U",
			"Hisense|LED48K681X3DU|msd6a918|LED48K681X3DU",
			"Hisense|LED48L288|mt5880|LED48L288",
			"Hisense|LED50G05|hisenseLED50G05|LED50G05",
			"Hisense|LED50K260X3D|mt5880|LED50K260X3D",
			"Hisense|LED50K360J|mt5880|LED50K360J",
			"Hisense|LED50K360X3D|mt5880|LED50K360X3D",
			"Hisense|LED50K370|mt5880|LED50K370",
			"Hisense|LED50K380U|msd6a918|LED50K380U",
			"Hisense|LED50K610X3D|mt5880|LED50K610X3D",
			"Hisense|LED50K660X3D|hisenseLED50K660X3D|LED50K660X3D",
			"Hisense|LED50K680X3DU|mt5327|LED50K680X3DU",
			"Hisense|LED50L288|mt5880|LED50L288",
			"Hisense|LED50XT880G3D|hisenseLED50XT880G3D|LED50XT880G3D",
			"Hisense|LED50XT880G3DU|mst901|LED50XT880G3DU",
			"Hisense|LED50XT900X3DU|mt5327|LED50XT900X3DU",
			"Hisense|LED55K260X3D|mt5880|LED55K260X3D",
			"Hisense|LED55K360X3D|mt5880|Hisense LED55K360X3D",
			"Hisense|LED55K370|mt5880|LED55K370",
			"Hisense|LED55K380U|msd6a918|LED55K380U",
			"Hisense|LED55K600A3D|mt5880|LED55K600A3D",
			"Hisense|LED55K600X3D|mt5880|LED55K600X3D",
			"Hisense|LED55K610X3D|mt5880|LED55K610X3D",
			"Hisense|LED55K680X3DU|mt5327|LED55K680X3DU",
			"Hisense|LED55L288|mt5880|LED55L288",
			"Hisense|LED55XT780G3D|hisenseLED55XT780G3D|LED55XT780G3D",
			"Hisense|LED55XT810X3DU|msd6a918|LED55XT810X3DU",
			"Hisense|LED55XT880G3DU|mst901|LED55XT880G3DU",
			"Hisense|LED55XT900X3DU|mt5327|LED55XT900X3DU",
			"Hisense|LED58K280J|mt5880|LED58K280J",
			"Hisense|LED58K280U|mt5327|LED58K280U",
			"Hisense|LED58K610X3D|mt5880|LED58K610X3D",
			"Hisense|LED58K680X3DU|mt5327|LED58K680X3DU",
			"Hisense|LED58XT880G3D|hisenseLED58XT880G3D|LED58XT880G3D",
			"Hisense|LED58XT880J3DU|mst901|LED58XT880J3DU",
			"Hisense|LED60K380|mt5327|LED60K380",
			"Hisense|LED65K560J3DTB|hisenseLED65K560J3DTB|LED65K560J3DTB",
			"Hisense|LED65K560J3DTR|hisenseLED65K560J3DTR|LED65K560J3DTR",
			"Hisense|LED65K600X3D|mt5880|LED65K600X3D",
			"Hisense|LED65K680X3DU|mt5327|LED65K680X3DU",
			"Hisense|LED65XT780G3D|hisenseLED65XT780G3D|LED65XT780G3D",
			"Hisense|LED65XT800X3DU|mt5327|LED65XT800X3DU",
			"Hisense|LED65XT810X3DU|msd6a918|LED65XT810X3DU",
			"Hisense|LED65XT880G3D|hisenseLED65XT880G3D|LED65XT880G3D",
			"Hisense|LED65XT880G3DF|hisenseLED65XT880G3DF|LED65XT880G3DF",
			"Hisense|LED65XT880G3DU|mst901|LED65XT880G3DU",
			"Hisense|LED65XT890G3D|hisenseLED65XT890G3D|LED65XT890G3D",
			"Hisense|LED65XT900G3DU|mst901|LED65XT900G3DU",
			"Hisense|LED65XT900X3DU|mt5327|LED65XT900X3DU",
			"Hisense|LED75XT890G3D|mst901|LED75XT890G3D",
			"Hisense|LED75XT910G3DU|mt5327|LED75XT910G3DU",
			"Hisense|LED84XT900G3D|hisenseLED84XT900G3D|LED84XT900G3D",
			"Hisense|LED85XT910G3DU|mt5327|Vision20",
			"Hisense|LT100K6900A|mt5327|LT100K6900A",
			"Hisense|M3101BCD|m3101bcd|M3101BCD",
			"Hisense|M470BSA|m470|M470BSA",
			"Hisense|M470BSD|m470bsd|M470BSD",
			"Hisense|M470BSE|m470bse|M470BSE",
			"Hisense|M470BSG|m470bsg|M470BSG",
			"Hisense|M470BSS|m470bss|M470BSS",
			"Hisense|M701|M701|M701",
			"Hisense|MEDION E4001|E4001|MEDION E4001",
			"Hisense|MEDION X4701|x4701|MEDION X4701",
			"Hisense|Oysters Pacific 800|x4701ae|Oysters Pacific 800",
			"Hisense|PX1900|atm7039c|PX1900",
			"Hisense|PX2000|hisensepx2000|PX2000",
			"Hisense|PX2700|g18ref|PX2700",
			"Hisense|PX2800|hisensepx2800|PX2800",
			"Hisense|PX3000|mt5880|Hisense PX3000",
			"Hisense|PX3000|mt5880|PX3000",
			"Hisense|PX3100|mt5880|PX3100",
			"Hisense|SKi630t|msm8226|SKi630t",
			"Hisense|Sero 7|rk3168|E2171CA",
			"Hisense|Sero 7|rk3168|E2171MX",
			"Hisense|Sero 7|rk3168|E2171TK",
			"Hisense|Sero 8|rk3188|E2281",
			"Hisense|Sero 8|rk3188|E2281CA",
			"Hisense|Sero 8|rk3188|E2281TK",
			"Hisense|Sero 8|rk3188|E2281UK",
			"Hisense|Sero 8 pro|rk3288|F5281",
			"Hisense|Sero7|rk3168|E2171SS",
			"Hisense|Sero7 LE|rk3026|E2371",
			"Hisense|Sero8|rk3188|E2281SS",
			"Hisense|SoundTab MA-317|rk3028a|Hisense SoundTab MA-317",
			"Hisense|U606AE|U606AE|HS-U606",
			"Hisense|U606AE|U606AE|PHS-402",
			"Hisense|U607|U607|Bouygues Telecom Bs 403",
			"Hisense|U609|U609|HS-U609",
			"Hisense|U610|U610|HS-U610",
			"Hisense|U800|U800|HS-U800",
			"Hisense|U800E-1|U800E_1|HS-U800E-1",
			"Hisense|U966|u966e_1|U966",
			"Hisense|U970|u970e_8|U970",
			"Hisense|U988|U988|HS-U988",
			"Hisense|U988E-1|U988E_1|HS-U988",
			"Hisense|U988E-2|U988E_2|HS-U988",
			"Hisense|VH777|VH777|VH777",
			"Hisense|Vidaa|helium3|H7",
			"Hisense|Vision|Vision|Vision",
			"Hisense|Vision 2.5|vision2_5|vision2_5",
			"Hisense|W270BD|W270BD|W270BD",
			"Hisense|WA912|U912ATW|WA912",
			"Hisense|WA960|U960Q_1|WA960",
			"Hisense|WM8|wm8980s|WM8",
			"Hisense|XOLO|us9230e1|XOLO T1000",
			"Hisense|l691|msm8226|HS-L691",
			"Hisense|vision2_1|vision2_1|vision2_1",
			"Honda|A-DA|e1853|14A-DA",
			"Honda|A-DA|e1853|MY15ADA",
			"Honda|MY15ADA|koelsch|MY15ADA",
			"Honeywell|Honeywell Dolphin 70e Black|dblack_gn|Dolphin 70e Black",
			"Honeywell|Honeywell Dolphin 70e Black|dblack_go|Dolphin 70e Black",
			"Honeywell|Honeywell Dolphin 70e Black|dblack_wn|Dolphin 70e Black",
			"Honeywell|Honeywell Dolphin 70e Black|dblack_wo|Dolphin 70e Black",
			"Huawei|A199|hwa199|HUAWEI A199",
			"Huawei|Ascend D|hwu9500|U9500",
			"Huawei|B199|hwB199|HUAWEI B199",
			"Huawei|C8812|hwc8812|HUAWEI C8812",
			"Huawei|C8813DQ|hwC8813DQ|HUAWEI C8813DQ",
			"Huawei|C8815|hwC8815|HUAWEI C8815",
			"Huawei|C8816|hwC8816|HUAWEI C8816",
			"Huawei|C8816D|hwC8816D|HUAWEI C8816D",
			"Huawei|C8817D|hwC8817D|C8817D",
			"Huawei|C8817E|hwC8817E|HUAWEI C8817E",
			"Huawei|C8817L|hwC8817L|HUAWEI C8817L",
			"Huawei|CM990|hwCM990|CM990",
			"Huawei|Che1-CL10|Che1|Che1-CL10",
			"Huawei|Che1-CL20|Che1|Che1-CL20",
			"Huawei|Che2-TL00|hwChe2|Che2-TL00",
			"Huawei|Che2-TL00H|hwChe2|Che2-TL00H",
			"Huawei|Che2-TL00M|hwChe2|Che2-TL00M",
			"Huawei|Che2-UL00|hwChe2|Che2-UL00",
			"Huawei|D2|hwD2-6070|HUAWEI D2-6070",
			"Huawei|G350|HWG350|HUAWEI G350-U20",
			"Huawei|G350|hwg350|HUAWEI G350",
			"Huawei|G350|hwg350|HUAWEI G350-U00",
			"Huawei|G506|hwG506-U151|HUAWEI G506-U151",
			"Huawei|G510|hwG510-0010|HUAWEI G510-0010",
			"Huawei|G510|hwG510-0100|HUAWEI G510-0100",
			"Huawei|G510|hwG510-0100|HuaweiG510-0100",
			"Huawei|G510|hwG510-0100|HuaweiG510-0100-orange",
			"Huawei|G520|hwG520-5000|HUAWEI G520-5000",
			"Huawei|G521-L076|HWG521-L|HUAWEI G521-L076",
			"Huawei|G521-L176|HWG521-L|HUAWEI G521-L176",
			"Huawei|G526|hwG526-L11|G526-L11",
			"Huawei|G526|hwG526-L22|G526-L22",
			"Huawei|G526|hwG526-L33|G526-L33",
			"Huawei|G527|hwG527-U081|G527-U081",
			"Huawei|G535-L11|hwG535-L11|HUAWEI G535-L11",
			"Huawei|G535-L11|hwG535-L11|Kestrel",
			"Huawei|G535-L11|hwG535-L11|Orange Gova",
			"Huawei|G535-L11|hwG535-L11|Speedsurfer",
			"Huawei|G535-L11|hwG535-L11|Ultym5",
			"Huawei|G6|hwG6-T00|HUAWEI G6-T00",
			"Huawei|G6-C00|hwG6-C00|HUAWEI G6-C00",
			"Huawei|G6-L11|hwG6-L11|HUAWEI G6-L11",
			"Huawei|G6-L22|hwG6-L22|HUAWEI G6-L22",
			"Huawei|G6-L33|hwG6-L33|HUAWEI G6-L33",
			"Huawei|G6-U00|hwG6-U00|HUAWEI G6-U00",
			"Huawei|G6-U10|hwG6-U10|HUAWEI G6-U10",
			"Huawei|G6-U251|hwG6-U251|HUAWEI G6-U251",
			"Huawei|G6-U34|hwG6-U34|HUAWEI G6-U34",
			"Huawei|G606|HWG606|HUAWEI G606-T00",
			"Huawei|G610|hwG610-T00|HUAWEI G610-T00",
			"Huawei|G610|hwG610-U00|G610-U00",
			"Huawei|G610|hwG610-U00|HUAWEI G610-U00",
			"Huawei|G610|hwG610-U30|HUAWEI G610-U30",
			"Huawei|G610-T01|hwG610-T01|HUAWEI G610-T01",
			"Huawei|G610-T11|hwG610-T11|HUAWEI G610-T11",
			"Huawei|G610-U15|hwG610-U15|HUAWEI G610-U15",
			"Huawei|G610-U20|hwG610-U20|HUAWEI G610-U20",
			"Huawei|G610C|hwG610-C00|HUAWEI G610-C00",
			"Huawei|G615-U10|hwG615-U10|G615-U10",
			"Huawei|G615-U10|hwG615-U10|HUAWEI G615-U10",
			"Huawei|G616-L076|HWG616-L|HUAWEI G616-L076",
			"Huawei|G620-L72|hwG620-L72|HUAWEI G620-L72",
			"Huawei|G620-L75|hwG620-L75|G620-L75",
			"Huawei|G620S-L01|hwG620S-L01|G620S-L01",
			"Huawei|G620S-L02|hwG620S-L02|G620S-L02",
			"Huawei|G620S-L03|hwG620S-L03|G620S-L03",
			"Huawei|G620S-L03|hwG620S-L03|HUAWEI G620",
			"Huawei|G620S-L03|hwG620S-L03|Personal Huawei G620S",
			"Huawei|G620S-UL00|hwG620S-UL00|G620S-UL00",
			"Huawei|G621-TL00|hwG621-TL00|G621-TL00",
			"Huawei|G621-TL00M|hwG621-TL00|G621-TL00M",
			"Huawei|G628-TL00|HWG628-TL|HUAWEI G628-TL00",
			"Huawei|G629-UL00|HWG629-UL|HUAWEI G629-UL00",
			"Huawei|G630-U00|hwG630-U00|HUAWEI G630-U00",
			"Huawei|G630-U10|hwG630-U10|G630-U10",
			"Huawei|G630-U20|hwG630-U20|G630-U20",
			"Huawei|G630-U251|hwG630-U251|G630-U251",
			"Huawei|G630-U251|hwG630-U251|HUAWEI G630-U251",
			"Huawei|G660-L075|hwG660-L075|HUAWEI G660-L075",
			"Huawei|G7-L03|hwG7-L03|G7-L03",
			"Huawei|G7-L03|hwG7-L03|HUAWEI G7",
			"Huawei|G7-L03|hwG7-L03|HUAWEI G7-L03",
			"Huawei|G7-TL00|hwG7-TL00|G7-TL00",
			"Huawei|G7-TL00|hwG7-TL00|HUAWEI G7-TL00",
			"Huawei|G700|hwG700-T00|HUAWEI G700-T00",
			"Huawei|G700|hwG700-U00|HUAWEI G700-U00",
			"Huawei|G700-T01|hwG700-T01|HUAWEI G700-T01",
			"Huawei|G700-U10|hwG700-U10|HUAWEI G700-U10",
			"Huawei|G700-U20|hwG700-U20|HUAWEI G700-U20",
			"Huawei|G716|hwG716-L070|HUAWEI G716-L070",
			"Huawei|G718|hwg718|HUAWEI G718",
			"Huawei|G730|hwG730-C00|HUAWEI G730-C00",
			"Huawei|G730|hwG730-T00|HUAWEI G730-T00",
			"Huawei|G730|hwG730-U00|HUAWEI G730-U00",
			"Huawei|G730-L075|hwG730-L075|HUAWEI G730-L075",
			"Huawei|G730-U10|hwG730-U10|HUAWEI G730-U10",
			"Huawei|G730-U251|hwG730-U251|HUAWEI G730-U251",
			"Huawei|G730-U27|hwG730-U27|HUAWEI G730-U27",
			"Huawei|G730-U30|hwG730-U30|HUAWEI G730-U30",
			"Huawei|G740|hwG740-L00|G740-L00",
			"Huawei|G740|hwG740-L00|Orange Yumo",
			"Huawei|G750-T00|hwG750-T00|HUAWEI G750-T00",
			"Huawei|G750-T01|hwG750-T01|HUAWEI G750-T01",
			"Huawei|G750-T01M|hwG750-T01M|HUAWEI G750-T01M",
			"Huawei|G750-T20|hwG750-T20|HUAWEI G750-T20",
			"Huawei|G750-U10|hwG750-U10|HUAWEI G750-U10",
			"Huawei|G7500|HWG7500|HUAWEI G7500",
			"Huawei|Gran_Roraima|hws7721u|Orinoquia Gran Roraima + S7-722u",
			"Huawei|H30-C00|hwH30-C00|H30-C00",
			"Huawei|H30-L01|hwH30|HONOR H30-L01",
			"Huawei|H30-L01M|hwH30|HONOR H30-L01M",
			"Huawei|H30-L02|hwH30|H30-L02",
			"Huawei|H30-L02|hwH30|HONOR H30-L02",
			"Huawei|H30-T00|hwH30-T00|H30-T00",
			"Huawei|H60|hwH60|H60-L01",
			"Huawei|H60-J1|hwH60|HW-H60-J1",
			"Huawei|H60-L01|hwH60|H60-L01",
			"Huawei|H60-L02|hwH60|H60-L02",
			"Huawei|H60-L03|hwH60|H60-L03",
			"Huawei|H60-L04|hwH60|H60-L04",
			"Huawei|H60-L11|hwH60|H60-L11",
			"Huawei|H60-L12|hwH60|H60-L12",
			"Huawei|H60-L21|hwH60|H60-L21",
			"Huawei|H60-L21|hwH60-L21|H60-L21",
			"Huawei|H882L|hwh882l|H882L",
			"Huawei|HUAWEI C199|hwC199|HUAWEI C199",
			"Huawei|HUAWEI G630|hwG630-T00|HUAWEI G630-T00",
			"Huawei|HUAWEI G7-L01|hwG7-L01|G7-L01",
			"Huawei|HUAWEI G7-UL20|hwG7-UL20|HUAWEI G7-UL20",
			"Huawei|HUAWEI H871G|hwh871g|HUAWEI H871G",
			"Huawei|HUAWEI MT7-J1|hwmt7|HUAWEI MT7-J1",
			"Huawei|HUAWEI MT7-L09|hwmt7|HUAWEI MT7-L09",
			"Huawei|HUAWEI MediaPad 10 Link+|hws10231l|402HW",
			"Huawei|HUAWEI MediaPad 10 Link+|hws10231l|MediaPad 10 Link+",
			"Huawei|HUAWEI MediaPad 10 Link+|hws10231l|S10-232L",
			"Huawei|HUAWEI MediaPad 10 Link+|hws10231l|SpeedTAB",
			"Huawei|HUAWEI MediaPad 7 Youth2|hws7721u|MediaPad 7 Youth 2",
			"Huawei|HUAWEI MediaPad 7 Youth2|hws7721u|S7-721u",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|403HW",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|CNPC Security Pad S1",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|MediaPad M1 8.0",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|MediaPad M1 8.0 (LTE)",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|MediaPad M1 8.0 (WIFI)",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|S8-303L",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|S8-303LT",
			"Huawei|HUAWEI MediaPad M1 8.0|hws8301l|S8-306L",
			"Huawei|HUAWEI MediaPad X1 7.0|hw7d501l|7D-501u",
			"Huawei|HUAWEI MediaPad X1 7.0|hw7d501l|MediaPad X1",
			"Huawei|HUAWEI MediaPad X1 7.0|hw7d501l|MediaPad X1 7.0",
			"Huawei|HUAWEI MediaPad X1 7.0|hw7d501l|X1 7.0",
			"Huawei|HUAWEI P6|hwp6-u06|HUAWEI Ascend P6",
			"Huawei|HUAWEI P6|hwp6-u06|HUAWEI P6-U06",
			"Huawei|HUAWEI P6|hwp6-u06|HUAWEI P6-U06-orange",
			"Huawei|HUAWEI P7-L11|hwp7|HUAWEI P7-L11",
			"Huawei|HUAWEI Y536-A1|hwY536A1|HUAWEI Y536A1",
			"Huawei|HUAWEI Y610|HWY610-U|HUAWEI Y610-U00",
			"Huawei|HUAWEI Y618|HWY618-T|HUAWEI Y618-T00",
			"Huawei|HW-03E|HW-03E|HW-03E",
			"Huawei|Hol-T00|HWHol-T|Hol-T00",
			"Huawei|Hol-U10|HWHol-U|Hol-U10",
			"Huawei|Hol-U19|HWHol-U|Hol-U19",
			"Huawei|Honor3|hwH30-T10|H30-T10",
			"Huawei|Honor3|hwH30-U10|H30-U10",
			"Huawei|Honor3|hwhn3-u00|HUAWEI HN3-U00",
			"Huawei|Honor3|hwhn3-u01|HUAWEI HN3-U01",
			"Huawei|IDEOS|U8150|Comet",
			"Huawei|IDEOS|U8150|Ideos",
			"Huawei|IDEOS|u8150|Ideos",
			"Huawei|Ice-Twist|Ice-Twist|Ice-Twist",
			"Huawei|M220|hwmediaqm220|M220",
			"Huawei|M220|hwmediaqm220|M220c",
			"Huawei|M220|hwmediaqm220|dTV01",
			"Huawei|M310|hwsingleboxm310w|M310",
			"Huawei|M620|M620|M620",
			"Huawei|M620|M620|TB01",
			"Huawei|M835|hwm835|HUAWEI-M835",
			"Huawei|M860|M860|M860",
			"Huawei|M865|hwm865|M865",
			"Huawei|M865|hwm865|USCCADR3305",
			"Huawei|M868|hwm868|HUAWEI M868",
			"Huawei|MS4C|hwMS4C|MS4C",
			"Huawei|MT2-L01|hwmt2-l01|HUAWEI MT2-L01",
			"Huawei|MT2-L02|hwmt2-l02|HUAWEI MT2-L02",
			"Huawei|MT2-L03|hwMT2L03|MT2L03",
			"Huawei|MT2-L05|hwmt2-l05|HUAWEI MT2-L05",
			"Huawei|MT2L03LITE|hwMT2L03LITE|MT2L03",
			"Huawei|MT7-CL00|hwmt7|HUAWEI MT7-CL00",
			"Huawei|MT7-TL00|hwmt7|HUAWEI MT7-TL00",
			"Huawei|MT7-TL10|hwmt7|HUAWEI MT7-TL10",
			"Huawei|MT7-UL00|hwmt7|HUAWEI MT7-UL00",
			"Huawei|Mate|hwmt1-t00|HUAWEI MT1-T00",
			"Huawei|Mate2|hwmt2-c00|HUAWEI MT2-C00",
			"Huawei|MediaPad Vogue|hws7601u|MediaPad 7 Lite II",
			"Huawei|MediaPad Vogue|hws7601u|MediaPad 7 Vogue",
			"Huawei|MediaPad Youth|hws7701w|MediaPad 7 Youth",
			"Huawei|MediaPad7|hws7601us|MediaPad 7 Vogue",
			"Huawei|MediaPad7|hws7601w|MediaPad 7 Vogue",
			"Huawei|P2|hwp2-6070|HUAWEI P2-6070",
			"Huawei|P6|hwp6-c00|HUAWEI P6-C00",
			"Huawei|P6|hwp6-t00|HUAWEI P6-T00",
			"Huawei|P6|hwp6-t00|HUAWEI P6-T00V",
			"Huawei|P6S|hwP6s-l04|P6 S-L04",
			"Huawei|P6S-L04|hwp6s-l04|302HW",
			"Huawei|P6S-U06|hwp6s-u06|HUAWEI P6 S-U06",
			"Huawei|P7 mini|hwP7Mini|HUAWEI P7 mini",
			"Huawei|P7-L00|hwp7|HUAWEI P7-L00",
			"Huawei|P7-L05|hwp7|HUAWEI P7-L05",
			"Huawei|P7-L07|hwp7|HUAWEI P7-L07",
			"Huawei|P7-L09|hwp7|HUAWEI P7-L09",
			"Huawei|P7-L10|hwp7|HUAWEI P7-L10",
			"Huawei|P7-L12|hwp7|HUAWEI P7-L12",
			"Huawei|Prism II|hwu8686|Prism II",
			"Huawei|S10|hws10103l|MediaPad 10 FHD",
			"Huawei|S10|hws10201l|MediaPad 10 LINK",
			"Huawei|S7|hws7601c|MediaPad 7 Vogue",
			"Huawei|S7|hws7601w|MediaPad 7 Vogue",
			"Huawei|S7|hws7701u|MediaPad 7 Youth",
			"Huawei|S7|hws7930u|Orinoquia Roraima S7-932u",
			"Huawei|S7|hws7951w|MediaPad 7 Lite+",
			"Huawei|S7|hws7951w|Telpad Dual S",
			"Huawei|S7|hws7961w|Telpad QS",
			"Huawei|S7|hws7961w|Telpad Quad S",
			"Huawei|SC-CL00|HWSC-CL00|HUAWEI SC-CL00",
			"Huawei|T-Mobile Pulse|U8100|Huawei_8100-9",
			"Huawei|T-Mobile Pulse|U8100|Tactile internet",
			"Huawei|T-Mobile Pulse|U8100|U8100",
			"Huawei|T-Mobile Pulse|U8100|Videocon_V7400",
			"Huawei|T101|hwt101|T-101",
			"Huawei|T101|hwt101|T101 PAD",
			"Huawei|T102|hwt102|QH-10",
			"Huawei|T102|hwt102|T102 PAD",
			"Huawei|T801|hwt801|T801 PAD",
			"Huawei|T802|hwt802|MT-803G",
			"Huawei|T802|hwt802|T802 PAD",
			"Huawei|U8120|U8120|Vodafone 845",
			"Huawei|U8220|U8220|Pulse",
			"Huawei|U8220|U8220|U8220",
			"Huawei|U8220|U8220|U8220PLUS",
			"Huawei|U8230|U8230|U8230",
			"Huawei|U8687|hwu8687|Huawei-U8687",
			"Huawei|V858|hwu8160|Huawei 858",
			"Huawei|V858|hwu8160|MTC 950",
			"Huawei|V858|hwu8160|MTC Mini",
			"Huawei|V858|hwu8160|Vodafone 858",
			"Huawei|Vogue7|hws7601u|MediaPad 7 Classic",
			"Huawei|Vogue7|hws7601u|MediaPad 7 Lite II",
			"Huawei|Vogue7|hws7601u|MediaPad 7 Vogue",
			"Huawei|Y210|oay210|Orinoquia Auyantepui Y210",
			"Huawei|Y220|HWY220-U|Y220-U00",
			"Huawei|Y220|HWY220-U|Y220-U05",
			"Huawei|Y220|HWY220-U|Y220-U17",
			"Huawei|Y220-T10|hwy220-t10|HUAWEI Y220-T10",
			"Huawei|Y220-U10|HWY220-U|Y220-U10",
			"Huawei|Y221-U03|HWY221-U|HUAWEI Y221-U03",
			"Huawei|Y221-U12|HWY221-U|HUAWEI Y221-U12",
			"Huawei|Y221-U22|HWY221-U|HUAWEI Y221-U22",
			"Huawei|Y221-U33|HWY221-U|HUAWEI Y221-U33",
			"Huawei|Y221-U43|HWY221-U|HUAWEI Y221-U43",
			"Huawei|Y221-U53|HWY221-U|HUAWEI Y221-U53",
			"Huawei|Y300|hwY300-0100|HUAWEI Ascend Y300",
			"Huawei|Y300|hwY300-0100|HUAWEI Y300-0100",
			"Huawei|Y300|hwY300-0151|HUAWEI Y300-0151",
			"Huawei|Y300|hwY300-0151|Pelephone-Y300-",
			"Huawei|Y300-0000|hwY300-0000|HUAWEI Y300-0000",
			"Huawei|Y301A1|hwY301A1|Huawei Y301A1",
			"Huawei|Y301A2|hwY301A2|Huawei Y301A2",
			"Huawei|Y320|hwy320-c00|HUAWEI Y320-C00",
			"Huawei|Y320-T00|HWY320-T00|HUAWEI Y320-T00",
			"Huawei|Y320-U01|HWY320|HUAWEI Y320-U01",
			"Huawei|Y320-U01|HWY320|Y320-U01",
			"Huawei|Y320-U10|HWY320-U|HUAWEI Y320-U10",
			"Huawei|Y320-U151|HWY320-U|HUAWEI Y320-U151",
			"Huawei|Y320-U30|HWY320-U|HUAWEI Y320-U30",
			"Huawei|Y320-U351|HWY320-U|HUAWEI Y320-U351",
			"Huawei|Y321|HWY321-U|HUAWEI Y321-U051",
			"Huawei|Y321|hwy321-c00|HUAWEI Y321-C00",
			"Huawei|Y325-T00|HWY325-T|HUAWEI Y325-T00",
			"Huawei|Y330|hwY330-U05|Bucare Y330-U05",
			"Huawei|Y330|hwY330-U05|HUAWEI Y330-U05",
			"Huawei|Y330|hwY330-U21|HUAWEI Y330-U21",
			"Huawei|Y330-C00|hwY330-C00|HUAWEI Y330-C00",
			"Huawei|Y330-U01|hwY330-U01|HUAWEI Y330-U01",
			"Huawei|Y330-U01|hwY330-U01|Luno",
			"Huawei|Y330-U07|hwY330-U07|HUAWEI Y330-U07",
			"Huawei|Y330-U08|hwY330-U08|HUAWEI Y330-U08",
			"Huawei|Y330-U11|hwY330-U11|HUAWEI Y330-U11",
			"Huawei|Y330-U11|hwY330-U11|V8510",
			"Huawei|Y330-U15|hwY330-U15|HUAWEI Y330-U15",
			"Huawei|Y330-U17|hwY330-U17|HUAWEI Y330-U17",
			"Huawei|Y336-A1|hwY336-A1|HUAWEI Y336-A1",
			"Huawei|Y340-U081|hwY340-U081|Y340-U081",
			"Huawei|Y360-U03|HWY360-U|HUAWEI Y360-U03",
			"Huawei|Y511-T00|HWY511-T|HUAWEI Y511-T00",
			"Huawei|Y511-T00|HWY511-T|Y511-T00",
			"Huawei|Y511-U00|HWY511-U|Y511-U00",
			"Huawei|Y511-U10|HWY511-U|HUAWEI Y511-U10",
			"Huawei|Y511-U251|HWY511-U|HUAWEI Y511-U251",
			"Huawei|Y511-U30|HWY511-U|HUAWEI Y511-U30",
			"Huawei|Y511-U30|HWY511-U|VIETTEL V8506",
			"Huawei|Y516-|HWY516-T|HUAWEI Y516-T00",
			"Huawei|Y518-T00|HWY518-T|HUAWEI Y518-T00",
			"Huawei|Y520-U03|HWY520-U|HUAWEI Y520-U03",
			"Huawei|Y520-U12|HWY520-U|HUAWEI Y520-U12",
			"Huawei|Y520-U22|HWY520-U|HUAWEI Y520-U22",
			"Huawei|Y520-U33|HWY520-U|HUAWEI Y520-U33",
			"Huawei|Y523-L076|HWY523|HUAWEI Y523-L076",
			"Huawei|Y523-L176|HWY523|HUAWEI Y523-L176",
			"Huawei|Y530|hwY530-U00|HUAWEI Y530-U00",
			"Huawei|Y530-U051|hwY530-U051|HUAWEI Y530",
			"Huawei|Y530-U051|hwY530-U051|HUAWEI Y530-U051",
			"Huawei|Y535|HWY535-C00|HUAWEI Y535-C00",
			"Huawei|Y535D-C00|HWY535D-C00|HUAWEI Y535D-C00",
			"Huawei|Y540-U01|HWY540-U|HUAWEI Y540-U01",
			"Huawei|Y550-L01|hwY550-L01|HUAWEI Y550-L01",
			"Huawei|Y550-L02|hwY550-L02|HUAWEI Y550-L02",
			"Huawei|Y550-L02|hwY550-L02|Y550-L02",
			"Huawei|Y550-L03|hwY550-L03|HUAWEI Y550",
			"Huawei|Y550-L03|hwY550-L03|HUAWEI Y550-L03",
			"Huawei|Y550-L03|hwY550-L03|Personal Huawei Y550",
			"Huawei|Y550-L03|hwY550-L03|Y550-L03",
			"Huawei|Y600|HWY600-U|HUAWEI Y600-U00",
			"Huawei|Y600|HWY600-U|HUAWEI Y600-U151",
			"Huawei|Y600|HWY600-U|HUAWEI Y600-U20",
			"Huawei|Y600-U351|HWY600-U|HUAWEI Y600-U351",
			"Huawei|Y600-U40|HWY600-U|HUAWEI Y600-U40",
			"Huawei|Y600D-C00|HWY600D-C00|HUAWEI Y600D-C00",
			"Huawei|Y625-U03|KVY625-U|Kavak Y625-U03",
			"Huawei|Y625-U13|HWY625-U|HUAWEI Y625-U13",
			"Huawei|Y625-U21|HWY625-U|HUAWEI Y625-U21",
			"Huawei|Y625-U32|HWY625-U|HUAWEI Y625-U32",
			"Huawei|Y625-U43|HWY625-U|HUAWEI Y625-U43",
			"Huawei|Youth|hws7701u|Orinoquia Gran Roraima S7-702u",
			"Huawei|eH811|hweH811|eH811",
			"Huawei|hw204HW|hw204HW|204HW",
			"Huawei|m881|hwm881|HUAWEI M881",
			"Huawei|t1_8p0|hws8701|MediaPad T1 8.0",
			"Huawei|t1_8p0|hws8701|S8-701u",
			"Huawei|t1_8p0|hws8701|S8-701w",
			"ICL-KME CS OJSC|TR10CD1|TR10CD1_7|TR10CD1",
			"ICRAIG|ICRAIG CMP 756|Omega9|CMP 756",
			"ICRAIG|ICRAIG CMP 759|Omega7|CMP 759",
			"IDEA|i4901|i4901|i4901",
			"IPRODA|T1144|T1144|T1144",
			"Ice Phone|Ice Phone Mini|ice_phone_mini|Ice-Phone Mini",
			"Impecca|ET7050D|ET7050D|ET7050D",
			"Inet-Tablet|D711HC_PG|D711HC_PG|D711HC_PG",
			"Inet-Tablet|D7911L3BC_OGS|D7911L3BC_OGS|D7911L3BC_OGS",
			"Infinity|Infinity-10.1-v2|Infinity-101-v2|Infinity-10.1-v2",
			"Infinix|Infinix X505|X505|Infinix X505",
			"Infinix|X405|up09_infinix_x405_gms|Infinix X405",
			"Infocus|Amazing A8|D78|Amazing A8",
			"Infocus|CS1 8.0|cs180|CS180",
			"Infocus|IN610|MG2|IN610",
			"Infocus|IN810|VKY|IN810",
			"Infocus|IN815|VK3|IN815",
			"Infocus|InFocus FP-U320-INF-WLTW|U320-WF|FP-U320-INF-WLTW",
			"Infocus|InFocus IN260|IN260|IN260",
			"Infocus|InFocus IN335|IVM|IN335",
			"Infocus|InFocus M2|LSO|InFocus M2",
			"Infocus|InFocus M2+|LSC|InFocus M2PLUS",
			"Infocus|InFocus M210|M210|M210",
			"Infocus|InFocus M310|H1W|InFocus M310",
			"Infocus|InFocus M320|M320|InFocus M320",
			"Infocus|InFocus M320e|M320e|InFocus M320e",
			"Infocus|InFocus M510t|TID|InFocus M510t",
			"Infocus|InFocus M810|VNA|InFocus M810",
			"Infocus|InFocus M810t|VNA|InFocus M810t",
			"Infocus|InFocus NewTab F1|f1|NewTab F1",
			"Infocus|Infocus M320e|M320e|InFocus M320e",
			"Infocus|M210|M210|M210",
			"Infocus|M2_3G|G10|InFocus M2_3G",
			"Infocus|M310|H1W|InFocus M310",
			"Infocus|M320|M320|InFocus M320",
			"Infocus|M320e|M320e|InFocus M320e",
			"Infocus|M320m|M320m|InFocus M320m",
			"Infocus|M320u|M320|InFocus M320u",
			"Infocus|M330|D77|InFocus M330",
			"Infocus|M510|TID_CHT|M510",
			"Infocus|M510t|TID|InFocus M510t",
			"Infocus|M511|TID_CHT|M510",
			"Infocus|M511|TID_TWN|M511",
			"Infocus|M512|MC2_CN|InFocus M512",
			"Infocus|M512|MC2_CN|M512",
			"Infocus|M518|MC2_APT|M518",
			"Infocus|Smart503|MC2|Smart503",
			"Infocus|T3000|P7M|T3000",
			"Inhon|L55|L55|L55",
			"Insignia|INSIGNIA FLEX 8\\xe2\\x80\\x99\\xe2\\x80\\x99 LTE Android Tablet|ns_15t8lte|NS-15T8LTE",
			"Insignia|INSIGNIA NS-14T002|ns_14t002|NS-14T002",
			"Insignia|Insignia Flex 10.1|ns_14t004|NS-14T004",
			"Insignia|\"Insignia Flex 8\"\"\"|ns_15at08|NS-15AT08",
			"Insignia|NS-15AT07|NS-15AT07|NS-15AT07",
			"Insignia|NS-15AT10|TA120|NS-15AT10",
			"Intel|AQ710A|Salitpa|AQ710A",
			"Intel|Etisalat E-20|zeeyabeach|BT230",
			"Intel|Orange San Diego|AZ210A|AZ210A",
			"Intel|Orange avec Intel Inside|AZ210B|AZ210B",
			"Intel|TR10CD1(PVT2)|TR10CD1_4|TR10CD1",
			"Intel|TR10CD1(PVT2)|TR10CD1_6|TR10CD1",
			"Intel|TR10CS1 (PVT2)|TR10CS1_4|TR10CS1",
			"Intel|Xolo X500|zeeyabeach|Xolo_X500",
			"Intel|Yolo|zeeyabeach|BT210",
			"Intenso|Intenso Tab814S|Tab814S|Tab814S",
			"Intenso|Tab734|Tab734|Tab734",
			"Intermec|CN51|CN51_NC0|CN51 NC0",
			"Intermec|CN51|CN51_NCF|CN51 NCF",
			"Intermec|CN51|CN51_NCU|CN51 NCU",
			"Intermec|CN51|CN51_NN0|CN51 NN0",
			"Intermec|CN51|CN51_QC0|CN51 QC0",
			"Intermec|CN51|CN51_QCF|CN51 QCF",
			"Intermec|CN51|CN51_QCU|CN51 QCU",
			"Intermec|CN51|CN51_QN0|CN51 QN0",
			"Itworks|TM705|TM705|NID_7010",
			"JLab|JLab_PRO-7|polaris-p1gms|JLab PRO-7",
			"JVC|DM65UXR\\\\DM65USR\\\\DM85UXR|mstarnapoli_atsc|DM65UXR",
			"KAZAM|KAZAM TV 4.5|KOT49H|KAZAM TV 45",
			"KAZAM|KAZAM TV 4.5|TV_45|KAZAM TV 45",
			"KAZAM|KAZAM Thunder 345|Thunder_345|KAZAM Thunder 345",
			"KAZAM|KAZAM Thunder 345 LTE|Thunder_345_LTE|KAZAM Thunder 345 LTE",
			"KAZAM|KAZAM Thunder 345 LTE|Thunder_345_LTE|Thunder3 45 LTE",
			"KAZAM|KAZAM Thunder 345L|Thunder_345L|KAZAM Thunder 345L",
			"KAZAM|KAZAM Thunder 347|Thunder_347|Thunder 347",
			"KAZAM|KAZAM Thunder 350L|Thunder_350L|KAZAM Thunder 350L",
			"KAZAM|KAZAM Thunder2 45L|Thunder2_45|KAZAM THUNDER2 45L",
			"KAZAM|KAZAM Thunder2 50|Thunder2_50|KAZAM Thunder2 50",
			"KAZAM|KAZAM Tornado 348|Tornado_348|Tornado 348",
			"KAZAM|KAZAM Tornado 350|KAZAM_Tornado_350|KAZAM Tornado 350",
			"KAZAM|KAZAM Tornado2 50|Tornado2_50|KAZAM Tornado2 50",
			"KAZAM|KAZAM Trooper 450|KAZAM_Trooper_450|KAZAM Trooper 450",
			"KAZAM|KAZAM Trooper 451|Trooper_451|KAZAM Trooper 451",
			"KAZAM|KAZAM Trooper2 40|Trooper2_40|KAZAM Trooper2 40",
			"KAZAM|KAZAM Trooper2 45|Trooper2_45|Trooper2 45",
			"KAZAM|KAZAM Trooper2 50|Trooper2_50|KAZAM Trooper2 50",
			"KAZAM|KAZAM Trooper2 60|Trooper2_60|Kazam Trooper2 60",
			"KAZAM|TROOPER X3.5|KAZAM|Trooper_X35",
			"KAZAM|TROOPER X4.0|KAZAM|Trooper_X40",
			"KAZAM|TROOPER X5.5|KAZAM|Trooper_X55",
			"KD Interactive|KD Interactive Kurio10S|C13300A|Kurio10S",
			"KD Interactive|KD Interactive Kurio4S|C13200A|Kurio4S",
			"KD Interactive|KD Interactive Kurio7S|C13000A|Kurio7S",
			"KD Interactive|Kurio Phone|C14500|KurioPhone",
			"KD Interactive|Kurio Tab - Extrem|g03k|C14100",
			"KLIPAD|KLIPAD_SMART_D71|KLIPAD_SMART_D71|KLIPAD_SMART_D71",
			"KLIPAD|KLIPAD_SMART_D791|KLIPAD_SMART_D791|KLIPAD_SMART_D791",
			"KOOOK|C600|GBC_bravo|C600",
			"KT Tech|EV-S100|s100|EV-S100",
			"KT Tech|TAKE SUIT|e100|KM-E100",
			"Karbonn|Sparkle V|Sparkle_V_sprout|Sparkle V",
			"Kennex|MD7053G|MD7053G|MD7053G",
			"Ketablet|Ketablet|TR10CS1_6|TR10CS1",
			"Kiano|Slim_Tab_8|Slim_Tab_8|Slim_Tab_8",
			"Kobo|Arc 10HD|macallan|arc 10HD",
			"Kobo|Arc 7|lbp8|arc 7",
			"Kobo|Arc 7HD|cardhu|arc 7HD",
			"Kobo|Kobo Arc|zeus|Arc",
			"Kobo|Kobo Vox|pegasus|Vox",
			"Kodak|SP4|SP4|SP4",
			"Kyocera|BASIO|KYV32|KYV32",
			"Kyocera|Brigadier|E6782|E6782",
			"Kyocera|DIGNO|KYI11|ISW11K",
			"Kyocera|DIGNO|KYL22|KYL22",
			"Kyocera|DIGNO Dual|WX04K|WX04K",
			"Kyocera|DIGNO R|202K|202K",
			"Kyocera|DIGNO S|KYL21|KYL21",
			"Kyocera|DIGNO T|302KC|302KC",
			"Kyocera|DM015K|DM015K|DM015K",
			"Kyocera|DuraForce|E6560|KYOCERA-E6560",
			"Kyocera|DuraForce|E6560C|E6560C",
			"Kyocera|DuraForce|E6560L|E6560L",
			"Kyocera|DuraForce|E6762|USCC-E6762",
			"Kyocera|Echo|KSP8000|KSP8000",
			"Kyocera|Echo|M9300|M9300",
			"Kyocera|Event|C5133|Event",
			"Kyocera|HONEY BEE|101K|101K",
			"Kyocera|Honeybee Touch|201K|201K",
			"Kyocera|Hydro|C5170|C5170",
			"Kyocera|Hydro|C5171|Hydro",
			"Kyocera|Hydro EDGE|C5215|C5215",
			"Kyocera|Hydro Elite|C6750|C6750",
			"Kyocera|Hydro ICON|C6730|C6730",
			"Kyocera|Hydro LIFE|C6530|C6530",
			"Kyocera|Hydro LIFE|C6530N|C6530N",
			"Kyocera|Hydro PLUS|C5171R|Hydro",
			"Kyocera|Hydro PLUS|C5171R|Hydro_PLUS",
			"Kyocera|Hydro VIBE|C6725|C6725",
			"Kyocera|Hydro XTRM|C6522|C6522",
			"Kyocera|Hydro XTRM|C6522N|C6522N",
			"Kyocera|Hydro XTRM|C6721|USCC-C6721",
			"Kyocera|KC-01|KC-01|KC-01",
			"Kyocera|LUCE|KCP01K|KCP01K",
			"Kyocera|Milano|C5120|C5120",
			"Kyocera|Milano|C5121|C5121",
			"Kyocera|Rise|C5155|C5155",
			"Kyocera|Rise|C5156|Rise",
			"Kyocera|TORQUE|SKT01|SKT01",
			"Kyocera|TORQUE G01|KYY24|KYY24",
			"Kyocera|Torque|E6710|Torque",
			"Kyocera|TorqueXT|E6715|E6715",
			"Kyocera|URBANO L01|KYY21|KYY21",
			"Kyocera|URBANO L02|KYY22|KYY22",
			"Kyocera|URBANO L03|KYY23|KYY23",
			"Kyocera|URBANO PROGRESSO|KYY04|URBANO PROGRESSO",
			"Kyocera|URBANO V01|KYV31|KYV31",
			"Kyocera|WX10K|WX10K|WX10K",
			"Kyocera|Zio|SCP-8600|Zio",
			"Kyocera|miraie|KYL23|KYL23",
			"Kyocera|zio|msm7627_kb60|Zio",
			"Kyocera|zio|msm7627_surf|Zio",
			"Kyocera|zio|pls8600|Zio",
			"Kyowon|All&G PAD|mypad2|KA-E410W",
			"LGE|070 touch|w3voip|LG-FL40L",
			"LGE|Ally|aloha|AS740",
			"LGE|Ally|aloha|Ally",
			"LGE|Ally|aloha|Aloha",
			"LGE|Ally|aloha|US740",
			"LGE|Android TV|cosmo|LG Google TV",
			"LGE|Android TV|eden|LG Android TV V4",
			"LGE|Android TV|eden|LG Google TV G3 KR",
			"LGE|Android TV G3|eden|LG Google TV G3",
			"LGE|Enact|fx3q|VS890 4G",
			"LGE|Escape|l1a|LG-P870",
			"LGE|Eve|EVE|GW620",
			"LGE|Eve|EVE|LG GW620",
			"LGE|Eve|EVE|LG GW620F",
			"LGE|Eve|EVE|LG GW620R",
			"LGE|Eve|EVE|LG GW620g",
			"LGE|Eve|EVE|LG KH5200",
			"LGE|Eve|EVE|LG-GW620",
			"LGE|Eve|EVE|LG-KH5200",
			"LGE|F60|e2|LG-D390",
			"LGE|F60|e2|LG-D390AR",
			"LGE|F60|e2ds|LG-D392",
			"LGE|F60|e2n|LG-D390n",
			"LGE|F60|e2nac|LG-D393",
			"LGE|F60|e2nam|LGMS395",
			"LGE|F60|e2nas|LGLS660",
			"LGE|F60|e2nav|VS810PP",
			"LGE|G Pad 10.1|e9wifi|LG-V700",
			"LGE|G Pad 10.1|e9wifin|LG-V700n",
			"LGE|G Pad 10.1 LTE|e9lte|LG-VK700",
			"LGE|G Pad 10.1 LTE|e9lte|VK700",
			"LGE|G Pad 7.0|e7wifi|LG-V400",
			"LGE|G Pad 7.0|e7wifi|LG-V400Y1",
			"LGE|G Pad 7.0 LTE|e7lte|LG-V410",
			"LGE|G Pad 7.0 LTE|e7lte|LGUK410",
			"LGE|G Pad 8.0|e8wifi|LG-V480",
			"LGE|G Pro Lite|luv90ds|LG-D685",
			"LGE|G Pro Lite|luv90ds|LG-D686",
			"LGE|G Pro Lite|luv90nfc|LG-D683",
			"LGE|G Pro Lite|luv90nfc|LG-D684",
			"LGE|G Pro Lite|luv90ss|LG-D680",
			"LGE|G Pro Lite|luv90ss|LG-D681",
			"LGE|G Pro Lite|luv90ss|LG-D682",
			"LGE|G Pro Lite|luv90ss|LG-D682TR",
			"LGE|G Pro2|b1|LG-F350K",
			"LGE|G Pro2|b1|LG-F350L",
			"LGE|G Pro2|b1|LG-F350S",
			"LGE|G Pro2|b1w|LG-D838",
			"LGE|G Vista|b2l|LG-D631",
			"LGE|G Vista|b2lds|LG-D690",
			"LGE|G Vista|x10|VS880",
			"LGE|G Watch|dory|G Watch",
			"LGE|G pad 8.0|e8wifi|LG-V480",
			"LGE|G pad 8.0 LTE|e8lte|LG-V490",
			"LGE|G2 MINI|g2m|LG-D620",
			"LGE|G2 MINI|g2mds|LG-D618",
			"LGE|G2 MINI|g2mss|LG-D610",
			"LGE|G2 MINI|g2mss|LG-D610AR",
			"LGE|G2 MINI|g2mss|LG-D610TR",
			"LGE|G2 MINI|g2mv|LG-D625",
			"LGE|G2 mini 4G LTE|g2mv|LG-D625",
			"LGE|G3 Beat|jagdsnm|LG-D726",
			"LGE|G3 Beat|jagdsnm|LG-D728",
			"LGE|G3 Beat|jagdsnm|LG-D729",
			"LGE|G3 Beat|jagn|LG-F470K",
			"LGE|G3 Beat|jagn|LG-F470L",
			"LGE|G3 Beat|jagn|LG-F470S",
			"LGE|G3 Beat|jagnm|LG-D722J",
			"LGE|G3 S|jag3gds|LG-D724",
			"LGE|G3 S|jag3gss|LG-D723",
			"LGE|G3 S|jagnm|LG-D722",
			"LGE|G3 S|jagnm|LG-D722AR",
			"LGE|G3 S|jagnm|LG-D725PR",
			"LGE|G3 Screen|liger|LG-F490L",
			"LGE|G3 Stylus|b2ldsn|LG-D690n",
			"LGE|G3 Stylus|b2lss|LG-D693",
			"LGE|G3 Stylus|b2lss|LG-D693AR",
			"LGE|G3 Stylus|b2lss|LG-D693TR",
			"LGE|G3 Stylus|b2lssn|LG-D693n",
			"LGE|G3 Vigor|jagc|LGLS885",
			"LGE|G3 Vigor|jagnm|LG-D725",
			"LGE|GA7800|eden|LG Android TV V4",
			"LGE|GPAD 7.0 LTE|e7lte|VK410",
			"LGE|Gpad 7.0|e7wifi|LG-V400",
			"LGE|Gx2|b2ln|LG-F430L",
			"LGE|Intuition|batman_vzw|VS950 4G",
			"LGE|L Bello|luv80ds|LG-D335",
			"LGE|L Bello|luv80ds|LG-D335E",
			"LGE|L Bello|luv80ds|LG-D337",
			"LGE|L Fino|l70p|LG-D290",
			"LGE|L Fino|l70p|LG-D290AR",
			"LGE|L Fino|l70pds|LG-D295",
			"LGE|L Fino|l70pn|LG-D290",
			"LGE|L-01F|g2|L-01F",
			"LGE|L20|luv20ds|LG-D105",
			"LGE|L20|luv20ss|LG-D100",
			"LGE|L20|luv20ss|LG-D100AR",
			"LGE|L20|luv20ts|LG-D107",
			"LGE|L30 Sporty|luv30ds|LG-D125",
			"LGE|L30 Sporty|luv30ss|LG-D120",
			"LGE|L30 Sporty|luv30ss|LG-D120AR",
			"LGE|L45|lo_2|LG-X130g",
			"LGE|L45|lo_2_ds|LG-X132",
			"LGE|L50 Sporty|luv50ds|LG-D221",
			"LGE|L50 Sporty|luv50ds|LG-D227",
			"LGE|L50 Sporty|luv50ss|LG-D213",
			"LGE|L50 Sporty|luv50ss|LG-D213AR",
			"LGE|L50 Sporty|luv50ssn|LG-D213",
			"LGE|L60|lo_1|LG-X135",
			"LGE|L60|lo_1|LG-X137",
			"LGE|L60|lo_1|LG-X140",
			"LGE|L60|lo_1|LG-X145",
			"LGE|L60|lo_1|LG-X147",
			"LGE|L65|w55|LG-D280",
			"LGE|L65|w55ds|LG-D285",
			"LGE|L65|w55n|LG-D280",
			"LGE|L7 II|vee7e|LG-P712",
			"LGE|L7 II|vee7e|LG-P714",
			"LGE|L7 II Dual|vee7ds|LG-P716",
			"LGE|L70|w5|LG-D320",
			"LGE|L70|w5|LG-D320AR",
			"LGE|L70|w5|LGAS323",
			"LGE|L70|w5c|LGLS620",
			"LGE|L70|w5ds|LG-D325",
			"LGE|L70|w5n|LG-D320",
			"LGE|L70|w5n|LG-D329",
			"LGE|L70|w5ts|LG-D340f8",
			"LGE|L70 CDMA|w5c|LGL41C",
			"LGE|L90|w7|LG-D400",
			"LGE|L90|w7|LG-D405",
			"LGE|L90|w7|LG-D415",
			"LGE|L90|w7ds|LG-D410",
			"LGE|L90|w7n|LG-D400",
			"LGE|L90|w7n|LG-D405",
			"LGE|L90 Dual|w7dsn|LG-D410",
			"LGE|LBello|luv80ss|LG-D331",
			"LGE|LBello|luv80ss|LG-D331AR",
			"LGE|LG Connect 4G|cayman|LG-MS840",
			"LGE|LG DOUBLEPLAY|lgc729|LG-C729",
			"LGE|LG Enlighten|VS700|LG-VS700",
			"LGE|LG Enlighten|VS700|LG-VS700PP",
			"LGE|LG Esteem|MS910|LG-MS910",
			"LGE|LG F70|f70|LGL31L",
			"LGE|LG F70|f70n|LG-D315",
			"LGE|LG F70|f70n|LG-D315l",
			"LGE|LG F70|f70n|LG-F370K",
			"LGE|LG F70|f70n|LG-F370L",
			"LGE|LG F70|f70n|LG-F370S",
			"LGE|LG G Flex|zee|LG-D950",
			"LGE|LG G Flex|zee|LG-D950G",
			"LGE|LG G Flex|zee|LG-D951",
			"LGE|LG G Flex|zee|LG-D955",
			"LGE|LG G Flex|zee|LG-D956",
			"LGE|LG G Flex|zee|LG-D958",
			"LGE|LG G Flex|zee|LG-D959",
			"LGE|LG G Flex|zee|LG-F340K",
			"LGE|LG G Flex|zee|LG-F340L",
			"LGE|LG G Flex|zee|LG-F340S",
			"LGE|LG G Flex|zee|LG-LS995",
			"LGE|LG G Flex|zee|LGL23",
			"LGE|LG G Flex2|z2|LG-F510K",
			"LGE|LG G Flex2|z2|LG-F510L",
			"LGE|LG G Flex2|z2|LG-F510S",
			"LGE|LG G Pad 8.0 LTE|e8lte|LG-P490L",
			"LGE|LG G Pad 8.3|awifi|LG-V500",
			"LGE|LG G Pad 8.3 Google Play Edition|palman|LG-V510",
			"LGE|LG G Pad 8.3 LTE|altev|VK810 4G",
			"LGE|LG G Pad 8.3 homeBoy|awifi070u|LG-V507L",
			"LGE|LG G2|g2|LG-D800",
			"LGE|LG G2|g2|LG-D801",
			"LGE|LG G2|g2|LG-D802",
			"LGE|LG G2|g2|LG-D802T",
			"LGE|LG G2|g2|LG-D802TR",
			"LGE|LG G2|g2|LG-D803",
			"LGE|LG G2|g2|LG-D805",
			"LGE|LG G2|g2|LG-D806",
			"LGE|LG G2|g2|LG-F320K",
			"LGE|LG G2|g2|LG-F320L",
			"LGE|LG G2|g2|LG-F320S",
			"LGE|LG G2|g2|LG-LS980",
			"LGE|LG G2|g2|VS980 4G",
			"LGE|LG G3|g3|AS985",
			"LGE|LG G3|g3|LG-AS990",
			"LGE|LG G3|g3|LG-D850",
			"LGE|LG G3|g3|LG-D851",
			"LGE|LG G3|g3|LG-D852",
			"LGE|LG G3|g3|LG-D852G",
			"LGE|LG G3|g3|LG-D855",
			"LGE|LG G3|g3|LG-D856",
			"LGE|LG G3|g3|LG-D857",
			"LGE|LG G3|g3|LG-D858",
			"LGE|LG G3|g3|LG-D858HK",
			"LGE|LG G3|g3|LG-D859",
			"LGE|LG G3|g3|LG-F400K",
			"LGE|LG G3|g3|LG-F400L",
			"LGE|LG G3|g3|LG-F400S",
			"LGE|LG G3|g3|LGL24",
			"LGE|LG G3|g3|LGLS990",
			"LGE|LG G3|g3|LGUS990",
			"LGE|LG G3|g3|LGV31",
			"LGE|LG G3|g3|VS985 4G",
			"LGE|LG G3 A|tigers|LG-F410S",
			"LGE|LG G3 Cat.6|tiger6|LG-F460K",
			"LGE|LG G3 Cat.6|tiger6|LG-F460L",
			"LGE|LG G3 Cat.6|tiger6|LG-F460S",
			"LGE|LG Gx|omega|LG-F310L",
			"LGE|LG Gx|omegar|LG-F310LR",
			"LGE|LG L39C|l4ii_cdma|LGL39C",
			"LGE|LG L80 Dual|w6ds|LG-D380",
			"LGE|LG L80 Dual|w6ds|LG-D385",
			"LGE|LG L80 Single|w6|LG-D370",
			"LGE|LG L80 Single|w6|LG-D373",
			"LGE|LG L80 Single|w6|LG-D375",
			"LGE|LG L80 Single|w6|LG-D375AR",
			"LGE|LG Lucid|cayman|VS840 4G",
			"LGE|LG Lucid|cayman|VS840PP",
			"LGE|LG Lucid 2|l1v|VS870 4G",
			"LGE|LG Lucid 3|x5|VS876",
			"LGE|LG Mach\\xe2\\x84\\xa2|l2s|LG-LS860",
			"LGE|LG Motion 4G|l0|LG-MS770",
			"LGE|LG OPTIMUS M+|m3_mpcs_us|LG-MS695",
			"LGE|LG OPTIMUS ZIP|lgl75c|LGL75C",
			"LGE|LG Optimus 3D|cosmopolitan|LG-P920",
			"LGE|LG Optimus 3D|cosmopolitan|LG-P925g",
			"LGE|LG Optimus 3D|cosmopolitan|LG-SU760",
			"LGE|LG Optimus ELITE\\xe2\\x84\\xa2|m3s|LG-VM696",
			"LGE|LG Optimus Elite|m3s|LG Optimus Elite",
			"LGE|LG Optimus Elite|m3s|LG-LS696",
			"LGE|LG Optimus F3Q|fx3q|LG-D520",
			"LGE|LG Optimus F7|fx1|LG-AS780",
			"LGE|LG Optimus F7|fx1|LG-LG870",
			"LGE|LG Optimus F7|fx1|LG-US780",
			"LGE|LG Optimus G|geeb|LG-E970",
			"LGE|LG Optimus G|geeb|LG-E971",
			"LGE|LG Optimus G|geeb|LG-E973",
			"LGE|LG Optimus G|geehdc|L-01E",
			"LGE|LG Optimus G|geehdc|LGL21",
			"LGE|LG Optimus G|geehrc|LG-E975",
			"LGE|LG Optimus G|geehrc|LG-E975K",
			"LGE|LG Optimus G|geehrc|LG-E975T",
			"LGE|LG Optimus G|geehrc|LG-E976",
			"LGE|LG Optimus G|geehrc|LG-E977",
			"LGE|LG Optimus G|geehrc|LG-E987",
			"LGE|LG Optimus G|geehrc|LG-F180K",
			"LGE|LG Optimus G|geehrc|LG-F180S",
			"LGE|LG Optimus G|geehrc4g|LG-F180L",
			"LGE|LG Optimus G|geehrc4g|LG-LS970",
			"LGE|LG Optimus G Pro|geefhd|LG-E980",
			"LGE|LG Optimus G Pro|geefhd|LG-E980h",
			"LGE|LG Optimus G Pro|geefhd|LG-E981h",
			"LGE|LG Optimus G Pro|geefhd|LG-E986",
			"LGE|LG Optimus G Pro|geefhd|LG-E988",
			"LGE|LG Optimus G Pro|geefhd|LG-E989",
			"LGE|LG Optimus G Pro|geefhd|LG-F240K",
			"LGE|LG Optimus G Pro|geefhd|LG-F240S",
			"LGE|LG Optimus G Pro|geefhd4g|LG-F240L",
			"LGE|LG Optimus HUB|univa_222-01|LG-E510",
			"LGE|LG Optimus HUB|univa_arb-xx|LG-E510",
			"LGE|LG Optimus HUB|univa_cis-xxx|LG-E510",
			"LGE|LG Optimus HUB|univa_esa-xx|LG-E510",
			"LGE|LG Optimus HUB|univa_eur-xx|LG-E510",
			"LGE|LG Optimus HUB|univa_open-eu|LG-E510",
			"LGE|LG Optimus HUB|univa_tur-xx|LG-E510",
			"LGE|LG Optimus L1II|v1|LG-E410",
			"LGE|LG Optimus L1II|v1|LG-E410B",
			"LGE|LG Optimus L1II|v1|LG-E410c",
			"LGE|LG Optimus L1II|v1|LG-E410f",
			"LGE|LG Optimus L1II|v1|LG-E410g",
			"LGE|LG Optimus L1II|v1|LG-E410i",
			"LGE|LG Optimus L1II|v1|LG-E411f",
			"LGE|LG Optimus L1II|v1|LG-E411g",
			"LGE|LG Optimus L1II|v1ds|LG-E415f",
			"LGE|LG Optimus L1II|v1ds|LG-E415g",
			"LGE|LG Optimus L1II|v1ds|LG-E420",
			"LGE|LG Optimus L1II|v1ds|LG-E420f",
			"LGE|LG Optimus L1II|v1ts|LG-E475f",
			"LGE|LG Optimus L3|e0|LG-E400",
			"LGE|LG Optimus L3|e0|LG-E400R",
			"LGE|LG Optimus L3|e0|LG-E400b",
			"LGE|LG Optimus L3|e0|LG-E400f",
			"LGE|LG Optimus L3|e0|LG-E400g",
			"LGE|LG Optimus L3|e0|LG-L38C",
			"LGE|LG Optimus L3|e0|LGL35G",
			"LGE|LG Optimus L3|e0_open_eur|LG-E400",
			"LGE|LG Optimus L3 Dual|e1|LG-E405",
			"LGE|LG Optimus L3 Dual|e1|LG-E405f",
			"LGE|LG Optimus L3 II|vee3ds|LG-E435",
			"LGE|LG Optimus L3 II|vee3ds|LG-E435f",
			"LGE|LG Optimus L3 II|vee3ds|LG-E435g",
			"LGE|LG Optimus L3 II|vee3ds|LG-E435k",
			"LGE|LG Optimus L3 II|vee3e|LG-E425",
			"LGE|LG Optimus L3 II|vee3e|LG-E425c",
			"LGE|LG Optimus L3 II|vee3e|LG-E425f",
			"LGE|LG Optimus L3 II|vee3e|LG-E425g",
			"LGE|LG Optimus L3 II|vee3e|LG-E425j",
			"LGE|LG Optimus L3 II|vee3e|LG-E430",
			"LGE|LG Optimus L3 II|vee3e|LG-E431g",
			"LGE|LG Optimus L4 II|vee4ss|LG-E440",
			"LGE|LG Optimus L4 II|vee4ss|LG-E440f",
			"LGE|LG Optimus L4 II|vee4ss|LG-E440g",
			"LGE|LG Optimus L4 II|vee4ss|LG-E465f",
			"LGE|LG Optimus L4 II|vee4ss|LG-E465g",
			"LGE|LG Optimus L4 II Dual|vee4ds|LG-E445",
			"LGE|LG Optimus L4 II Dual|vee4ds|LG-E467f",
			"LGE|LG Optimus L4 II Tri|vee4ts|LG-E470f",
			"LGE|LG Optimus L5 Dual|m4ds|LG-E615",
			"LGE|LG Optimus L5 Dual|m4ds|LG-E615f",
			"LGE|LG Optimus L5 II|vee5ds|LG-E455",
			"LGE|LG Optimus L5 II|vee5ds|LG-E455f",
			"LGE|LG Optimus L5 II|vee5ds|LG-E455g",
			"LGE|LG Optimus L5 II|vee5nfc|LG-E460",
			"LGE|LG Optimus L5 II|vee5nfc|LG-E460f",
			"LGE|LG Optimus L5 II|vee5ss|LG-E450",
			"LGE|LG Optimus L5 II|vee5ss|LG-E450B",
			"LGE|LG Optimus L5 II|vee5ss|LG-E450f",
			"LGE|LG Optimus L5 II|vee5ss|LG-E450g",
			"LGE|LG Optimus L5 II|vee5ss|LG-E450j",
			"LGE|LG Optimus L5 II|vee5ss|LG-E451g",
			"LGE|LG Optimus L5 II|vee5ss|LG-E460",
			"LGE|LG Optimus L7|u0|LG-P700",
			"LGE|LG Optimus L7|u0|LG-P705",
			"LGE|LG Optimus L7|u0|LG-P705f",
			"LGE|LG Optimus L7|u0|LG-P705g",
			"LGE|LG Optimus L7|u0|LG-P708g",
			"LGE|LG Optimus L7|u0|LG-T280",
			"LGE|LG Optimus L7|u0|LGL96G",
			"LGE|LG Optimus L7|vee7ds|LG-P715",
			"LGE|LG Optimus L7 II|vee7e|LG-P710",
			"LGE|LG Optimus L7 II|vee7e|LG-P712",
			"LGE|LG Optimus L7 II|vee7e|LG-P713",
			"LGE|LG Optimus L7 II|vee7e|LG-P713GO",
			"LGE|LG Optimus L7 II|vee7e|LG-P713TR",
			"LGE|LG Optimus L7 II|vee7e|LG-P714",
			"LGE|LG Optimus L70|w5|LG-D321",
			"LGE|LG Optimus L70|w5|LGMS323",
			"LGE|LG Optimus L7II|vee7ds|LG-P715",
			"LGE|LG Optimus L7II|vee7ds|LG-P716",
			"LGE|LG Optimus L9|u2|LG-D700",
			"LGE|LG Optimus L9|u2|LG-P760",
			"LGE|LG Optimus L9|u2|LG-P765",
			"LGE|LG Optimus L9|u2|LG-P768",
			"LGE|LG Optimus L9|u2|LG-P769",
			"LGE|LG Optimus L9|u2|LG-P778",
			"LGE|LG Optimus L9|u2|LGMS769",
			"LGE|LG Optimus L9 (NFC)|u2|LG-P760",
			"LGE|LG Optimus L9 II|l9ii|LG-D605",
			"LGE|LG Optimus LTE Tag|cayman|LG-AS840",
			"LGE|LG Optimus LTE Tag|cayman|LG-F120K",
			"LGE|LG Optimus LTE Tag|cayman|LG-F120L",
			"LGE|LG Optimus LTE Tag|cayman|LG-F120S",
			"LGE|LG Optimus LTE Tag|lge_120_kt|LG-F120K",
			"LGE|LG Optimus LTE Tag|lge_120_skt|LG-F120S",
			"LGE|LG Optimus LTE3|fx1sk|LG-F260S",
			"LGE|LG Optimus One|thunderg|LG-P500",
			"LGE|LG Optimus One|thunderg|LG-P500h",
			"LGE|LG Optimus One|thunderg|LG-P503",
			"LGE|LG Optimus One|thunderg|LG-P504",
			"LGE|LG Optimus One|thunderg|LG-P505",
			"LGE|LG Optimus One|thunderg|LG-P505CH",
			"LGE|LG Optimus One|thunderg|LG-P505R",
			"LGE|LG Optimus One|thunderg|LG-P506",
			"LGE|LG Optimus One|thunderg|LG-P509",
			"LGE|LG Optimus Regard|l0|LG-LW770",
			"LGE|LG Optimus Select|u0_cdma|LG-AS730",
			"LGE|LG Optimus Zone 2|w3c|VS415PP",
			"LGE|LG Pecan|pecan|LG-P350",
			"LGE|LG Pecan|pecan|LG-P350f",
			"LGE|LG Pecan|pecan|LG-P350g",
			"LGE|LG Revolution|bryce|VS910 4G",
			"LGE|LG Spectrum|VS920|VS920 4G",
			"LGE|LG Thrill 4G|cosmopolitan|LG-P925",
			"LGE|LG Venice|u0_cdma|LG-LG730",
			"LGE|LG Viper 4G LTE|cayman|LG-LS840",
			"LGE|LG Volt|my70ds|LG-H422",
			"LGE|LG Volt|x5|LGLS740",
			"LGE|LG optimus LTE2|d1lkt|LG-F160K",
			"LGE|LG optimus LTE2|d1lsk|LG-F160S",
			"LGE|LG optimus LTE2|d1lu|LG-F160L",
			"LGE|LG optimus LTE2|d1lu|LG-F160LV",
			"LGE|LG optimus it L-05D|l_dcm|L-05D",
			"LGE|LG optimus it L-05E|L05E|L-05E",
			"LGE|LG-AS876|x5|AS876",
			"LGE|LG-D150|w35|LG-D150",
			"LGE|LG-D157f|w35ds|LG-D157f",
			"LGE|LG-E985T|gvarfhd|LG-E985",
			"LGE|LG-E985T|gvarfhd|LG-E985T",
			"LGE|LGL22|g2|LGL22",
			"LGE|Marquee|L-07C|L-07C",
			"LGE|Marquee|LG855|LG-LG855",
			"LGE|Marquee|LS855|LG-LS855",
			"LGE|Marquee|bproj_CIS-xxx|LG-P970",
			"LGE|My touch 4G|e739|LG-E739",
			"LGE|Nexus 4|mako|Nexus 4",
			"LGE|Nexus 5|hammerhead|Nexus 5",
			"LGE|Optiimus Black|bproj_208-01|LG-P970",
			"LGE|Optimus  EX|x2|LG-SU880",
			"LGE|Optimus  LTE|l1a|LG-P870",
			"LGE|Optimus 2|as680|LG-AS680",
			"LGE|Optimus 2X|p990|LG-P990",
			"LGE|Optimus 2X|p990|LG-P990H",
			"LGE|Optimus 2X|p990|LG-P990h",
			"LGE|Optimus 2X|p990_262-xx|LG-P990",
			"LGE|Optimus 2X|p990_CIS-xxx|LG-P990",
			"LGE|Optimus 2X|p990_EUR-xx|LG-P990",
			"LGE|Optimus 2X|p990hN|LG-P990hN",
			"LGE|Optimus 2X|p999|LG-P999",
			"LGE|Optimus 2X|star|LG-P990",
			"LGE|Optimus 2X|star|LG-SU660",
			"LGE|Optimus 2X|star_450-05|LG-SU660",
			"LGE|Optimus 2X|su660|LG-SU660",
			"LGE|Optimus 3D|cosmo_450-05|LG-SU760",
			"LGE|Optimus 3D|cosmo_EUR-XXX|LG-P920",
			"LGE|Optimus 3D|cosmo_MEA-XXX|LG-P920",
			"LGE|Optimus 3D|p920|LG-P920",
			"LGE|Optimus 3D|p920|LG-P920h",
			"LGE|Optimus 3D|su760|LG-SU760",
			"LGE|Optimus 3D Cube|cx2|LG-SU870",
			"LGE|Optimus 3D MAX|cx2|LG-P720",
			"LGE|Optimus 3D MAX|cx2|LG-P720h",
			"LGE|Optimus 3D MAX|cx2|LG-P725",
			"LGE|Optimus 3D MAX|cx2|LG-SU870",
			"LGE|Optimus 4X HD|x3|LG-P880",
			"LGE|Optimus 4X HD|x3|LG-P880g",
			"LGE|Optimus Big|justin|LG-LU6800",
			"LGE|Optimus Big|lu6800|LG-LU6800",
			"LGE|Optimus Black|LGL85C|LGL85C",
			"LGE|Optimus Black|black|LG-KU5900",
			"LGE|Optimus Black|blackg|LG-P970",
			"LGE|Optimus Black|blackg|LG-P970h",
			"LGE|Optimus Black|bproj_214-03|LG-P970",
			"LGE|Optimus Black|bproj_262-XXX|LG-P970",
			"LGE|Optimus Black|bproj_302-220|LG-P970g",
			"LGE|Optimus Black|bproj_334-020|LG-P970h",
			"LGE|Optimus Black|bproj_724-xxx|LG-P970h",
			"LGE|Optimus Black|bproj_ARE-XXX|LG-P970",
			"LGE|Optimus Black|bproj_EUR-XXX|LG-P970",
			"LGE|Optimus Black|bproj_sea-xxx|LG-P970",
			"LGE|Optimus Black|ku5900|LG-KU5900",
			"LGE|Optimus Black|lgp970|LG-P970",
			"LGE|Optimus Black|lgp970|LG-P970g",
			"LGE|Optimus Black|lgp970|LG-P970h",
			"LGE|Optimus Chat|hazel|LG-C550",
			"LGE|Optimus Chat|hazel|LG-C555",
			"LGE|Optimus Core|u0_cdma|LGL86C",
			"LGE|Optimus EX|x2|IS11LG",
			"LGE|Optimus EX|x2_450-05|LG-SU880",
			"LGE|Optimus Exceed 2|w5c|LG-VS450PP",
			"LGE|Optimus F3|fx3|LG-LS720",
			"LGE|Optimus F3|fx3|LG-P655H",
			"LGE|Optimus F3|fx3|LG-P655K",
			"LGE|Optimus F3|fx3|LG-P659",
			"LGE|Optimus F3|fx3|LG-P659H",
			"LGE|Optimus F3|fx3|LGL25L",
			"LGE|Optimus F3|fx3|LGMS659",
			"LGE|Optimus F5|l1e|LG-P870h",
			"LGE|Optimus F5|l1e|LG-P875",
			"LGE|Optimus F5|l1e|LG-P875h",
			"LGE|Optimus F5|l1v|AS870 4G",
			"LGE|Optimus F6|f6|LG-D500",
			"LGE|Optimus F6|f6|LG-D505",
			"LGE|Optimus F6|f6|LGMS500",
			"LGE|Optimus Fuel|w3c|LGL34C",
			"LGE|Optimus G Pro|geevl04e|L-04E",
			"LGE|Optimus GJ|geehdc|LG-E975w",
			"LGE|Optimus GK|gvfhd|LG-F220K",
			"LGE|Optimus Hub|lgc800|LG-C800",
			"LGE|Optimus Hub|lgc800g|LG-C800G",
			"LGE|Optimus Hub|univa_214-04|LG-E510",
			"LGE|Optimus Hub|univa_724-05|LG-E510f",
			"LGE|Optimus Hub|univa_730-01|LG-E510g",
			"LGE|Optimus Hub|univa_730-03|LG-E510g",
			"LGE|Optimus Hub|univa_740-01|LG-E510g",
			"LGE|Optimus Hub|univa_clr-br|LG-E510f",
			"LGE|Optimus Hub|univa_ctm-xxx|LG-E510g",
			"LGE|Optimus Hub|univa_ent-cl|LG-E510g",
			"LGE|Optimus Hub|univa_open-br|LG-E510f",
			"LGE|Optimus Hub|univa_open-de|LG-E510",
			"LGE|Optimus Hub|univa_ssv-xxx|LG-E510g",
			"LGE|Optimus Hub|univa_tcl-mx|LG-E510f",
			"LGE|Optimus Hub|univa_tlf-es|LG-E510",
			"LGE|Optimus Hub|univa_ufn-mx|LG-E510g",
			"LGE|Optimus Hub|univa_usc-mx|LG-E510g",
			"LGE|Optimus Hub|univa_viv-br|LG-E510f",
			"LGE|Optimus L40|w3|LG-D160",
			"LGE|Optimus L40|w3|LG-D165",
			"LGE|Optimus L40|w3|LG-D165AR",
			"LGE|Optimus L40|w3ds|LG-D170",
			"LGE|Optimus L40|w3ds|LG-D175f",
			"LGE|Optimus L40|w3ts|LG-D180f",
			"LGE|Optimus L5|m4|LG-E610",
			"LGE|Optimus L5|m4|LG-E610v",
			"LGE|Optimus L5|m4|LG-E612",
			"LGE|Optimus L5|m4|LG-E612f",
			"LGE|Optimus L5|m4|LG-E612g",
			"LGE|Optimus L5|m4|LG-E617G",
			"LGE|Optimus L5|m4|LG-L40G",
			"LGE|Optimus L7II|vee7ds|LG-P716",
			"LGE|Optimus LIFE|l1_dcm|L-02E",
			"LGE|Optimus LTE|i_dcm|L-01D",
			"LGE|Optimus LTE|i_skt|LG-SU640",
			"LGE|Optimus LTE|i_u|LG-LU6200",
			"LGE|Optimus LTE|iproj|LG-P936",
			"LGE|Optimus LTE|lgp930|LG-P930",
			"LGE|Optimus LTE|lgp935|LG-P935",
			"LGE|Optimus Mach|LU3000|LG-LU3000",
			"LGE|Optimus Mach|hub|LG-LU3000",
			"LGE|Optimus Mach|lu3000|LG-LU3000",
			"LGE|Optimus Net|gelato_302-610|LG-P690b",
			"LGE|Optimus Net|gelato_cis-xx|LG-P690",
			"LGE|Optimus Net|gelato_sea-xx|LG-P690",
			"LGE|Optimus Net|lgl45c|LGL45C",
			"LGE|Optimus Net Dual|gelatods_are-xxx|LG-P698",
			"LGE|Optimus Net Dual|gelatods_cis-xxx|LG-P698",
			"LGE|Optimus Net Dual|gelatods_ind-xxx|LG-P698",
			"LGE|Optimus Net Dual|gelatods_open-br|LG-P698f",
			"LGE|Optimus Net Dual|gelatods_sea-xxx|LG-P698",
			"LGE|Optimus One|ku3700|LG-KU3700",
			"LGE|Optimus One|lu3700|LG-LU3700",
			"LGE|Optimus One|su370|LG-SU370",
			"LGE|Optimus One|thunder_kor-05|LG-SU370",
			"LGE|Optimus One|thunder_kor-08|LG-KU3700",
			"LGE|Optimus One|thunder_kor-08|LG-LU3700",
			"LGE|Optimus One|thunderc|LG-CX670",
			"LGE|Optimus One|thunderc|LG-LW690",
			"LGE|Optimus One|thunderc|LG-MS690",
			"LGE|Optimus One|thunderc|LG-US670",
			"LGE|Optimus One|thunderc|LS670",
			"LGE|Optimus One|thunderc|VM670",
			"LGE|Optimus One|thunderc|Vortex",
			"LGE|Optimus One|thunderc|thunderc",
			"LGE|Optimus PAD LTE|express|LG-LU8300",
			"LGE|Optimus Pad|l06c|L-06C",
			"LGE|Optimus Pad|v900|LG-V900",
			"LGE|Optimus Pad|v900asia|LG-V900",
			"LGE|Optimus Pad|v901ar|LG-V901",
			"LGE|Optimus Pad|v901kr|LG-V901",
			"LGE|Optimus Pad|v901tr|LG-V901",
			"LGE|Optimus Pad|v905r|LG-V905R",
			"LGE|Optimus Pad|v909|LG-V909",
			"LGE|Optimus Pad|v909mkt|LG-V909",
			"LGE|Optimus Plus|m3_acg_us|LG-AS695",
			"LGE|Optimus Pro|muscat|LG-C660",
			"LGE|Optimus Pro|muscat|LG-C660R",
			"LGE|Optimus Pro|muscat|LG-C660h",
			"LGE|Optimus Q|lgl55c|LGL55C",
			"LGE|Optimus Q2|bssq|LG-LU6500",
			"LGE|Optimus Q2|bssq_450-06|LG-LU6500",
			"LGE|Optimus Slider|VM701|LG-VM701",
			"LGE|Optimus Sol|victor|LG-E730",
			"LGE|Optimus Sol|victor|LG-E730f",
			"LGE|Optimus Spirit|gelato_505-01|LG-P690f",
			"LGE|Optimus Spirit|gelato_eur-xx|LG-P690",
			"LGE|Optimus Vu|325|LG-F100L",
			"LGE|Optimus Vu|batman|LG-F100L",
			"LGE|Optimus Vu|batman|LG-F100S",
			"LGE|Optimus Vu|batman_dcm|L-06DJOJO",
			"LGE|Optimus Vu|batman_lgu|LG-F100L",
			"LGE|Optimus Vu|batman_skt|LG-F100S",
			"LGE|Optimus Vu|lge_325_skt|LG-F100S",
			"LGE|Optimus Vu|vu10|LG-P895",
			"LGE|Optimus Vu|vu10|LG-P895qb",
			"LGE|Optimus Vu2|vu2kt|LG-F200K",
			"LGE|Optimus Vu2|vu2sk|LG-F200S",
			"LGE|Optimus Vu2|vu2u|LG-F200L",
			"LGE|Optimus Vu2|vu2u|LG-F200LS",
			"LGE|Optimus Vu:|batman_dcm|L-06D",
			"LGE|Optimus Z|su950|SU950",
			"LGE|Optimus Zone|e0v|LG-VS410PP",
			"LGE|Optimus chat|elini|L-04C",
			"LGE|PRADA 3.0|p2|L-02D",
			"LGE|PRADA 3.0|p2|LG-KU5400",
			"LGE|PRADA 3.0|p2|LG-LU5400",
			"LGE|PRADA 3.0|p2|LG-P940",
			"LGE|PRADA 3.0|p2|LG-P940h",
			"LGE|PRADA 3.0|p2|LG-SU540",
			"LGE|PecanV|pecanV|LG-P355",
			"LGE|Shine Plus with Google|alohag|LG-C710h",
			"LGE|Smart Dios V8700|SE_TF|ref_SCTF",
			"LGE|Spectrum|i_vzw|VS920 4G",
			"LGE|Spectrum 2|d1lv|VS930 4G",
			"LGE|Spirit 4G|l1m|LG-MS870",
			"LGE|Splendor|u0_cdma|LG-US730",
			"LGE|Spray|e2jps|402LG",
			"LGE|Swift|swift|GT540",
			"LGE|Swift|swift|GT540GO",
			"LGE|Swift|swift|GT540R",
			"LGE|Swift|swift|GT540f",
			"LGE|Thrill 4G|cosmo_310-410|LG-P925",
			"LGE|Thrill 4G|p925|LG-P925",
			"LGE|VU3|vu3|LG-F300K",
			"LGE|VU3|vu3|LG-F300L",
			"LGE|VU3|vu3|LG-F300S",
			"LGE|Wine Smart|vfp|LG-D486",
			"LGE|Wine Smart|vfp|LG-F480K",
			"LGE|Wine Smart|vfp|LG-F480L",
			"LGE|Wine Smart|vfp|LG-F480S",
			"LGU+|LAP250U|hg2|LAP250U",
			"LGU+|ST940I-UP|tvg2|ST940I-UP",
			"LGUplus|TV G|smartbox|TI320-DU",
			"LIAONING YIYATONG|DISNEY_Tablet_PC|DISNEY_Tablet_PC|DS2310-70LP",
			"LIAONING YIYATONG|IRULU_MID|IRULU_MID|X10",
			"LT|LT610|LT610|LT610",
			"LT|W2|EG606_YE|W2",
			"LUVO|LUVO 001|luvo_001|Luvo 001",
			"LUVO|LUVO-001L|LUVO-001L|LUVO 001L",
			"LUXYA|LUXYA MID704DC Tablet / Bitmore Tab770|MY7317P|MID704DC",
			"Lanix|ILIUM PAD E10Si|iLium_Pad_E10Si_1|ILIUM PAD E10Si",
			"Lanix|ILIUM S670|Ilium_S670|ILIUM S670",
			"Lava|XOLO LT900|LT900|LT900",
			"Lazer|Lazer MD1005 Tablet|MD1005|MD1005",
			"Lazer|Lazer MY1306P|MY1306P|MY1306P",
			"Lazer|Lazer X4508|X4508|X4508",
			"Lazer|MID9526CM|MID9526CM|S952",
			"Lazer|MW-7615P|MW-7615P|MW-7615P",
			"Lazer|MW6617|MW6617|MW6617",
			"Le Pan|Le Pan TC1010|FG6A-LP|Le Pan TC1010",
			"Le Pan|Le Pan TC1020|FG6Q|Le Pan TC1020",
			"Le Pan|Le Pan TC802A|UY8|Le Pan TC802A",
			"Lenovo|60K72|ideatv_K72|ideatv K72",
			"Lenovo|A10|A10|IdeaPadA10",
			"Lenovo|A1000|A1000F|IdeaTabA1000-F",
			"Lenovo|A1000-G|A1000G|IdeaTabA1000-G",
			"Lenovo|A1000L|A1000LF|IdeaTabA1000L-F",
			"Lenovo|A208t|A208t|Lenovo A208t",
			"Lenovo|A2107A|A2107A-H|A2107A-H",
			"Lenovo|A218t|A218t|Lenovo A218t",
			"Lenovo|A269|A269|Lenovo A269",
			"Lenovo|A269i|A269i|Lenovo A269i",
			"Lenovo|A278t|A278t|Lenovo A278t",
			"Lenovo|A3000|A3000|Lenovo A3000-H",
			"Lenovo|A305E|A305e|LNV-Lenovo A305e",
			"Lenovo|A305E|A305e|Lenovo A305e",
			"Lenovo|A308t|A308t|Lenovo A308t",
			"Lenovo|A318t|A318t|Lenovo A318t",
			"Lenovo|A369|A369|Lenovo A369",
			"Lenovo|A369i|A369i|Lenovo A369i",
			"Lenovo|A370e|A370e|LNV-Lenovo A370e",
			"Lenovo|A375e|A375e|LNV-Lenovo A375e",
			"Lenovo|A376|A376|Lenovo A376",
			"Lenovo|A378t|A378t|Lenovo A378t",
			"Lenovo|A390|A390|Lenovo A390_ROW",
			"Lenovo|A390t|A390t|Lenovo A390t",
			"Lenovo|A398t|A398t|Lenovo A398t",
			"Lenovo|A516|A516|Lenovo A516",
			"Lenovo|A516|A516_ROW|Lenovo A516",
			"Lenovo|A630|A630e|LNV-Lenovo A630e",
			"Lenovo|A656|A656|Lenovo A656",
			"Lenovo|A658T|A658t|Lenovo A658t",
			"Lenovo|A670t|A670t|Lenovo A670t",
			"Lenovo|A678t|A678t|Lenovo A678t",
			"Lenovo|A706|armani|Lenovo A706",
			"Lenovo|A706_ROW|armani_row|Lenovo A706_ROW",
			"Lenovo|A720e|andorrap|Lenovo A720e",
			"Lenovo|A750e|athenae|Lenovo A750e",
			"Lenovo|A760|audi|Lenovo A760",
			"Lenovo|A766|A766|Lenovo A766",
			"Lenovo|A770e|athenaep|Lenovo A770e",
			"Lenovo|A820|A820|Lenovo A820",
			"Lenovo|A820e|andorra|Lenovo A820e",
			"Lenovo|A830|A830|Lenovo A830",
			"Lenovo|A850|A850|Lenovo A850",
			"Lenovo|A850|A850_ROW|Lenovo A850",
			"Lenovo|A860e|artini|Lenovo A860e",
			"Lenovo|B6000-F|B6000|Lenovo B6000-F",
			"Lenovo|B6000-H|B6000|Lenovo B6000-H",
			"Lenovo|B6000-HV|B6000|Lenovo B6000-HV",
			"Lenovo|B8000-F|B8000|Lenovo B8000-F",
			"Lenovo|B8000-H|B8000|Lenovo B8000-H",
			"Lenovo|E4002|E4002|MEDION E4002",
			"Lenovo|EveryPad|A3000|EveryPad",
			"Lenovo|EveryPad|A3000|IdeaTab A3000-F",
			"Lenovo|IdeaTV|msm8660_surf|ideatv K91",
			"Lenovo|IdeaTab A1000|A1000LF|LenovoA1000L-F",
			"Lenovo|IdeaTab A1010|A1010T|IdeaTabA1010-T",
			"Lenovo|IdeaTab A1020|A1020T|IdeaTabA1020-T",
			"Lenovo|IdeaTab A3000|A3000|IdeaTab A3000-H",
			"Lenovo|IdeaTab A3000|A3000|Vodafone Smart Tab III 7",
			"Lenovo|IdeaTab A5000|A5000E|IdeaTabA5000-E",
			"Lenovo|IdeaTab S6000|S6000|IdeaTab S6000-F",
			"Lenovo|IdeaTab S6000|S6000|IdeaTab S6000-H",
			"Lenovo|IdeaTab S6000|S6000|Vodafone Smart Tab III 10",
			"Lenovo|Indigo|Indigo|ThinkPad Tablet",
			"Lenovo|K1|K1|K1",
			"Lenovo|K800|K800|Lenovo K800",
			"Lenovo|K900|K900|Lenovo K900",
			"Lenovo|K900|K900_ROW|Lenovo K900_ROW",
			"Lenovo|LIFETAB E10310|LIFETAB_E10310|LIFETAB_E10310",
			"Lenovo|LIFETAB E7310|LIFETAB_E7310|LIFETAB_E7310",
			"Lenovo|LIFETAB E7310|LIFETAB_E7310|LIFETAB_E7312",
			"Lenovo|Lenovo|A708t|Lenovo A708t",
			"Lenovo|Lenovo|A880|Lenovo A880",
			"Lenovo|Lenovo  TAB 2 A7-30HC|A7-30HC|Lenovo TAB 2 A7-30HC",
			"Lenovo|Lenovo  X2|X2-AP|Lenovo X2-AP",
			"Lenovo|Lenovo A2105|Lenovo_A2105|Lenovo_A2105",
			"Lenovo|Lenovo A228t|A228t|Lenovo A228t",
			"Lenovo|Lenovo A238t|A238t|Lenovo A238t",
			"Lenovo|Lenovo A300t|A300t|Lenovo A300t",
			"Lenovo|Lenovo A316|A316|Lenovo A316",
			"Lenovo|Lenovo A316i|A316i|Lenovo A316i",
			"Lenovo|Lenovo A319|A319|Lenovo A319",
			"Lenovo|Lenovo A320t|A320t|Lenovo A320t",
			"Lenovo|Lenovo A328|A328|Lenovo A328",
			"Lenovo|Lenovo A328t|A328t|Lenovo A328t",
			"Lenovo|Lenovo A3300|A3300-GV|LenovoA3300-GV",
			"Lenovo|Lenovo A3300|A3300-H|LenovoA3300-H",
			"Lenovo|Lenovo A3300|A3300-HV|LenovoA3300-HV",
			"Lenovo|Lenovo A3300|A3300-T|Lenovo A3300-T",
			"Lenovo|Lenovo A3300-GV|A3300-GV|LenovoA3300-GV",
			"Lenovo|Lenovo A3300-HV|A3300-HV|Lenovo A3300-HV",
			"Lenovo|Lenovo A330e|A330e|Lenovo A330e",
			"Lenovo|Lenovo A338t|A338t|Lenovo A338t",
			"Lenovo|Lenovo A3500|A3500HV|Lenovo A3500-HV",
			"Lenovo|Lenovo A3500-F|A3500F|EveryPad2",
			"Lenovo|Lenovo A3500-F|A3500F|Lenovo A3500-F",
			"Lenovo|Lenovo A3500-FL|A3500FL|Lenovo A3500-FL",
			"Lenovo|Lenovo A3500-H|A3500H|Lenovo A3500-H",
			"Lenovo|Lenovo A355e|A355e|Lenovo A355e",
			"Lenovo|Lenovo A358t|A358t|Lenovo A358t",
			"Lenovo|Lenovo A3600-D|A3600-d|Lenovo A3600-d",
			"Lenovo|Lenovo A360e|A360e|Lenovo A360e",
			"Lenovo|Lenovo A360t|A360t|Lenovo A360t",
			"Lenovo|Lenovo A368t|A368t|Lenovo A368t",
			"Lenovo|Lenovo A3800-D|A3800-d|Lenovo A3800-d",
			"Lenovo|Lenovo A380e|A380e|LNV-Lenovo A380e",
			"Lenovo|Lenovo A380t|A380t|Lenovo A380t",
			"Lenovo|Lenovo A385e|A385e|LNV-Lenovo A385e",
			"Lenovo|Lenovo A388t|A388t|Lenovo A388t",
			"Lenovo|Lenovo A390|A390|Lenovo A390",
			"Lenovo|Lenovo A395e|A395e|LNV-Lenovo A395e",
			"Lenovo|Lenovo A396|A396|Lenovo A396",
			"Lenovo|Lenovo A396|A396_TY|Lenovo A396_TY",
			"Lenovo|Lenovo A398t+|A398tp|Lenovo A398t+",
			"Lenovo|Lenovo A399|Aiken|Lenovo A399",
			"Lenovo|Lenovo A5000|A5000|Lenovo A5000",
			"Lenovo|Lenovo A505e|A505e|LNV-Lenovo A505e",
			"Lenovo|Lenovo A516|A516_ROW|Lenovo A516",
			"Lenovo|Lenovo A526|A526|Lenovo A526",
			"Lenovo|Lenovo A529|A529|Lenovo A529",
			"Lenovo|Lenovo A536|A536|Lenovo A536",
			"Lenovo|Lenovo A5500-F|A5500-F|Lenovo A5500-F",
			"Lenovo|Lenovo A5500-H|A5500-H|Lenovo A5500-H",
			"Lenovo|Lenovo A5500-HV|A5500-HV|Lenovo A5500-HV",
			"Lenovo|Lenovo A560|A560_msm8212|Lenovo A560",
			"Lenovo|Lenovo A560|A560_msm8610|LNV-Lenovo A560",
			"Lenovo|Lenovo A5800-D|A5800-D|Lenovo A5800-D",
			"Lenovo|Lenovo A588t|LenovoA588t|LenovoA588t",
			"Lenovo|Lenovo A6000|Kraft-A6000|Lenovo A6000",
			"Lenovo|Lenovo A6000-l|Kraft-A6000-l|Lenovo A6000-l",
			"Lenovo|Lenovo A606|A606|Lenovo A606",
			"Lenovo|Lenovo A616|A616|Lenovo A616",
			"Lenovo|Lenovo A628t|A628t|Lenovo A628t",
			"Lenovo|Lenovo A680|A680|Lenovo A680",
			"Lenovo|Lenovo A680|A680|Lenovo A680_ROW",
			"Lenovo|Lenovo A688t|A688t|Lenovo A688t",
			"Lenovo|Lenovo A690e|A690e|LNV-Lenovo A690e",
			"Lenovo|Lenovo A7-30GC|A7-30GC|Lenovo TAB 2 A7-30GC",
			"Lenovo|Lenovo A7-30H|A7-30H|Lenovo TAB 2 A7-30H",
			"Lenovo|Lenovo A7-60HC|A760HC|Lenovo A7-60HC",
			"Lenovo|Lenovo A7600-F|A7600-F|Lenovo A7600-F",
			"Lenovo|Lenovo A7600-H|A7600-H|Lenovo A7600-H",
			"Lenovo|Lenovo A7600-HV|A7600-HV|Lenovo A7600-HV",
			"Lenovo|Lenovo A768t|airplayt|Lenovo A768t",
			"Lenovo|Lenovo A780e|A780e|LNV-Lenovo A780e",
			"Lenovo|Lenovo A785e|A785e|LNV-Lenovo A785e",
			"Lenovo|Lenovo A788t|A788t|Lenovo A788t",
			"Lenovo|Lenovo A805e|airplayep|Lenovo A805e",
			"Lenovo|Lenovo A806|A806|Lenovo A806",
			"Lenovo|Lenovo A808t|A808t|Lenovo A808t",
			"Lenovo|Lenovo A808t-i|A808t-i|Lenovo A808t-i",
			"Lenovo|Lenovo A816|airplayw|Lenovo A816",
			"Lenovo|Lenovo A820t|A820t|Lenovo A820t",
			"Lenovo|Lenovo A828|A858|Lenovo A858",
			"Lenovo|Lenovo A828t|A828t|Lenovo A828t",
			"Lenovo|Lenovo A850|A850|Lenovo A850",
			"Lenovo|Lenovo A850+|A850p|Lenovo A850+",
			"Lenovo|Lenovo A858t|A858t|Lenovo A858t",
			"Lenovo|Lenovo A859|A859_ROW|Lenovo A859",
			"Lenovo|Lenovo A889|A889|Lenovo A889",
			"Lenovo|Lenovo A890e|airbuse|Lenovo A890e",
			"Lenovo|Lenovo A916|A916|Lenovo A916",
			"Lenovo|Lenovo A936|A936|Lenovo A936",
			"Lenovo|Lenovo B8080|B8080|Lenovo B8080-F",
			"Lenovo|Lenovo B8080-H|B8080|Lenovo B8080-H",
			"Lenovo|Lenovo B8080-HV|B8080|Lenovo B8080-HV",
			"Lenovo|Lenovo K30-T|Kraft-T|Lenovo K30-T",
			"Lenovo|Lenovo K30-TM|Kraft-TM|Lenovo K30-TM",
			"Lenovo|Lenovo K30-W|Kraft-W|Lenovo K30-W",
			"Lenovo|Lenovo K80M|K80M|Lenovo K80M",
			"Lenovo|Lenovo K860|stuttgart|Lenovo K860",
			"Lenovo|Lenovo K860i|K860i|Lenovo K860i",
			"Lenovo|Lenovo K910|kiton|Lenovo K910",
			"Lenovo|Lenovo K910L|K910L|Lenovo K910L",
			"Lenovo|Lenovo K910e|kitone|LNV-Lenovo K910e",
			"Lenovo|Lenovo K910e|kitone|Lenovo K910e",
			"Lenovo|Lenovo K920/VIBE Z2|kingdom_row|Lenovo K920",
			"Lenovo|Lenovo K920/VIBE Z2 Pro|kingdomt|Lenovo K920",
			"Lenovo|Lenovo N300|Lindos2|Lenovo N300",
			"Lenovo|Lenovo N308|SmartAIO|Lenovo N308",
			"Lenovo|Lenovo P70|P70-A|Lenovo P70-A",
			"Lenovo|Lenovo P70|P70-t|Lenovo P70-t",
			"Lenovo|Lenovo P780|P780|Lenovo P780",
			"Lenovo|Lenovo P90|P90|Lenovo P90",
			"Lenovo|Lenovo S580|S580|Lenovo S580",
			"Lenovo|Lenovo S60|sisleylt|Lenovo S60-t",
			"Lenovo|Lenovo S60|sisleylw|Lenovo S60-w",
			"Lenovo|Lenovo S650|S650|Lenovo S650",
			"Lenovo|Lenovo S650_ROW|S650_ROW|Lenovo S650",
			"Lenovo|Lenovo S658t|S658t|Lenovo S658t",
			"Lenovo|Lenovo S660|S660|Lenovo S660",
			"Lenovo|Lenovo S668t|S668t|Lenovo S668t",
			"Lenovo|Lenovo S810t|shellt|Lenovo S810t",
			"Lenovo|Lenovo S820|S820|Lenovo S820",
			"Lenovo|Lenovo S820|S820_AMX_ROW|Lenovo S820",
			"Lenovo|Lenovo S850|S850|Lenovo S850",
			"Lenovo|Lenovo S850t|S850t|Lenovo S850t",
			"Lenovo|Lenovo S856|shellamx|Lenovo S856",
			"Lenovo|Lenovo S856|shellr|Lenovo S856",
			"Lenovo|Lenovo S856|shellr_s|Lenovo S856",
			"Lenovo|Lenovo S856|shellw|Lenovo S856",
			"Lenovo|Lenovo S858t|Selected|Lenovo S858t",
			"Lenovo|Lenovo S860|S860|Lenovo S860",
			"Lenovo|Lenovo S860e|shelle|Lenovo S860e",
			"Lenovo|Lenovo S898t+|S898tp|Lenovo S898t+",
			"Lenovo|Lenovo S90|sisleye|Lenovo S90-e",
			"Lenovo|Lenovo S90-T|sisleyt|Lenovo S90-t",
			"Lenovo|Lenovo S90-U|sisleyw|Lenovo S90-u",
			"Lenovo|Lenovo S920|S920|Lenovo S920",
			"Lenovo|Lenovo S930|S930|Lenovo S930",
			"Lenovo|Lenovo S930_ROW|S930_ROW|Lenovo S930",
			"Lenovo|Lenovo S938t|S938t|Lenovo S938t",
			"Lenovo|Lenovo S939|S939|Lenovo S939",
			"Lenovo|Lenovo S960|S960|Lenovo S960",
			"Lenovo|Lenovo S960|S960_AMX_ROW|Lenovo S960",
			"Lenovo|Lenovo S968t|S968t|Lenovo S968t",
			"Lenovo|Lenovo TAB 2 A7-10F|Tab2A7-10F|Tab2A7-10F",
			"Lenovo|Lenovo TAB 2 A7-30F|A7-30F|Lenovo TAB 2 A7-30F",
			"Lenovo|Lenovo TAB 2 A7-30HC|A7-30HC|Lenovo 2 A7-30HC",
			"Lenovo|Lenovo TAB 2 A7-30TC|A7-30TC|Lenovo 2 A7-30TC",
			"Lenovo|Lenovo TAB A10-80HC|A10_80HC|TAB A10-80HC",
			"Lenovo|Lenovo TAB S8-50F|S8-50F|Lenovo TAB S8-50F",
			"Lenovo|Lenovo TAB S8-50L|S8-50L|Lenovo TAB S8-50L",
			"Lenovo|Lenovo TAB S8-50LC|S8-50LC|Lenovo S8-50LC",
			"Lenovo|Lenovo TAB S8-50LC|S8-50LC|Lenovo TAB S8-50LC",
			"Lenovo|Lenovo X2|X2-EU|Lenovo X2-EU",
			"Lenovo|Lenovo X2-CU/VIBE X2|X2-CU|Lenovo X2-CU",
			"Lenovo|Lenovo X2-TO/VIBE X2|X2-TO|Lenovo X2-TO",
			"Lenovo|Lenovo X2-TR/VIBE X2|X2-TR|Lenovo X2-TR",
			"Lenovo|Lenovo YOGA Tablet Pro-1050L/Yoga Tablet 2|YT2|YOGA Tablet 2-1050L",
			"Lenovo|Lenovo YOGA Tablet Pro-1050LC/Yoga Tablet 2|YT2|YOGA Tablet 2-1050LC",
			"Lenovo|Lenovo YOGA Tablet Pro-830L/Yoga Tablet 2|YT2|YOGA Tablet 2-830L",
			"Lenovo|Lenovo YOGA Tablet Pro-830LC/Yoga Tablet 2|YT2|YOGA Tablet 2-830LC",
			"Lenovo|Lenovo YogaTablet2 -1050F|YT2|YOGA Tablet 2-1050F",
			"Lenovo|Lenovo YogaTbalet2-830F|YT2|YOGA Tablet 2-830F",
			"Lenovo|Lenovo Z2|z2r|Lenovo Z2",
			"Lenovo|Lenovo Z2|z2t|Lenovo Z2",
			"Lenovo|Lenovo Z2w|z2w|Lenovo Z2w",
			"Lenovo|LenovoTV 50S52;AQUOS LCD-50S1A|bumblebee|AQUOS 50S1",
			"Lenovo|LenovoTV 50S52;AQUOS LCD-50S1A|bumblebee|LenovoTV 50S52",
			"Lenovo|Lenvo S960|S960_ROW|Lenovo S960",
			"Lenovo|P780|P780|Lenovo P780",
			"Lenovo|P780|P780_ROW|Lenovo P780",
			"Lenovo|P780|P780_ROW|Lenovo P780_ROW",
			"Lenovo|S5000|S5000|Lenovo S5000-F",
			"Lenovo|S5000|S5000|Lenovo S5000-H",
			"Lenovo|S6000L|S6000L|Lenovo S6000L-F",
			"Lenovo|S61|ideatv_S61|ideatv S61",
			"Lenovo|S680|seoul|Lenovo S680",
			"Lenovo|S686|Alaska|Lenovo S686",
			"Lenovo|S720|S720|Lenovo S720",
			"Lenovo|S750|S750|Lenovo S750",
			"Lenovo|S820|S820|Lenovo S820",
			"Lenovo|S820|S820_ROW|Lenovo S820",
			"Lenovo|S820|S820_ROW|Lenovo S820_ROW",
			"Lenovo|S820e|applee|Lenovo S820e",
			"Lenovo|S850e|sichuan|Lenovo S850e",
			"Lenovo|S868|S868t|Lenovo S868t",
			"Lenovo|S870e|S870e|LNV-Lenovo S870e",
			"Lenovo|S870e|S870e|Lenovo S870e",
			"Lenovo|S898t|S898t|Lenovo S898t",
			"Lenovo|S920|S920|Lenovo S920",
			"Lenovo|S960|S960|Lenovo S960",
			"Lenovo|ThinkVision28|ThinkVision28|ThinkVision28",
			"Lenovo|ideatv S52|ideatv_S52|ideatv S52",
			"Lexibook|MFC191|MFC191|S952",
			"LinkNet|Smart Box HD|SH940C-LN|SH940C-LN",
			"Listo|WEBPAD1002|Listo|WEBPAD1002",
			"Logic Instrument|FieldBook E1|FieldBook_E1|FieldBook_E1",
			"Logicom|E350|E350|E350",
			"Logicom|E400|E400|E400",
			"Logicom|LOGICOM E1052GP|E1052GP|E1052GP",
			"Logicom|LOGICOM E852GP|E852GP|E852GP",
			"Logicom|Logicom S450|S450|S450",
			"Logicom|Logicom S504|S504|Connect 5",
			"Logicom|Logicom S504|S504|S504",
			"Logicom|S1052|S1052|S1052",
			"Logicom|S732|rk3026|S732",
			"Logicom|S7842|S7842|S7842",
			"Logicom|S952|S952|S952",
			"Logitech|Revue|ka|Revue",
			"Lumigon|T2|msm7630_fw8911|T2",
			"Lumigon|T2 HD|T2HD|T2HD",
			"MAD CATZ|Mad Catz M.O.J.O.|mojo|Mad Catz M.O.J.O.",
			"MG Series|Any 302|TR10CS1_15|TR10CS1",
			"MG Series|Any 302|TR10CS2_3|TR10CS2",
			"MG series|Any 302|TR10CS2_2|TR10CS2",
			"MG series|Any 303|TR10CD2_2|TR10CD2",
			"MG series|Any 303|TR10CD2_3|TR10CD2",
			"MPman|MPDC1006|MPDC1006|MPDC1006",
			"MPman|MPDC706|MPDC706|MPDC706",
			"MSI|Primo73|N71J|Primo73",
			"MSI|Primo76|N728|Primo76",
			"MSI|Primo81|N821|Primo 81",
			"MSI|Primo81|N821|Primo81",
			"MSI|Primo81|Primo81|Primo81",
			"MTC|MTC 982|MTC_982|MTC 982",
			"MTC|Vodafone Smart mini|Vodafone_875|MTC 970H",
			"MTN|MTN-5982C3|MTN-TBW5982C3|MTN-TBW5982C3",
			"MTN|MTN-8978P|MTN-8978P|MTN-8978P",
			"MTN|MTN-S620|MTN-S620|MTN-S620",
			"MTN|MTN-S720i|MTN-S720i|MTN-S720i",
			"MTN-E70|MTN-E70|MTN-E70|MTN-E70",
			"MTT|Master|Master|M.T.T. Master",
			"MTT|Smart Fun|SmartFun|M.T.T. Smart Fun",
			"Mach Speed|xtreme|X_treme_Play_Tab|X-treme Play Tab",
			"MachSpeed|MachSpeed Trio Stealth_10 Tablet|D_10AL|Trio Stealth_10",
			"MachSpeed|MachSpeed Trio Stealth_8 Tablet|D_8AL|Trio Stealth_8",
			"MachSpeed|MachSpeed Trio Stealth_9 Tablet|D_9AL|Trio Stealth_9",
			"MachSpeed|MachSpeed TrioStealth-7|TrioStealth-7|TrioStealth-7",
			"Mad Catz|Mad Catz M.O.J.O.|mojo|Mad Catz M.O.J.O.",
			"Medion|LIFETAB E10316|LIFETAB_E10316|BOSCH_E10316",
			"Medion|LIFETAB E10316|LIFETAB_E10316|LIFETAB_E10316",
			"Medion|LIFETAB E10320|LIFETAB_E10320|LIFETAB_E10320",
			"Medion|LIFETAB E10320|LIFETAB_E10320|MICROSTAR_E10319",
			"Medion|LIFETAB E7313|LIFETAB_E7313|LIFETAB_E7313",
			"Medion|LIFETAB E7316|LIFETAB_E7316|LIFETAB_E7316",
			"Medion|LIFETAB E7316|LIFETAB_E7316|LIFETAB_S7316",
			"Medion|LIFETAB E732X|LIFETAB_E732X|LIFETAB_E732X",
			"Medion|LIFETAB S1033X|LIFETAB_S1033X|LIFETAB_S1033X",
			"Medion|LIFETAB S785X|LIFETAB_S785X|LIFETAB_S785X",
			"Medion|Lifetab P9514|LIFETAB_P9514|LIFETAB_P9514",
			"Medion|MEDION E4502|E4502|MEDION E4502",
			"Medion|P4501|P4501|P4501",
			"Medion|microstar E10319|E10319|E10319",
			"MegaFon|MFLogin3T|MFLogin3T|MFLogin3T",
			"MegaFon|MFLoginPh|MFLoginPh|MFLoginPh",
			"MegaFon|MS4B|Viper_LTE|ALCATEL ONE TOUCH 7030R",
			"MegaFon|MS4B|Viper_LTE|ALCATEL ONE TOUCH 7030Y",
			"MegaFon|MS4B|Viper_LTE|MS4B",
			"MegaFon|MegaFon Login 3|MFLogin3|MegaFon Login 3",
			"MegaFon|MegaFon MT3A|MT3A|MT3A",
			"Megahouse|CP-D403|CP-D403|CP-D403",
			"Mexico SEP tender|TR10CS1|TR10CS1_7|TR10CS1",
			"Mexico SEP tender|TR10CS1|TR10CS1_8|TR10CS1",
			"Micromax|A110|s9081|Micromax A110",
			"Micromax|A111|A111|A111",
			"Micromax|A116|A116|A116",
			"Micromax|A240|A240|A240",
			"Micromax|A27|A27|A27",
			"Micromax|A44|tinnoes13_s7050|A44",
			"Micromax|A45|tinnoes73_s8030_2g|A45",
			"Micromax|A50|kpt73_gb|Micromax A50",
			"Micromax|A54|A54|A54",
			"Micromax|A56|A56|Micromax A56",
			"Micromax|A57|A57|A57",
			"Micromax|A57|A57|Micromax A57",
			"Micromax|A72|A72|Micromax A72",
			"Micromax|A73|A73|A73",
			"Micromax|A73|A73|Micromax A73",
			"Micromax|A75|A75|A75",
			"Micromax|A78|Micromax|A78",
			"Micromax|A84|A84|A84",
			"Micromax|A87|A87|A87",
			"Micromax|A87|A87|Micromax A87",
			"Micromax|A88|A88|A88",
			"Micromax|A89|A89|A89",
			"Micromax|A91|A91|A91",
			"Micromax|A91|A91|Micromax A91",
			"Micromax|BOLT|A068|A068",
			"Micromax|BOLT|A069|Micromax A069",
			"Micromax|BOLT|A24|Micromax A24",
			"Micromax|BOLT|A34|Micromax A34",
			"Micromax|BOLT|A67|Micromax A67",
			"Micromax|BOLT|A69|Micromax A69",
			"Micromax|BOLT|A69_IN|Micromax A69",
			"Micromax|Bolt|A066|Micromax A066",
			"Micromax|Bolt|A067|Micromax A067",
			"Micromax|Bolt|AD4500|Micromax AD4500",
			"Micromax|CANVAS 2 COLOURS|A120|Micromax A120",
			"Micromax|CANVAS 2 PLUS|A110Q|A110Q",
			"Micromax|CANVAS 2 PLUS|s9086b|Micromax A110Q",
			"Micromax|CANVAS 4|A210|A210",
			"Micromax|CANVAS 4|s9111b|A210",
			"Micromax|CANVAS BEAT|A114R|Micromax A114R",
			"Micromax|CANVAS BLAZE|MT500|MT500",
			"Micromax|CANVAS DOODLE 3|MicromaxA102|Micromax A102",
			"Micromax|CANVAS DUET|AE90|Micromax AE90",
			"Micromax|CANVAS DUET 2|EG111|Micromax EG111",
			"Micromax|CANVAS ELANZA 2|A121|Micromax A121",
			"Micromax|CANVAS ENGAGE|A091|Micromax A091",
			"Micromax|CANVAS ENTICE|A105|Micromax A105",
			"Micromax|CANVAS GOLD|A300|Micromax A300",
			"Micromax|CANVAS KNIGHT|A350|Micromax A350",
			"Micromax|CANVAS KNIGHT|s9320ae|Micromax A350",
			"Micromax|CANVAS MAGNUS|s9203|Micromax A117",
			"Micromax|CANVAS POWER|A96|Micromax A96",
			"Micromax|CANVAS TUBE|A118R|Micromax A118R",
			"Micromax|CANVAS TURBO|A250|A250",
			"Micromax|CANVAS TURBO|s9311|A250",
			"Micromax|CANVAS TURBO MINI|s8513a|Micromax A200",
			"Micromax|CANVAS UNITE 2|A106|Micromax A106",
			"Micromax|Canvas 4+|A315|Micromax A315",
			"Micromax|Canvas A1|AQ4501_sprout|Micromax AQ4501",
			"Micromax|Canvas Fire|A093|Micromax A093",
			"Micromax|Canvas Fire|A104|Micromax A104",
			"Micromax|Canvas HD Plus|A190|Micromax A190",
			"Micromax|Canvas Knight Cameo|A290|Micromax A290",
			"Micromax|Canvas L|A108|Micromax A108",
			"Micromax|Canvas Nitro|A310|Micromax A310",
			"Micromax|Canvas Nitro|A311|Micromax A311",
			"Micromax|Canvas XL2|A109|Micromax A109",
			"Micromax|MICROMAX MAD|A94|Micromax A94",
			"Micromax|MICROMAX UNITE|A092|Micromax A092",
			"Micromax|Micromax Bolt|A064|Micromax A064",
			"Micromax|Micromax Bolt|A065|Micromax A065",
			"Micromax|Micromax Bolt|A082|Micromax A082",
			"Micromax|P275|P275|P275",
			"Micromax|P300|crane-M701C_mmx|P300",
			"MiiA|MiiA|TA10CA1_2|TA10CA1",
			"MiiA|MiiA MT-733G|MT-733G|MT-733G",
			"Mint|M4|M4|M4",
			"Mint|M5|MINT-M5|M5",
			"MobiWire|Cygnus|Cygnus45|Cygnus",
			"MobiWire|Cygnus|Cygnus45|Cygnus45",
			"Mobicell|Air|Mobicel_Air|Air",
			"Mobily|MDB342X|qnbml|MDB342X",
			"Mobiwire|Ahiga|Ahiga|Ahiga",
			"Mobiwire|Cygnus mini|Cygnus_mini|Cygnus mini",
			"Mobiwire|Cygnus mini|Cygnus_mini|Cygnus_mini",
			"Mobiwire|Pegasus|Pegasus|Pegasus",
			"Mobiwire|VSN V.45|up06_h26_v45|V.45",
			"Mohu|MH-CHANNELS|MH-CHANNELS|MH-CHANNELS",
			"Monster|Monster M10|M10|M10",
			"Motorola|Atrix|olympus|MB860",
			"Motorola|Atrix|olympus|MB861",
			"Motorola|Atrix|olympus|ME860",
			"Motorola|Atrix HD|qinara|MB886",
			"Motorola|Backflip|motus|MB300",
			"Motorola|Backflip|motus|ME600",
			"Motorola|CLIQ|morrison|MB200",
			"Motorola|CLIQ|morrison|morrison",
			"Motorola|Charm|umts_basil|MB502",
			"Motorola|Citrus|cdma_ciena|WX442",
			"Motorola|Citrus|cdma_ciena|WX445",
			"Motorola|Citrus|cdma_ciena|XT301",
			"Motorola|Cliq-XT|zeppelin|MB501",
			"Motorola|Cliq-XT|zeppelin|ME501",
			"Motorola|DROID RAZR HD|vanquish|DROID RAZR HD",
			"Motorola|DROID RAZR HD|vanquish_u|RAZR HD",
			"Motorola|DROID RAZR HD|vanquish_u|XT925",
			"Motorola|DROID RAZR M|scorpion_mini|XT907",
			"Motorola|DROID RAZR i|smi|XT890",
			"Motorola|DROID Turbo|quark|XT1254",
			"Motorola|DROID4|cdma_maserati|DROID4",
			"Motorola|Defy|umts_jordan|MB525",
			"Motorola|Defy|umts_jordan|MB526",
			"Motorola|Defy|umts_jordan|ME525",
			"Motorola|Defy|umts_jordan|ME525+",
			"Motorola|Defy|umts_jordan|unknown",
			"Motorola|Defy Mini|TinBoost|XT320",
			"Motorola|Defy Mini|TinBoost|XT321",
			"Motorola|Defy Mini|tinboost_umts|XT320",
			"Motorola|Defy Mini|tinboost_umts|XT321",
			"Motorola|Defy Pro|XT560|XT560",
			"Motorola|Devour|calgary|Devour",
			"Motorola|Devour|calgary|calgary",
			"Motorola|Droid|miler|A854",
			"Motorola|Droid|sholes|Droid",
			"Motorola|Droid|umts_sholes|A853",
			"Motorola|Droid|umts_sholes|Milestone",
			"Motorola|Droid|umts_sholes|XT701",
			"Motorola|Droid|umts_sholes|XT702",
			"Motorola|Droid|umts_sholes|XT720",
			"Motorola|Droid|umts_sholes|umts",
			"Motorola|Droid 3|cdma_solana|DROID3",
			"Motorola|Droid 4|cdma_maserati|DROID4",
			"Motorola|Droid Bionic|cdma_targa|DROID BIONIC",
			"Motorola|Droid II|cdma_droid2|A955",
			"Motorola|Droid II|cdma_droid2|DROID2",
			"Motorola|Droid II|cdma_droid2we|DROID2 GLOBAL",
			"Motorola|Droid MAXX|obake-maxx|XT1080",
			"Motorola|Droid Mini|obakem|XT1030",
			"Motorola|Droid Pro|cdma_venus2|DROID PRO",
			"Motorola|Droid Pro|cdma_venus2|DROID Pro",
			"Motorola|Droid Pro|cdma_venus2|Milestone PLUS",
			"Motorola|Droid Pro|cdma_venus2|XT610",
			"Motorola|Droid RAZR|umts_spyder|XT910",
			"Motorola|Droid Razr M|scorpion_mini_t|201M",
			"Motorola|Droid Ultra|obake|XT1080",
			"Motorola|Droid X|cdma_shadow|DROIDX",
			"Motorola|Droid X|cdma_shadow|MB810",
			"Motorola|Droid X|cdma_shadow|ME811",
			"Motorola|Droid X|cdma_shadow|Milestone X",
			"Motorola|Droid X|cdma_shadow|MotoroiX",
			"Motorola|Droid X2|daytona|DROID X2",
			"Motorola|Droid X2|daytona|Milestone X2",
			"Motorola|Electrify M|solstice|XT901",
			"Motorola|Flipout|umts_ruth|MB511",
			"Motorola|Flipout|umts_ruth|ME511",
			"Motorola|Flipout|umts_ruth|MotoMB511",
			"Motorola|Glam XT800|titanium|XT800",
			"Motorola|Iron Rock|umts_irock|XT627",
			"Motorola|MOTO E|condor_cdma|XT1019",
			"Motorola|MOTO E|condor_cdma|XT830C",
			"Motorola|MOTO E|condor_umts|XT1021",
			"Motorola|MOTO E|condor_umts|XT1023",
			"Motorola|MOTO E|condor_umtsds|XT1022",
			"Motorola|MOTO G|falcon_umts|XT1002",
			"Motorola|MOTO G|falcon_umts|XT1003",
			"Motorola|MOTO G|falcon_umts|XT1032",
			"Motorola|MOTO G|peregrine|XT1039",
			"Motorola|MOTO G|peregrine|XT1042",
			"Motorola|MOTO G|peregrine|XT1045",
			"Motorola|MOTO G|titan_umts|XT1063",
			"Motorola|MOTO G|titan_umts|XT1064",
			"Motorola|MOTO G|titan_umtsds|XT1068",
			"Motorola|MOTO G|titan_umtsds|titan_niibr_ds",
			"Motorola|MOTO G LTE|peregrine|XT1039",
			"Motorola|MOTO G LTE|peregrine|XT1040",
			"Motorola|MOTO G LTE|peregrine|XT1045",
			"Motorola|MOTO G w/4G LTE|peregrine|XT1039",
			"Motorola|MOTO G w/4G LTE|peregrine|XT1040",
			"Motorola|MOTO X|ghost|XT1053",
			"Motorola|MOTO X|ghost|XT1058",
			"Motorola|MOTO X|victara|XT1092",
			"Motorola|MOTO X|victara|XT1093",
			"Motorola|MOTO X|victara|XT1094",
			"Motorola|MOTO X|victara|XT1095",
			"Motorola|MOTO X|victara|XT1096",
			"Motorola|MOTO X|victara|XT1097",
			"Motorola|MOTOROLA ELECTRIFY 2|cdma_yangtze|XT881",
			"Motorola|MOTOROLA RAZR HD|vanquish_u|RAZR HD",
			"Motorola|MOTOROLA RAZR M|smq_t|201M",
			"Motorola|MOTOROLA RAZR i|smi|XT890",
			"Motorola|Master Touch|umts_primus|XT621",
			"Motorola|Milestone2|umts_milestone2|A953",
			"Motorola|Milestone2|umts_milestone2|ME722",
			"Motorola|Milestone2|umts_milestone2|MotoA953",
			"Motorola|Moto 360|minnow|Moto 360",
			"Motorola|Moto Defy XT|XT535|XT535",
			"Motorola|Moto Defy XT|tinboostplus_cdma|XT555C",
			"Motorola|Moto Defy XT|tinboostplus_cdma|XT556",
			"Motorola|Moto Defy XT|tinboostplus_cdma|XT557",
			"Motorola|Moto Defy XT|tinboostplus_umts|XT535",
			"Motorola|Moto E|condor_cdma|XT1019",
			"Motorola|Moto E|condor_udstv|XT1025",
			"Motorola|Moto E|condor_umts|XT1021",
			"Motorola|Moto E|condor_umts|XT1023",
			"Motorola|Moto E|condor_umtsds|XT1022",
			"Motorola|Moto G|falcon_cdma|XT1028",
			"Motorola|Moto G|falcon_cdma|XT1031",
			"Motorola|Moto G|falcon_cdma|XT937C",
			"Motorola|Moto G|falcon_umts|XT1008",
			"Motorola|Moto G|falcon_umts|XT1032",
			"Motorola|Moto G|falcon_umts|XT1034",
			"Motorola|Moto G|falcon_umts|XT939G",
			"Motorola|Moto G|falcon_umtsds|XT1033",
			"Motorola|Moto G|titan_udstv|XT1069",
			"Motorola|Moto G|titan_udstv|titan_retbr_dstv",
			"Motorola|Moto G|titan_umts|XT1063",
			"Motorola|Moto G|titan_umts|XT1064",
			"Motorola|Moto G|titan_umtsds|XT1068",
			"Motorola|Moto G Google Play edition|falcon_umts|XT1032",
			"Motorola|Moto G w/ 4G LTE|peregrine|XT1045",
			"Motorola|Moto G w/4G LTE|peregrine|XT1040",
			"Motorola|Moto MAXX|quark_umts|XT1225",
			"Motorola|Moto X|ghost|XT1049",
			"Motorola|Moto X|ghost|XT1050",
			"Motorola|Moto X|ghost|XT1052",
			"Motorola|Moto X|ghost|XT1053",
			"Motorola|Moto X|ghost|XT1055",
			"Motorola|Moto X|ghost|XT1056",
			"Motorola|Moto X|ghost|XT1058",
			"Motorola|Moto X|ghost|XT1060",
			"Motorola|Moto X|victara|XT1097",
			"Motorola|Moto X|victara|XT1098",
			"Motorola|Motoluxe|XT611|XT611",
			"Motorola|Motoluxe|XT615|XT615",
			"Motorola|Motoluxe|XT682|XT682",
			"Motorola|Motoluxe|ironmax_umts|XT615",
			"Motorola|Motoluxe|ironmax_umts|XT685",
			"Motorola|Motoluxe|ironmaxct_cdma|Motorola MOT-XT681",
			"Motorola|Motoluxe|ironmaxtv_umts|XT687",
			"Motorola|Motoluxe|umts_irock|XT626",
			"Motorola|Motoluxe|umts_irock|XT627",
			"Motorola|Motoroi|sholest|Milestone XT720",
			"Motorola|Motoroi|sholest|Motorola XT720",
			"Motorola|Motoroi|sholest|XT720",
			"Motorola|Motorola Atrix HD|qinara|MB886",
			"Motorola|Motorola Master Touch|umts_primus|XT621",
			"Motorola|Motorola Master Touch XT621|umts_primus|XT621",
			"Motorola|Motorola Photon|asanti_c|XT897",
			"Motorola|Motorola Photon|asanti_c|XT897S",
			"Motorola|Motorola RAZR HD|vanquish|DROID RAZR HD",
			"Motorola|Motorola RAZR M|smq|XT907",
			"Motorola|Motorola Razr V|umts_yangtze|XT885",
			"Motorola|Motorola Razr V|umts_yangtze|XT886",
			"Motorola|Motorola XOOM 2|fleming|MZ609",
			"Motorola|Motorola XOOM 2|pasteur|MZ617",
			"Motorola|Motosmart|XT389|XT389",
			"Motorola|Motosmart|XT390|XT390",
			"Motorola|Motosmart|argonmini_umts|XT389",
			"Motorola|Motosmart|argonmini_umts|XT390",
			"Motorola|Motosmart|silversmart_umts|XT303",
			"Motorola|Motosmart|silversmart_umts|XT305",
			"Motorola|Nexus 6|shamu|Nexus 6",
			"Motorola|Opus One|rubicon|Motorola Titanium",
			"Motorola|Opus One|rubicon|Titanium",
			"Motorola|Photon 4G|sunfire|ISW11M",
			"Motorola|Photon 4G|sunfire|MB855",
			"Motorola|Photon 4G|sunfire|Motorola Electrify",
			"Motorola|Quench XT3|XT502|Motorola-XT502",
			"Motorola|RAZR D1|hawk35_umts|XT914",
			"Motorola|RAZR D1|hawk35_umts|XT915",
			"Motorola|RAZR D1|hawk35_umts|XT916",
			"Motorola|RAZR D1|hawk35_umts|XT918",
			"Motorola|RAZR D3|hawk40_umts|XT919",
			"Motorola|RAZR D3|hawk40_umts|XT920",
			"Motorola|Spice|sesame|XT300",
			"Motorola|XOOM|stingray|Xoom",
			"Motorola|XOOM|umts_everest|MZ601",
			"Motorola|XOOM|umts_hubble|MZ601",
			"Motorola|XOOM|umts_hubble|MZ605",
			"Motorola|XOOM|wifi_hubble|MZ604",
			"Motorola|XOOM|wifi_hubble|MZ606",
			"Motorola|XOOM|wingray|Xoom",
			"Motorola|XT605|umts_jorian|XT605",
			"Motorola|XT627|umts_irock|XT627",
			"Motorola Solutions|TC55CH|TC55CH|TC55CH",
			"MotorolaSolutionsInc.|TC55|TC55|TC55",
			"MotorolaSolutionsInc.|TC55CH|TC55CH|TC55CH",
			"Multilaser|M7s Dual Core|NB116_NB117_NB118|M7s Dual ML03",
			"My Go|GoTab GBT10|GBT10|GoTab_GBT10",
			"NAXA|NAXA NID-7011|NID-7011|NID-7011",
			"NAXA|NID_7010|NID_7010|NID_7010",
			"NCREDIBLE|NV8-HD|NV8-HD|NV8",
			"NEC|101T \\xe3\\x80\\x80MEDIAS|NEC-101|NEC-101T",
			"NEC|AGT10|AGT10|N8730-411",
			"NEC|AGT10|AGT10|N8730-41101",
			"NEC|AGT10|AGT10|N8730-41102",
			"NEC|CASIO G\\'zOne Commando 4G LTE|C811|C811 4G",
			"NEC|Casio G\\'zOne Commando|C771|C771",
			"NEC|Disney Mobile on docomoN-03E|N-03E|N-03E",
			"NEC|G\\'z One IS11CA|IS11CA|IS11CA",
			"NEC|G\\'zOne TYPE-L CAL21|CAL21|CAL21",
			"NEC|G\\xe2\\x80\\x99zOne CA-201L|CA201L|CA-201L",
			"NEC|LaVieTab PC-TE508S1W/LaVieTab PC- TE508S1L|PC-TE508S1_nec|LaVieTab PC-TE508S1",
			"NEC|LaVieTab PC-TE510S1L|PC-TE510S1_nec|LaVieTab PC-TE510S1",
			"NEC|LifeTouch B|LTB013|D000-000013-101",
			"NEC|LifeTouch B|LTB018|D000-000018-001",
			"NEC|LifeTouch B|LTB018|D000-000018-002",
			"NEC|LifeTouch B|LTB018|D000-000018-003",
			"NEC|LifeTouch B|LTB018|D000-000018-004",
			"NEC|LifeTouch B|LTB018|D000-000018-101",
			"NEC|LifeTouch B|LTB018|D000-000018-102",
			"NEC|LifeTouch B|LTB018|D000-000018-104",
			"NEC|LifeTouch B|LTB019G|D000-000019-002",
			"NEC|LifeTouch B|LTB019W|D000-000019-001",
			"NEC|LifeTouch B|LTB028|LTB-HS",
			"NEC|LifeTouch L|D000000023|LT-TLA",
			"NEC|LifeTouch L|D000000035|NEC-STR",
			"NEC|LifeTouch L|D000000039|NEC-STR",
			"NEC|LifeTouch L|LTTLA16|LT-TLA",
			"NEC|LifeTouch L|LTTLA32|LT-TLA",
			"NEC|LifeTouch Note|D000000010N|D000-000010-N",
			"NEC|LifeTouch Note|D000000011N|D000-000011-N",
			"NEC|LifeTouch Note|LTNA7|LT-NA7",
			"NEC|LifeTouch Note|LTNA7F|LT-NA7F",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-B01",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-B02",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-C01",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-K01",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-R01",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-R02",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-R03",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-R04",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-S00",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-S01",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-S05",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-S25",
			"NEC|LifeTouch S|LifeTouch_D1|D000-000001-S85",
			"NEC|LifeTouch S|LifeTouch_D7|D000-000007-D01",
			"NEC|LifeTouch W|LT-W1_1|D000-000002-W01",
			"NEC|LifeTouch W|LT-W1_2|D000-000002-W02",
			"NEC|LifeTouch W|LifeTouch_W1|D000-000002-001",
			"NEC|MEDIAS  X N-06E|N-06E|N-06E",
			"NEC|MEDIAS BR IS11N|IS11N|IS11N",
			"NEC|MEDIAS CH 101N|101N|101N",
			"NEC|MEDIAS ES N-05D|N-05D|N-05D",
			"NEC|MEDIAS LTE N-04D|N-04D|N-04D",
			"NEC|MEDIAS N-04C|N-04C|N-04C",
			"NEC|MEDIAS NE-202|NE-202|NE-202",
			"NEC|MEDIAS NEC-101S|MAGNUM|NEC-101S",
			"NEC|MEDIAS NEC-101S|NEC-101|NEC-101S",
			"NEC|MEDIAS NEC-102|NEC-102|NEC-102",
			"NEC|MEDIAS PP N-01D|N-01D|N-01D",
			"NEC|MEDIAS TAB N-06D|N-06D|N-06D",
			"NEC|MEDIAS TAB UL N-08D|N-08D|N-08D",
			"NEC|MEDIAS U N-02E|N-02E|N-02E",
			"NEC|MEDIAS U NE-103T|NE-103|NE-103T",
			"NEC|MEDIAS W N-05E|N-05E|N-05E",
			"NEC|MEDIAS WP N-06C|N-06C|N-06C",
			"NEC|MEDIAS X N-04E|N-04E|N-04E",
			"NEC|MEDIAS X N-07D|N-07D|N-07D",
			"NEC|NE-201|NE-201|NEC-NE-201A1A",
			"NEC|NEC909e|NEC909e|NEC909e",
			"NEC|PC- TS507N1S|PC-TS507N1S|LaVieTab PC-TS507N1S",
			"NEC|PC-508T1W|508T1W|PC-TS508T1W",
			"NEC|PC-708T1W|708T1W|PC-TS708T1W",
			"NEC|PC-TE307N1W|PC-TE307N1W|PC-TE307N1W",
			"NEC|PC-TE510N1B|LaVieTab|LaVieTab PC-TE510N1B",
			"NESO|NESO N810 i7|N810_i7_1|N810 i7",
			"NGM|Dynamic Racing 3|NGM_Dynamic_Racing_3|Dynamic Racing 3",
			"NGM|Forward_Prime|Forward_Prime|Forward_Prime",
			"NGM|Infinity|Infinity|Infinity",
			"NGM|NGM_Dynamic_Stylo|NGM_Dynamic_Stylo|NGM_Dynamic_Stylo",
			"NVidia|SHIELD Tablet|shieldtablet|SHIELD Tablet",
			"NVidia|Shield|roth|SHIELD",
			"NVidia|TegraNote|tegranote|Tegra Note 7",
			"NVidia|TegraNote|tegranote|TegraNote-P1640",
			"NVidia|TegraNote|tegranote|TegraNote-Premium",
			"NVidia|TegraNote|tegranote7c|Tegra Note 7",
			"NanoTech|Nuvola NP-1|nuvola|Nuvola NP-1",
			"Ncredible|Ncredible NV8|NV8|NV8",
			"Netgear|NeoTVPrime|NeoTV|GTV100",
			"Next Learning|Nexttab N3|N3|N3",
			"Nextbook|M-MP7NB3G|7_Plus_HD3G|M-MP7NB3G",
			"Nextbook|NX700QC|nxm7100lvd_wm|NX700QC",
			"Nextbook|NX785QC|nxm865lvd_wm|NX785QC",
			"Nextbook|NX785QC8G|NXM865FD|NX785QC8G",
			"Nextel|V.35|up11_vsn_h1_single|Nextel V.35",
			"Nextel|V.45|H26-VSN|NII Nextel",
			"Nextel|V.45|H26-VSN|V.45",
			"Nikon|COOLPIX|S800c|COOLPIX S800c",
			"Nikon|COOLPIX S810c|s810c|COOLPIX S810c",
			"Nobis|NB07|NB07|NB07",
			"Nobis|NB1022|RCT6203W46|NB1022",
			"Nobis|NB7022S|NB7022S|NB7022S",
			"Nobis|NB7850S|NB7850S|NB7850S",
			"Nobis|Nobis NB09|M9025|NB09",
			"Noblex|NBX-T1014N|TA10CA2|TA10CA2",
			"Noblex|NBX-T7014|E270BSA|NBX-T7014",
			"Nubia|NX401|NX40X|NX401",
			"Nubia|NX501|NX501|NX501",
			"Nubia|NX503A|NX503A|NX503A",
			"Nuvision|TM1088|fiber-wifibt|CT1000",
			"Nuvision|TM1088|fiber-wifibt|Endeavour101",
			"Nuvision|TM1088|fiber-wifibt|TM1088",
			"Nuvision|TM1088|fiber-wifibt|WEBPAD1002",
			"Nuvision|TM800A510L|TM800A|TM800A510L",
			"Nyx mobile|NOBA|NOBA|NOBA",
			"OPPO|N1|N1|N1",
			"OPPO|N1T|N1T|N1T",
			"OPPO|N1W|N1W|N1W",
			"OPPO|R1001|R1001|R1001",
			"OPPO|R2001|R2001|R2001",
			"OPPO|R2010|R2010|R2010",
			"OPPO|R2017|R2017|R2017",
			"OPPO|R6007|R6007|R6007",
			"OPPO|R7007|R7007|R7007",
			"OPPO|R8001|R8001|R8001",
			"OPPO|R8006|R8006|R8006",
			"OPPO|R8007|R8007|R8007",
			"OPPO|R815|R815|R815",
			"OPPO|R819|R819|R819",
			"OPPO|R821|R821|R821",
			"OPPO|R827|R827|R827",
			"OPPO|R827T|R827T|R827T",
			"OPPO|R829|R829|R829",
			"OPPO|R829T|R829T|R829T",
			"OPPO|R830|R830|R830",
			"OPPO|R830S|R830S|R830S",
			"OPPO|R831|R831|R831",
			"OPPO|R831L|R831L|R831L",
			"OPPO|R831S|R831S|R831S",
			"OPPO|R831T|R831T|R831T",
			"OPPO|R850|R850|R850",
			"OPPO|U707|U707|U707",
			"OPPO|U708|U708|U708",
			"OPPO|X9006|X9006|X9006",
			"OPPO|X9007|X9007|X9007",
			"OPPO|X9070|X9070|X9070",
			"OPPO|X9076|X9076|X9076",
			"OPPO|X9077|X9077|X9077",
			"OPPO|X909|FIND5|X909",
			"Odys|ODYS UNO X10|UNO_X10|UNO X10",
			"Odys|ODYS UNO X8|UNO_X8|UNO_X8",
			"Olivetti|Graphos A10|Graphos_A10|Graphos A10",
			"Olivetti|OP110|OP110|OP110",
			"OnePlus|OnePlus One|A0001|A0001",
			"OnePlus|OnePlus One|A0001|One",
			"Oppo|1100|1100|1100",
			"Oppo|1105|1105|1105",
			"Oppo|1107|1107|1107",
			"Oppo|3000|3000|3000",
			"Oppo|3001|Mirror3|3001",
			"Oppo|3005|3005|3005",
			"Oppo|3006|Mirror3|3006",
			"Oppo|3007|3007|3007",
			"Oppo|3008|3008|3008",
			"Oppo|N1|N1|N1",
			"Oppo|N1 mimi|N1mini|N5111",
			"Oppo|N1 mimi|N1mini|N5116",
			"Oppo|N3|N3|N5206",
			"Oppo|N3|N3|N5207",
			"Oppo|N5110|N5110|N5110",
			"Oppo|N5117|N5117|N5117",
			"Oppo|OPPO Find5|FIND5|X909",
			"Oppo|OPPO Find5|FIND5|X909T",
			"Oppo|OPPO Ulike2|Ulike2|U705T",
			"Oppo|R1001|R1001|R1001",
			"Oppo|R5|R5|R8106",
			"Oppo|R5|R5|R8107",
			"Oppo|R6006|R3|R6006",
			"Oppo|R7005|R3|R7005",
			"Oppo|R8000|R1S|R8000",
			"Oppo|R809T|R809T|R809T",
			"Oppo|R815T|R815T|R815T",
			"Oppo|R815W|R815W|R815W",
			"Oppo|R819T|R819T|R819T",
			"Oppo|R820|R820|R820",
			"Oppo|R8201|R1x|R8201",
			"Oppo|R8206|R1x|R8206",
			"Oppo|R8207|R1C|R8207",
			"Oppo|R821T|R821T|R821T",
			"Oppo|R823T|R823T|R823T",
			"Oppo|R831K|R831K|R831K",
			"Oppo|R833T|OPPO82_13067|R833T",
			"Oppo|U3|U3|U3",
			"Oppo|U707T|U707T|U707T",
			"Oppo|Ulike2|U705W|U705W",
			"Oppo|X9000|X9000|X9000",
			"Oppo|X9077|FIND7|X9077",
			"Opticon|H-27|H-27|H-27",
			"Orange|Luno|Luno|Luno",
			"Orange|NOS Roya|Alto45|NOS Roya",
			"Orange|Orange Nura|M812|Orange Nura",
			"Orange|Orange Roya|Alto45|Orange Roya",
			"Orange|Orange Zilo|Orange_Zilo|Zilo",
			"Orange|idol S|Diablo_LTE|Orange Niva",
			"Orbic|R370H|R370H|R370L",
			"Panasonic|003P|pana1|003P",
			"Panasonic|101P|pana2_2s|101P",
			"Panasonic|102P|pana2_4s|102P",
			"Panasonic|48AX600C|mst918_4G|48AX600C",
			"Panasonic|55AX600C|mst918_4k2k|55AX600C",
			"Panasonic|AS650C|AS650C|AS650C",
			"Panasonic|Altus 18|Altus18|Altus 18",
			"Panasonic|Altus 24|Altus24|Altus 24",
			"Panasonic|Boukenkun-reciever|VW_RCBKK1|VW_RCBKK1",
			"Panasonic|EB-4063-X|EB-4063-X|EB-4063-X",
			"Panasonic|ELUGA|pana2_4o|dL1",
			"Panasonic|ELUGA P|P-03E|P-03E",
			"Panasonic|ELUGA Power|P-07D|P-07D",
			"Panasonic|ELUGA U|ELUGA_U|Panasonic ELUGA U",
			"Panasonic|ELUGA V|P-06D|P-06D",
			"Panasonic|ELUGA X|P-02E|P-02E",
			"Panasonic|ELUGA dL1|pana2_4v|Panasonic_dL1",
			"Panasonic|Eco11 v2|Eco11v2|Eco11 v2",
			"Panasonic|Eco9 v1|Eco|Eco9 v1",
			"Panasonic|Eco9 v2|Ecov2|Eco9 v2",
			"Panasonic|Elite 11|Elite11|Elite",
			"Panasonic|Elite 11|Elite11|Elite 11",
			"Panasonic|Elite 16|Elite16|Elite",
			"Panasonic|FZ-A1|toughpad|FZ-A1B",
			"Panasonic|FZ-B2|fz_b2bb|FZ-B2B",
			"Panasonic|FZ-X1|FZ-X1|FZ-X1",
			"Panasonic|FZ-X1|FZ-X1VU|FZ-X1",
			"Panasonic|FZ-X1|FZ-X1VUC|FZ-X1",
			"Panasonic|HDPSEB v2|HDPSEBv2|HD PSEB v2",
			"Panasonic|JT-B1|B1|JT-B1",
			"Panasonic|KX-PRXA10|KX-PRXA10|Panasonic KX-PRXA10",
			"Panasonic|KX-PRXA15|KX-PRXA15|Panasonic KX-PRXA15",
			"Panasonic|LUMIX CM1|DMC-CM1|DMC-CM1",
			"Panasonic|P-01D|pana2_1|P-01D",
			"Panasonic|P-02D|pana2_2|P-02D",
			"Panasonic|P-04D|pana2_4d|P-04D",
			"Panasonic|P-05D|pana2_4a|P-05D",
			"Panasonic|P-07C|pana1|P-07C",
			"Panasonic|Panasonic ELUGA A|ELUGA_A|Panasonic ELUGA A",
			"Panasonic|Panasonic ELUGA I|ELUGA_I|Panasonic ELUGA I",
			"Panasonic|Panasonic ELUGA S|ELUGA_S|Panasonic ELUGA S",
			"Panasonic|Panasonic P11|P11|Panasonic P11",
			"Panasonic|Panasonic P31|P31|Panasonic P31",
			"Panasonic|Panasonic P41|P41|Panasonic P41",
			"Panasonic|Panasonic P41 HD|P41HD|Panasonic P41HD",
			"Panasonic|Panasonic P51|P51|Panasonic P51",
			"Panasonic|Panasonic P55|P55|Panasonic P55",
			"Panasonic|Panasonic P61|P61|Panasonic P61",
			"Panasonic|Panasonic P61|P61|Panasonic_P61",
			"Panasonic|Panasonic P81|P81|Panasonic P81",
			"Panasonic|Panasonic T11|T11|Panasonic T11",
			"Panasonic|Panasonic T21|T21|Panasonic T21",
			"Panasonic|Panasonic T31|T31|Panasonic T31",
			"Panasonic|Panasonic T40|T40|Panasonic T40",
			"Panasonic|Panasonic T41|T41|Panasonic T41",
			"Panasonic|Panasonic T9|T9|Panasonic T9",
			"Panasonic|SM17|SM17|Smart Monitor 17",
			"Panasonic|SV-ME1000|K1|Panasonic SV-ME1000",
			"Panasonic|SmartTVBox|C01AS|SMARTTVBOX",
			"Panasonic|UN-MT300|A1|Panasonic UN-MT300",
			"Panasonic|UN-W700|F1|Panasonic UN-W700",
			"Panasonic|VTSH v1|VTSHv1|VTSH v1",
			"Pantech|ADR910L|ADR910L|ADR910L",
			"Pantech|ADR930L|ADR930L|ADR930L",
			"Pantech|AT1|at1|IM-T100K",
			"Pantech|Apache|ADR8995|ADR8995",
			"Pantech|Burst|presto|PantechP9070",
			"Pantech|Crossover|moon|PantechP8000",
			"Pantech|EF33S|ef33s|IM-A760S",
			"Pantech|EF34K|ef34k|IM-A770K",
			"Pantech|EF35L|ef35l|IM-A780L",
			"Pantech|Element|pororo|PantechP4100",
			"Pantech|Flex|oscar|PantechP8010",
			"Pantech|IM-810K|ef40k|IM-A810K",
			"Pantech|IM-840SP|IM-A840SP|IM-A840SP",
			"Pantech|IM-A730S|ef30s|IM-A730S",
			"Pantech|IM-A740S|ef31s|IM-A740S",
			"Pantech|IM-A760S|ef33s|IM-A760S",
			"Pantech|IM-A770K|ef34k|IM-A770K",
			"Pantech|IM-A775C|ef34c|IM-A775C",
			"Pantech|IM-A780L|ef35l|IM-A780L",
			"Pantech|IM-A800S|ef39s|IM-A800S",
			"Pantech|IM-A810K|ef40k|IM-A810K",
			"Pantech|IM-A810S|ef40s|IM-A810S",
			"Pantech|IM-A830K|ef45k|IM-A830K",
			"Pantech|IM-A830KE|ef45kv|IM-A830KE",
			"Pantech|IM-A830L|ef46l|IM-A830L",
			"Pantech|IM-A830S|ef47s|IM-A830S",
			"Pantech|IM-A840S|IM-A840S|IM-A840S",
			"Pantech|IM-A850K|ef49k|IM-A850K",
			"Pantech|IM-A850L|ef50l|IM-A850L",
			"Pantech|IM-A850S|ef48s|IM-A850S",
			"Pantech|IM-A860K|ef51k|IM-A860K",
			"Pantech|IM-A860L|ef51l|IM-A860L",
			"Pantech|IM-A860S|ef51s|IM-A860S",
			"Pantech|IM-A870K|ef52k|IM-A870K",
			"Pantech|IM-A870L|ef52l|IM-A870L",
			"Pantech|IM-A870S|ef52s|IM-A870S",
			"Pantech|IM-A880S|EF56S|IM-A880S",
			"Pantech|IM-A910K|ef63k|IM-A910K",
			"Pantech|IM-A910L|ef63l|IM-A910L",
			"Pantech|IM-A910S|ef63s|IM-A910S",
			"Pantech|IM-A920S|ef65s|IM-A920S",
			"Pantech|Izar|sp33k|SKY IM-A630K",
			"Pantech|MIRACH_J|mirachj|EIS01PT",
			"Pantech|MIRACH_J|mirachj|IS11PT",
			"Pantech|Mirach|ef13l|IM-A690L",
			"Pantech|Mirach|ef13s|IM-A690S",
			"Pantech|Mirach A|ef32k|IM-A750K",
			"Pantech|P9090|magnus|PantechP9090",
			"Pantech|PLANET|mini|PLANET",
			"Pantech|PTL21|maruko|PTL21",
			"Pantech|Pocket|mini|PantechP9060",
			"Pantech|SIRIUS \\xce\\xb1|jmasai|IS06",
			"Pantech|Sirius|ef10s|SKY IM-A600S",
			"Pantech|VEGA RACER|ef34k|IM-A770K",
			"Pantech|VEGA RACER|ef35l|IM-A780L",
			"Pantech|VEGA Secret Note|EF59K|IM-A890K",
			"Pantech|VEGA Secret Note|EF59L|IM-A890L",
			"Pantech|VEGA Secret Note|EF59S|IM-A890S",
			"Pantech|VEGA Secret UP|EF60S|IM-A900S",
			"Pantech|VEGA Secret UP|EF61K|IM-A900K",
			"Pantech|VEGA Secret UP|EF62L|IM-A900L",
			"Pantech|VEGA Secret UP|ef60s|IM-A900S",
			"Pantech|VEGA Secret UP|ef61k|IM-A900K",
			"Pantech|VEGA Secret UP|ef62l|IM-A900L",
			"Pantech|Vega|ef12s|SKY IM-A650S",
			"Pantech|Vega LTE M|ef65l|IM-A820L",
			"Pantech|Vega X|ef14l|IM-A720L",
			"Pantech|Vega X|ef18k|IM-A710K",
			"Pantech|ef14lv|ef14lv|IM-A725L",
			"Partner|OT-310|OT-310|OT-310",
			"Pegatron|MaxiIQ|MaxiIQ|MaxiIQ",
			"Pegatron|Olipad|OP111|chagall",
			"Pegatron|chagall|chagall|Chagall 10.1 WiFi",
			"Pegatron|chagall|chagall|E-Tab 4G",
			"Pegatron|chagall|chagall|chagall",
			"Philips|HMP8100/93|HMP8100_ATV_93|HMP8100_ATV_93",
			"Philips|HMP8100/98|HMP8100_ATV|HMP8100_ATV_INT",
			"Philips|PHP-S221C4AFD|MMD_WW|G3SMNTS22",
			"Philips|PHP-S231C4AFD|G3SMNTS23|G3SMNTS23",
			"Philips|PI2000|PI2000|PI2000",
			"Philips|PI2010|PI2010|PI2010",
			"Philips|PI2011|PI2011|PI2011",
			"Philips|PI3100/51|PI3100_51|PI3100",
			"Philips|PI3100/58|PI3100_58|PI3100",
			"Philips|PI3100/93|PI3100_93|PI3100-93",
			"Philips|PI3100/98|PI3100_98|PI3100",
			"Philips|PI3100Z3/93|PI3100Z3_93|PI3100Z3_93",
			"Philips|PI3105|PI3105|PI3105",
			"Philips|PI3106|PI3106|PI3106",
			"Philips|PI3110|PI3110|PI3110",
			"Philips|PI3205G|PI3205G_93|PI3205G",
			"Philips|PI3210G|PI3210G|PI3210G",
			"Philips|PI3900|T7p_Duo_93|PI3900-93",
			"Philips|PI3900/51|PI3900_51|PI3900",
			"Philips|PI3900/58|PI3900_58|PI3900",
			"Philips|PI3900/98|PI3900_98|PI3900",
			"Philips|PI3910|PI3910|PI3910",
			"Philips|PI4010G|PI4010G|PI4010G",
			"Philips|PI7100/93|PI7100_93|PI7100_93",
			"Philips|Philips Android 2014|philipstv|AND1E",
			"Philips|Philips W8510|Topaz|Philips W8510",
			"Philips|Philips_W3500|Philips_W3500|Philips_W3500",
			"Philips|S221C3A|MMD_WW|G2SMNT",
			"Philips|T8 PI7000|Tablet_8|PI7000",
			"Philips|W337|W337|W337",
			"Philips|W3568|Philips_W3568|Philips W3568",
			"Philips|W3620|W3620|W3620",
			"Philips|W6500|Philips_W6500_A|W6500",
			"Philips|W6610|Marathon|Philips W6610",
			"Philips|W6620|W6620|W6620",
			"Philips|W8555|Pride|Philips W8555",
			"Pioneer|PIONEER R1|RCK8726Y703H|R1",
			"Polaris|polaris_Condroid_X7|polaris-Condroid_X7|Condroid_X7",
			"Polaroid|A9x/PTAB935x|A9xPTAB935x|A9x/PTAB935x",
			"Polaroid|Infinite|MID1324|MID 1324",
			"Polaroid|MID4X07|RCT6773W22|MID4X07",
			"Polaroid|MID4X10|RCT6203W46|MID4X10",
			"Polaroid|P10/Q10/PTAB1040/PTAB1041|pq10-Polaroid|P10/Q10/PTAB1040/PTAB1041",
			"Polaroid|P1001|P1001|P1001",
			"Polaroid|P7/Q7/PTAB740/PTAB741|pq7-Polaroid|P7/Q7/PTAB740/PTAB741",
			"Polaroid|P791|P791|P791",
			"Polaroid|P9/Q9/PTAB940/PTAB941|pq9-Polaroid|P9/Q9/PTAB940/PTAB941",
			"Polaroid|POLAROID A7_PTAB735|A7_PTAB735|HS-7DTB39",
			"Polaroid|PRO5023PW|up06_h25_polaroid|PRO5023PW",
			"Polaroid|PTAB1051-PTAB1055|RCT6203W46|PTAB1051_PTAB1055",
			"Polaroid|Polaroid A/S10|wing-AS10|A/S10",
			"Polaroid|Polaroid A/S7|wing-AS7|A/S7",
			"Polaroid|Polaroid A/S8|wing-AS8|A/S8",
			"Polaroid|Polaroid A7X_PTAB735X|PTAB735X|A7X_PTAB735X",
			"Polaroid|Polaroid K7|PTAB782|K7/PTAB782",
			"Polaroid|Polaroid MID1028|MID1028|MID1028",
			"Polaroid|Polaroid P900|P900|P900/Q900",
			"Polaroid|Polaroid PTAB751|STJR76|PTAB751",
			"Polaroid|Polaroid V7|nuclear-M7021|V7",
			"Positivo|Positivo AB7|YPY_AB7D|YPY_AB7D",
			"Positivo|Positivo AB7D|YPY_AB7DC|YPY_AB7DC",
			"Positivo|Positivo Mini|mini|Positivo Mini",
			"Positivo|Positivo Mini|mini|Positivo Mini TE",
			"Positivo|Positivo S380|S380|S380",
			"Positivo|Positivo S450|S450|S450",
			"Positivo|Positivo SX1000|SX1000|Positivo SX1000",
			"Positivo|Positivo T1060|t1060|T1060",
			"Positivo|Positivo T1060|t1060|T1060B",
			"Positivo|Positivo T701 TV|T701|T701",
			"Positivo|Positivo T705|T705|T705",
			"Positivo|Positivo T705|T705|T705K",
			"Positivo|Positivo T710|T710|Positivo T710",
			"Positivo|Positivo T750|t750|Positivo T750",
			"Positivo|Positivo YPY AB7|YPY_AB7|Positivo Ypy AB7E",
			"Positivo|Positivo YPY AB7|YPY_AB7|Positivo Ypy AB7EC",
			"Positivo|Positivo YPY7 3G|TB07FTA|Positivo Ypy 7 - TB07FTA",
			"Positivo|Positivo YPY7 3G|TB07FTA|Ypy 7 - TB07FTA",
			"Positivo|Positivo YPY7 3G|YPY_07FTA|YPY_07FTA",
			"Positivo|Positivo YPY7 wifi|TB07STA|Positivo Ypy 7 - TB07STA",
			"Positivo|Positivo YPY7 wifi|TB07STA|Ypy 7 - TB07STA",
			"Positivo|Positivo YPY7 wifi|YPY_07STA|YPY_07STA",
			"Positivo|Positivo Ypy 07FTB|YPY_07FTB|POSITIVO TABLET YPY 07FTB PM BEL\\xc3\\x89M",
			"Positivo|Positivo Ypy 07FTB|YPY_07FTB|YPY_07FTB",
			"Positivo|Positivo Ypy 07FTBF|YPY_07FTBF|YPY_07FTBF",
			"Positivo|Positivo Ypy 07STB|YPY_07STB|YPY_07STB",
			"Positivo|Positivo Ypy 07STBF|YPY_07STBF|YPY_07STBF",
			"Positivo|Positivo Ypy 10FTB|YPY_10FTB|YPY_10FTB",
			"Positivo|Positivo Ypy 10FTBF|YPY_10FTBF|YPY_10FTBF",
			"Positivo|Positivo Ypy 10STB|YPY_10STB|YPY_10STB",
			"Positivo|Positivo Ypy 10STBF|YPY_10STBF|YPY_10STBF",
			"Positivo|Positivo Ypy AB10|YPY_AB10D|YPY_AB10D",
			"Positivo|Positivo Ypy AB10D|YPY_AB10DC|YPY_AB10DC",
			"Positivo|Positivo Ypy AB10E|YPY_AB10EC|Positivo Ypy AB10EC",
			"Positivo|Positivo Ypy AB10H|YPY_AB10H|Positivo Ypy AB10H",
			"Positivo|Positivo Ypy AB7F|YPY_AB7F|Positivo Ypy AB7F",
			"Positivo|Positivo Ypy L700|YPY_L700|Positivo Ypy L700",
			"Positivo|Positivo Ypy L700|YPY_L700|Positivo Ypy L700 Ed. Especial",
			"Positivo|Positivo Ypy L700|YPY_L700|Positivo Ypy L700+",
			"Positivo|Positivo Ypy S350p|YPY_S350|YPY_S350_PLUS",
			"Positivo|Positivo Ypy S405|YPY_S405|YPY_S405",
			"Positivo|Positivo Ypy TQ7|YPY_TQ7|YPY_TQ7",
			"Positivo|Positivo mini I|mini_i|Positivo Mini I",
			"Positivo|S440|S440|S440",
			"Positivo|S480|S480|S480",
			"Positivo|YPY 10 3G|YPY_10FTA|YPY_10FTA",
			"Positivo|YPY S400|YPY_S400|YPY_S400",
			"Positivo|Ypy|T701_AR|L701 TV",
			"Positivo|Ypy|YPY_L700|Positivo BGH Ypy L700",
			"Positivo|Ypy AB10DP|YPY_AB10DP|YPY_AB10DP",
			"Positivo|Ypy AB10E|YPY_AB10E|Positivo Ypy AB10E",
			"Positivo|Ypy AB7K|YPY_AB7K|YPY_AB7K",
			"Positivo|Ypy Kids|YPY_L700|Positivo BGH Ypy L700 Kids",
			"Positivo|Ypy Kids|YPY_L700|Positivo Ypy L700 Kids",
			"Positivo|Ypy Kids|YPY_L700|Positivo Ypy L700+ Kids",
			"Positivo|Ypy L1000|YPY_L1000|Positivo Ypy L1000",
			"Positivo|Ypy L1000|YPY_L1000|Positivo Ypy L1000F",
			"Positivo|Ypy L1050|YPY_L1050|Positivo Ypy L1050",
			"Positivo|Ypy L1050|YPY_L1050|Positivo Ypy L1050F",
			"Positivo|Ypy L1050E|YPY_L1050E|Positivo Ypy L1050E",
			"Positivo|Ypy Mini|mini|Positivo BGH Mini",
			"Positivo|Ypy S350|YPY_S350|YPY_S350",
			"Positivo|Ypy S450|YPY_S450|YPY_S450",
			"Positivo|Ypy S460|YPY_S460|YPY_S460",
			"Positivo|Ypy S500|YPY_S500|YPY_S500",
			"Prestige|ELITE10Q|Elite10Q|Elite10Q",
			"Prestige|ELITE7Q|Elite7Q|Elite7Q",
			"Prestige|ELITE8Q|ELITE8Q|ELITE8Q",
			"Prestige|ELITE8QS|Elite8QS|Elite8QS",
			"Prestige|ELITE9Q|ELITE9Q|ELITE9Q",
			"Prestige|PRO7DS|PRO7DS|PRO7DS",
			"Prestige|PRO8D|PRO8D|PRO8D",
			"Prestige|PRO9D|PRO9D|PRO9D",
			"Prestige|Prestige 7G|Prestige-7G|Prestige 7G",
			"Prestige|Prestige PRO10D|PRO10D|PRO10D",
			"Prestigio|GV7790|GV7790|GV7790",
			"Prestigio|MultiPad Thunder 7.0i|g01p|PMT3377_Wi",
			"Prestigio|MultiPad Wize 3037 3G|PMT3038_3G|PMT3037_3G",
			"Prestigio|MultiPad Wize 3038 3G|PMT3038_3G|PMT3038_3G",
			"Prestigio|MultiPhone 5453 DUO|PSP5453DUO|PSP5453DUO",
			"Prestigio|MultiPhone 5455 DUO|PSP5455DUO|PSP5455DUO",
			"Prestigio|MultiPhone 5504 DUO|PSP5504DUO|PSP5504DUO",
			"Prestigio|MultiPhone 5505 DUO|PSP5505DUO|PSP5505DUO",
			"Prestigio|MultiPhone 5517 DUO|PSP5517DUO|PSP5517DUO",
			"Prestigio|Multiphone 5508 DUO|PSP5508DUO|PSP5508DUO",
			"Prestigio|PMT3008_Wi|PMT3008_Wi_C|PMT3008_Wi_C",
			"Prestigio|PMT3047_3G|PMT3047_3G|PMT3047_3G",
			"Prestigio|PMT3177_3G|PMT3177_3G|PMT3177_3G",
			"Prestigio|PMT3277_3G|PMT3277_3G|PMT3277_3G",
			"Prestigio|PMT3287_3G|PMT3287_3G|PMT3287_3G",
			"Prestigio|PMT3677_Wi|PMT3677|PMT3677_Wi",
			"Prestigio|PMT5001_3G|PMT5001_3G|PMT5001_3G",
			"Prestigio|PMT5002_Wi|PMT5002_Wi|PMT5002_Wi",
			"Prestigio|PMT5008_3G|PMT5008_3G|PMT5008_3G",
			"Prestigio|PMT5011_3G|PMT5001_3G|PMT5011_3G",
			"Prestigio|PMT5018_3G|PMT5008_3G|PMT5018_3G",
			"Prestigio|PMT5021_3G|PMT5001_3G|PMT5021_3G",
			"Prestigio|PMT5287_4G|PMT5287_4G|PMT5287_4G",
			"Prestigio|PMT5487_3G|PMT5487_3G|PMT5487_3G",
			"Prestigio|PMT5777_3G|PMT5777_3G|PMT5777_3G",
			"Prestigio|PMT5887_3G|PMT5887_3G|PMT5887_3G",
			"Prestigio|PMT7008_4G|PMT7008_4G|PMT7008_4G",
			"Prestigio|PMT7177_3G|PMT7177_3G|PMT7177_3G",
			"Prestigio|PMT7287_3G|PMT7287_3G|PMT7287_3G",
			"Prestigio|PMT7787_3G|PMT7787_3G|PMT7787_3G",
			"Prestigio|PSP3404DUO|PSP3404DUO|PSP3404DUO",
			"Prestigio|PSP3405DUO|PSP3405DUO|PSP3405DUO",
			"Prestigio|PSP3450DUO|PSP3450DUO|PSP3450DUO",
			"Prestigio|PSP3502DUO|PSP3502DUO|PSP3502DUO",
			"Prestigio|PSP5047DUO|MT6589|PSP5047DUO",
			"Prestigio|PSP5454DUO|PSP5454DUO|PSP5454DUO",
			"Prestigio|PSP5457DUO|MT6572|PSP5457DUO",
			"Prestigio|PSP5507DUO|PSP5507DUO|PSP5507DUO",
			"Prestigio|PSP5550DUO|PSP5550DUO|PSP5550DUO",
			"Prestigio|Prestigio Grace|PSP7557|PSP7557",
			"Prestigio|Prestigio MultiPad Wize 3009|PMT3009_Wi_C|PMT3009_Wi_C",
			"Prestigio|Prestigio MultiPad Wize 3017|PMT3017_WI|PMT3017_WI",
			"Prestigio|Prestigio MultiPad Wize 3018|PMT3018_WI|PMT3018_WI",
			"Prestigio|Prestigio PMT5587_Wi|AML8726-MXS|PMT5587_Wi",
			"Prestigio|Prestigio PMT5877C|PMT5877C|PMT5877C",
			"Prestigio|Prestigio PMT7077_3G|PMT7077_3G|PMT7077_3G",
			"Prestigio|Prestigio PMT7287C3G|PMT7287C3G|PMT7287C3G",
			"Prestigio|Prestigio PSP5307DUO|PSP5307DUO|PSP5307DUO",
			"Proscan|PLT1077G(1GB/8GB)|PLT1077G|PLT1077G",
			"Proscan|PLT7100G|PLT7100G|PLT7100G",
			"Proscan|PLT7109G|PLT7109G|PLT7109G",
			"Proscan|PLT7130G|PLT7130G|PLT7130G",
			"Proscan|PLT7602G|PLT7602G|PLT7602G",
			"Proscan|PLT7803G|PLT7803G|PLT7803G",
			"Proscan|PLT7804G|PLT7804G|PLT7804G",
			"Proscan|PLT8802G|PLT8802G|PLT8802G",
			"Proscan|PLT9602G|PLT9602G|PLT9602G",
			"Proscan|PLT9606G|PLT9606G|PLT9606G",
			"Proscan|PLT9609G|PLT9609G|PLT9609G",
			"Proscan|PROSCAN PLT8235G Tablet|MID807_K|PLT8235G",
			"QLA|C600|GBC_bravo|C600",
			"QOOQ|QOOQV3|QOOQ|QOOQ",
			"Qilive|Q.3778|Q3778|Q.3778",
			"Qilive|Qilive W55|W55|W55",
			"Quanta|CT2200|CT2200|CT2200",
			"Quanta|FT103|FT103|FT103",
			"Quanta|True Beyond 4G|al7|TRUE BEYOND 4G",
			"Quanta|VSD220|VSD220|VSD220",
			"RAINBOW|WIKO|s5501|RAINBOW",
			"RCA|RCA DAA730R / RCA DAA738R|MD7081|DAA730R",
			"RCA|RCA DAA730R / RCA DAA738R|MD7081|DAA738R",
			"RCA|RCA RCS13101T|RCS13101T|RCS13101T",
			"RCA|RCA RCT6077W2|RCT6077W2|RCT6077W2",
			"RCA|RCA RCT6077W22|RCT6077W22|RCT6077W22",
			"RCA|RCA RCT6272W23|RCT6272W23|RCT6272W23",
			"RCA|RCA RCT6378W2|RCT6378W2|RCT6378W2",
			"RCA|RCA RCT6691W3|RCT6691W3|RCT6691W3",
			"RCA|RCT6002W46|RCT6002W46|RCT6002W46",
			"RCA|RCT6103W46|RCT6103W46|RCT6103W46",
			"RCA|RCT6203W46|RCT6203W46|RCT6203W46",
			"RCA|RCT6273W26|RCT6273W26|RCT6273W26",
			"RCA|RCT6293W23|RCT6293W23|RCT6293W23",
			"RCA|RCT6573W23|RCT6573W23|RCT6573W23",
			"RCA|RCT6672W23|RCT6672W23|RCT6672W23",
			"RCA|RCT6773W22|RCT6773W22|RCT6773W22",
			"REACH|Q882|Q882|Q882",
			"Revtel|R45|Revel_R45|R45",
			"SAKAISIO|SAKAISIO FP-U320-711-WIFI|U320-WF|FP-U320-711-WIFI",
			"SAKAISIO|SAKAISIO FP-U320-711-WWAN|U320-3G|FP-U320-711-WWAN",
			"SELECLINE|MID7526CM|MID7526CM|NID_7010",
			"SELECLINE|MID_9526CM|MID_9526CM|S952",
			"SENSEIT|SENSEIT L301|SENSEIT-L301|SENSEIT L301",
			"SFR|SFR-G8800|stb_dvb_sfr|SFR-G8800",
			"SFR|STARADDICT 4|staraddict4|STARADDICT 4",
			"SFR|Star Shine II|up11_sfr_mensa|STARSHINE II",
			"SFR|Star Trail 5|STARTRAIL5|STARTRAIL5",
			"SFR|Star Trail III|mobiwire_startrail3|StarTrail III",
			"SK Broadband|B tv smart|BHX-S100|BHX-S100",
			"SK Broadband|B tv smart|BKO-S200|BKO-S200",
			"SK Telesys|AN200|AN200|AN200",
			"SK Telesys|Fly|htt77_ics2|Fly",
			"SK Telesys|ITP-E410W|willow|ITP-E410W",
			"SK Telesys|ITP-R208W|rk30sdk|ITP-R208W",
			"SK Telesys|Ice-Phone Forever|z262_wvga_mge|Ice-Phone Forever",
			"SK Telesys|K2401|K2401|S201",
			"SK Telesys|S150|msm8255_k5|S150",
			"SK Telesys|T720|T720|Clear",
			"SK Telesys|UTA200|htt75_nand_6628_ics|UTA200",
			"SK Telesys|icube-830|icube-830|icube-830",
			"SKnetworks|EG68BE|EG668BE|EG668",
			"SKnetworks|EG978TW|EG978TW|EG978",
			"SKnetworks|EG980|eg980tw|EG980",
			"SKnetworks|WA966|u966e_2|WA966",
			"SKnetworks|WA978|u978e_1|WA978",
			"SMARTAB|SMARTAB ST700|ST700|ST700",
			"SOSMART|SOSMART T5|SOSMART_T5|SOSMART_T5",
			"STOREX|eZee\\'Tab7Q11-M|Tab7Q11|Tab7Q11",
			"STOREX|eZee\\xe2\\x80\\x99Tab7D14-S|STOREX7526|NID_7010",
			"Samsung|Absolute|GT-B9120|GT-B9120",
			"Samsung|Acclaim|SCH-R880|SCH-R880",
			"Samsung|Admire|SCH-R720|SCH-R720",
			"Samsung|Amazing|amazingtrf|SGH-S730M",
			"Samsung|Baffin|baffinltelgt|SHV-E270L",
			"Samsung|Captivate Glide|SGH-I927|SAMSUNG-SGH-I927",
			"Samsung|Captivate Glide|SGH-I927|SGH-I927",
			"Samsung|China Telecom|kylevectc|SCH-I699I",
			"Samsung|Conquer|SPH-D600|SPH-D600",
			"Samsung|DoubleTime|SGH-I857|SAMSUNG-SGH-I857",
			"Samsung|Droid Charge|SCH-I510|SCH-I510",
			"Samsung|Europa|GT-I5500B|GT-I5500B",
			"Samsung|Europa|GT-I5500L|GT-I5500L",
			"Samsung|Europa|GT-I5500M|GT-I5500M",
			"Samsung|Europa|GT-I5503T|GT-I5503T",
			"Samsung|Europa|GT-I5510L|GT-I5510L",
			"Samsung|Exhibit|SGH-T759|SGH-T759",
			"Samsung|GALAXY Camera|gd1|EK-GC100",
			"Samsung|Galaxy (China)|GT-B9062|GT-B9062",
			"Samsung|Galaxy 070|hendrix|YP-GI2",
			"Samsung|Galaxy A|archer|SHW-M100S",
			"Samsung|Galaxy A|archer|archer",
			"Samsung|Galaxy A3|a33g|SM-A300H",
			"Samsung|Galaxy A3|a3lte|SM-A300F",
			"Samsung|Galaxy A3|a3lte|SM-A300M",
			"Samsung|Galaxy A3|a3ltechn|SM-A3000",
			"Samsung|Galaxy A3|a3ltechn|SM-A300X",
			"Samsung|Galaxy A3|a3ltectc|SM-A3009",
			"Samsung|Galaxy A3|a3ltezh|SM-A3000",
			"Samsung|Galaxy A3|a3ulte|SM-A300FU",
			"Samsung|Galaxy A3|a3ulte|SM-A300XU",
			"Samsung|Galaxy A5|a53g|SM-A500H",
			"Samsung|Galaxy A5|a5lte|SM-A500F",
			"Samsung|Galaxy A5|a5lte|SM-A500G",
			"Samsung|Galaxy A5|a5lte|SM-A500M",
			"Samsung|Galaxy A5|a5ltechn|SM-A5000",
			"Samsung|Galaxy A5|a5ltechn|SM-A500X",
			"Samsung|Galaxy A5|a5ltectc|SM-A5009",
			"Samsung|Galaxy A5|a5ltezh|SM-A5000",
			"Samsung|Galaxy A5|a5ltezt|SM-A500YZ",
			"Samsung|Galaxy A5|a5ultektt|SM-A500K",
			"Samsung|Galaxy A5|a5ultelgt|SM-A500L",
			"Samsung|Galaxy A5|a5ulteskt|SM-A500S",
			"Samsung|Galaxy A7|a7alte|SM-A700F",
			"Samsung|Galaxy A7|a7ltektt|SM-A700K",
			"Samsung|Galaxy A7|a7ltelgt|SM-A700L",
			"Samsung|Galaxy A7|a7lteskt|SM-A700S",
			"Samsung|Galaxy Ace|GT-S5830|GT-S5830",
			"Samsung|Galaxy Ace|GT-S5830B|GT-S5830B",
			"Samsung|Galaxy Ace|GT-S5830C|GT-S5830C",
			"Samsung|Galaxy Ace|GT-S5830D|GT-S5830D",
			"Samsung|Galaxy Ace|GT-S5830F|GT-S5830F",
			"Samsung|Galaxy Ace|GT-S5830G|GT-S5830G",
			"Samsung|Galaxy Ace|GT-S5830L|GT-S5830L",
			"Samsung|Galaxy Ace|GT-S5830M|GT-S5830M",
			"Samsung|Galaxy Ace|GT-S5830T|GT-S5830T",
			"Samsung|Galaxy Ace|GT-S5830i|GT-S5830i",
			"Samsung|Galaxy Ace|GT-S5831i|GT-S5831i",
			"Samsung|Galaxy Ace|GT-S5838|GT-S5838",
			"Samsung|Galaxy Ace|GT-S5839i|GT-S5839i",
			"Samsung|Galaxy Ace|GT-S6358|GT-S6358",
			"Samsung|Galaxy Ace|SCH-I619|SCH-I619",
			"Samsung|Galaxy Ace|SHW-M240S|SHW-M240S",
			"Samsung|Galaxy Ace|heat3gou|SM-G310R5",
			"Samsung|Galaxy Ace|heatlte|SM-G357M",
			"Samsung|Galaxy Ace 4|vivaltods5m|SM-G313HU",
			"Samsung|Galaxy Ace 4|vivaltods5m|SM-G313HY",
			"Samsung|Galaxy Ace 4|vivaltods5m|SM-G313M",
			"Samsung|Galaxy Ace 4|vivaltods5m|SM-G313MY",
			"Samsung|Galaxy Ace 4 Lite|vivalto3g|SM-G313U",
			"Samsung|Galaxy Ace Advance|GT-S6800|GT-S6800",
			"Samsung|Galaxy Ace Duos|GT-S6352|GT-S6352",
			"Samsung|Galaxy Ace Duos|GT-S6802|GT-S6802",
			"Samsung|Galaxy Ace Duos|GT-S6802B|GT-S6802B",
			"Samsung|Galaxy Ace Duos|SCH-I579|SCH-i579",
			"Samsung|Galaxy Ace Duos|SCH-I589|SCH-I589",
			"Samsung|Galaxy Ace Duos|SCH-I589|SCH-i589",
			"Samsung|Galaxy Ace Duos|SCH-i579|SCH-i579",
			"Samsung|Galaxy Ace Duos|SCH-i589|SCH-i589",
			"Samsung|Galaxy Ace Plus|GT-S7500|GT-S7500",
			"Samsung|Galaxy Ace Plus|GT-S7500L|GT-S7500L",
			"Samsung|Galaxy Ace Plus|GT-S7500T|GT-S7500T",
			"Samsung|Galaxy Ace Plus|GT-S7500W|GT-S7500W",
			"Samsung|Galaxy Ace Plus|GT-S7508|GT-S7508",
			"Samsung|Galaxy Ace Q|SGH-I827D|SGH-I827D",
			"Samsung|Galaxy Ace Style|heat3gtfnvzw|SM-S765C",
			"Samsung|Galaxy Ace Style|heat3gtfnvzw|SM-S766C",
			"Samsung|Galaxy Ace Style|heatnfc3g|SM-G310HN",
			"Samsung|Galaxy Ace Style|heatqlte|SM-G357FZ",
			"Samsung|Galaxy Ace2|GT-I8160|GT-I8160",
			"Samsung|Galaxy Ace2|GT-I8160L|GT-I8160L",
			"Samsung|Galaxy Ace2|GT-I8160P|GT-I8160P",
			"Samsung|Galaxy Ace2 X|kylessopen|GT-S7560",
			"Samsung|Galaxy Ace2 X|kylessopen|GT-S7560M",
			"Samsung|Galaxy Ace3|logan|GT-S7270",
			"Samsung|Galaxy Ace3|logan|GT-S7270L",
			"Samsung|Galaxy Ace3|logan|SCH-I679",
			"Samsung|Galaxy Ace3|logan3gcmcc|GT-S7278",
			"Samsung|Galaxy Ace3|logands|GT-S7272",
			"Samsung|Galaxy Ace3|loganlte|GT-S7275",
			"Samsung|Galaxy Ace3|loganrelte|GT-S7275B",
			"Samsung|Galaxy Ace3|loganrelte|GT-S7275R",
			"Samsung|Galaxy Ace3|loganrelte|GT-S7275T",
			"Samsung|Galaxy Ace3 Duos|loganlite3g|GT-S7272C",
			"Samsung|Galaxy Ace3 Duos|loganu3gcmcc|GT-S7278U",
			"Samsung|Galaxy Ace3 Duos TV|logandsdtv|GT-S7273T",
			"Samsung|Galaxy Ace4|vivalto3g|SM-G313ML",
			"Samsung|Galaxy Ace4|vivalto5mve3g|SM-G316M",
			"Samsung|Galaxy Ace4|vivaltolte|SM-G313F",
			"Samsung|Galaxy Ace4|vivaltolte|SM-G313MU",
			"Samsung|Galaxy Ace4|vivaltonfc3g|SM-G313HN",
			"Samsung|Galaxy Ace4 Lite|vivalto|SM-G3139D",
			"Samsung|Galaxy Ace4 Lite|vivalto3g|SM-G313H",
			"Samsung|Galaxy Ace4 Neo|vivalto3mve3gltn|SM-G316ML",
			"Samsung|Galaxy Admire|SCH-R820|SCH-R820",
			"Samsung|Galaxy Admire 2|goghcri|SCH-R830C",
			"Samsung|Galaxy Alpha|slte|SM-G850F",
			"Samsung|Galaxy Alpha|slte|SM-G850FQ",
			"Samsung|Galaxy Alpha|slte|SM-G850M",
			"Samsung|Galaxy Alpha|slte|SM-G850X",
			"Samsung|Galaxy Alpha|slte|SM-G850Y",
			"Samsung|Galaxy Alpha|slteatt|SAMSUNG-SM-G850A",
			"Samsung|Galaxy Alpha|sltecan|SM-G850W",
			"Samsung|Galaxy Alpha|sltechn|SM-G8508S",
			"Samsung|Galaxy Alpha|sltektt|SM-G850K",
			"Samsung|Galaxy Alpha|sltelgt|SM-G850L",
			"Samsung|Galaxy Alpha|slteskt|SM-G850S",
			"Samsung|Galaxy Amp|kyleatt|SAMSUNG-SGH-I407",
			"Samsung|Galaxy Apollo|GT-I5800|GT-I5800",
			"Samsung|Galaxy Apollo|GT-I5800|GT-I5800L",
			"Samsung|Galaxy Apollo|GT-I5800D|GT-I5800D",
			"Samsung|Galaxy Apollo|GT-I5801|GT-I5801",
			"Samsung|Galaxy Appeal|SGH-I827|SAMSUNG-SGH-I827",
			"Samsung|Galaxy Attain|SCH-R920|SCH-R920",
			"Samsung|Galaxy Avant|afyonltetmo|SM-G386T",
			"Samsung|Galaxy Axiom|infiniteusc|SCH-R830",
			"Samsung|Galaxy Beam|GT-I8250|GT-I8250",
			"Samsung|Galaxy Beam|GT-I8530|GT-I8530",
			"Samsung|Galaxy Camera|gd1|EK-GC100",
			"Samsung|Galaxy Camera|gd1att|SAMSUNG-EK-GC100",
			"Samsung|Galaxy Camera|gd1can|EK-GC100",
			"Samsung|Galaxy Camera|gd1ktt|EK-KC100K",
			"Samsung|Galaxy Camera|gd1ltelgt|EK-KC120L",
			"Samsung|Galaxy Camera|gd1lteskt|EK-KC120S",
			"Samsung|Galaxy Camera|gd1ltevzw|EK-GC120",
			"Samsung|Galaxy Camera|gd1skt|EK-KC100S",
			"Samsung|Galaxy Camera|gd1wifi|EK-GC110",
			"Samsung|Galaxy Camera|gd1wifiany|EK-GC110",
			"Samsung|Galaxy Camera|u0lte|EK-GN100",
			"Samsung|Galaxy Camera|u0lte|EK-GN120",
			"Samsung|Galaxy Camera 2|sf2wifi|EK-GC200",
			"Samsung|Galaxy Centura|amazing3gtrf|SCH-S738C",
			"Samsung|Galaxy Chat|zanin|GT-B5330",
			"Samsung|Galaxy Chat|zanin|GT-B5330B",
			"Samsung|Galaxy Chat|zanin|GT-B5330L",
			"Samsung|Galaxy Core|afyonlteMetroPCS|SM-G386T1",
			"Samsung|Galaxy Core|afyonltecan|SM-G386W",
			"Samsung|Galaxy Core|arubaslim|GT-I8262",
			"Samsung|Galaxy Core|arubaslimss|GT-I8260",
			"Samsung|Galaxy Core|arubaslimss|GT-I8260L",
			"Samsung|Galaxy Core 2|kanas|SM-G355H",
			"Samsung|Galaxy Core Advance|cane3g|GT-I8580",
			"Samsung|Galaxy Core Advance|cane3gskt|SHW-M570S",
			"Samsung|Galaxy Core LTE|afyonlte|SM-G386F",
			"Samsung|Galaxy Core LTE|cs03lte|SM-G3518",
			"Samsung|Galaxy Core Lite|victorlte|SM-G3586V",
			"Samsung|Galaxy Core Lite|victorlte|SM-G3589W",
			"Samsung|Galaxy Core Max|kleoslte|SM-G5108",
			"Samsung|Galaxy Core Max Duos|kleoslte|SM-G5108Q",
			"Samsung|Galaxy Core Plus|cs02|SM-G350",
			"Samsung|Galaxy Core Plus|cs023g|SM-G3502",
			"Samsung|Galaxy Core Plus|cs02ve3g|SM-G3502L",
			"Samsung|Galaxy Core Plus|cs02ve3gdtv|SM-G3502T",
			"Samsung|Galaxy Core Plus|cs02ve3gss|SM-G350L",
			"Samsung|Galaxy Core Plus|cs02ve3gss|SM-G350M",
			"Samsung|Galaxy Core Prime|core33g|SM-G360H",
			"Samsung|Galaxy Core Prime|coreprimelte|SM-G360M",
			"Samsung|Galaxy Core Prime|rossalte|SM-G3606",
			"Samsung|Galaxy Core Prime|rossalte|SM-G3608",
			"Samsung|Galaxy Core Prime|rossaltectc|SM-G3609",
			"Samsung|Galaxy Core Safe|arubaslimss|GT-I8260E",
			"Samsung|Galaxy Core Safe|arubaslimss|SHW-M580D",
			"Samsung|Galaxy Core Safe|arubaslimss|SHW-M585D",
			"Samsung|Galaxy Core2|kanas|SM-G355H",
			"Samsung|Galaxy Core2|kanas|SM-G355HQ",
			"Samsung|Galaxy Core2|kanas|SM-G355M",
			"Samsung|Galaxy Core2|kanas3g|SM-G3556D",
			"Samsung|Galaxy Core2|kanas3gcmcc|SM-G3558",
			"Samsung|Galaxy Core2|kanas3gctc|SM-G3559",
			"Samsung|Galaxy Core2|kanas3gnfc|SM-G355HN",
			"Samsung|Galaxy Discover|amazing3gcri|SCH-R740C",
			"Samsung|Galaxy Discover|amazingtrfcd|SCH-S735C",
			"Samsung|Galaxy Duos|aruba3gcmcc|GT-I8268",
			"Samsung|Galaxy E7|e7lte|SM-E700M",
			"Samsung|Galaxy Europa|GT-I5500|GT-I5500",
			"Samsung|Galaxy Europa|GT-I5503|GT-I5503",
			"Samsung|Galaxy Europa|GT-I5508|GT-I5508",
			"Samsung|Galaxy Europa|GT-I5510|GT-I5510",
			"Samsung|Galaxy Exhibit|codinaMetroPCS|SGH-T599N",
			"Samsung|Galaxy Exhibit|codinatmo|SGH-T599",
			"Samsung|Galaxy Exhibit|codinavid|SGH-T599V",
			"Samsung|Galaxy Exhibit2|SGH-T679|SGH-T679",
			"Samsung|Galaxy Exhilarate|SGH-I577|SAMSUNG-SGH-I577",
			"Samsung|Galaxy Express|expressatt|SAMSUNG-SGH-I437",
			"Samsung|Galaxy Express|expressatt|SAMSUNG-SGH-I437P",
			"Samsung|Galaxy Express|expresslte|GT-I8730",
			"Samsung|Galaxy Express|expresslte|GT-I8730T",
			"Samsung|Galaxy Express|expressziglteatt|SAMSUNG-SGH-I437Z",
			"Samsung|Galaxy Express2|wilcoxlte|SM-G3815",
			"Samsung|Galaxy Fame|SCH-I629|SCH-I629",
			"Samsung|Galaxy Fame|nevis|GT-S6810",
			"Samsung|Galaxy Fame|nevis|GT-S6810B",
			"Samsung|Galaxy Fame|nevis|GT-S6810E",
			"Samsung|Galaxy Fame|nevis|GT-S6810L",
			"Samsung|Galaxy Fame|nevis3g|GT-S6812i",
			"Samsung|Galaxy Fame|nevis3gcmcc|GT-S6818",
			"Samsung|Galaxy Fame|nevis3gcmcc|GT-S6818V",
			"Samsung|Galaxy Fame|nevisds|GT-S6812",
			"Samsung|Galaxy Fame|nevisds|GT-S6812B",
			"Samsung|Galaxy Fame|nevisnvess|GT-S6790N",
			"Samsung|Galaxy Fame|nevisp|GT-S6810M",
			"Samsung|Galaxy Fame|nevisp|GT-S6810P",
			"Samsung|Galaxy Fame|nevisvess|GT-S6790",
			"Samsung|Galaxy Fame|nevisvess|GT-S6790E",
			"Samsung|Galaxy Fame|nevisvess|GT-S6790L",
			"Samsung|Galaxy Fame|nevisw|GT-S6812C",
			"Samsung|Galaxy Fame Lite Duos|nevisw|GT-S6792L",
			"Samsung|Galaxy Fit|GT-S5670|GT-S5670",
			"Samsung|Galaxy Fit|GT-S5670B|GT-S5670B",
			"Samsung|Galaxy Fit|GT-S5670L|GT-S5670L",
			"Samsung|Galaxy Fresh|kylevess|GT-S7390",
			"Samsung|Galaxy Fresh|kylevess|GT-S7390E",
			"Samsung|Galaxy Fresh|kylevess|GT-S7390G",
			"Samsung|Galaxy Gio|GT-S5660|GT-S5660",
			"Samsung|Galaxy Gio|GT-S5660B|GT-S5660B",
			"Samsung|Galaxy Gio|GT-S5660L|GT-S5660L",
			"Samsung|Galaxy Gio|GT-S5660M|GT-S5660M",
			"Samsung|Galaxy Gio|GT-S5660V|GT-S5660V",
			"Samsung|Galaxy Gio|SCH-i569|SCH-i569",
			"Samsung|Galaxy Gio|SHW-M290K|SHW-M290K",
			"Samsung|Galaxy Gio|SHW-M290S|SHW-M290S",
			"Samsung|Galaxy Golden|ks02lte|GT-I9230",
			"Samsung|Galaxy Golden|ks02lte|GT-I9235",
			"Samsung|Galaxy Golden|ks02ltektt|SHV-E400K",
			"Samsung|Galaxy Golden|ks02lteskt|SHV-E400S",
			"Samsung|Galaxy Golden 2|pateklte|SM-W2015",
			"Samsung|Galaxy Grand|baffin3gduosctc|SCH-I879",
			"Samsung|Galaxy Grand|baffincmcc|GT-I9128",
			"Samsung|Galaxy Grand|baffincmcc|GT-I9128V",
			"Samsung|Galaxy Grand|baffinltektt|SHV-E270K",
			"Samsung|Galaxy Grand|baffinlteskt|SHV-E270S",
			"Samsung|Galaxy Grand|baffinrd|GT-I9118",
			"Samsung|Galaxy Grand|baffinss|GT-I9080E",
			"Samsung|Galaxy Grand|baffinss|GT-I9080L",
			"Samsung|Galaxy Grand|baffinvektt|SHV-E275K",
			"Samsung|Galaxy Grand|baffinveskt|SHV-E275S",
			"Samsung|Galaxy Grand|baffinvetd3g|GT-I9128E",
			"Samsung|Galaxy Grand|baffinvetd3g|GT-I9128I",
			"Samsung|Galaxy Grand Duos|baffin|GT-I9082",
			"Samsung|Galaxy Grand Duos|baffin|GT-I9082L",
			"Samsung|Galaxy Grand Max|grandmaxltechn|SM-G720AX",
			"Samsung|Galaxy Grand Neo|baffinlite|GT-I9060",
			"Samsung|Galaxy Grand Neo|baffinlite|GT-I9060L",
			"Samsung|Galaxy Grand Neo|baffinlite|GT-I9082C",
			"Samsung|Galaxy Grand Neo|baffinlitedtv|GT-I9063T",
			"Samsung|Galaxy Grand Neo|baffinq3g|GT-I9168",
			"Samsung|Galaxy Grand Neo|baffinq3g|GT-I9168I",
			"Samsung|Galaxy Grand Neo Plus|grandneove3g|GT-I9060C",
			"Samsung|Galaxy Grand Neo+|baffinq3gduosctc|SCH-I879E",
			"Samsung|Galaxy Grand Prime|fortuna3g|SM-G530H",
			"Samsung|Galaxy Grand Prime|fortuna3gdtv|SM-G530BT",
			"Samsung|Galaxy Grand Prime|fortunalte|SM-G5306W",
			"Samsung|Galaxy Grand Prime|fortunalte|SM-G5308W",
			"Samsung|Galaxy Grand Prime|fortunalte|SM-G530F",
			"Samsung|Galaxy Grand Prime|fortunalte|SM-G530M",
			"Samsung|Galaxy Grand Prime|fortunaltectc|SM-G5309W",
			"Samsung|Galaxy Grand Prime|fortunaltezh|SM-G5308W",
			"Samsung|Galaxy Grand Prime|fortunaltezt|SM-G530Y",
			"Samsung|Galaxy Grand Prime|grandprimelte|SM-G530FZ",
			"Samsung|Galaxy Grand duos|baffin|GT-I9082i",
			"Samsung|Galaxy Grand-Max|grandmaxltekx|SM-G720N0",
			"Samsung|Galaxy Grand2|ms013g|SM-G7102",
			"Samsung|Galaxy Grand2|ms013g|SM-G7106",
			"Samsung|Galaxy Grand2|ms013g|SM-G7108",
			"Samsung|Galaxy Grand2|ms013g|SM-G7109",
			"Samsung|Galaxy Grand2|ms013gdtv|SM-G7102T",
			"Samsung|Galaxy Grand2|ms013gss|SM-G710",
			"Samsung|Galaxy Grand2|ms01lte|SM-G7105",
			"Samsung|Galaxy Grand2|ms01lte|SM-G7105L",
			"Samsung|Galaxy Grand2|ms01ltektt|SM-G710K",
			"Samsung|Galaxy Grand2|ms01ltelgt|SM-G710L",
			"Samsung|Galaxy Grand2|ms01lteskt|SM-G710S",
			"Samsung|Galaxy Indulge|SCH-R910|SCH-R910",
			"Samsung|Galaxy Indulge|SCH-R915|SCH-R915",
			"Samsung|Galaxy Infinite|infinite3gduosctc|SCH-I759",
			"Samsung|Galaxy J|hltejs01tw|SGH-N075T",
			"Samsung|Galaxy K|SHW-M130K|SHW-M130K",
			"Samsung|Galaxy K Zoom|m2a3g|SM-C111",
			"Samsung|Galaxy K Zoom|m2a3g|SM-C111M",
			"Samsung|Galaxy K Zoom|m2alte|SM-C115",
			"Samsung|Galaxy K Zoom|m2alte|SM-C115M",
			"Samsung|Galaxy K Zoom|m2altecan|SM-C115W",
			"Samsung|Galaxy K Zoom|m2altelgt|SM-C115L",
			"Samsung|Galaxy M Pro2|lucas|GT-B7810",
			"Samsung|Galaxy M Style|SHW-M340L|SHW-M340L",
			"Samsung|Galaxy M Style|SHW-M340S|SHW-M340S",
			"Samsung|Galaxy M Style|vastoicmcc|GT-I8258",
			"Samsung|Galaxy Mega 2|mega23g|SM-G750H",
			"Samsung|Galaxy Mega 5.8|crater|GT-I9152",
			"Samsung|Galaxy Mega 5.8|crater3gctc|SCH-P709",
			"Samsung|Galaxy Mega 5.8|craterss|GT-I9150",
			"Samsung|Galaxy Mega 5.8|cratertd3g|GT-I9158",
			"Samsung|Galaxy Mega 6.3|melius3g|GT-I9200",
			"Samsung|Galaxy Mega 6.3|melius3g|GT-I9208",
			"Samsung|Galaxy Mega 6.3|melius3gduosctc|SCH-P729",
			"Samsung|Galaxy Mega 6.3|meliuslte|GT-I9205",
			"Samsung|Galaxy Mega 6.3|meliuslteMetroPCS|SGH-M819N",
			"Samsung|Galaxy Mega 6.3|meliuslteatt|SAMSUNG-SGH-I527",
			"Samsung|Galaxy Mega 6.3|meliusltecan|SGH-I527M",
			"Samsung|Galaxy Mega 6.3|meliusltektt|SHV-E310K",
			"Samsung|Galaxy Mega 6.3|meliusltelgt|SHV-E310L",
			"Samsung|Galaxy Mega 6.3|meliuslteskt|SHV-E310S",
			"Samsung|Galaxy Mega 6.3|meliusltespr|SPH-L600",
			"Samsung|Galaxy Mega 6.3|meliuslteusc|SCH-R960",
			"Samsung|Galaxy Mega Plus|craterq3g|GT-I9152P",
			"Samsung|Galaxy Mega Plus|craterq3g|GT-I9158P",
			"Samsung|Galaxy Mega Plus|megapluslte|GT-I9158V",
			"Samsung|Galaxy Mega2|mega2lte|SM-G750F",
			"Samsung|Galaxy Mega2|mega2lteatt|SAMSUNG-SM-G750A",
			"Samsung|Galaxy Mega2|vasta3g|SM-G750H",
			"Samsung|Galaxy Mega2|vastalte|SM-G7508Q",
			"Samsung|Galaxy Mega2|vastaltezh|SM-G7508Q",
			"Samsung|Galaxy Mini|GT-S5570|GT-S5570",
			"Samsung|Galaxy Mini|GT-S5570B|GT-S5570B",
			"Samsung|Galaxy Mini|GT-S5570I|GT-S5570I",
			"Samsung|Galaxy Mini|GT-S5570L|GT-S5570L",
			"Samsung|Galaxy Mini|GT-S5578|GT-S5578",
			"Samsung|Galaxy Mini|SGH-T499|SGH-T499",
			"Samsung|Galaxy Mini|SGH-T499V|SGH-T499V",
			"Samsung|Galaxy Mini|SGH-T499Y|SGH-T499Y",
			"Samsung|Galaxy Mini2|GT-S6500|GT-S6500",
			"Samsung|Galaxy Mini2|GT-S6500D|GT-S6500D",
			"Samsung|Galaxy Mini2|GT-S6500L|GT-S6500L",
			"Samsung|Galaxy Mini2|GT-S6500T|GT-S6500T",
			"Samsung|Galaxy Music|ivoryss|GT-S6010",
			"Samsung|Galaxy Music|ivoryss|GT-S6010L",
			"Samsung|Galaxy Music Duos|ivory|GT-S6012",
			"Samsung|Galaxy Music Duos|ivory|GT-S6012B",
			"Samsung|Galaxy NX|u0lte|EK-GN120",
			"Samsung|Galaxy NX|u0lte|EK-GN120A",
			"Samsung|Galaxy NX|u0lteue|EK-GN120A",
			"Samsung|Galaxy Neo|SHW-M220L|SHW-M220L",
			"Samsung|Galaxy Nexus|maguro|Galaxy Nexus",
			"Samsung|Galaxy Nexus|maguro|Galaxy X",
			"Samsung|Galaxy Nexus|toro|Galaxy Nexus",
			"Samsung|Galaxy Nexus|toroplus|Galaxy Nexus",
			"Samsung|Galaxy Note|GT-I9220|GT-I9220",
			"Samsung|Galaxy Note|GT-I9228|GT-I9228",
			"Samsung|Galaxy Note|GT-N7000|GT-N7000",
			"Samsung|Galaxy Note|GT-N7005|GT-N7005",
			"Samsung|Galaxy Note|SC-05D|SC-05D",
			"Samsung|Galaxy Note|SCH-i889|SCH-i889",
			"Samsung|Galaxy Note|SGH-I717|SAMSUNG-SGH-I717",
			"Samsung|Galaxy Note|SGH-I717|SGH-I717",
			"Samsung|Galaxy Note|SGH-I717D|SGH-I717D",
			"Samsung|Galaxy Note|SGH-I717M|SGH-I717M",
			"Samsung|Galaxy Note|SGH-I717R|SGH-I717R",
			"Samsung|Galaxy Note|SGH-T879|SGH-T879",
			"Samsung|Galaxy Note|SHV-E160K|SHV-E160K",
			"Samsung|Galaxy Note|SHV-E160L|SHV-E160L",
			"Samsung|Galaxy Note|SHV-E160S|SHV-E160S",
			"Samsung|Galaxy Note 10.1|lt033g|SM-P601",
			"Samsung|Galaxy Note 10.1|lt033g|SM-P602",
			"Samsung|Galaxy Note 10.1|lt03ltektt|SM-P605K",
			"Samsung|Galaxy Note 10.1|lt03ltelgt|SM-P605L",
			"Samsung|Galaxy Note 10.1|lt03lteskt|SM-P605S",
			"Samsung|Galaxy Note 10.1|p4notelte|GT-N8020",
			"Samsung|Galaxy Note 10.1|p4noteltektt|SHV-E230K",
			"Samsung|Galaxy Note 10.1|p4noteltelgt|SHV-E230L",
			"Samsung|Galaxy Note 10.1|p4notelteskt|SHV-E230S",
			"Samsung|Galaxy Note 10.1|p4noteltespr|SPH-P600",
			"Samsung|Galaxy Note 10.1|p4notelteusc|SCH-I925U",
			"Samsung|Galaxy Note 10.1|p4noteltevzw|SCH-I925",
			"Samsung|Galaxy Note 10.1|p4noterf|GT-N8000",
			"Samsung|Galaxy Note 10.1|p4noterf|GT-N8005",
			"Samsung|Galaxy Note 10.1|p4noterfktt|SHW-M480K",
			"Samsung|Galaxy Note 10.1|p4notewifi|GT-N8013",
			"Samsung|Galaxy Note 10.1|p4notewifi43241any|SHW-M486W",
			"Samsung|Galaxy Note 10.1|p4notewifiany|SHW-M480W",
			"Samsung|Galaxy Note 10.1|p4notewifiktt|SHW-M485W",
			"Samsung|Galaxy Note 10.1|p4notewifiww|GT-N8010",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt033g|SM-P601",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03lte|SM-P605",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03lte|SM-P605M",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03ltevzw|SM-P605V",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03wifi|SM-P600",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03wifikx|SM-P600",
			"Samsung|Galaxy Note 10.1 2014 Edition|lt03wifiue|SM-P600",
			"Samsung|Galaxy Note 8|konalteatt|SAMSUNG-SGH-I467",
			"Samsung|Galaxy Note 8.0|kona3g|GT-N5100",
			"Samsung|Galaxy Note 8.0|kona3g|GT-N5105",
			"Samsung|Galaxy Note 8.0|konalte|GT-N5120",
			"Samsung|Galaxy Note 8.0|konaltecan|SGH-I467M",
			"Samsung|Galaxy Note 8.0|konawifi|GT-N5110",
			"Samsung|Galaxy Note 8.0|konawifiany|SHW-M500W",
			"Samsung|Galaxy Note Edge|SCL24|SCL24",
			"Samsung|Galaxy Note Edge|tbeltektt|SM-N915K",
			"Samsung|Galaxy Note Edge|tbeltelgt|SM-N915L",
			"Samsung|Galaxy Note Edge|tbelteskt|SM-N915S",
			"Samsung|Galaxy Note Edge|tblte|SM-N9150",
			"Samsung|Galaxy Note Edge|tblte|SM-N915F",
			"Samsung|Galaxy Note Edge|tblte|SM-N915FY",
			"Samsung|Galaxy Note Edge|tblte|SM-N915G",
			"Samsung|Galaxy Note Edge|tblte|SM-N915X",
			"Samsung|Galaxy Note Edge|tblteatt|SAMSUNG-SM-N915A",
			"Samsung|Galaxy Note Edge|tbltecan|SM-N915W8",
			"Samsung|Galaxy Note Edge|tbltechn|SM-N9150",
			"Samsung|Galaxy Note Edge|tbltespr|SM-N915P",
			"Samsung|Galaxy Note Edge|tbltetmo|SM-N915T",
			"Samsung|Galaxy Note Edge|tblteusc|SM-N915R4",
			"Samsung|Galaxy Note Edge|tbltevzw|SM-N915V",
			"Samsung|Galaxy Note II|t0lteatt|SAMSUNG-SGH-I317",
			"Samsung|Galaxy Note Pro 12.2|v1a3g|SM-P901",
			"Samsung|Galaxy Note Pro 12.2|v1awifi|SM-P900",
			"Samsung|Galaxy Note Pro 12.2|v1awifikx|SM-P900",
			"Samsung|Galaxy Note Pro 12.2|viennalte|SM-P905",
			"Samsung|Galaxy Note Pro 12.2|viennalte|SM-P905M",
			"Samsung|Galaxy Note Pro 12.2|viennalteatt|SAMSUNG-SM-P907A",
			"Samsung|Galaxy Note Pro 12.2|viennaltekx|SM-P905F0",
			"Samsung|Galaxy Note Pro 12.2|viennaltevzw|SM-P905V",
			"Samsung|Galaxy Note2|SC-02E|SC-02E",
			"Samsung|Galaxy Note2|t03g|GT-N7100",
			"Samsung|Galaxy Note2|t03g|GT-N7100T",
			"Samsung|Galaxy Note2|t03gchn|GT-N7100",
			"Samsung|Galaxy Note2|t03gchnduos|GT-N7102",
			"Samsung|Galaxy Note2|t03gchnduos|GT-N7102i",
			"Samsung|Galaxy Note2|t03gcmcc|GT-N7108",
			"Samsung|Galaxy Note2|t03gctc|SCH-N719",
			"Samsung|Galaxy Note2|t03gcuduos|GT-N7102",
			"Samsung|Galaxy Note2|t03gcuduos|GT-N7102i",
			"Samsung|Galaxy Note2|t0lte|GT-N7105",
			"Samsung|Galaxy Note2|t0lte|GT-N7105T",
			"Samsung|Galaxy Note2|t0lteatt|SAMSUNG-SGH-I317",
			"Samsung|Galaxy Note2|t0ltecan|SGH-I317M",
			"Samsung|Galaxy Note2|t0ltecan|SGH-T889V",
			"Samsung|Galaxy Note2|t0ltecmcc|GT-N7108D",
			"Samsung|Galaxy Note2|t0ltedcm|SC-02E",
			"Samsung|Galaxy Note2|t0ltektt|SHV-E250K",
			"Samsung|Galaxy Note2|t0ltelgt|SHV-E250L",
			"Samsung|Galaxy Note2|t0lteskt|SHV-E250S",
			"Samsung|Galaxy Note2|t0ltespr|SPH-L900",
			"Samsung|Galaxy Note2|t0ltetmo|SGH-T889",
			"Samsung|Galaxy Note2|t0lteusc|SCH-R950",
			"Samsung|Galaxy Note2|t0ltevzw|SCH-I605",
			"Samsung|Galaxy Note3|SC-02F|SC-02F",
			"Samsung|Galaxy Note3|SCL22|SCL22",
			"Samsung|Galaxy Note3|ha3g|SM-N900",
			"Samsung|Galaxy Note3|ha3g|SM-N9000Q",
			"Samsung|Galaxy Note3|hlte|SM-N9005",
			"Samsung|Galaxy Note3|hlte|SM-N9006",
			"Samsung|Galaxy Note3|hlte|SM-N9007",
			"Samsung|Galaxy Note3|hlte|SM-N9008V",
			"Samsung|Galaxy Note3|hlte|SM-N9009",
			"Samsung|Galaxy Note3|hlte|SM-N900U",
			"Samsung|Galaxy Note3|hlteatt|SAMSUNG-SM-N900A",
			"Samsung|Galaxy Note3|hltecan|SM-N900W8",
			"Samsung|Galaxy Note3|hltektt|SM-N900K",
			"Samsung|Galaxy Note3|hltelgt|SM-N900L",
			"Samsung|Galaxy Note3|hlteskt|SM-N900S",
			"Samsung|Galaxy Note3|hltespr|SM-N900P",
			"Samsung|Galaxy Note3|hltetmo|SM-N900T",
			"Samsung|Galaxy Note3|hlteusc|SM-N900R4",
			"Samsung|Galaxy Note3|hltevzw|SM-N900V",
			"Samsung|Galaxy Note3 Duos|hlte|SM-N9002",
			"Samsung|Galaxy Note3 Duos|hlte|SM-N9008",
			"Samsung|Galaxy Note3 Neo|frescoltektt|SM-N750K",
			"Samsung|Galaxy Note3 Neo|frescoltelgt|SM-N750L",
			"Samsung|Galaxy Note3 Neo|frescolteskt|SM-N750S",
			"Samsung|Galaxy Note3 Neo|hl3g|SM-N750",
			"Samsung|Galaxy Note3 Neo|hl3g|SM-N7500Q",
			"Samsung|Galaxy Note3 Neo|hl3gds|SM-N7502",
			"Samsung|Galaxy Note3 Neo|hllte|SM-N7505",
			"Samsung|Galaxy Note3 Neo|hllte|SM-N7505L",
			"Samsung|Galaxy Note3 Neo|hllte|SM-N7507",
			"Samsung|Galaxy Note4|tre3caltektt|SM-N916K",
			"Samsung|Galaxy Note4|tre3caltelgt|SM-N916L",
			"Samsung|Galaxy Note4|tre3calteskt|SM-N916S",
			"Samsung|Galaxy Note4|tre3g|SM-N910H",
			"Samsung|Galaxy Note4|trelte|SM-N910C",
			"Samsung|Galaxy Note4|treltektt|SM-N910K",
			"Samsung|Galaxy Note4|treltelgt|SM-N910L",
			"Samsung|Galaxy Note4|trelteskt|SM-N910S",
			"Samsung|Galaxy Note4|trhplte|SM-N910U",
			"Samsung|Galaxy Note4|trlte|SM-N910F",
			"Samsung|Galaxy Note4|trlte|SM-N910G",
			"Samsung|Galaxy Note4|trlte|SM-N910X",
			"Samsung|Galaxy Note4|trlteatt|SAMSUNG-SM-N910A",
			"Samsung|Galaxy Note4|trlteatt|SM-N910F",
			"Samsung|Galaxy Note4|trltecan|SM-N910W8",
			"Samsung|Galaxy Note4|trltechn|SM-N9100",
			"Samsung|Galaxy Note4|trltechn|SM-N9106W",
			"Samsung|Galaxy Note4|trltechn|SM-N9108V",
			"Samsung|Galaxy Note4|trltechn|SM-N9109W",
			"Samsung|Galaxy Note4|trltechnzh|SM-N9100",
			"Samsung|Galaxy Note4|trltespr|SM-N910P",
			"Samsung|Galaxy Note4|trltetmo|SM-N910T",
			"Samsung|Galaxy Note4|trlteusc|SM-N910R4",
			"Samsung|Galaxy Note4|trltevzw|SM-N910V",
			"Samsung|Galaxy Player|YP-GB70|YP-GB70",
			"Samsung|Galaxy Player 3.6|YP-GS1|YP-GS1",
			"Samsung|Galaxy Player 4|YP-GB1|YP-GB1",
			"Samsung|Galaxy Player 4.0|YP-G1|YP-G1",
			"Samsung|Galaxy Player 4.2|YP-GI1|YP-GI1",
			"Samsung|Galaxy Player 5|YP-G70|YP-G70",
			"Samsung|Galaxy Player 5.8|harrison|YP-GP1",
			"Samsung|Galaxy Player 5.8|harrisonkrktt|YP-GP1",
			"Samsung|Galaxy Player 5.8|harrisonkrlgt|YP-GP1",
			"Samsung|Galaxy Player 50|YP-G50|YP-G50",
			"Samsung|Galaxy Pocket|GT-S5300|GT-S5300",
			"Samsung|Galaxy Pocket|GT-S5300B|GT-S5300B",
			"Samsung|Galaxy Pocket|GT-S5300L|GT-S5300L",
			"Samsung|Galaxy Pocket|GT-S5302|GT-S5302",
			"Samsung|Galaxy Pocket|GT-S5302B|GT-S5302B",
			"Samsung|Galaxy Pocket|coriplus|GT-S5301",
			"Samsung|Galaxy Pocket|coriplusds|GT-S5303",
			"Samsung|Galaxy Pocket Neo|corsica|GT-S5312",
			"Samsung|Galaxy Pocket Neo|corsica|GT-S5312B",
			"Samsung|Galaxy Pocket Neo|corsica|GT-S5312L",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310B",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310E",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310G",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310L",
			"Samsung|Galaxy Pocket Neo|corsicass|GT-S5310T",
			"Samsung|Galaxy Pocket Neo|corsicave3g|GT-S5310I",
			"Samsung|Galaxy Pocket Neo|corsicave3g|GT-S5310N",
			"Samsung|Galaxy Pocket Neo|corsicaveds3gvj|GT-S5312C",
			"Samsung|Galaxy Pocket Neo|corsicaveds3gvj|GT-S5312M",
			"Samsung|Galaxy Pocket Neo|d2aio|SAMSUNG-SGH-I747Z",
			"Samsung|Galaxy Pocket Plus|coriplus|GT-S5301",
			"Samsung|Galaxy Pocket Plus|coriplus|GT-S5301B",
			"Samsung|Galaxy Pocket Plus|coriplus|GT-S5301L",
			"Samsung|Galaxy Pocket SS|corsicave3g|GT-S5310C",
			"Samsung|Galaxy Pocket SS|corsicave3g|GT-S5310M",
			"Samsung|Galaxy Pocket2|pocket23g|SM-G110B",
			"Samsung|Galaxy Pocket2|pocket23g|SM-G110M",
			"Samsung|Galaxy Pocket2|pocket2ss3g|SM-G110H",
			"Samsung|Galaxy Pop|superiorlteskt|SHV-E220S",
			"Samsung|Galaxy Pop (CDMA)|SCH-i559|SCH-i559",
			"Samsung|Galaxy Precedent|SCH-M828C|SCH-M828C",
			"Samsung|Galaxy Premier|superior|GT-I9260",
			"Samsung|Galaxy Premier|superiorcmcc|GT-I9268",
			"Samsung|Galaxy Prevail|SPH-M820|SPH-M820-BST",
			"Samsung|Galaxy Prevail2|raybst|SPH-M840",
			"Samsung|Galaxy Pro|GT-B7510|GT-B7510",
			"Samsung|Galaxy Pro|GT-B7510B|GT-B7510B",
			"Samsung|Galaxy Pro|GT-B7510L|GT-B7510L",
			"Samsung|Galaxy Proclaim|SCH-S720C|SCH-S720C",
			"Samsung|Galaxy Q|SGH-T589|SGH-T589",
			"Samsung|Galaxy Q|SGH-T589R|SGH-T589R",
			"Samsung|Galaxy Q|SGH-T589W|SGH-T589W",
			"Samsung|Galaxy R-Style|jaguark|SHV-E170K",
			"Samsung|Galaxy R-Style|jaguarl|SHV-E170L",
			"Samsung|Galaxy R-Style|jaguars|SHV-E170S",
			"Samsung|Galaxy Reverb|iconvmu|SPH-M950",
			"Samsung|Galaxy Round|flteskt|SM-G910S",
			"Samsung|Galaxy Rugby|comanchecan|SGH-I547C",
			"Samsung|Galaxy Rugby Pro|comancheatt|SAMSUNG-SGH-I547",
			"Samsung|Galaxy Rush|prevail2spr|SPH-M830",
			"Samsung|Galaxy S|GT-I9000|GT-I9000",
			"Samsung|Galaxy S|GT-I9000B|GT-I9000B",
			"Samsung|Galaxy S|GT-I9000M|GT-I9000M",
			"Samsung|Galaxy S|GT-I9000T|GT-I9000T",
			"Samsung|Galaxy S|GT-I9003|GT-I9003",
			"Samsung|Galaxy S|GT-I9003L|GT-I9003L",
			"Samsung|Galaxy S|GT-I9008L|GT-I9008L",
			"Samsung|Galaxy S|GT-I9010|GT-I9010",
			"Samsung|Galaxy S|GT-I9018|GT-I9018",
			"Samsung|Galaxy S|GT-I9050|GT-I9050",
			"Samsung|Galaxy S|SC-02B|SC-02B",
			"Samsung|Galaxy S|SCH-I500|SCH-I500",
			"Samsung|Galaxy S|SCH-S950C|SCH-S950C",
			"Samsung|Galaxy S|SCH-i909|SCH-i909",
			"Samsung|Galaxy S|SGH-I897|SAMSUNG-SGH-I897",
			"Samsung|Galaxy S|SGH-T959V|SGH-T959V",
			"Samsung|Galaxy S|SGH-T959W|SGH-T959W",
			"Samsung|Galaxy S|SHW-M110S|SHW-M110S",
			"Samsung|Galaxy S|SHW-M190S|SHW-M190S",
			"Samsung|Galaxy S|SPH-D700|SPH-D700",
			"Samsung|Galaxy S|loganlte|GT-S7275",
			"Samsung|Galaxy S Advance|GT-I9070|GT-I9070",
			"Samsung|Galaxy S Advance|GT-I9070P|GT-I9070P",
			"Samsung|Galaxy S Aviator|SCH-R930|SCH-R930",
			"Samsung|Galaxy S Blaze|SGH-T769|SGH-T769",
			"Samsung|Galaxy S BlazeQ|apexqtmo|SGH-T699",
			"Samsung|Galaxy S Captivate|SGH-I896|SAMSUNG-SGH-I896",
			"Samsung|Galaxy S Captivate|SGH-I896|SGH-I896",
			"Samsung|Galaxy S Continuum|SCH-I400|SCH-I400",
			"Samsung|Galaxy S DUOS|kyleopen|GT-S7562",
			"Samsung|Galaxy S DUOS|kyleopen|GT-S7562L",
			"Samsung|Galaxy S Duos|kylechn|GT-S7562",
			"Samsung|Galaxy S Duos|kyleopen|GT-S7562",
			"Samsung|Galaxy S Duos|kyletdcmcc|GT-S7568",
			"Samsung|Galaxy S Duos2|kyleprods|GT-S7582",
			"Samsung|Galaxy S Duos2|kyleprods|GT-S7582L",
			"Samsung|Galaxy S Duos3|vivalto3gvn|SM-G313HZ",
			"Samsung|Galaxy S Epic|SPH-D700|SPH-D700",
			"Samsung|Galaxy S Fascinate|SGH-T959P|SGH-T959P",
			"Samsung|Galaxy S Glide|SGH-I927R|SAMSUNG-SGH-I927R",
			"Samsung|Galaxy S Lightray|SCH-R940|SCH-R940",
			"Samsung|Galaxy S Plus|GT-I9001|GT-I9001",
			"Samsung|Galaxy S Stratosphere|SCH-I405|SCH-I405",
			"Samsung|Galaxy S Vibrant|SGH-T959|SGH-T959",
			"Samsung|Galaxy S Vibrant|SGH-T959D|SGH-T959D",
			"Samsung|Galaxy S duos|kyleichn|GT-S7566",
			"Samsung|Galaxy S2|GT-I9100|GT-I9100",
			"Samsung|Galaxy S2|GT-I9100M|GT-I9100M",
			"Samsung|Galaxy S2|GT-I9100P|GT-I9100P",
			"Samsung|Galaxy S2|GT-I9100T|GT-I9100T",
			"Samsung|Galaxy S2|GT-I9103|GT-I9103",
			"Samsung|Galaxy S2|GT-I9108|GT-I9108",
			"Samsung|Galaxy S2|GT-I9210T|GT-I9210T",
			"Samsung|Galaxy S2|SC-02C|SC-02C",
			"Samsung|Galaxy S2|SCH-R760X|SCH-R760X",
			"Samsung|Galaxy S2|SGH-I777|SAMSUNG-SGH-I777",
			"Samsung|Galaxy S2|SGH-S959G|SGH-S959G",
			"Samsung|Galaxy S2|SGH-T989|SGH-T989",
			"Samsung|Galaxy S2|SHV-E110S|SHV-E110S",
			"Samsung|Galaxy S2|SHW-M250K|SHW-M250K",
			"Samsung|Galaxy S2|SHW-M250L|SHW-M250L",
			"Samsung|Galaxy S2|SHW-M250S|SHW-M250S",
			"Samsung|Galaxy S2|t1cmcc|GT-I9108",
			"Samsung|Galaxy S2 Duos|SCH-i929|SCH-i929",
			"Samsung|Galaxy S2 Duos TV|logandsdtv|GT-S7273T",
			"Samsung|Galaxy S2 Epic|SCH-R760|SCH-R760",
			"Samsung|Galaxy S2 Epic|SPH-D710|SPH-D710",
			"Samsung|Galaxy S2 Epic|SPH-D710BST|SPH-D710BST",
			"Samsung|Galaxy S2 Epic|SPH-D710VMUB|SPH-D710VMUB",
			"Samsung|Galaxy S2 HD LTE|SGH-I757M|SGH-I757M",
			"Samsung|Galaxy S2 HD LTE|SHV-E120K|SHV-E120K",
			"Samsung|Galaxy S2 HD LTE|SHV-E120L|SHV-E120L",
			"Samsung|Galaxy S2 HD LTE|SHV-E120S|SHV-E120S",
			"Samsung|Galaxy S2 LTE|GT-I9210|GT-I9210",
			"Samsung|Galaxy S2 LTE|SC-03D|SC-03D",
			"Samsung|Galaxy S2 LTE|SGH-I727R|SGH-I727R",
			"Samsung|Galaxy S2 Plus|GT-I9100G|GT-I9100G",
			"Samsung|Galaxy S2 Plus|s2ve|GT-I9105",
			"Samsung|Galaxy S2 Plus|s2vep|GT-I9105P",
			"Samsung|Galaxy S2 Skyrocket|SGH-I727|SAMSUNG-SGH-I727",
			"Samsung|Galaxy S2 Skyrocket|SGH-I727|SGH-I727",
			"Samsung|Galaxy S2 Wimax|ISW11SC|ISW11SC",
			"Samsung|Galaxy S2 X|SGH-T989D|SGH-T989D",
			"Samsung|Galaxy S3|SC-03E|SC-03E",
			"Samsung|Galaxy S3|c1att|SGH-I748",
			"Samsung|Galaxy S3|c1ktt|SHV-E210K",
			"Samsung|Galaxy S3|c1lgt|SHV-E210L",
			"Samsung|Galaxy S3|c1skt|SHV-E210S",
			"Samsung|Galaxy S3|d2att|SAMSUNG-SGH-I747",
			"Samsung|Galaxy S3|d2can|SGH-I747M",
			"Samsung|Galaxy S3|d2can|SGH-T999V",
			"Samsung|Galaxy S3|d2cri|SCH-R530C",
			"Samsung|Galaxy S3|d2dcm|Gravity",
			"Samsung|Galaxy S3|d2dcm|SC-06D",
			"Samsung|Galaxy S3|d2lteMetroPCS|SGH-T999N",
			"Samsung|Galaxy S3|d2lterefreshspr|SPH-L710T",
			"Samsung|Galaxy S3|d2ltetmo|SGH-T999L",
			"Samsung|Galaxy S3|d2mtr|SCH-R530M",
			"Samsung|Galaxy S3|d2spi|SCH-L710",
			"Samsung|Galaxy S3|d2spr|SPH-L710",
			"Samsung|Galaxy S3|d2tfnspr|SCH-S960L",
			"Samsung|Galaxy S3|d2tfnvzw|SCH-S968C",
			"Samsung|Galaxy S3|d2tmo|SGH-T999",
			"Samsung|Galaxy S3|d2usc|SCH-R530U",
			"Samsung|Galaxy S3|d2vmu|SPH-L710",
			"Samsung|Galaxy S3|d2vzw|SCH-I535",
			"Samsung|Galaxy S3|d2vzw|SCH-I535PP",
			"Samsung|Galaxy S3|d2xar|SCH-R530X",
			"Samsung|Galaxy S3|m0|GT-I9300",
			"Samsung|Galaxy S3|m0|GT-I9300T",
			"Samsung|Galaxy S3|m0apt|SCH-I939",
			"Samsung|Galaxy S3|m0chn|GT-I9300",
			"Samsung|Galaxy S3|m0cmcc|GT-I9308",
			"Samsung|Galaxy S3|m0ctc|SCH-I939",
			"Samsung|Galaxy S3|m0ctcduos|SCH-I939D",
			"Samsung|Galaxy S3|m0skt|SHW-M440S",
			"Samsung|Galaxy S3|m3|GT-I9305",
			"Samsung|Galaxy S3|m3|GT-I9305N",
			"Samsung|Galaxy S3|m3|GT-I9305T",
			"Samsung|Galaxy S3|m3dcm|GravityQuad",
			"Samsung|Galaxy S3|m3dcm|SC-03E",
			"Samsung|Galaxy S3 Duos|arubaslim|GT-I8262B",
			"Samsung|Galaxy S3 Mini|golden|GT-I8190",
			"Samsung|Galaxy S3 Mini|golden|GT-I8190L",
			"Samsung|Galaxy S3 Mini|golden|GT-I8190N",
			"Samsung|Galaxy S3 Mini|golden|GT-I8190T",
			"Samsung|Galaxy S3 Mini|goldenlteatt|SAMSUNG-SM-G730A",
			"Samsung|Galaxy S3 Mini|goldenltebmc|SM-G730W8",
			"Samsung|Galaxy S3 Mini|goldenltevzw|SM-G730V",
			"Samsung|Galaxy S3 Mini|goldenve3g|GT-I8200L",
			"Samsung|Galaxy S3 Mini Value Edition|goldenve3g|GT-I8200N",
			"Samsung|Galaxy S3 Mini Value Edition|goldenvess3g|GT-I8200",
			"Samsung|Galaxy S3 Mini Value Edition|goldenvess3g|GT-I8200Q",
			"Samsung|Galaxy S3 Neo|s3ve3g|GT-I9300I",
			"Samsung|Galaxy S3 Neo|s3ve3g|GT-I9301I",
			"Samsung|Galaxy S3 Neo|s3ve3g|GT-I9301Q",
			"Samsung|Galaxy S3 Neo|s3ve3gdd|GT-I9300I",
			"Samsung|Galaxy S3 Neo|s3ve3gds|GT-I9300I",
			"Samsung|Galaxy S3 Neo|s3ve3gdsdd|GT-I9300I",
			"Samsung|Galaxy S3 Neo Plus|s3ve3g|GT-I9300I",
			"Samsung|Galaxy S3 Neo Plus|s3ve3g|GT-I9308I",
			"Samsung|Galaxy S3 Progre|SCL21|SCL21",
			"Samsung|Galaxy S3 Slim|wilcoxds|SM-G3812B",
			"Samsung|Galaxy S4|SC-04E|SC-04E",
			"Samsung|Galaxy S4|ja3g|GT-I9500",
			"Samsung|Galaxy S4|ja3gduosctc|SCH-I959",
			"Samsung|Galaxy S4|jaltektt|SHV-E300K",
			"Samsung|Galaxy S4|jaltelgt|SHV-E300L",
			"Samsung|Galaxy S4|jalteskt|SHV-E300S",
			"Samsung|Galaxy S4|jflte|GT-I9505",
			"Samsung|Galaxy S4|jflte|GT-I9508",
			"Samsung|Galaxy S4|jflte|GT-I9508C",
			"Samsung|Galaxy S4|jflteaio|SAMSUNG-SGH-I337Z",
			"Samsung|Galaxy S4|jflteatt|SAMSUNG-SGH-I337",
			"Samsung|Galaxy S4|jfltecan|SGH-I337M",
			"Samsung|Galaxy S4|jfltecan|SGH-M919V",
			"Samsung|Galaxy S4|jfltecri|SCH-R970C",
			"Samsung|Galaxy S4|jfltecsp|SCH-R970X",
			"Samsung|Galaxy S4|jfltelra|SCH-I545L",
			"Samsung|Galaxy S4|jflterefreshspr|SPH-L720T",
			"Samsung|Galaxy S4|jfltespr|SPH-L720",
			"Samsung|Galaxy S4|jfltetfnatt|SM-S975L",
			"Samsung|Galaxy S4|jfltetfntmo|SGH-S970G",
			"Samsung|Galaxy S4|jfltetmo|SGH-M919",
			"Samsung|Galaxy S4|jflteusc|SCH-R970",
			"Samsung|Galaxy S4|jfltevzw|SCH-I545",
			"Samsung|Galaxy S4|jftdd|GT-I9507",
			"Samsung|Galaxy S4|jftdd|GT-I9507V",
			"Samsung|Galaxy S4|jfvelte|GT-I9515",
			"Samsung|Galaxy S4|jfvelte|GT-I9515L",
			"Samsung|Galaxy S4|jfwifi|GT-I9505X",
			"Samsung|Galaxy S4|ks01lte|GT-I9506",
			"Samsung|Galaxy S4|ks01ltektt|SHV-E330K",
			"Samsung|Galaxy S4|ks01ltelgt|SHV-E330L",
			"Samsung|Galaxy S4 Active|jactivelte|GT-I9295",
			"Samsung|Galaxy S4 Active|jactivelteatt|SAMSUNG-SGH-I537",
			"Samsung|Galaxy S4 Active|jactivelteatt|SGH-I537",
			"Samsung|Galaxy S4 Active|jactivelteskt|SHV-E470S",
			"Samsung|Galaxy S4 Duos|ja3gchnduos|GT-I9502",
			"Samsung|Galaxy S4 Google Play Edition|jgedlte|GT-I9505G",
			"Samsung|Galaxy S4 LTE-A|ks01lteskt|SHV-E330S",
			"Samsung|Galaxy S4 Mini|serrano3g|GT-I9190",
			"Samsung|Galaxy S4 Mini|serranods|GT-I9192",
			"Samsung|Galaxy S4 Mini|serranolte|GT-I9195",
			"Samsung|Galaxy S4 Mini|serranolte|GT-I9195L",
			"Samsung|Galaxy S4 Mini|serranolte|GT-I9195T",
			"Samsung|Galaxy S4 Mini|serranolte|GT-I9195X",
			"Samsung|Galaxy S4 Mini|serranolte|GT-I9197",
			"Samsung|Galaxy S4 Mini|serranoltebmc|SGH-I257M",
			"Samsung|Galaxy S4 Mini|serranoltektt|SHV-E370K",
			"Samsung|Galaxy S4 Mini|serranoltekx|SHV-E370D",
			"Samsung|Galaxy S4 Mini|serranoltelra|SCH-I435L",
			"Samsung|Galaxy S4 Mini|serranoltespr|SPH-L520",
			"Samsung|Galaxy S4 Mini|serranolteusc|SCH-R890",
			"Samsung|Galaxy S4 Mini|serranoltevzw|SCH-I435",
			"Samsung|Galaxy S4 Mini|serranove3g|GT-I9192I",
			"Samsung|Galaxy S4 Mini|serranovelte|GT-I9195I",
			"Samsung|Galaxy S4 Mini|serranovolteatt|SAMSUNG-SGH-I257",
			"Samsung|Galaxy S4 Zoom|mproject3g|SM-C101",
			"Samsung|Galaxy S4 Zoom|mprojectlteatt|SAMSUNG-SM-C105A",
			"Samsung|Galaxy S4 Zoom|mprojectltelgt|SM-C105L",
			"Samsung|Galaxy S4 Zoom|mprojectlteskt|SM-C105S",
			"Samsung|Galaxy S4 Zoom|mprojectqlte|SM-C105",
			"Samsung|Galaxy S5|SCL23|SCL23",
			"Samsung|Galaxy S5|k3g|SM-G900H",
			"Samsung|Galaxy S5|klte|SM-G9008W",
			"Samsung|Galaxy S5|klte|SM-G9009W",
			"Samsung|Galaxy S5|klte|SM-G900F",
			"Samsung|Galaxy S5|klte|SM-G900FQ",
			"Samsung|Galaxy S5|klte|SM-G900I",
			"Samsung|Galaxy S5|klte|SM-G900M",
			"Samsung|Galaxy S5|klte|SM-G900MD",
			"Samsung|Galaxy S5|klteMetroPCS|SM-G900T1",
			"Samsung|Galaxy S5|klteacg|SM-G900R7",
			"Samsung|Galaxy S5|klteaio|SAMSUNG-SM-G900AZ",
			"Samsung|Galaxy S5|klteatt|SAMSUNG-SM-G900A",
			"Samsung|Galaxy S5|kltecan|SM-G900W8",
			"Samsung|Galaxy S5|klteduoszn|SM-G9006W",
			"Samsung|Galaxy S5|kltektt|SM-G900K",
			"Samsung|Galaxy S5|kltelgt|SM-G900L",
			"Samsung|Galaxy S5|kltelra|SM-G900R6",
			"Samsung|Galaxy S5|klteskt|SM-G900S",
			"Samsung|Galaxy S5|kltespr|SM-G900P",
			"Samsung|Galaxy S5|kltetmo|SM-G900T",
			"Samsung|Galaxy S5|klteusc|SM-G900R4",
			"Samsung|Galaxy S5|kltevzw|SM-G900V",
			"Samsung|Galaxy S5|kwifi|SM-G900X",
			"Samsung|Galaxy S5|lentisltektt|SM-G906K",
			"Samsung|Galaxy S5|lentisltelgt|SM-G906L",
			"Samsung|Galaxy S5|lentislteskt|SM-G906S",
			"Samsung|Galaxy S5 Active|klteactive|SM-G870F",
			"Samsung|Galaxy S5 Active|klteattactive|SAMSUNG-SM-G870A",
			"Samsung|Galaxy S5 Active|kltecanactive|SM-G870W",
			"Samsung|Galaxy S5 Dual SIM|klte|SM-G900FD",
			"Samsung|Galaxy S5 Google Play Edition|kgedlte|SM-G900FG",
			"Samsung|Galaxy S5 K Sport|kltesprsports|SM-G860P",
			"Samsung|Galaxy S5 LTE-A|kccat6|SM-G901F",
			"Samsung|Galaxy S5 Mini|kmini3g|SM-G800H",
			"Samsung|Galaxy S5 Mini|kminilte|SM-G800F",
			"Samsung|Galaxy S5 Mini|kminilte|SM-G800Y",
			"Samsung|Galaxy S5 Mini|kminilteusc|SM-G800R4",
			"Samsung|Galaxy S5 mini|kmini3g|SM-G800HQ",
			"Samsung|Galaxy S5 mini|kminilte|SM-G800M",
			"Samsung|Galaxy S5 mini|kminilteatt|SAMSUNG-SM-G800A",
			"Samsung|Galaxy S5 mini|kminiwifi|SM-G800X",
			"Samsung|Galaxy S5 mini LTE|kminilte|SM-G800Y",
			"Samsung|Galaxy Spica|GT-I5700|GT-I5700",
			"Samsung|Galaxy Spica|GT-I5700|GT-I5700L",
			"Samsung|Galaxy Spica|GT-I5700|GT-I5700R",
			"Samsung|Galaxy Spica|GT-I5700L|GT-I5700L",
			"Samsung|Galaxy Spica|spica|GT-I5700",
			"Samsung|Galaxy Spica|spica|GT-I5700L",
			"Samsung|Galaxy Spica|spica|GT-I5700R",
			"Samsung|Galaxy Spica|spica|GT-i5700",
			"Samsung|Galaxy Star|mint|GT-S5282",
			"Samsung|Galaxy Star|mintss|GT-S5280",
			"Samsung|Galaxy Star Plus|logan2g|GT-S7262",
			"Samsung|Galaxy Star Trios|mintts|GT-S5283B",
			"Samsung|Galaxy Star2 Plus|higgs2g|SM-G350E",
			"Samsung|Galaxy Stellar|jaspervzw|SCH-I200",
			"Samsung|Galaxy Stellar|jaspervzw|SCH-I200PP",
			"Samsung|Galaxy Style Duos|aruba3gduosctc|SCH-I829",
			"Samsung|Galaxy Tab|GT-P1000|GT-P1000",
			"Samsung|Galaxy Tab|GT-P1000L|GT-P1000L",
			"Samsung|Galaxy Tab|GT-P1000M|GT-P1000M",
			"Samsung|Galaxy Tab|GT-P1000N|GT-P1000N",
			"Samsung|Galaxy Tab|GT-P1000R|GT-P1000R",
			"Samsung|Galaxy Tab|GT-P1000T|GT-P1000T",
			"Samsung|Galaxy Tab|GT-P1010|GT-P1010",
			"Samsung|Galaxy Tab|GT-P1013|GT-P1013",
			"Samsung|Galaxy Tab|SC-01C|SC-01C",
			"Samsung|Galaxy Tab|SCH-I800|SCH-I800",
			"Samsung|Galaxy Tab|SGH-T849|SGH-T849",
			"Samsung|Galaxy Tab|SHW-M180K|SHW-M180K",
			"Samsung|Galaxy Tab|SHW-M180L|SHW-M180L",
			"Samsung|Galaxy Tab|SHW-M180S|SHW-M180S",
			"Samsung|Galaxy Tab|SHW-M180W|SHW-M180W",
			"Samsung|Galaxy Tab|SMT-i9100|SMT-i9100",
			"Samsung|Galaxy Tab 10.1|GT-P7500|GT-P7500",
			"Samsung|Galaxy Tab 10.1|GT-P7500D|GT-P7500D",
			"Samsung|Galaxy Tab 10.1|GT-P7503|GT-P7503",
			"Samsung|Galaxy Tab 10.1|GT-P7510|GT-P7510",
			"Samsung|Galaxy Tab 10.1|SC-01D|SC-01D",
			"Samsung|Galaxy Tab 10.1|SCH-I905|SCH-I905",
			"Samsung|Galaxy Tab 10.1|SGH-T859|SGH-T859",
			"Samsung|Galaxy Tab 10.1|SHW-M300W|SHW-M300W",
			"Samsung|Galaxy Tab 10.1|SHW-M380K|SHW-M380K",
			"Samsung|Galaxy Tab 10.1|SHW-M380S|SHW-M380S",
			"Samsung|Galaxy Tab 10.1|SHW-M380W|SHW-M380W",
			"Samsung|Galaxy Tab 10.1 N|GT-P7501|GT-P7501",
			"Samsung|Galaxy Tab 10.1 N|GT-P7511|GT-P7511",
			"Samsung|Galaxy Tab 10.1 v|p3|GT-P7100",
			"Samsung|Galaxy Tab 4 8.0|milletwifi|SM-T330X",
			"Samsung|Galaxy Tab 4 8.0|milletwifikx|SM-T330",
			"Samsung|Galaxy Tab 4 Active|rubenslte|SM-T365",
			"Samsung|Galaxy Tab 7.0|SPH-P100|SPH-P100",
			"Samsung|Galaxy Tab 7.0 Plus|GT-P6200|GT-P6200",
			"Samsung|Galaxy Tab 7.0 Plus|GT-P6200L|GT-P6200L",
			"Samsung|Galaxy Tab 7.0 Plus|GT-P6201|GT-P6201",
			"Samsung|Galaxy Tab 7.0 Plus|GT-P6210|GT-P6210",
			"Samsung|Galaxy Tab 7.0 Plus|GT-P6211|GT-P6211",
			"Samsung|Galaxy Tab 7.0 Plus|SC-02D|SC-02D",
			"Samsung|Galaxy Tab 7.0 Plus|SGH-T869|SGH-T869",
			"Samsung|Galaxy Tab 7.0 Plus|SHW-M430W|SHW-M430W",
			"Samsung|Galaxy Tab 7.7|GT-P6800|GT-P6800",
			"Samsung|Galaxy Tab 7.7|GT-P6810|GT-P6810",
			"Samsung|Galaxy Tab 7.7|SCH-I815|SCH-I815",
			"Samsung|Galaxy Tab 7.7 Plus|SC-01E|SC-01E",
			"Samsung|Galaxy Tab 8.9|GT-P7300|GT-P7300",
			"Samsung|Galaxy Tab 8.9|GT-P7310|GT-P7310",
			"Samsung|Galaxy Tab 8.9|GT-P7320|GT-P7320",
			"Samsung|Galaxy Tab 8.9|SCH-P739|SCH-P739",
			"Samsung|Galaxy Tab 8.9|SGH-I957|SAMSUNG-SGH-I957",
			"Samsung|Galaxy Tab 8.9|SGH-I957D|SAMSUNG-SGH-I957D",
			"Samsung|Galaxy Tab 8.9|SGH-I957D|SGH-I957D",
			"Samsung|Galaxy Tab 8.9|SGH-I957M|SAMSUNG-SGH-I957M",
			"Samsung|Galaxy Tab 8.9|SGH-I957M|SGH-I957M",
			"Samsung|Galaxy Tab 8.9|SGH-I957R|SAMSUNG-SGH-I957R",
			"Samsung|Galaxy Tab 8.9|SGH-I957R|SGH-I957R",
			"Samsung|Galaxy Tab 8.9|SHV-E140K|SHV-E140K",
			"Samsung|Galaxy Tab 8.9|SHV-E140L|SHV-E140L",
			"Samsung|Galaxy Tab 8.9|SHV-E140S|SHV-E140S",
			"Samsung|Galaxy Tab 8.9|SHW-M305W|SHW-M305W",
			"Samsung|Galaxy Tab Pro 10.1|picassolte|SM-T525",
			"Samsung|Galaxy Tab Pro 10.1|picassowifi|SM-T520",
			"Samsung|Galaxy Tab Pro 12.2|v2lte|SM-T905",
			"Samsung|Galaxy Tab Pro 12.2|v2wifi|SM-T900",
			"Samsung|Galaxy Tab Pro 12.2|v2wifi|SM-T900X",
			"Samsung|Galaxy Tab Pro 8.4|mondrianlte|SM-T325",
			"Samsung|Galaxy Tab Pro 8.4|mondrianwifi|SM-T320",
			"Samsung|Galaxy Tab Pro 8.4|mondrianwifi|SM-T320X",
			"Samsung|Galaxy Tab Pro 8.4|mondrianwifikx|SM-T320",
			"Samsung|Galaxy Tab Pro 8.4|mondrianwifiue|SM-T320",
			"Samsung|Galaxy Tab Q|q7|SM-T2519",
			"Samsung|Galaxy Tab S 10.5|chagallwifi|SM-T800",
			"Samsung|Galaxy Tab S 8.4|SC-03G|SC-03G",
			"Samsung|Galaxy Tab S 8.4|klimtlte|SM-T705",
			"Samsung|Galaxy Tab S 8.4|klimtlte|SM-T705C",
			"Samsung|Galaxy Tab S 8.4|klimtlte|SM-T705Y",
			"Samsung|Galaxy Tab S 8.4|klimtltevzw|SM-T707V",
			"Samsung|Galaxy Tab S 8.4|klimtwifi|SM-T700",
			"Samsung|Galaxy Tab2 10.1|espresso10att|SAMSUNG-SGH-I497",
			"Samsung|Galaxy Tab2 10.1|espresso10can|SGH-I497",
			"Samsung|Galaxy Tab2 10.1|espresso10rf|GT-P5100",
			"Samsung|Galaxy Tab2 10.1|espresso10spr|SPH-P500",
			"Samsung|Galaxy Tab2 10.1|espresso10tmo|SGH-T779",
			"Samsung|Galaxy Tab2 10.1|espresso10vzw|SCH-I915",
			"Samsung|Galaxy Tab2 10.1|espresso10wifi|GT-P5110",
			"Samsung|Galaxy Tab2 10.1|espresso10wifi|GT-P5113",
			"Samsung|Galaxy Tab2 7.0|espressorf|GT-P3100",
			"Samsung|Galaxy Tab2 7.0|espressorf|GT-P3100B",
			"Samsung|Galaxy Tab2 7.0|espressorf|GT-P3105",
			"Samsung|Galaxy Tab2 7.0|espressovzw|SCH-I705",
			"Samsung|Galaxy Tab2 7.0|espressovzw|SCH-i705",
			"Samsung|Galaxy Tab2 7.0|espressowifi|GT-P3110",
			"Samsung|Galaxy Tab2 7.0|espressowifi|GT-P3113",
			"Samsung|Galaxy Tab3|lt01wifikx|SM-T310",
			"Samsung|Galaxy Tab3 10.1|santos103g|GT-P5200",
			"Samsung|Galaxy Tab3 10.1|santos10lte|GT-P5220",
			"Samsung|Galaxy Tab3 10.1|santos10wifi|GT-P5210",
			"Samsung|Galaxy Tab3 10.1|santos10wifi|GT-P5210XD1",
			"Samsung|Galaxy Tab3 7.0|lt023g|SM-T211",
			"Samsung|Galaxy Tab3 7.0|lt023g|SM-T212",
			"Samsung|Galaxy Tab3 7.0|lt023gdtv|SM-T211M",
			"Samsung|Galaxy Tab3 7.0|lt02lte|SM-T215",
			"Samsung|Galaxy Tab3 7.0|lt02lteatt|SAMSUNG-SM-T217A",
			"Samsung|Galaxy Tab3 7.0|lt02ltespr|SM-T217S",
			"Samsung|Galaxy Tab3 7.0|lt02ltetmo|SM-T217T",
			"Samsung|Galaxy Tab3 7.0|lt02wifi|SM-T210",
			"Samsung|Galaxy Tab3 7.0|lt02wifi|SM-T210R",
			"Samsung|Galaxy Tab3 7.0|lt02wifilgt|SM-T210L",
			"Samsung|Galaxy Tab3 8.0|lt013g|SM-T311",
			"Samsung|Galaxy Tab3 8.0|lt013g|SM-T312",
			"Samsung|Galaxy Tab3 8.0|lt01lte|SM-T315",
			"Samsung|Galaxy Tab3 8.0|lt01lte|SM-T315T",
			"Samsung|Galaxy Tab3 8.0|lt01wifi|SM-T310",
			"Samsung|Galaxy Tab3 8.0|lt01wifi|SM-T310X",
			"Samsung|Galaxy Tab3 8.0|lt02lduwifi|SM-T210X",
			"Samsung|Galaxy Tab3 Kids|lt02kidswifi|SM-T2105",
			"Samsung|Galaxy Tab3 Lite|goya3g|SM-T111",
			"Samsung|Galaxy Tab3 Lite|goya3g|SM-T111M",
			"Samsung|Galaxy Tab3 Lite|goyawifi|SM-T110",
			"Samsung|Galaxy Tab4|matissewifigoogle|SM-T530NN",
			"Samsung|Galaxy Tab4 10.0|matisse3g|SM-T531",
			"Samsung|Galaxy Tab4 10.0|matisselte|SM-T535",
			"Samsung|Galaxy Tab4 10.0|matisselteatt|SAMSUNG-SM-T537A",
			"Samsung|Galaxy Tab4 10.0|matisselteusc|SM-T537R4",
			"Samsung|Galaxy Tab4 10.0|matisseltevzw|SM-T537V",
			"Samsung|Galaxy Tab4 10.1|matissewifi|SM-T530",
			"Samsung|Galaxy Tab4 10.1|matissewifi|SM-T530X",
			"Samsung|Galaxy Tab4 10.1|matissewifikx|SM-T530",
			"Samsung|Galaxy Tab4 10.1|matissewifiue|SM-T530NU",
			"Samsung|Galaxy Tab4 7.0|403SC|403SC",
			"Samsung|Galaxy Tab4 7.0|degas3g|SM-T231",
			"Samsung|Galaxy Tab4 7.0|degas3g|SM-T232",
			"Samsung|Galaxy Tab4 7.0|degaslte|SM-T235",
			"Samsung|Galaxy Tab4 7.0|degaslte|SM-T235Y",
			"Samsung|Galaxy Tab4 7.0|degasltespr|SM-T237P",
			"Samsung|Galaxy Tab4 7.0|degasltevzw|SM-T237V",
			"Samsung|Galaxy Tab4 7.0|degaswifi|SM-T230",
			"Samsung|Galaxy Tab4 7.0|degaswifi|SM-T230X",
			"Samsung|Galaxy Tab4 7.0|degaswifidtv|SM-T230NT",
			"Samsung|Galaxy Tab4 7.0|degaswifiopenbnn|SM-T230NU",
			"Samsung|Galaxy Tab4 7.0|degaswifiue|SM-T230NU",
			"Samsung|Galaxy Tab4 8.0|millet3g|SM-T331",
			"Samsung|Galaxy Tab4 8.0|milletlte|SM-T335",
			"Samsung|Galaxy Tab4 8.0|milletlteatt|SAMSUNG-SM-T337A",
			"Samsung|Galaxy Tab4 8.0|milletltektt|SM-T335K",
			"Samsung|Galaxy Tab4 8.0|milletltelgt|SM-T335L",
			"Samsung|Galaxy Tab4 8.0|milletltetmo|SM-T337T",
			"Samsung|Galaxy Tab4 8.0|milletltevzw|SM-T337V",
			"Samsung|Galaxy Tab4 8.0|milletwifi|SM-T330",
			"Samsung|Galaxy Tab4 8.0|milletwifiue|SM-T330NU",
			"Samsung|Galaxy Tab4 Active|rubenslte|SM-T365M",
			"Samsung|Galaxy Tab4 Active|rubenswifi|SM-T360",
			"Samsung|Galaxy Tab4 Nook 10.1|matissewifiopenbnn|SM-T530NU",
			"Samsung|Galaxy TabS 10.5|SCT21|SCT21",
			"Samsung|Galaxy TabS 10.5|chagallhltektt|SM-T805K",
			"Samsung|Galaxy TabS 10.5|chagallhltelgt|SM-T805L",
			"Samsung|Galaxy TabS 10.5|chagallhlteskt|SM-T805S",
			"Samsung|Galaxy TabS 10.5|chagalllte|SM-T805",
			"Samsung|Galaxy TabS 10.5|chagalllte|SM-T805M",
			"Samsung|Galaxy TabS 10.5|chagalllte|SM-T805Y",
			"Samsung|Galaxy TabS 10.5|chagalllteatt|SAMSUNG-SM-T807A",
			"Samsung|Galaxy TabS 10.5|chagallltecan|SM-T805W",
			"Samsung|Galaxy TabS 10.5|chagallltespr|SM-T807P",
			"Samsung|Galaxy TabS 10.5|chagallltetmo|SM-T807T",
			"Samsung|Galaxy TabS 10.5|chagalllteusc|SM-T807R4",
			"Samsung|Galaxy TabS 10.5|chagallltevzw|SM-T807V",
			"Samsung|Galaxy TabS 10.5|chagallwifi|SM-T800",
			"Samsung|Galaxy TabS 10.5|chagallwifi|SM-T800X",
			"Samsung|Galaxy TabS 10.5|chagallwifikx|SM-T800",
			"Samsung|Galaxy TabS 8.4|klimtlte|SM-T705",
			"Samsung|Galaxy TabS 8.4|klimtlte|SM-T705M",
			"Samsung|Galaxy TabS 8.4|klimtlteatt|SAMSUNG-SM-T707A",
			"Samsung|Galaxy TabS 8.4|klimtltecan|SM-T705W",
			"Samsung|Galaxy TabS 8.4|klimtwifi|SM-T700",
			"Samsung|Galaxy TabS 8.4|klimtwifikx|SM-T700",
			"Samsung|Galaxy Tap Pro 10.1|picassolte|SM-T525",
			"Samsung|Galaxy Tap Pro 8.4|mondrianwifi|SM-T320",
			"Samsung|Galaxy Trend|kyleve|GT-S7392",
			"Samsung|Galaxy Trend|kyleve|GT-S7392L",
			"Samsung|Galaxy Trend|kyleve3gcmcc|GT-S7568I",
			"Samsung|Galaxy Trend Duos|kyleichn|GT-S7562i",
			"Samsung|Galaxy Trend Duos|kylepluschn|GT-S7572",
			"Samsung|Galaxy Trend Duos|kyleve|GT-S7390",
			"Samsung|Galaxy Trend Duos|kyleve|GT-S7392",
			"Samsung|Galaxy Trend Duos|kyleve|GT-S7562C",
			"Samsung|Galaxy Trend Lite|kylevess|GT-S7390",
			"Samsung|Galaxy Trend Lite|kylevess|GT-S7390L",
			"Samsung|Galaxy Trend Plus|kylepro|GT-S7580",
			"Samsung|Galaxy Trend Plus|kylepro|GT-S7580E",
			"Samsung|Galaxy Trend Plus|kylepro|GT-S7580L",
			"Samsung|Galaxy Trend2|garda3gcmcc|GT-S7898",
			"Samsung|Galaxy Trend2|gardave3gcmcc|GT-S7898I",
			"Samsung|Galaxy Trend2|kyleplusctc|SCH-I739",
			"Samsung|Galaxy Trend3|cs023g|SM-G3502U",
			"Samsung|Galaxy Trend3|cs02cmcc|SM-G3508",
			"Samsung|Galaxy Trend3|cs02ctc|SM-G3509",
			"Samsung|Galaxy Trend3|cs02ve|SM-G3508I",
			"Samsung|Galaxy Trend3|cs02ve3g|SM-G3502C",
			"Samsung|Galaxy Trend3|cs02ve3g|SM-G3502I",
			"Samsung|Galaxy Trend3|cs02ve3g|SM-G3508J",
			"Samsung|Galaxy Trend3|cs02ve3g|SM-G3509I",
			"Samsung|Galaxy U|SHW-M130L|SHW-M130L",
			"Samsung|Galaxy Victory|goghspr|SPH-L300",
			"Samsung|Galaxy Victory|goghvmu|SPH-L300",
			"Samsung|Galaxy W|GT-I8150|GT-I8150",
			"Samsung|Galaxy W|GT-I8150B|GT-I8150B",
			"Samsung|Galaxy W|GT-I8150T|GT-I8150T",
			"Samsung|Galaxy W|SGH-T679M|SGH-T679M",
			"Samsung|Galaxy Win|delos3gcmcc|GT-I8558",
			"Samsung|Galaxy Win|delos3gduosctc|SCH-I869",
			"Samsung|Galaxy Win|delos3geur|GT-I8552",
			"Samsung|Galaxy Win|delos3geur|GT-I8552B",
			"Samsung|Galaxy Win|delos3gss|GT-I8550E",
			"Samsung|Galaxy Win|delosltelgt|SHV-E500L",
			"Samsung|Galaxy Win|deloslteskt|SHV-E500S",
			"Samsung|Galaxy Win Duos|delos3gchn|GT-I8552",
			"Samsung|Galaxy Win Pro|wilcox3g|SM-G3818",
			"Samsung|Galaxy Win Pro|wilcoxctc|SM-G3819",
			"Samsung|Galaxy Win Pro|wilcoxctc|SM-G3819D",
			"Samsung|Galaxy Win Pro|wilcoxds|SM-G3812",
			"Samsung|Galaxy Win2|coreprimeltedtv|SM-G360BT",
			"Samsung|Galaxy Xcover|GT-S5690|GT-S5690",
			"Samsung|Galaxy Xcover|GT-S5690L|GT-S5690L",
			"Samsung|Galaxy Xcover|GT-S5690M|GT-S5690M",
			"Samsung|Galaxy Xcover|GT-S5690R|GT-S5690R",
			"Samsung|Galaxy Xcover2|skomer|GT-S7710",
			"Samsung|Galaxy Xcover2|skomer|GT-S7710L",
			"Samsung|Galaxy Y|GT-S5360|GT-S5360",
			"Samsung|Galaxy Y|GT-S5360B|GT-S5360B",
			"Samsung|Galaxy Y|GT-S5360L|GT-S5360L",
			"Samsung|Galaxy Y|GT-S5360T|GT-S5360T",
			"Samsung|Galaxy Y|GT-S5363|GT-S5363",
			"Samsung|Galaxy Y|GT-S5368|GT-S5368",
			"Samsung|Galaxy Y|GT-S5369|GT-S5369",
			"Samsung|Galaxy Y|SCH-I509|SCH-I509",
			"Samsung|Galaxy Y|SCH-i509|SCH-i509",
			"Samsung|Galaxy Y Duos|GT-S6102|GT-S6102",
			"Samsung|Galaxy Y Duos|GT-S6102B|GT-S6102B",
			"Samsung|Galaxy Y Duos|GT-S6102E|GT-S6102E",
			"Samsung|Galaxy Y Plus|coriplusds|GT-S5303",
			"Samsung|Galaxy Y Plus|coriplusds|GT-S5303B",
			"Samsung|Galaxy Y Pop|GT-S6108|GT-S6108",
			"Samsung|Galaxy Y Pro|GT-B5510|GT-B5510",
			"Samsung|Galaxy Y Pro|GT-B5510B|GT-B5510B",
			"Samsung|Galaxy Y Pro|GT-B5510L|GT-B5510L",
			"Samsung|Galaxy Y Pro Duos|GT-B5512|GT-B5512",
			"Samsung|Galaxy Y Pro Duos|GT-B5512B|GT-B5512B",
			"Samsung|Galaxy Y TV|GT-S5367|GT-S5367",
			"Samsung|Galaxy Young|roy|GT-S6312",
			"Samsung|Galaxy Young|roydtv|GT-S6313T",
			"Samsung|Galaxy Young|royss|GT-S6310",
			"Samsung|Galaxy Young|royss|GT-S6310B",
			"Samsung|Galaxy Young|royss|GT-S6310L",
			"Samsung|Galaxy Young|royss|GT-S6310T",
			"Samsung|Galaxy Young|royssdtv|GT-S6313T",
			"Samsung|Galaxy Young|royssnfc|GT-S6310N",
			"Samsung|Galaxy Young2|young23g|SM-G130H",
			"Samsung|Galaxy Young2|young23g|SM-G130M",
			"Samsung|Galaxy Young2|young23g|SM-G130U",
			"Samsung|Galaxy Young2|young23gdtv|SM-G130BT",
			"Samsung|Galaxy Young2|young2ds2g|SM-G130E",
			"Samsung|Galaxy Young2|young2nfc3g|SM-G130HN",
			"Samsung|Galaxy player 70 Plus|YP-GB70D|YP-GB70D",
			"Samsung|Galaxy win|delos3gss|GT-I8550L",
			"Samsung|Garda|gardalteMetroPCS|SGH-T399N",
			"Samsung|Gear Live|sprat|Gear Live",
			"Samsung|Gem|SCH-I100|SCH-I100",
			"Samsung|Hennessy|hennessy3gduosctc|SCH-W789",
			"Samsung|Homesync|spcwifi|GT-B9150",
			"Samsung|IceTouch|gokey|YP-GH1",
			"Samsung|Illusion|SCH-I110|SCH-I110",
			"Samsung|Infuse|SGH-I997|SAMSUNG-SGH-I997",
			"Samsung|Infuse|SGH-I997R|SAMSUNG-SGH-I997R",
			"Samsung|Moment|SPH-M900|SPH-M900",
			"Samsung|Montblanc|montblanc3gctc|SM-W2014",
			"Samsung|Nexus 10|manta|Nexus 10",
			"Samsung|Nexus S|crespo|Nexus S",
			"Samsung|Nexus S|crespo4g|Nexus S 4G",
			"Samsung|ProXpress M4580|fiber-athena|samsung-printer-tablet",
			"Samsung|Replenish|SPH-M580|SPH-M580",
			"Samsung|Replenish|SPH-M580BST|SPH-M580BST",
			"Samsung|Repp|SCH-R680|SCH-R680",
			"Samsung|Roy VE DTV|royssvedtv|GT-S6293T",
			"Samsung|Roy VE DTV|royvedtv|GT-S6293T",
			"Samsung|Rugby Smart|SGH-I847|SAMSUNG-SGH-I847",
			"Samsung|SC-01F|SC-01F|SC-01F",
			"Samsung|SC-01G|SC-01G|SC-01G",
			"Samsung|SC-02G|SC-02G|SC-02G",
			"Samsung|SC-04F|SC-04F|SC-04F",
			"Samsung|Samsung Galaxy S4|jflteMetroPCS|SGH-M919N",
			"Samsung|Samsung Galaxy S4 Zoom|mprojectltektt|SM-C105K",
			"Samsung|Samsung Galaxy Stratosphere2|aegis2vzw|SCH-I415",
			"Samsung|Samsung Garda|gardaltetmo|SGH-T399",
			"Samsung|Samsung Homesync|spcwifi|GT-B9150",
			"Samsung|Samsung Homesync|spcwifiany|GT-B9150",
			"Samsung|Sidekick|SGH-T839|SGH-T839",
			"Samsung|Transfix|SCH-R730|SCH-R730",
			"Samsung|Transform|SPH-M920|SPH-M920",
			"Samsung|Transform Ultra|SPH-M930|SPH-M930",
			"Samsung|Transform Ultra|SPH-M930BST|SPH-M930BST",
			"Samsung|VinsQ|SPH-M910|SPH-M910",
			"Samsung|VinsQ(M910)|SPH-M910|SPH-M910",
			"Samsung|olleh|ik1|SMT-E5015",
			"Sanyo|Benesse|31TL04|31TL04",
			"Sanyo|Benesse|40TL04|40TL04",
			"Sanyo|Benesse|41EA04|41EA04",
			"Sanyo|CEA1|CEA1|CEA1",
			"Sanyo|CEA3|CEA3|CEA3",
			"Sharp|306SH|SG306SH|306SH",
			"Sharp|401SH|SG401SH|401SH",
			"Sharp|A01SH|A01SH|A01SH",
			"Sharp|ADS1|ADS1|SHARP-ADS1",
			"Sharp|AQUOS CRYSTAL X|SG402SH|402SH",
			"Sharp|AQUOS PAD  SHT22|SHT22|SHT22",
			"Sharp|AQUOS PAD SH-06F|SH-06F|SH-06F",
			"Sharp|AQUOS PAD SH-08E|SH-08E|SH-08E",
			"Sharp|AQUOS PAD SHT21|SHT21|SHT21",
			"Sharp|AQUOS PHONE  SERIE SHL22|SHL22|SHL22",
			"Sharp|AQUOS PHONE  SERIE SHL23|SHL23|SHL23",
			"Sharp|AQUOS PHONE  SERIE mini SHL24|SHL24|SHL24",
			"Sharp|AQUOS PHONE CL IS17SH|SHI17|IS17SH",
			"Sharp|AQUOS PHONE EX SH-02F|SH-02F|SH-02F",
			"Sharp|AQUOS PHONE EX SH-04E|SH04E|SH-04E",
			"Sharp|AQUOS PHONE IS11SH|SHI11|IS11SH",
			"Sharp|AQUOS PHONE IS12SH|SHI12|IS12SH",
			"Sharp|AQUOS PHONE IS13SH|SHI13|IS13SH",
			"Sharp|AQUOS PHONE IS14SH|SHI14|IS14SH",
			"Sharp|AQUOS PHONE SERIE ISW16SH|SHI16|ISW16SH",
			"Sharp|AQUOS PHONE SERIE SHL21|SHL21|SHL21",
			"Sharp|AQUOS PHONE SH-01D|SH01D|SH-01D",
			"Sharp|AQUOS PHONE SH-06D|SH06D|SH-06D",
			"Sharp|AQUOS PHONE SH-12C|SH12C|SH-12C",
			"Sharp|AQUOS PHONE SH90B|SH90B|SH90B",
			"Sharp|AQUOS PHONE SL IS15SH|SHI15|IS15SH",
			"Sharp|AQUOS PHONE SoftBank  102SH II|SBM102SH2|SBM102SH2",
			"Sharp|AQUOS PHONE SoftBank 006SH|SBM006SH|SBM006SH",
			"Sharp|AQUOS PHONE SoftBank 102SH|SBM102SH|SBM102SH",
			"Sharp|AQUOS PHONE SoftBank 103SH|SBM103SH|SBM103SH",
			"Sharp|AQUOS PHONE SoftBank 104SH|SBM104SH|SBM104SH",
			"Sharp|AQUOS PHONE THE HYBRID SoftBank 007SH|SBM007SH|SBM007SH",
			"Sharp|AQUOS PHONE THE HYBRID SoftBank 007SH J|SBM007SHJ|SBM007SHJ",
			"Sharp|AQUOS PHONE THE HYBRID SoftBank 101SH|SBM101SH|SBM101SH",
			"Sharp|AQUOS PHONE WX05SH|WX05SH|WX05SH",
			"Sharp|AQUOS PHONE Xx 206SH|SBM206SH|SBM206SH",
			"Sharp|AQUOS PHONE Xx 302SH|SBM302SH|SBM302SH",
			"Sharp|AQUOS PHONE Xx SoftBank 106SH|SBM106SH|SBM106SH",
			"Sharp|AQUOS PHONE Xx mini 303SH|SBM303SH|SBM303SH",
			"Sharp|AQUOS PHONE ZETA  SH-01F|SH-01F|SH-01F",
			"Sharp|AQUOS PHONE ZETA SH-02E|SH02E|SH-02E",
			"Sharp|AQUOS PHONE ZETA SH-06E|SH-06E|SH-06E",
			"Sharp|AQUOS PHONE ZETA SH-09D|SH09D|SH-09D",
			"Sharp|AQUOS PHONE es WX04SH|WX04SH|WX04SH",
			"Sharp|AQUOS PHONE f SH-13C|SH13C|SH-13C",
			"Sharp|AQUOS PHONE si SH-01E|SH01E|SH-01E",
			"Sharp|AQUOS PHONE si SH-07E|SH-07E|SH-07E",
			"Sharp|AQUOS PHONE slider SH-02D|SH02D|SH-02D",
			"Sharp|AQUOS PHONE ss 205SH|SBM205SH|SBM205SH",
			"Sharp|AQUOS PHONE st SH-07D|SH07D|SH-07D",
			"Sharp|AQUOS PHONE sv SH-10D|SH10D|SH-10D",
			"Sharp|AQUOS SERIE SHL25|SHL25|SHL25",
			"Sharp|AQUOS SERIE mini SHV31|SHV31|SHV31",
			"Sharp|AQUOS SH-M01|SH-M01|SH-M01",
			"Sharp|AQUOS Xx 304SH|SG304SH|304SH",
			"Sharp|AQUOS ZETA SH-01G|SH-01G|SH-01G",
			"Sharp|AQUOS ZETA SH-04F|SH-04F|SH-04F",
			"Sharp|DM009SH|DM009SH|DM009SH",
			"Sharp|DM010SH|DM010SH|DM010SH",
			"Sharp|DM011SH|DM011SH|DM011SH",
			"Sharp|DM012SH|DM012SH|DM012SH",
			"Sharp|DM013SH|DM013SH|DM013SH",
			"Sharp|DM016SH|DM016SH|DM016SH",
			"Sharp|Disney Mobile DM014SH|DM014SH|DM014SH",
			"Sharp|Disney Mobile on docomo SH-02G|SH-02G|SH-02G",
			"Sharp|Disney Mobile on docomo SH-05F|SH-05F|SH-05F",
			"Sharp|EB-A71GJ|EB-A71GJ|EB-A71GJ",
			"Sharp|EB-WX1GJ/EB-W51GJ|Galapagos|EB-W51GJ",
			"Sharp|EB-WX1GJ/EB-W51GJ|Galapagos|EB-WX1GJ",
			"Sharp|GALAPAGOS SoftBank 003SH|SBM003SH|SBM003SH",
			"Sharp|GALAPAGOS SoftBank 005SH|SBM005SH|SBM005SH",
			"Sharp|INFOBAR A01|SHX11|INFOBAR A01",
			"Sharp|INFOBAR C01|SHX12|INFOBAR C01",
			"Sharp|IS01|SHI01|IS01",
			"Sharp|IS03|SHI03|IS03",
			"Sharp|IS05|SHI05|IS05",
			"Sharp|LC-42LE860M|LC-42LE860M|LC-42LE860M",
			"Sharp|LC-50LE860M|LC-50LE860M|LC-50LE860M",
			"Sharp|LC-55LE860M|LC-55LE860M|LC-55LE860M",
			"Sharp|LC-A11A|a11a|LC-A11A",
			"Sharp|LC-LX565H|lx565h|LC-LX565H",
			"Sharp|LCD-LX460A|lx460a|LCD-LX460A",
			"Sharp|LCD-LX560A|lx560a|LCD-LX560A",
			"Sharp|LCD-LX565A|msd818sharp|LCD-LX565A",
			"Sharp|LCD-V3A|v3a|LCD-V3A",
			"Sharp|LYNX 3D SH-03C|SH03C|SH-03C",
			"Sharp|LYNX SH-10B|SH10B|SH-10B",
			"Sharp|PANTONE 5 SoftBank 107SH|SBM107SH|SBM107SH",
			"Sharp|PANTONE 6 SoftBank 200SH|SBM200SH|SBM200SH",
			"Sharp|Q-pot.Phone SH-04D|SH04D|SH-04D",
			"Sharp|RW107|RW107|RW107",
			"Sharp|SBM204SH|SBM204SH|SBM204SH",
			"Sharp|SC-S01|scallop|SC-S01",
			"Sharp|SH-01E Vivienne Westwood|SH01EVW|SH-01EVW",
			"Sharp|SH-01F DRAGON QUEST|SH-01FDQ|SH-01FDQ",
			"Sharp|SH-03F|SH03F|SH-03F",
			"Sharp|SH-05E|SH05E|SH-05E",
			"Sharp|SH-06D NERV|SH06DNERV|SH-06DNERV",
			"Sharp|SH80F|SH80F|SH80F",
			"Sharp|SH8118U|msm7627|SH8118U",
			"Sharp|SH8128U|msm7627|SH8128U",
			"Sharp|SH8188U|SH8188U|SH8188U",
			"Sharp|SoftBank 007SH KT|SBM007SHK|SBM007SHK",
			"Sharp|SoftBank 107SH B|SBM107SHB|SBM107SHB",
			"Sharp|SoftBank 305SH|SG305SH|305SH",
			"Sharp|SoftBank AQUOS PHONE Xx 203SH|SBM203SH|SBM203SH",
			"Sharp|Yahoo! Phone SoftBank 009SH Y|SBM009SHY|SBM009SHY",
			"SiAL|Bic camera|Si01BB|Si01BB",
			"SiAL|Si01BE|Si01BE|Si01BE",
			"Sico SmartPhone|SSR1-5-8M|GBC_bravo|SSR1-5-8M",
			"Sico SmartPhone|SSR1-5-8M|GBC_bravo|sico pro",
			"Siragon|Siragon SP-5000|SP-5000|Siragon SP-5000",
			"Smartab|Smartab SRF79|SRF79|SRF79",
			"Smartab|Smartab STJR76|STJR76|STJR76",
			"Smartfren|AD688G|AD688G|Smartfren Andromax AD688G",
			"Smartfren|AD689G|AD689G|Smartfren Andromax AD689G",
			"Smartfren|Andromax AD687G|AD687G|Andromax AD687G",
			"Smartfren|EG680|eg680|EG680",
			"Smartfren|NEWAD688G|NEWAD688G|Smartfren Andromax AD688G",
			"Smartfren|Smartfren Andromax AD682H|AD682H|Smartfren Andromax AD682H",
			"Sonim|Ex-Handy 09|XP6700Z1|Ex-Handy 09",
			"Sonim|Smart-Ex 01|XP7700Z1|Smart-Ex 01",
			"Sonim|XP6|XP6700|XP6700",
			"Sonim|XP7|XP7700|XP7700",
			"Sony|BRAVIA Smart Stick|NSZGU1|NSZ-GU1",
			"Sony|Internet TV|asura|Internet TV",
			"Sony|Internet TV|eagle|Internet TV Box",
			"Sony|NSZGS7|NSZGS7|NSZ-GS7/GX70",
			"Sony|NW-F800|icx1227|WALKMAN",
			"Sony|NW-F880 Series|icx1237|WALKMAN",
			"Sony|NW-Z1000|icx1216|NW-Z1000Series",
			"Sony|NW-ZX1|icx1240|WALKMAN",
			"Sony|NWZ-Z1000|icx1216|NWZ-Z1000Series",
			"Sony|SmartWatch 3|tetra|SmartWatch 3",
			"Sony|Sony|D2302|D2302",
			"Sony|Sony Ericsson Live with Walkman|WT19a|WT19a",
			"Sony|Sony Ericsson Live with Walkman|WT19i|WT19i",
			"Sony|Sony Smartphone Z Ultra Google Play edition|C6806|C6806_GPe",
			"Sony|Sony Tablet S|nbx03|Sony Tablet S",
			"Sony|Tablet P|nbx02|Sony Tablet P",
			"Sony|Tablet S|nbx03|Sony Tablet S",
			"Sony|Xperia  E1 dual|D2114|D2114",
			"Sony|Xperia A|SO-04E|SO-04E",
			"Sony|Xperia A2|SO-04F|SO-04F",
			"Sony|Xperia C|C2304|C2304",
			"Sony|Xperia C|C2305|C2305",
			"Sony|Xperia C3|D2533|D2533",
			"Sony|Xperia C3 Dual|D2502|D2502",
			"Sony|Xperia E|C1504|C1504",
			"Sony|Xperia E|C1505|C1505",
			"Sony|Xperia E dual|C1604|C1604",
			"Sony|Xperia E dual|C1605|C1605",
			"Sony|Xperia E1|D2004|D2004",
			"Sony|Xperia E1|D2005|D2005",
			"Sony|Xperia E1|D2114|D2114",
			"Sony|Xperia E1 Dual|D2104|D2104",
			"Sony|Xperia E1 Dual|D2105|D2105",
			"Sony|Xperia E1 dual|D2104|D2104",
			"Sony|Xperia E1 dual|D2105|D2105",
			"Sony|Xperia E3|D2202|D2202",
			"Sony|Xperia E3|D2203|D2203",
			"Sony|Xperia E3|D2206|D2206",
			"Sony|Xperia E3|D2243|D2243",
			"Sony|Xperia E3 Dual|D2212|D2212",
			"Sony|Xperia Go|ST27a|ST27a",
			"Sony|Xperia Go|ST27i|ST27i",
			"Sony|Xperia J|ST26a|ST26a",
			"Sony|Xperia J|ST26i|ST26i",
			"Sony|Xperia L|C2104|C2104",
			"Sony|Xperia L|C2105|C2105",
			"Sony|Xperia M|C1904|C1904",
			"Sony|Xperia M|C1905|C1905",
			"Sony|Xperia M dual|C2004|C2004",
			"Sony|Xperia M dual|C2005|C2005",
			"Sony|Xperia M2|D2303|D2303",
			"Sony|Xperia M2|D2305|D2305",
			"Sony|Xperia M2|D2306|D2306",
			"Sony|Xperia M2 Aqua|D2403|D2403",
			"Sony|Xperia M2 Aqua|D2406|D2406",
			"Sony|Xperia M2 dual|D2302|D2302",
			"Sony|Xperia Miro|ST23i|ST23i",
			"Sony|Xperia P|LT22i|LT22i",
			"Sony|Xperia PLAY|R800a|R800a",
			"Sony|Xperia PLAY|R800i|R800i",
			"Sony|Xperia S|LT26i|LT26i",
			"Sony|Xperia SL|LT26ii|LT26ii",
			"Sony|Xperia SP|C5302|C5302",
			"Sony|Xperia SP|C5303|C5303",
			"Sony|Xperia SP|C5306|C5306",
			"Sony|Xperia SP|M35h|M35h",
			"Sony|Xperia SP|M35t|M35t",
			"Sony|Xperia T|LT30a|LT30a",
			"Sony|Xperia T|LT30p|LT30p",
			"Sony|Xperia T2 Ultra|D5303|D5303",
			"Sony|Xperia T2 Ultra|D5306|D5306",
			"Sony|Xperia T2 Ultra|D5316|D5316",
			"Sony|Xperia T2 Ultra|D5316N|D5316N",
			"Sony|Xperia T2 Ultra|D5322|D5322",
			"Sony|Xperia T2 Ultra dual|D5322|D5322",
			"Sony|Xperia T3|D5102|D5102",
			"Sony|Xperia T3|D5103|D5103",
			"Sony|Xperia T3|D5106|D5106",
			"Sony|Xperia TX|LT29i|LT29i",
			"Sony|Xperia Tablet S|txs03|SGPT12",
			"Sony|Xperia Tablet S|txs03|SGPT13",
			"Sony|Xperia Tablet Z|SGP311|SGP311",
			"Sony|Xperia Tablet Z|SGP312|SGP312",
			"Sony|Xperia Tablet Z|SGP321|SGP321",
			"Sony|Xperia Tablet Z|SGP351|SGP351",
			"Sony|Xperia Tipo|ST21i|ST21i",
			"Sony|Xperia Tipo|ST21i2|ST21i2",
			"Sony|Xperia U|ST25a|ST25a",
			"Sony|Xperia U|ST25i|ST25i",
			"Sony|Xperia V|LT25i|LT25i",
			"Sony|Xperia Z|C6602|C6602",
			"Sony|Xperia Z|C6603|C6603",
			"Sony|Xperia Z|C6606|C6606",
			"Sony|Xperia Z|C6616|C6616",
			"Sony|Xperia Z|L36h|L36h",
			"Sony|Xperia Z|SO-02E|SO-02E",
			"Sony|Xperia Z Ultra|C6802|C6802",
			"Sony|Xperia Z Ultra|C6806|C6806",
			"Sony|Xperia Z Ultra|C6833|C6833",
			"Sony|Xperia Z Ultra|C6843|C6843",
			"Sony|Xperia Z Ultra|SGP412|SGP412",
			"Sony|Xperia Z Ultra|SOL24|SOL24",
			"Sony|Xperia Z Ultra|XL39h|XL39h",
			"Sony|Xperia Z1|C6902|C6902",
			"Sony|Xperia Z1|C6903|C6903",
			"Sony|Xperia Z1|C6906|C6906",
			"Sony|Xperia Z1|C6916|C6916",
			"Sony|Xperia Z1|C6943|C6943",
			"Sony|Xperia Z1|L39h|L39h",
			"Sony|Xperia Z1|L39t|L39t",
			"Sony|Xperia Z1|L39u|L39u",
			"Sony|Xperia Z1|SO-01F|SO-01F",
			"Sony|Xperia Z1|SOL23|SOL23",
			"Sony|Xperia Z1 Compact|D5503|D5503",
			"Sony|Xperia Z1 Compact|M51w|M51w",
			"Sony|Xperia Z1f|SO-02F|SO-02F",
			"Sony|Xperia Z2|D6502|D6502",
			"Sony|Xperia Z2|D6503|D6503",
			"Sony|Xperia Z2|D6543|D6543",
			"Sony|Xperia Z2|SO-03F|SO-03F",
			"Sony|Xperia Z2 Tablet|SGP511|SGP511",
			"Sony|Xperia Z2 Tablet|SGP512|SGP512",
			"Sony|Xperia Z2 Tablet|SGP521|SGP521",
			"Sony|Xperia Z2 Tablet|SGP551|SGP551",
			"Sony|Xperia Z2 Tablet|SGP561|SGP561",
			"Sony|Xperia Z2 Tablet|SO-05F|SO-05F",
			"Sony|Xperia Z2 Tablet|SOT21|SOT21",
			"Sony|Xperia Z2a|D6563|D6563",
			"Sony|Xperia Z3|401SO|401SO",
			"Sony|Xperia Z3|D6603|D6603",
			"Sony|Xperia Z3|D6616|D6616",
			"Sony|Xperia Z3|D6643|D6643",
			"Sony|Xperia Z3|D6653|D6653",
			"Sony|Xperia Z3|SO-01G|SO-01G",
			"Sony|Xperia Z3|SOL26|SOL26",
			"Sony|Xperia Z3 Compact|D5803|D5803",
			"Sony|Xperia Z3 Compact|D5833|D5833",
			"Sony|Xperia Z3 Compact|SO-02G|SO-02G",
			"Sony|Xperia Z3 Dual|D6633|D6633",
			"Sony|Xperia Z3 Tablet Compact|SGP611|SGP611",
			"Sony|Xperia Z3 Tablet Compact|SGP612|SGP612",
			"Sony|Xperia Z3 Tablet Compact|SGP621|SGP621",
			"Sony|Xperia Z3 Tablet Compact|SGP641|SGP641",
			"Sony|Xperia Z3v|D6708|D6708",
			"Sony|Xperia ZL|C6502|C6502",
			"Sony|Xperia ZL|C6503|C6503",
			"Sony|Xperia ZL|C6506|C6506",
			"Sony|Xperia ZL|L35h|L35h",
			"Sony|Xperia ZL2|SOL25|SOL25",
			"Sony|Xperia ZR|C5306|C5306",
			"Sony|Xperia ZR|C5502|C5502",
			"Sony|Xperia ZR|C5503|C5503",
			"Sony|Xperia acro HD|SO-03D|SO-03D",
			"Sony|Xperia acro S|LT26w|LT26w",
			"Sony|Xperia active|ST17i|ST17i",
			"Sony|Xperia arc S|LT18i|LT18i",
			"Sony|Xperia go|ST27i|ST27i",
			"Sony|Xperia ion|LT28h|LT28h",
			"Sony|Xperia ion|LT28i|LT28i",
			"Sony|Xperia mini pro|SK17a|SK17a",
			"Sony|Xperia miro|ST23a|ST23a",
			"Sony|Xperia miro|ST23i|ST23i",
			"Sony|Xperia neo L|MT25i|MT25i",
			"Sony|Xperia pro|MK16i|MK16i",
			"Sony|Xperia ray|ST18i|ST18i",
			"Sony|Xperia sola|MT27i|MT27i",
			"Sony|Xperia tipo|ST21a|ST21a",
			"Sony|Xperia tipo dual|ST21a2|ST21a2",
			"Sony Ericsson|Live with Walkman|WT19a|WT19a",
			"Sony Ericsson|Live with Walkman|WT19i|WT19i",
			"Sony Ericsson|Live with Walkman(TM)|WT19a|WT19a",
			"Sony Ericsson|X10 Xperia Mini|robyn|E10i",
			"Sony Ericsson|Xperia A|SO-04E|SO-04E",
			"Sony Ericsson|Xperia AX|SO-01E|SO-01E",
			"Sony Ericsson|Xperia Acro|IS11S|IS11S",
			"Sony Ericsson|Xperia Acro|SO-02C|SO-02C",
			"Sony Ericsson|Xperia Arc|LT15i|LT15i",
			"Sony Ericsson|Xperia Arc|SO-01C|SO-01C",
			"Sony Ericsson|Xperia C|S39h|S39h",
			"Sony Ericsson|Xperia GX|SO-04D|SO-04D",
			"Sony Ericsson|Xperia Go|ST27a|ST27a",
			"Sony Ericsson|Xperia Go|ST27i|ST27i",
			"Sony Ericsson|Xperia J|ST26i|ST26i",
			"Sony Ericsson|Xperia Neo|MT15i|MT15i",
			"Sony Ericsson|Xperia P|LT22i|LT22i",
			"Sony Ericsson|Xperia PLAY|R800a|R800a",
			"Sony Ericsson|Xperia PLAY|R800at|R800at",
			"Sony Ericsson|Xperia PLAY|R800i|R800i",
			"Sony Ericsson|Xperia PLAY|R800x|R800x",
			"Sony Ericsson|Xperia PLAY|SO-01D|SO-01D",
			"Sony Ericsson|Xperia Play|R800i|R800i",
			"Sony Ericsson|Xperia Play|Zeus|Zeus",
			"Sony Ericsson|Xperia S|LT26i|LT26i",
			"Sony Ericsson|Xperia S|SO-02D|SO-02D",
			"Sony Ericsson|Xperia SL|LT26ii|LT26ii",
			"Sony Ericsson|Xperia SP|M35c|M35c",
			"Sony Ericsson|Xperia SX|SO-05D|SO-05D",
			"Sony Ericsson|Xperia T|LT30a|LT30a",
			"Sony Ericsson|Xperia T|LT30at|LT30at",
			"Sony Ericsson|Xperia T|LT30p|LT30p",
			"Sony Ericsson|Xperia TX|LT29i|LT29i",
			"Sony Ericsson|Xperia Tablet Z|SGP311|SGP311",
			"Sony Ericsson|Xperia Tablet Z|SGP321|SGP321",
			"Sony Ericsson|Xperia Tablet Z|SGP341|SGP341",
			"Sony Ericsson|Xperia Tablet Z|SO-03E|SO-03E",
			"Sony Ericsson|Xperia U|ST25i|ST25i",
			"Sony Ericsson|Xperia UL|SOL22|SOL22",
			"Sony Ericsson|Xperia V|LT25c|LT25c",
			"Sony Ericsson|Xperia V|LT25i|LT25i",
			"Sony Ericsson|Xperia VL|SOL21|SOL21",
			"Sony Ericsson|Xperia X10|SO-01B|SO-01B",
			"Sony Ericsson|Xperia X10|SonyEricssonSO-01B|SO-01B",
			"Sony Ericsson|Xperia X10|SonyEricssonX10iv|X10i",
			"Sony Ericsson|Xperia X10|X10a|X10a",
			"Sony Ericsson|Xperia X10|X10i|X10i",
			"Sony Ericsson|Xperia X10 Mini|E10i|E10i",
			"Sony Ericsson|Xperia X10 Mini|SonyEricssonE10i|E10i",
			"Sony Ericsson|Xperia X10 Mini Pro|U20i|U20i",
			"Sony Ericsson|Xperia X10 mini|SonyEricssonE10a|E10a",
			"Sony Ericsson|Xperia X10 mini pro|SonyEricssonU20a|U20a",
			"Sony Ericsson|Xperia X8|E15i|E15i",
			"Sony Ericsson|Xperia Z|SO-02E|SO-02E",
			"Sony Ericsson|Xperia Z Ultra|C6806|C6806",
			"Sony Ericsson|Xperia Z1|C6903|C6903",
			"Sony Ericsson|Xperia Z1|C6906|C6906",
			"Sony Ericsson|Xperia Z1|C6943|C6943",
			"Sony Ericsson|Xperia acro HD|IS12S|IS12S",
			"Sony Ericsson|Xperia acro S|LT26w|LT26w",
			"Sony Ericsson|Xperia active|ST17i|ST17i",
			"Sony Ericsson|Xperia arc|LT15a|LT15a",
			"Sony Ericsson|Xperia arc|LT15i|LT15i",
			"Sony Ericsson|Xperia arc S|LT18a|LT18a",
			"Sony Ericsson|Xperia arc S|LT18i|LT18i",
			"Sony Ericsson|Xperia go|ST27i|ST27i",
			"Sony Ericsson|Xperia ion|LT28at|LT28at",
			"Sony Ericsson|Xperia ion|LT28h|LT28h",
			"Sony Ericsson|Xperia ion|LT28i|LT28i",
			"Sony Ericsson|Xperia live|WT19a|WT19a",
			"Sony Ericsson|Xperia mini|S51SE|S51SE",
			"Sony Ericsson|Xperia mini|ST15a|ST15a",
			"Sony Ericsson|Xperia mini|ST15i|ST15i",
			"Sony Ericsson|Xperia mini pro|SK17a|SK17a",
			"Sony Ericsson|Xperia mini pro|SK17i|SK17i",
			"Sony Ericsson|Xperia miro|ST23i|ST23i",
			"Sony Ericsson|Xperia neo|MT15a|MT15a",
			"Sony Ericsson|Xperia neo|MT15i|MT15i",
			"Sony Ericsson|Xperia neo L|MT25i|MT25i",
			"Sony Ericsson|Xperia neo V|MT11a|MT11a",
			"Sony Ericsson|Xperia neo V|MT11i|MT11i",
			"Sony Ericsson|Xperia pro|MK16a|MK16a",
			"Sony Ericsson|Xperia pro|MK16i|MK16i",
			"Sony Ericsson|Xperia ray|SO-03C|SO-03C",
			"Sony Ericsson|Xperia ray|ST18a|ST18a",
			"Sony Ericsson|Xperia ray|ST18i|ST18i",
			"Sony Ericsson|Xperia sola|MT27i|MT27i",
			"Sony Ericsson|Xperia tipo|ST21i|ST21i",
			"Sony Ericsson|Xperia tipo dual|ST21i2|ST21i2",
			"Sophix_Digix|TAB-750_G|TAB-750_G|TAB-750_G",
			"Spice|Dream Uno|Mi-498_sprout|Spice Mi-498",
			"Spice|Dream Uno|Mi-498_sprout|Spice Mi-498H",
			"Spice|Smart Flo Mi-449|SpiceMi449|SpiceMi-449",
			"Spice|Smart Flo Mi-449|hongyu72_wet_jb3|SpiceMi-449",
			"Spice|Stellar Mi-438|Spice|MI-438",
			"Spice|Stellar Mi-506|SpiceMi506|Spice Mi-506",
			"Storex|eZeeTab10D12-S|eZeeTab10D12-S|eZeeTab10D12-S",
			"Storex|eZeeTab7D15-M|eZeeTab7D15-M|eZeeTab7D15-M",
			"Sugar|Sugar|s9121|SUGAR",
			"TCL|L55E6700UDS|movo_la|movo_la",
			"TCL|LE50UHDE5692G|movo|movo",
			"TCT (Alcatel)|5038D|SOUL45_GSM|5038D",
			"TCT (Alcatel)|6010A|Miata_3G|6016A",
			"TCT (Alcatel)|6014X|Miata_3G|6014X",
			"TCT (Alcatel)|6016D|Miata_3G|6016D",
			"TCT (Alcatel)|6016E|Miata_3G|6016E",
			"TCT (Alcatel)|6016X|Miata_3G|6016X",
			"TCT (Alcatel)|6037B|Eclipse|6037B",
			"TCT (Alcatel)|6037K|Eclipse|6037K",
			"TCT (Alcatel)|6037Y|Eclipse|6037Y",
			"TCT (Alcatel)|6042D|CROSSAPAC|6042D",
			"TCT (Alcatel)|6043A|DIABLOXPLUS|6043A",
			"TCT (Alcatel)|6043D|DIABLOXPLUS|6043D",
			"TCT (Alcatel)|7040N|Yaris5TMO|7040N",
			"TCT (Alcatel)|7040R|Yaris5TMO|7040R",
			"TCT (Alcatel)|7040T|Yaris5TMO|7040T",
			"TCT (Alcatel)|7045Y|RIO5|7045Y",
			"TCT (Alcatel)|730U|Rio5C|J730U",
			"TCT (Alcatel)|8030Y|HERO2|8030Y",
			"TCT (Alcatel)|9005X|PIXO8_3G|9005X",
			"TCT (Alcatel)|A851L|Viper_gsm|Alcatel 7030L",
			"TCT (Alcatel)|A851L|Viper_gsm|Alcatel A851L",
			"TCT (Alcatel)|ALCATEL 4015T|Yaris35_GSM|ALCATEL 4015T",
			"TCT (Alcatel)|ALCATEL A564C|Yaris5NA|ALCATEL A564C",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4005D|Beetle_Lite_Edge_GSM|ALCATEL ONE TOUCH 4005D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4005D|Beetle_Lite_Edge_GSM|ALCATEL_ONE_TOUCH_4005D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|ALCATEL ONE TOUCH 4010A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|ALCATEL ONE TOUCH 4010D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|ALCATEL ONE TOUCH 4010E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|ALCATEL ONE TOUCH 4010X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|TCL_J210",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4010X|Beetle_Lite_GSM|Telenor Smart 2",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4029A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4030D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4030E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4030X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4030Y",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|ALCATEL ONE TOUCH 4030Y_orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|Infinity POP",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4030X|Beetle_GSM|Orange Runo",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4037N|SOUL4NA|ALCATEL ONE TOUCH 4037N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 4037N|SOUL4NA|ALCATEL ONE TOUCH 4037T",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020T",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020W",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5020X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL ONE TOUCH 5021E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL_ONE_TOUCH_5020D_Orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|ALCATEL_ONE_TOUCH_5020X_Orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|AURUS III",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|MTC 972",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|Orange Kivo",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020D|Megane_GSM|Telenor Smart Pro 2",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5020N|Megane|ALCATEL ONE TOUCH 5020N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|ALCATEL ONE TOUCH 5035A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|ALCATEL ONE TOUCH 5035D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|ALCATEL ONE TOUCH 5035E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|ALCATEL ONE TOUCH 5035X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|TCL J610",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5035D|Camry_GSM|TCL S810",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5036A|YarisL_GSM|ALCATEL ONE TOUCH 5036A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5036D|YarisL_GSM|ALCATEL ONE TOUCH 5036D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5036D|YarisL_GSM|ONE TOUCH 5036X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5036F|YarisL_GSM|ALCATEL ONE TOUCH 5036F",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5036X|YarisL_GSM|ALCATEL ONE TOUCH 5036X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 5037E|YarisL_GSM|ALCATEL ONE TOUCH 5037E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 601|Telsa|ALCATEL ONE TOUCH 6010",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010A|Telsa|ALCATEL ONE TOUCH 6010A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010D|Telsa|ALCATEL ONE TOUCH 6010D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010X|Telsa|ALCATEL ONE TOUCH 6010",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010X|Telsa|ALCATEL ONE TOUCH 6010D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010X|Telsa|ALCATEL ONE TOUCH 6010X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010X|Telsa|ALCATEL ONE TOUCH 6010X-orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6010X|Telsa|TCL S520",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030A|Diablo|ALCATEL ONE TOUCH 6030A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030D|Diablo|ALCATEL ONE TOUCH 6030D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030N|Diablo|ALCATEL ONE TOUCH 6030N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|ALCATEL ONE TOUCH 6030A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|ALCATEL ONE TOUCH 6030D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|ALCATEL ONE TOUCH 6030X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|ALCATEL ONE TOUCH 6030X-orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|ALCATEL_ONE_TOUCH_6030X_Orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|Optimus_San_Remo",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X|Diablo|TCL S820",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6030X-orange|Diablo|ALCATEL ONE TOUCH 6030X-orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6032|Alpha|ALCATEL ONE TOUCH 6032",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6032A|Alpha|ALCATEL ONE TOUCH 6032A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6032X|Alpha|ALCATEL ONE TOUCH 6032X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6033A|DiabloHD|ALCATEL ONE TOUCH 6033A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6033M|DiabloHD|ALCATEL ONE TOUCH 6033M",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6033X|DiabloHD|ALCATEL ONE TOUCH 6033M",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6033X|DiabloHD|ALCATEL ONE TOUCH 6033X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6033X|DiabloHD|TCL S850",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6037B|Eclipse|ALCATEL ONE TOUCH 6037B",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6040A|DIABLOX|ALCATEL ONE TOUCH 6040A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6040D|DIABLOX|ALCATEL ONE TOUCH 6040D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6040D|DIABLOX|TCL S950",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6040E|DIABLOX|ALCATEL ONE TOUCH 6040E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 6040X|DIABLOX|ALCATEL ONE TOUCH 6040X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7024W|Rav4|ALCATEL ONE TOUCH 7024R",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7024W|Rav4|ALCATEL ONE TOUCH 7024W",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7024W|Rav4|ALCATEL ONE TOUCH FIERCE",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7025D|Rav4|ALCATEL ONE TOUCH 7025D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7040A|YARISXL|ALCATEL ONE TOUCH 7040A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7040D|YARISXL|ALCATEL ONE TOUCH 7040D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7040E|YARISXL|ALCATEL ONE TOUCH 7040E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7040K|YARISXL|ALCATEL ONE TOUCH 7040K",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7040X|YARISXL|ALCATEL ONE TOUCH 7040X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7041D|YARISXL|ALCATEL ONE TOUCH 7041D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7041X|YARISXL|ALCATEL ONE TOUCH 7041X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7042A|YARISXL|ALCATEL ONE TOUCH 7042A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7042D|YARISXL|ALCATEL ONE TOUCH 7042D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7047D|YARIS_55|ALCATEL ONE TOUCH 7047D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 7047E|YARIS_55|ALCATEL ONE TOUCH 7047E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8008D|Scribe5HD|ALCATEL ONE TOUCH 8008D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8008D|Scribe5HD|ALCATEL ONE TOUCH 8008X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8008D|Scribe5HD|Orange Infinity 8008X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8020A|SCRIBEPRO|ALCATEL ONE TOUCH 8020A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8020D|SCRIBEPRO|ALCATEL ONE TOUCH 8020D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8020E|SCRIBEPRO|ALCATEL ONE TOUCH 8020E",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 8020X|SCRIBEPRO|ALCATEL ONE TOUCH 8020X",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903|one_touch_903A_gsm|ALCATEL_ONE_TOUCH_903",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903|one_touch_903A_gsm|ALCATEL_one_touch_903",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903|one_touch_903D_gsm|ALCATEL ONE TOUCH 903D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903|one_touch_903_gsm|ALCATEL ONE TOUCH 903",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903|one_touch_903_gsm|ALCATEL_ONE_TOUCH_903",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903A|one_touch_903A_gsm|ALCATEL ONE TOUCH 903A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 903D|one_touch_903D_gsm|ALCATEL ONE TOUCH 903D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918|one_touch_918_gsm|ALCATEL ONE TOUCH 918",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918|one_touch_918_gsm|ALCATEL one touch 918",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918|one_touch_918_gsm|ALCATEL_one_touch_918",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918|one_touch_918_gsm|Alcatel_one_touch_918_Orange",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918A|one_touch_918A_gsm|ALCATEL ONE TOUCH 918A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918A|one_touch_918A_gsm|ALCATEL_one_touch_918A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918D|ONE_TOUCH_918D_umts|ALCATEL ONE TOUCH 918D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918D|one_touch_918D_gsm|ALCATEL ONE TOUCH 918D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918N|ONE_TOUCH_918N_umts|ALCATEL ONE TOUCH 918N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918N|one_touch_918N_gsm|ALCATEL ONE TOUCH 918N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 918S|one_touch_918S_gsm|ALCATEL ONE TOUCH 918S",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 922|one_touch_922_gsm|ALCATEL ONE TOUCH 922",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 930D|OT-930|ALCATEL ONE TOUCH 930D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 930D|OT-930|ALCATEL ONE TOUCH 930N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985|one_touch_985_gsm|ALCATEL ONE TOUCH 985",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985|one_touch_985_gsm|ALCATEL_ONE_TOUCH_985",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985|one_touch_985_gsm|ALCATEL_one_touch_985",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985|one_touch_985_gsm|Telenor_One_Touch_C",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985D|one_touch_985D_gsm|ALCATEL ONE TOUCH 985D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 985N|one_touch_985N_gsm|ALCATEL ONE TOUCH 985N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991|ONE_TOUCH_991D_gsm|ALCATEL ONE TOUCH 991D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991|ONE_TOUCH_991_gsm|ALCATEL ONE TOUCH 991",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991|ONE_TOUCH_991_gsm|ALCATEL_one_touch_991",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991|ONE_TOUCH_991_gsm|Telenor_Smart_Pro",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991|one_touch_991_gsm|ALCATEL ONE TOUCH 991",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991A|ONE_TOUCH_991A_gsm|ALCATEL ONE TOUCH 991A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991D|ONE_TOUCH_991D_gsm|ALCATEL ONE TOUCH 991D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991D|one_touch_991D_gsm|ALCATEL ONE TOUCH 991D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 991T|one_touch_991T_gsm|ALCATEL ONE TOUCH 991T",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 992D|Martell_lite_GSM|ALCATEL ONE TOUCH 992",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 992D|Martell_lite_GSM|ALCATEL ONE TOUCH 992D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 992D|Martell_lite_GSM|TCL S500",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 992D|Martell_lite_GSM|TCL S600",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 995|one_touch_995_gsm|ALCATEL_one_touch_995",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 995|one_touch_995_gsm|Optimus_Madrid",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 995|one_touch_995_gsm|Telenor_Smart_HD",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 995A|one_touch_995_gsm|ALCATEL_one_touch_995A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 997D|Martell_GSM|ALCATEL ONE TOUCH 997",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 997D|Martell_GSM|ALCATEL ONE TOUCH 997D",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 997D|Martell_GSM|BASE_Lutea_3",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 997D|Martell_GSM|TCL S710",
			"TCT (Alcatel)|ALCATEL ONE TOUCH 997D|Martell_GSM|TCL S800",
			"TCT (Alcatel)|ALCATEL ONE TOUCH D668|DANIEL|ALCATEL ONE TOUCH D668",
			"TCT (Alcatel)|ALCATEL ONE TOUCH D668|DANIEL|TCL D668",
			"TCT (Alcatel)|ALCATEL ONE TOUCH D668|DANIEL|TCL-D668",
			"TCT (Alcatel)|ALCATEL ONE TOUCH Fierce|Rav4|ALCATEL ONE TOUCH 7024N",
			"TCT (Alcatel)|ALCATEL ONE TOUCH Fierce|Rav4|ALCATEL ONE TOUCH 7024W",
			"TCT (Alcatel)|ALCATEL ONE TOUCH Fierce|Rav4|ALCATEL ONE TOUCH Fierce",
			"TCT (Alcatel)|ALCATEL ONE TOUCH P310A|POP7|ALCATEL ONE TOUCH P310A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH P320A|POP8|ALCATEL ONE TOUCH P320A",
			"TCT (Alcatel)|ALCATEL ONE TOUCH P321|POP8|ALCATEL ONE TOUCH P321",
			"TCT (Alcatel)|ALCATEL ONE TOUCH918N|one_touch_918_gsm|ALCATEL ONE TOUCH 918N",
			"TCT (Alcatel)|ALCATEL ONETOUCH 4037R|SOUL4NA|ALCATEL ONE TOUCH 4037R",
			"TCT (Alcatel)|ALCATEL ONETOUCH 4037R|SOUL4NA|ALCATEL ONETOUCH 4037R",
			"TCT (Alcatel)|ALCATEL ONETOUCH 6032|Alpha|ALCATEL ONETOUCH 6032",
			"TCT (Alcatel)|ALCATEL ONETOUCH 6032A|Alpha|ALCATEL ONETOUCH 6032A",
			"TCT (Alcatel)|ALCATEL ONETOUCH 6032X|Alpha|ALCATEL ONETOUCH 6032X",
			"TCT (Alcatel)|ALCATEL ONETOUCH 6043D|DIABLOXPLUS|ALCATEL ONETOUCH 6043D",
			"TCT (Alcatel)|ALCATEL ONETOUCH P310A|POP7|ALCATEL ONE TOUCH P310A",
			"TCT (Alcatel)|ALCATEL ONETOUCH P310A|POP7|ALCATEL ONETOUCH P310A",
			"TCT (Alcatel)|ALCATEL ONETOUCH POP D3|SOUL4NA|ALCATEL ONETOUCH POP D3",
			"TCT (Alcatel)|ALCATEL one touch 986+|OT986|ALCATEL one touch 986+",
			"TCT (Alcatel)|ALCATEL one touch 990C+|one_touch_990C_Plus_cdma|ALCATEL one touch 990C+",
			"TCT (Alcatel)|ALCATEL one touch J320|Iris|ALCATEL one touch J320",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_5037A|YarisL_GSM|ALCATEL ONE TOUCH 5037A",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_5037X|YarisL_GSM|ALCATEL ONE TOUCH 5037X",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_6010X_Orange|ONE_TOUCH_6010X_gsm|ALCATEL_ONE_TOUCH_6010X_Orange",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_6030X_Orange|ONE_TOUCH_6030X_gsm|ALCATEL_ONE_TOUCH_6030X_Orange",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_7047A|YARIS_55|ALCATEL ONE TOUCH 7047A",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_903|one_touch_903A_gsm|ALCATEL_ONE_TOUCH_903",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_903|one_touch_903_gsm|ALCATEL_ONE_TOUCH_903",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_991_Orange|one_touch_991_gsm|ALCATEL_ONE_TOUCH_991_Orange",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_P310X|POP7|ALCATEL ONE TOUCH P310X",
			"TCT (Alcatel)|ALCATEL_ONE_TOUCH_P320X|POP8|ALCATEL ONE TOUCH P320X",
			"TCT (Alcatel)|ALCATEL_one_touch_918D|one_touch_918D_umts|ALCATEL_one_touch_918D",
			"TCT (Alcatel)|ALCATEL_one_touch_991|ONE_TOUCH_991A_gsm|ALCATEL_one_touch_991",
			"TCT (Alcatel)|ALCATEL_one_touch_991|one_touch_991_gsm|ALCATEL_one_touch_991",
			"TCT (Alcatel)|ALCATEL_one_touch_995|one_touch_995_gsm|ALCATEL_one_touch_995",
			"TCT (Alcatel)|ALCATEL_one_touch_995|one_touch_995_gsm|RPSPE4301",
			"TCT (Alcatel)|ALCATEL_one_touch_995|one_touch_995_gsm|Telenor_One_Touch_S",
			"TCT (Alcatel)|ALCATEL_one_touch_995|one_touch_995_gsm|Telenor_Smart_HD",
			"TCT (Alcatel)|ALCATEL_one_touch_995A|one_touch_995_gsm|ALCATEL_one_touch_995A",
			"TCT (Alcatel)|ALCATEL_one_touch_995S|one_touch_995_gsm|ALCATEL_one_touch_995S",
			"TCT (Alcatel)|ALOO 6032|Alpha|ALOO 6032",
			"TCT (Alcatel)|ATEL ONE TOUCH 995|one_touch_995_gsm|ALCATEL_one_touch_995",
			"TCT (Alcatel)|Alcatel one touch 985|one_touch_985A_gsm|ALCATEL_one_touch_985",
			"TCT (Alcatel)|Alcatel_one_touch_918_Orange|one_touch_918_gsm|Alcatel_one_touch_918_Orange",
			"TCT (Alcatel)|Boost View 5.0|YARISXL|Boost View 5.0",
			"TCT (Alcatel)|D819|HERO8|D819",
			"TCT (Alcatel)|D820|HERO8|D820",
			"TCT (Alcatel)|D820X|HERO8|D820X",
			"TCT (Alcatel)|DL900|YARISXL|DL900",
			"TCT (Alcatel)|E500|X50D|E500",
			"TCT (Alcatel)|EVO7|EVO7|MTC 1065",
			"TCT (Alcatel)|EVO7|EVO7|onetouch EVO7",
			"TCT (Alcatel)|EVO7|EVO7|starTIM tab 7",
			"TCT (Alcatel)|Hero2C|Hero2C|7055A",
			"TCT (Alcatel)|I216A|PIXI7|I216A",
			"TCT (Alcatel)|I216X|PIXI7|I216X",
			"TCT (Alcatel)|I220|PIXI8|I220",
			"TCT (Alcatel)|I221|PIXI8|I221",
			"TCT (Alcatel)|IDOL 2 S|EOS4G|6050Y",
			"TCT (Alcatel)|IDOL 2 S|EOS4G|ALCATEL ONE TOUCH 6050Y",
			"TCT (Alcatel)|IDOL 2 S|EOS4G|Idol2S_Orange",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|6050A",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|6050F",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|6050W",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|6050Y",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|ALCATEL ONE TOUCH 6050Y",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|ALCATEL ONETOUCH 6050A",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|OWN S5030",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|TCL S830U",
			"TCT (Alcatel)|IDOL 2 S|EOS_lte|TCL S838M",
			"TCT (Alcatel)|Idol S|DiabloHD_LTE|ALCATEL ONE TOUCH 6035L",
			"TCT (Alcatel)|Idol S|DiabloHD_LTE|BS472",
			"TCT (Alcatel)|Idol S|DiabloHD_LTE|MTC 978",
			"TCT (Alcatel)|Idol S|Diablo_LTE|ALCATEL ONE TOUCH 6034L",
			"TCT (Alcatel)|Idol S|Diablo_LTE|ALCATEL ONE TOUCH 6034M",
			"TCT (Alcatel)|Idol S|Diablo_LTE|ALCATEL ONE TOUCH 6034R",
			"TCT (Alcatel)|Idol S|Diablo_LTE|ALCATEL ONE TOUCH 6034Y",
			"TCT (Alcatel)|Idol S|Diablo_LTE|San Remo 4G",
			"TCT (Alcatel)|Idol S|Diablo_LTE|TCL-S850L",
			"TCT (Alcatel)|Idol X|DiabloHD_LTE|ALCATEL ONE TOUCH 6035R",
			"TCT (Alcatel)|Idol2 MINI S|MIATA_lte|6036A",
			"TCT (Alcatel)|Idol2 MINI S|MIATA_lte|6036X",
			"TCT (Alcatel)|Idol2 MINI S|MIATA_lte|6036Y",
			"TCT (Alcatel)|Juke-A554C|A554C|A554C",
			"TCT (Alcatel)|M812C|M812|M812C",
			"TCT (Alcatel)|MTC 960|MTC_960_gsm|MTC 960",
			"TCT (Alcatel)|MTC 970|Beetle_GSM|MTC 970",
			"TCT (Alcatel)|MTC975|Camry_GSM|MTC975",
			"TCT (Alcatel)|MegaFon_SP-A10|one_touch_995_gsm|MegaFon_SP-A10",
			"TCT (Alcatel)|Megafon Login|OT-930|MegaFon_SP-AI",
			"TCT (Alcatel)|Moov2|move_2_gsm|move 2",
			"TCT (Alcatel)|ONE TOUCH 4007X|Pixo_GSM|ONE TOUCH 4007A",
			"TCT (Alcatel)|ONE TOUCH 4007X|Pixo_GSM|ONE TOUCH 4007D",
			"TCT (Alcatel)|ONE TOUCH 4007X|Pixo_GSM|ONE TOUCH 4007E",
			"TCT (Alcatel)|ONE TOUCH 4007X|Pixo_GSM|ONE TOUCH 4007X",
			"TCT (Alcatel)|ONE TOUCH 4011X|BeetleliteJB|ONE TOUCH 4011X",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|4032A",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|4032D",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|4032E",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|4032X",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|4033L",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|MS3B",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|ONE TOUCH 4032A",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|ONE TOUCH 4033A",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|ONE TOUCH 4033D",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|ONE TOUCH 4033E",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|ONE TOUCH 4033X",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|OWN S3010",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|OWN S3010D",
			"TCT (Alcatel)|ONE TOUCH 4033X|Yaris_M_GSM|TCLJ330",
			"TCT (Alcatel)|ONE TOUCH 6012A|California_GSM|ONE TOUCH 6012A",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ALCATEL ONE TOUCH 6012D",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ALCATEL ONE TOUCH 6012X",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|Mobile Sosh",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ONE TOUCH 6012A",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ONE TOUCH 6012D",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ONE TOUCH 6012E",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ONE TOUCH 6012X",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|ONE TOUCH IDOL MINI",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|Orange Covo",
			"TCT (Alcatel)|ONE TOUCH 6012D|California|Orange Hiro",
			"TCT (Alcatel)|ONE TOUCH 6012X|California|ONE TOUCH 6012X",
			"TCT (Alcatel)|ONE TOUCH 8020D|SCRIBEPRO|ONE TOUCH 8020D",
			"TCT (Alcatel)|ONE TOUCH 983|ONE_TOUCH_983_gsm|ALCATEL_one_touch_983",
			"TCT (Alcatel)|ONE TOUCH 983|ONE_TOUCH_983_gsm|ONE TOUCH 983",
			"TCT (Alcatel)|ONE TOUCH 983|ONE_TOUCH_983_gsm|VALENCIA",
			"TCT (Alcatel)|ONE TOUCH 983|one_touch_983_gsm|ALCATEL_one_touch_983",
			"TCT (Alcatel)|ONE TOUCH C505C|ONE_TOUCH_C505C_cdma|ONE TOUCH C505C",
			"TCT (Alcatel)|ONE TOUCH EVO7HD|E710|MTC 1078",
			"TCT (Alcatel)|ONE TOUCH EVO7HD|E710|ONE TOUCH E710",
			"TCT (Alcatel)|ONE TOUCH EVO7HD|E710|ONE TOUCH EVO7HD",
			"TCT (Alcatel)|ONE TOUCH EVO8HD|E720|ONE TOUCH EVO7HD",
			"TCT (Alcatel)|ONE TOUCH EVO8HD|E720|ONE TOUCH EVO8HD",
			"TCT (Alcatel)|ONE TOUCH SCRIBE 5|Scribe5_gsm|ALCATEL ONE TOUCH 8000A",
			"TCT (Alcatel)|ONE TOUCH SCRIBE 5|Scribe5_gsm|ALCATEL ONE TOUCH 8000D",
			"TCT (Alcatel)|ONE TOUCH SCRIBE 5|Scribe5_gsm|TCL Y710",
			"TCT (Alcatel)|ONE TOUCH T10|GR-TB7|GR-TB7",
			"TCT (Alcatel)|ONE TOUCH TAB 7|T011|ONE TOUCH TAB 7",
			"TCT (Alcatel)|ONE TOUCH TAB 7|T011|TCL TAB 7",
			"TCT (Alcatel)|ONE TOUCH TAB 7HD|T016|ONE TOUCH TAB 7HD",
			"TCT (Alcatel)|ONE TOUCH TAB 8HD|T021|ONE TOUCH TAB 8HD",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|Alltel|ONE_TOUCH_960C",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|ONE_TOUCH_960C|ADR3010",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|ONE_TOUCH_960C|ONE TOUCH 960C",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|RadioShack|ADR3010",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|nTelos|ONE_TOUCH_960C",
			"TCT (Alcatel)|ONE TOUCH Ultra 960c|vulcan|ONE_TOUCH_960C",
			"TCT (Alcatel)|ONE_TOUCH_P310A|POP7|ONE TOUCH P310A",
			"TCT (Alcatel)|ONE_TOUCH_P310X|POP7|ALCATEL ONE TOUCH P310X",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|4016D",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4015A",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4015D",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4015N",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4015X",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4015X-orange",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|ONE TOUCH 4016A",
			"TCT (Alcatel)|ONT TOUCH 4015X|Yaris35_GSM|Orange Yomi",
			"TCT (Alcatel)|OT-990C|OT-990C_cdma|OT-990C",
			"TCT (Alcatel)|OWN S4010|Telsa|OWN S4010",
			"TCT (Alcatel)|OWN S5010|DiabloHD|OWN S5010",
			"TCT (Alcatel)|OWN_S4025|YARIS_55|OWN_S4025",
			"TCT (Alcatel)|One Touch 890D|one_touch_890D_gsm|ALCATEL one touch 890D",
			"TCT (Alcatel)|One Touch 906|one_touch_906_gsm|ALCATEL_one_touch_906Y",
			"TCT (Alcatel)|One Touch 908|one_touch_908_gsm|ALCATEL ONE TOUCH 908",
			"TCT (Alcatel)|One Touch 908|one_touch_908_gsm|ALCATEL_one_touch_908",
			"TCT (Alcatel)|One Touch 908|one_touch_908_gsm|Alcatel_one_touch_908F_Orange",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|ALCATEL ONE TOUCH 990",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|ALCATEL one touch 990",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|ALCATEL_one_touch_990",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|Alcatel one touch 908F Orange",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|Alcatel one touch 990 Orange",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|Los Angeles",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|TCL A990",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|TCL ONE TOUCH 990",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|Telenor_OneTouch",
			"TCT (Alcatel)|One Touch 990|one_touch_990_gsm|Vodafone 958",
			"TCT (Alcatel)|One Touch 990A|one_touch_990A_gsm|ALCATEL_one_touch_990A",
			"TCT (Alcatel)|One Touch 990S|one_touch_990S_gsm|ALCATEL_one_touch_990S",
			"TCT (Alcatel)|One Touch 990S|one_touch_990S_gsm|Alcatel one touch 990S",
			"TCT (Alcatel)|Orange_Sego|Orange_Sego|Orange Sego",
			"TCT (Alcatel)|P330X|Pop7_LTE|P330X",
			"TCT (Alcatel)|P330X_orange|Pop7_LTE|P330X_orange",
			"TCT (Alcatel)|P350X|POP8_LTE|P350X",
			"TCT (Alcatel)|P688L|Rio5C|TCL-P688L",
			"TCT (Alcatel)|P688L|Rio5C|TOM007",
			"TCT (Alcatel)|PIXI3(3.5)|PIXI3_35|4009D",
			"TCT (Alcatel)|PIXI3(3.5)|PIXI3_35|4009E",
			"TCT (Alcatel)|POP 2|Alto45|5042A",
			"TCT (Alcatel)|POP 2|Alto45|5042D",
			"TCT (Alcatel)|POP 2|Alto45|5042X",
			"TCT (Alcatel)|POP S3|RIO4G_TF|ALCATEL A845L",
			"TCT (Alcatel)|POP S3|RIO_4G|5050A",
			"TCT (Alcatel)|POP S3|RIO_4G|5050S",
			"TCT (Alcatel)|POP S3|RIO_4G|5050X",
			"TCT (Alcatel)|POP S3|RIO_4G|5050Y",
			"TCT (Alcatel)|POP S3|RIO_4G|ALCATEL ONETOUCH 5050X",
			"TCT (Alcatel)|POP S3|RIO_4G|Siru",
			"TCT (Alcatel)|POP S9|RIO6_lte|7050Y",
			"TCT (Alcatel)|POP S9|RIO6_lte|ALCATEL A995L",
			"TCT (Alcatel)|POP S9|RIO6_lte|STARXTREM II",
			"TCT (Alcatel)|POP S9|RIO6_lte|Smartphone Android by SFR STARXTREM II",
			"TCT (Alcatel)|PandA_m14|YARIS_55|PandA_m14",
			"TCT (Alcatel)|Pixo 7|PIXO7|I211",
			"TCT (Alcatel)|Pixo 7|PIXO7|I212",
			"TCT (Alcatel)|Pixo 7|PIXO7|I213",
			"TCT (Alcatel)|SMART 4G 5.5 Enterprise|M812|SMART 4G 5.5 Enterprise",
			"TCT (Alcatel)|SOUL 4.5|SOUL45_GSM|5038X",
			"TCT (Alcatel)|Smartphone Android by SFR STARADDICT II|one_touch_996_gsm|Smartphone Android by SFR STARADDICT II",
			"TCT (Alcatel)|Smartphone Android by SFR STARADDICT II|one_touch_996_gsm|Smartphone_Android_by_SFR_STARADDICT_II",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018A",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018D",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018E",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018F",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018M",
			"TCT (Alcatel)|Soul 3.5|SOUL35|4018X",
			"TCT (Alcatel)|Soul 3.5|SOUL35|T4018",
			"TCT (Alcatel)|Soul 4|SOUL4|4035A",
			"TCT (Alcatel)|Soul 4|SOUL4|4035D",
			"TCT (Alcatel)|Soul 4|SOUL4|4035X",
			"TCT (Alcatel)|Soul 4|SOUL4|4035X_Orange",
			"TCT (Alcatel)|Soul 4|SOUL4|4035Y",
			"TCT (Alcatel)|Soul 4|SOUL4|4036E",
			"TCT (Alcatel)|Soul 4.5|SOUL45_GSM|5038A",
			"TCT (Alcatel)|Soul 4.5|SOUL45_GSM|5038E",
			"TCT (Alcatel)|Soul 4.5|SOUL45_GSM|D45",
			"TCT (Alcatel)|T10|T10|one touch T10",
			"TCT (Alcatel)|TAB 7 DUAL CORE|T015|TAB 7 DUAL CORE",
			"TCT (Alcatel)|TAB 7 DUAL CORE|T017|TAB 7 DUAL CORE",
			"TCT (Alcatel)|TCL 736L|Rio5C|TCL-J736L",
			"TCT (Alcatel)|TCL 738M|Rio5C|TCL J738M",
			"TCT (Alcatel)|TCL A506|TCL_A506|TCL A506",
			"TCT (Alcatel)|TCL A865|TCL_A865|TCL A865",
			"TCT (Alcatel)|TCL A988|one_touch_993D_gsm|ALCATEL ONE TOUCH 993D",
			"TCT (Alcatel)|TCL A988|one_touch_993_gsm|ALCATEL ONE TOUCH 993",
			"TCT (Alcatel)|TCL A988|one_touch_993_gsm|MTC_968",
			"TCT (Alcatel)|TCL C995|TCL_C995_cdma|TCL C995",
			"TCT (Alcatel)|TCL D35|SOUL35|TCL D35",
			"TCT (Alcatel)|TCL D40 DUAL|D40|TCL D40 DUAL",
			"TCT (Alcatel)|TCL D662|TCL_D662_cdma|TCL D662",
			"TCT (Alcatel)|TCL D706|EG502|TCL D706",
			"TCT (Alcatel)|TCL D768|EG501|TCL D768",
			"TCT (Alcatel)|TCL D920|TCL_D920|TCL D920",
			"TCT (Alcatel)|TCL D920|TCL_D920|TCL-D920",
			"TCT (Alcatel)|TCL DL750|SOUL4|DL750",
			"TCT (Alcatel)|TCL H900M|HERO2|TCL H900M",
			"TCT (Alcatel)|TCL J210C|TCL_J210C_cdma|TCL J210C",
			"TCT (Alcatel)|TCL J210C|TCL_J210C_cdma|TCL-J210C",
			"TCT (Alcatel)|TCL J300|Megane|ALCATEL ONE TOUCH 5020T",
			"TCT (Alcatel)|TCL J300|Megane|TCL J300",
			"TCT (Alcatel)|TCL J310|MeganeB|TCL J310",
			"TCT (Alcatel)|TCL J320|Iris|TCL J320",
			"TCT (Alcatel)|TCL J320C|Daniel_lite|TCL J320C",
			"TCT (Alcatel)|TCL J320T|IrisTD|TCL J320T",
			"TCT (Alcatel)|TCL J600T|Iris2TD|TCL J600T",
			"TCT (Alcatel)|TCL J620|md501|TCL J620",
			"TCT (Alcatel)|TCL J630|Iris2PlusUMTS|J630",
			"TCT (Alcatel)|TCL J630T|Iris2PlusTD|TCL J630T",
			"TCT (Alcatel)|TCL J720|YARISXL|TCL J720",
			"TCT (Alcatel)|TCL J726T|YARISXL|TCL J726T",
			"TCT (Alcatel)|TCL J900|Camry2|TCL J900",
			"TCT (Alcatel)|TCL J900T|Camry2_TD|TCL J900T",
			"TCT (Alcatel)|TCL J920|YARIS_55|TCL J920",
			"TCT (Alcatel)|TCL J926T|YARIS_55|TCL J926T",
			"TCT (Alcatel)|TCL J928|Prius|TCL J928",
			"TCT (Alcatel)|TCL J929L|Diablo_LTE|TCL-J929L",
			"TCT (Alcatel)|TCL J938M|RIO55_LTE|TCL J938M",
			"TCT (Alcatel)|TCL M2M|CROSS2|TCL M2M",
			"TCT (Alcatel)|TCL M2U|CROSS2|TCL M2U",
			"TCT (Alcatel)|TCL P301C|Cooper_40|TCL P301C",
			"TCT (Alcatel)|TCL P301M|TCL_P301M|TCL P301M",
			"TCT (Alcatel)|TCL P302C|Cooper40|TCL P302C",
			"TCT (Alcatel)|TCL P331M|TCL_P331M|TCL P331M",
			"TCT (Alcatel)|TCL P360W|Focus5_CU|TCL P360W",
			"TCT (Alcatel)|TCL P606|Rav4_GSM|TCL P600",
			"TCT (Alcatel)|TCL P606|Rav4_GSM|TCL P606",
			"TCT (Alcatel)|TCL P606|Rav4_GSM|TCL P606T",
			"TCT (Alcatel)|TCL P631M|Civic_X|TCL P631M",
			"TCT (Alcatel)|TCL P728M|CROSS_LTE|TCL P728M",
			"TCT (Alcatel)|TCL S300T|TCLS300T|TCL S300T",
			"TCT (Alcatel)|TCL S500|TCL_S500_GSM|TCL S500",
			"TCT (Alcatel)|TCL S520|TCL_S520_GSM|TCL S520",
			"TCT (Alcatel)|TCL S530T|California|TCL S530T",
			"TCT (Alcatel)|TCL S600|TCL_S600_GSM|TCL S600",
			"TCT (Alcatel)|TCL S700|TCLS700|TCL S700",
			"TCT (Alcatel)|TCL S700T|TCLS700T|TCL S700T",
			"TCT (Alcatel)|TCL S720|Cross55|TCL S720",
			"TCT (Alcatel)|TCL S720T|Cross55|TCL S720T",
			"TCT (Alcatel)|TCL S725T|Cross55_TD_Plus|TCL_S725T",
			"TCT (Alcatel)|TCL S820|Diablo|TCL S820",
			"TCT (Alcatel)|TCL S850|DiabloHD|TCL S850",
			"TCT (Alcatel)|TCL S860|Alpha|TCL S860",
			"TCT (Alcatel)|TCL S900|S900|TCL S900",
			"TCT (Alcatel)|TCL S950|DIABLOX|TCL S950",
			"TCT (Alcatel)|TCL S950T|DIABLOX|TCL S950T",
			"TCT (Alcatel)|TCL S960|DIABLOXPLUS|TCL S960",
			"TCT (Alcatel)|TCL S960T|DIABLOXPLUS|TCL S960T",
			"TCT (Alcatel)|TCL Y900|Scribe5HD_GSM|TCL Y900",
			"TCT (Alcatel)|TCL Y910|SCRIBEPRO|TCL Y910",
			"TCT (Alcatel)|TCL Y910T|SCRIBEPRO|TCL Y910T",
			"TCT (Alcatel)|TCL i718M|EOS2|TCL i718M",
			"TCT (Alcatel)|TCL i718M|EOS_Plus|TCL i709M",
			"TCT (Alcatel)|TCL-J320D|Daniel_lite_2S|TCL-J320D",
			"TCT (Alcatel)|TCL-J900C|TCL_J900C|TCL-J900C",
			"TCT (Alcatel)|TCL-P306C|Ford50|TCL-P306C",
			"TCT (Alcatel)|TCL_6110A|Telsa|TCL 6110A",
			"TCT (Alcatel)|TCL_6110A|Telsa|TCL_6110A",
			"TCT (Alcatel)|TCL_J636D|msm7627a_a5y_j636d|TCL_J636D",
			"TCT (Alcatel)|TCL_J706T|Focus_L|TCL_J706T",
			"TCT (Alcatel)|TCL_U980|TCL_U980|TCL_U980",
			"TCT (Alcatel)|Telenor_Smart|one_touch_985_gsm|Telenor_Smart",
			"TCT (Alcatel)|VF 860|Vodafone_Smart_II_gsm|Vodafone Smart II",
			"TCT (Alcatel)|VF685|VF685|VF685",
			"TCT (Alcatel)|Vodafone 785|Vodafone_785|MTC 982",
			"TCT (Alcatel)|Vodafone 785|Vodafone_785|Vodafone 785",
			"TCT (Alcatel)|Vodafone 861|one_touch_922_gsm|ALCATEL ONE TOUCH 922",
			"TCT (Alcatel)|Vodafone 861|one_touch_922_gsm|Vodafone 861",
			"TCT (Alcatel)|Vodafone 985N|Vodafone985N|Vodafone 985N",
			"TCT (Alcatel)|Vodafone Smart III|Vodafone_975|Vodafone 975",
			"TCT (Alcatel)|Vodafone Smart III (with NFC)|Vodafone_975N|Vodafone 975N",
			"TCT (Alcatel)|Vodafone Smart Tab 4|Vodafone_Smart_Tab_4|Vodafone Smart Tab 4",
			"TCT (Alcatel)|Vodafone_Smart_Tab_3G|Vodafone_Smart_Tab_3G|Vodafone Smart Tab 3G",
			"TCT (Alcatel)|Vodafone_Smart_Tab_4G|Vodafone_Smart_Tab_4G|Vodafone Smart Tab 4G",
			"TCT (Alcatel)|X35E|X35E|X35E",
			"TCT (Alcatel)|X50D|X50D|X50D",
			"TCT (Alcatel)|freebit PandA_m14|YARIS_55|freebit PandA_m14",
			"TCT (Alcatel)|one touch 995C+|one_touch_995C_cdma|one touch 995C+",
			"TCT (Alcatel)|one touch D920|one_touch_D920|ALCATEL one touch D920",
			"TCT (Alcatel)|one touch D920|one_touch_D920|one touch D920",
			"TCT (Alcatel)|one_touch_D920_ALIQ|one_touch_D920|one touch D920",
			"TCT (Alcatel)|tmn smarta8|ONE_TOUCH_991D_gsm|moche smart a8",
			"TCT (Alcatel)|tmn smarta8|ONE_TOUCH_991D_gsm|tmn smart a8",
			"TMAX|TMAX TM9S775|MID908A-K|TM9S775",
			"TPS|SMART Sprint|SMARTSprint|SMART Sprint",
			"TSUNAMi|Tablet Tsunami TSTA080D1|MG080D1T|MG080D1T",
			"TUNTUN English|DREAM PAD|delight|ITP-R408W",
			"TUPAD|TU7_58212_18222|TU7_58212|TU7_18222",
			"Taiwan Mobile|Amazing X3|TX3|Amazing X3",
			"TaiwanMobile|A6S|A6S|A6S",
			"TaiwanMobile|Amazing A8|D78|Amazing A8",
			"TaiwanMobile|TaiwanMobile A5C|A5C|A5C",
			"TaiwanMobile|TaiwanMobile A6S|A6S|A6S",
			"Techno|H5S|up09_tecno_h5s|TECNO H5S",
			"Techno|H7S|qp16_tecno_h7s|H7S",
			"Techno|M3|up11_tecno_m3s|TECNO M3S",
			"Techno|P5|up08_tecno_p5s|TECNO P5S",
			"Techno|TECNO-A7S|TECNO-A7S|TECNO-A7S",
			"Tecno|RACE Blot2|Infinix_X454|Infinix X454",
			"Tecno|TECNO-M6S|g335_b1|TECNO-M6S",
			"Tele2|Tele2fon V4|Tele2fon_v4|Tele2fon v4",
			"Tele2|Tele2fon V5|Tele2fon_v5|Tele2fon v5",
			"Teleepoch|Essentiel B SmartTab 1002|T9660R|SmartTAB 1002",
			"Teleepoch|U675|U675|U675",
			"Teleepoch|UMX U680C|U680C|U680C",
			"Telefunken|VP73_Telefunken|Telefunken|VP73_Telefunken",
			"Tesco|Hudl 2|HTF8A4|Hudl 2",
			"Tesco|Hudl HT7S3|ht7s3|Hudl HT7S3",
			"Tesco|Hudl HT7S3|ht7s3|hudl ht7s3",
			"Tesco|W032I_C3|w032i_c3|W032I_C3",
			"Tesco|hudl 2|HTF8A4|Hudl 2",
			"Thomson|THOMSON PRIMO7 Tablet|PRIMO7|PRIMO7",
			"Thomson|THOMSON PRIMO8 Tablet|AMLMY8306P|PRIMO8",
			"Thomson|TLink 455|TLINK455|TLINK455",
			"Tinhvan|TA80TA1|TA80TA1_1|TA80TA1",
			"Tonino Lamborghini|Antares|TL66|TL66",
			"Tonino Lamborghini|Antares|TL_Antares|Antares",
			"Toshiba|A204|tos14ast10|A204",
			"Toshiba|AT10-A/AT15-A (Japan: AT503)|tostab12BL|AT10-A",
			"Toshiba|AT10-A/AT15-A (Japan: AT503)|tostab12BL|AT502",
			"Toshiba|AT10-A/AT15-A (Japan: AT503)|tostab12BL|AT503",
			"Toshiba|AT10-A/AT15-A (Japan: AT503)|tostab12BL|AT503_HEMS",
			"Toshiba|AT100|tostab03|AT100",
			"Toshiba|AT100|tostab03|Tostab03",
			"Toshiba|AT10LE-A/AT15LE-A/AT10PE-A/AT15PE-A (Japan: AT703)|tostab12BA|AT10LE-A",
			"Toshiba|AT10LE-A/AT15LE-A/AT10PE-A/AT15PE-A (Japan: AT703)|tostab12BA|AT10PE-A",
			"Toshiba|AT10LE-A/AT15LE-A/AT10PE-A/AT15PE-A (Japan: AT703)|tostab12BA|AT702",
			"Toshiba|AT10LE-A/AT15LE-A/AT10PE-A/AT15PE-A (Japan: AT703)|tostab12BA|AT703",
			"Toshiba|AT400|tostab12AL|AT300SE",
			"Toshiba|AT400|tostab12AL|AT400",
			"Toshiba|AT7-A|tost7t|AT374",
			"Toshiba|AT7-A|tost7t|AT7-A",
			"Toshiba|AT7-B|tos13t7gt|AT374",
			"Toshiba|AT7-B|tos13t7gt|AT7-B",
			"Toshiba|AT7-C|tos14riy20|A17",
			"Toshiba|AT7-C|tos14riy20|AT7-C",
			"Toshiba|JP:REGZA Tablet AT570  Others:TOSHIBA AT270|tostab11BS|AT270",
			"Toshiba|JP:REGZA Tablet AT570  Others:TOSHIBA AT270|tostab11BS|AT470",
			"Toshiba|JP:REGZA Tablet AT570  Others:TOSHIBA AT270|tostab11BS|AT570",
			"Toshiba|JPN:REGZA Tablet AT500  Other countries:TOSHIBA AT300|tostab11BA|AT300",
			"Toshiba|JPN:REGZA Tablet AT500  Other countries:TOSHIBA AT300|tostab11BA|AT500",
			"Toshiba|JPN:REGZA Tablet AT500  Other countries:TOSHIBA AT300|tostab11BA|AT500a",
			"Toshiba|JPN:REGZA Tablet AT830 Other countries:TOSHIBA AT330|tostab11BT|AT330",
			"Toshiba|JPN:REGZA Tablet AT830 Other countries:TOSHIBA AT330|tostab11BT|AT830",
			"Toshiba|L4300|l4300|l4300",
			"Toshiba|L5450/L5400|tostv14rtk1|l5400",
			"Toshiba|L5450/L5400|tostv14rtk1|l5450",
			"Toshiba|L5450/L5400|tostv14rtk2|l5400",
			"Toshiba|L5450/L5400|tostv14rtk2|l5450",
			"Toshiba|L5450C|l5450|l3453",
			"Toshiba|L5450C|l5450|l5450",
			"Toshiba|L9450|l9450|l9450",
			"Toshiba|TOSHIBA STB10|TOSPASB|TOSPASB",
			"Toshiba|Thrive 7|tostab04|AT1S0",
			"Toshiba|tt300/tt301/tt302|Mozart|Mozart",
			"Toughshield|R200|R200|R200",
			"Toughshield|T700_TABLET|T700_TABLET|T700",
			"TrekStor|TrekStor tolino tab 7|tolino7|tolino tab 7",
			"TrekStor|TrekStor tolino tab 8.9|tolino89|tolino tab 8.9",
			"Trekstor|SurfTab ST70204-3|TREKM7100KLD|SurfTab ST70204-3",
			"Trekstor|SurfTab xintron i 7.0|st70408_4_coho|st70408_4_coho",
			"Trekstor|tolino tab 8|tolino8|tolino tab 8",
			"Trio|TRIO-7.85|astar-y3|TRIO-7.85",
			"Trio|Trio 7.85 vQ|Trio-785-vQ|Trio 7.85 vQ",
			"Trio|Trio AXS 3G|Trio_AXS_3G|Trio AXS 3G",
			"Trio|Trio MINI|Trio-MINI|Trio MINI",
			"Trio|Trio MST711|nuclear-MST711|MST711",
			"Trio|Trio-Stealth G4 7|Trio-Stealth-G4-7|Trio-Stealth-G4-7",
			"Trio|Trio-Stealth-G4-101|Trio-Stealth-G4-101|Trio-Stealth G4 10.1",
			"Trio|Trio-Stealth-G4-7.85|Trio-Stealth-G4-785|Trio-Stealth-G4-7.85",
			"Tunisie Telecom|Star Trail by TT|startrail3_tt|StarTrail TT",
			"Turk Telekom|Turk Telekom E4|722T|E4",
			"U606|Hisense|U606|STARSHINE III",
			"U688|Hisense|U688|HS-U688",
			"VENSTAR|VENSTAR2050|VENSTAR2050|VENSTAR2050",
			"VF|Vodafone Smart mini|Vodafone_875|Vodafone 875",
			"VIBE Z|K910|kitonw|Lenovo K910",
			"VSN|V.35|up11_h1_vsn|V.35",
			"Venturer|i7|RCT6773W22|i7",
			"Verizon Wireless|Ellipsis 7|QMV7A|QMV7A",
			"Verizon Wireless|Ellipsis 7|QMV7A|QMV7B",
			"Verizon Wireless|Ellipsis 7 (QMV7A)|QMV7A|QMV7A",
			"Verizon Wireless|Ellipsis 7 (QMV7B)|QMV7A|QMV7B",
			"Verizon Wireless|Ellipsis 8|QTAQZ3|QTAQZ3",
			"Verssed|FW8977-ED|FW8977-ED|FW8977-ED",
			"Vertu|Aster|alexa|Aster",
			"Vertu|Constellation V|gambit|Constellation V",
			"Vertu|Signature Touch|odin|Signature Touch",
			"Vertu|VERTU Ti|hermione|VERTU Ti",
			"Vestel|VP100+|ephesus|10.1Myros",
			"Vestel|VP100+|ephesus|VP100+",
			"Vestel|VP73|wing-ibt|VP73",
			"Vestel|VP73|wing-ibt|VP73_Hyundai",
			"Vestel|VP73|wing-ibt|VP73_Myros",
			"Vestel|VP73|wing-ibt|VP73_Vox",
			"Vestel|VP73|wing-ibt|VP73_le",
			"Vestel|VP74|wifionly-gms|VP74",
			"Vestel|VP74|wifionly-gms|VP74-Celcus",
			"Vestel|VP74|wifionly-gms|VP74-Finlux",
			"Vestel|VP74|wifionly-gms|VP74-Orava",
			"Vestel|VP74|wifionly-gms|VP74-Telefunken",
			"Vestel|VP74|wifionly-gms|VP74-Vox",
			"Vestel|VSP145M|VSP145M|VSP145M",
			"Vestel|VSP250g|VSP250g|VSP250g",
			"Vestel|VSP250s|VSP250s|VSP250s",
			"Vestel|VSP355g|g55|VSP355g",
			"Vestel|VSP355s|s55|VSP355s",
			"ViewSonic|VSD221|VSD221|VSD221",
			"ViewSonic|VSD231|VSD231-VSA|VSD231",
			"ViewSonic|VSD241 Smart Display|VSD241|VSD241",
			"Vizio|StreamPlayer|VAP430|VAP430",
			"Vulcan|TA10TA2|TA10TA2_1|TA10TA2",
			"Vulcan|TA71CA1|TA71CA1|TA71CA1",
			"Vulcan|TA71CA2|TA71CA2_1|TA71CA2",
			"W.e.|WETAB700DG|WETAB700DG|WETAB700DG",
			"W.e.|WeTab1004B|WeTab1004B|WeTab1004B",
			"WACOM|Cintiq Companion Hybrid 13HD|CintiqCompanionHybrid13HD|Cintiq Companion Hybrid 13HD",
			"WIKO|Barry|s7811|BARRY",
			"WIKO|CINK SLIM 2|s8074|CINK SLIM 2",
			"WIKO|DARKFULL|s9311|DARKFULL",
			"WIKO|DARKFULL|s9311|Dream",
			"WIKO|DARKNIGHT|s9203|DARKNIGHT",
			"WIKO|DARKNIGHT|s9203|X-tremer",
			"WIKO|HIGHWAY|s9320|HIGHWAY",
			"WIKO|IGGY|s7521|Cynus F4",
			"WIKO|IGGY|s7521|Golf",
			"WIKO|IGGY|s7521|IGGY",
			"WeTek|Play|Play|Play",
			"Welgate|WA-U420D|WA-U420D|WA-U420D",
			"Wenu|Kate|Wenu|Kate",
			"Wiko|BLOOM|s4700|BLOOM",
			"Wiko|GETAWAY|s8812|GETAWAY",
			"Wiko|GOA|s3511|GOA",
			"Wiko|HIGHWAY SIGNS|s4750|HIGHWAY SIGNS",
			"Wiko|KITE|l4020|KITE",
			"Wiko|LENNY|s5201ap|LENNY",
			"Wiko|RAINBOW|s5501|RAINBOW",
			"Wiko|RAINBOW 4G|l5503|RAINBOW 4G",
			"Wiko|SLIDE|s8321|SLIDE",
			"Wiko|SUNSET|s4011|SUNSET",
			"Wintec|T730|T730|Clear",
			"Wintec|T730|T730|T730",
			"Wintec|T750|T750|Clear",
			"Wistron|W1011A|w1011a|W1011A",
			"Xiaomi|China|HM2013023|2013023",
			"Xiaomi|China|braveheart|MiTV",
			"Xiaomi|China|casablanca|MiBOX1S",
			"Xiaomi|HM 1SC|armani|HM 1AC",
			"Xiaomi|HM 1SC|armani|HM 1SC",
			"Xiaomi|HM 1SC|armani|HM 1SW",
			"Xiaomi|HM 1SLTETD|HM2014501|2014501",
			"Xiaomi|HM 1STD|HM2014011|2014011",
			"Xiaomi|HM 2LTE-BR|HM2014819|2014819",
			"Xiaomi|HM 2LTE-CMCC|HM2014813|2014813",
			"Xiaomi|HM 2LTE-CT|HM2014812|2014812",
			"Xiaomi|HM 2LTE-CU|HM2014811|2014811",
			"Xiaomi|HM 2LTE-IN|HM2014818|2014818",
			"Xiaomi|HM 2LTE-SA|HM2014817|2014817",
			"Xiaomi|HM NOTE 1LTETD|dior|HM NOTE 1LTE",
			"Xiaomi|HM NOTE 1LTETD|dior|HM NOTE 1LTETD",
			"Xiaomi|HM NOTE 1LTETD|dior|HM NOTE 1LTEW",
			"Xiaomi|HM NOTE 1S CT|gucci|gucci",
			"Xiaomi|HM NOTE 1TD|lcsh92_wet_tdd|HM NOTE 1TD",
			"Xiaomi|HM NOTE 1W|lcsh92_wet_jb9|HM NOTE 1W",
			"Xiaomi|Hong Mi|HM2013022|2013022",
			"Xiaomi|MI 2A|taurus|MI 2A",
			"Xiaomi|MI 2S|aries|MI 2",
			"Xiaomi|MI 2S|aries|MI 2S",
			"Xiaomi|MI 3W|cancro|MI 3W",
			"Xiaomi|MI 4LTE|cancro|MI 4LTE",
			"Xiaomi|MI 4LTE-CT|cancro|MI 4LTE",
			"Xiaomi|MI PAD|mocha|MI PAD",
			"Xiaomi|Mi 3|pisces|MI 3",
			"Xiaomi|MiBOX2|dredd|MiBOX2",
			"Xiaomi|MiTV2|entrapment|MiTV2",
			"Xplore_Technologies|RangerX|iX101T1|iX101T1",
			"Xplore_Technologies|RangerX|iX101T1_2G|iX101T1-2G",
			"Yota Devices|YotaPhone2|yotaphone2|YD201",
			"YotaPhone|YotaPhone C9660|yotaphone|C9660",
			"ZTE|009Z|bladeplus|009Z",
			"ZTE|009Z|bladeplus|ZTE V882",
			"ZTE|009Z|bladeplus|ZTE V886J",
			"ZTE|402ZT|ZTE-402ZT|402ZT",
			"ZTE|A2|A2|A2",
			"ZTE|Amazing A1|roamer2|Amazing A1",
			"ZTE|Amazing P6|msm8226|Amazing P6",
			"ZTE|Amazing X2|msm8226|Amazing X2",
			"ZTE|Amazing p5_Lite|V77|Amazing p5_Lite",
			"ZTE|Android edition by sfr STARADDICT|skate|Android edition by sfr STARADDICT",
			"ZTE|Avea inTouch 2|atlas40|Avea inTouch 2",
			"ZTE|Avea inTouch 4|msm8916_32|Avea inTouch 4",
			"ZTE|Avea inTouch 4|msm8916_32|ZTE Blade V220",
			"ZTE|BASE Lutea 2|skate|BASE Lutea 2",
			"ZTE|BLADE III_IL|atlas40|BLADE III_IL",
			"ZTE|Baker|ZTE-Baker|Baker",
			"ZTE|Beeline E600|roamer2|Beeline E600",
			"ZTE|Blade|blade|003Z",
			"ZTE|Blade|blade|Android Edition StarTrail",
			"ZTE|Blade|blade|BASE lutea",
			"ZTE|Blade|blade|BLADE_N880",
			"ZTE|Blade|blade|Beeline E400",
			"ZTE|Blade|blade|Kyivstar Spark",
			"ZTE|Blade|blade|MF8604",
			"ZTE|Blade|blade|Movistar Prime",
			"ZTE|Blade|blade|N880",
			"ZTE|Blade|blade|Netphone 701",
			"ZTE|Blade|blade|Optimus San Francisco",
			"ZTE|Blade|blade|Orange San Francisco",
			"ZTE|Blade|blade|Orange Tactile internet 2",
			"ZTE|Blade|blade|RTK V8",
			"ZTE|Blade|blade|San Francisco",
			"ZTE|Blade|blade|V8502",
			"ZTE|Blade|blade|WayteQ Libra",
			"ZTE|Blade|blade|XCD35",
			"ZTE|Blade|blade|ZTE Libra",
			"ZTE|Blade|blade|ZTE V880",
			"ZTE|Blade|blade|ZTE-BLADE",
			"ZTE|Blade|blade|ZTE-C N880S",
			"ZTE|Blade|blade|ZTE-LIBRA",
			"ZTE|Blade|blade|ZTE-Libra",
			"ZTE|Blade|blade|ZTE-U V880",
			"ZTE|Blade|blade|a5",
			"ZTE|Blade Apex|coeus|Blade Apex",
			"ZTE|Blade Apex2|msm8226|Blade Apex2",
			"ZTE|Blade G LTE|oceanus|Blade G LTE",
			"ZTE|Blade III|atlas40|Blade III",
			"ZTE|Blade III|atlas40|Skate Pro",
			"ZTE|Blade S|blade2|Blade S",
			"ZTE|Bouygues Telecom Bs 351|nice|Bouygues Telecom Bs 351",
			"ZTE|Bouygues Telecom Bs 402|P172G10|A4C",
			"ZTE|Bouygues Telecom Bs 402|P172G10|Amazing A4C",
			"ZTE|Bouygues Telecom Bs 402|P172G10|Bouygues Telecom Bs 402",
			"ZTE|Bouygues Telecom Bs 402|P172G10|ZTE Blade Q Mini",
			"ZTE|Cosmote Smart Share|nice|Cosmote Smart Share",
			"ZTE|Cosmote Xplore|Cosmote|Cosmote Xplore",
			"ZTE|Cosmote Xplore|Cosmote-Xplore|Cosmote Xplore",
			"ZTE|Dialog Q143L|frosty|Dialog Q143L",
			"ZTE|FT142D|msm8226|FT142D",
			"ZTE|Grand Memo|prindle|Grand Memo",
			"ZTE|Grand S Flex|iris|Grand S Flex",
			"ZTE|Grand X|P175A20|Grand X",
			"ZTE|Grand X|P175A20|Grand X(M)",
			"ZTE|Grand X|P175A20|ZTE Grand X Classic",
			"ZTE|Grand X|P175A20|ZTE V970",
			"ZTE|Grand X|P175A20|ZTE V970M",
			"ZTE|Grand X|P175A20|ZTE-U V970M",
			"ZTE|Grand X|P175A20|tmn smart a18",
			"ZTE|Grand X In|mfld_pr2|Grand X In",
			"ZTE|Grand X In|mfld_pr2|STARADDICT II Plus",
			"ZTE|Grand X In|mfld_pr2|ZTE Grand X In",
			"ZTE|Grand X Pro|P177A20|Blade Super",
			"ZTE|Grand X Pro|P177A20|Grand X Pro",
			"ZTE|Grand X Pro|P177A20|KPN Smart 300",
			"ZTE|Infinity X^2 mini|msm8226|Infinity X^2 mini",
			"ZTE|K97|K97|K97",
			"ZTE|K97|K97|ZTE K97",
			"ZTE|KIS|roamer2|KIS",
			"ZTE|KIS PLUS|roamer2|KIS PLUS",
			"ZTE|KPN Smart 400|msm8226|KPN Smart 400",
			"ZTE|Karbonn_A1|N721|Karbonn_A1",
			"ZTE|Kis T3|roamer2|Kis T3",
			"ZTE|Leopard MF800|roamer2|Leopard MF800",
			"ZTE|Light Tab 2|v9plus|Light Tab 2",
			"ZTE|M8402|roamer2|M8402",
			"ZTE|MEDION LIFE P4310|skate|MEDION LIFE P4310",
			"ZTE|MEDION Smartphone LIFE E3501|roamer2|MEDION Smartphone LIFE E3501",
			"ZTE|MF97A|MF97A|MF97A",
			"ZTE|MF97W|MF97W|MF97W",
			"ZTE|MS4A|kiska|MS4A",
			"ZTE|MS4A|roundtop|MS4A",
			"ZTE|Momodesign MD Droid|atlas40|Momodesign MD Droid",
			"ZTE|Movistar Express|QB7211|Movistar Express",
			"ZTE|Movistar Motion|roamer2|Movistar Motion",
			"ZTE|N720|N720|N720",
			"ZTE|N720|N720|ZTE-U N720",
			"ZTE|N720|N720|ZTE_U N720",
			"ZTE|N762|roamer|N762",
			"ZTE|N765_APT|N765|N765_APT",
			"ZTE|N790|deepblue|N790",
			"ZTE|N800|nex|N800",
			"ZTE|N8000_USA_Cricket|N8000|N8000_USA_Cricket",
			"ZTE|N8000_USA_RS|N8000|N8000_USA_RS",
			"ZTE|N8000_WHTE_CKT|N8000|N8000_WHTE_CKT",
			"ZTE|N8010_APT|N8010_APT|N8010_APT",
			"ZTE|N810|fluid|N810",
			"ZTE|N8300_Reliance|N8300_Reliance|N8300_Reliance",
			"ZTE|N8303|roamer2|N8303",
			"ZTE|N850|sean|N850",
			"ZTE|N850L|seanplus|N850L",
			"ZTE|N855D|hope|N855D",
			"ZTE|N860|arthur|N860",
			"ZTE|N860|arthur|Warp",
			"ZTE|N861|warp2|N861",
			"ZTE|N910|arthur4g|ZTE-N910",
			"ZTE|N9100|hayes|N9100",
			"ZTE|N9101|apollo|N9101",
			"ZTE|N9130|speed|N9130",
			"ZTE|N9180|N9180|N9180",
			"ZTE|N918St|N918St|N918St",
			"ZTE|N918St|N918St|ZTE N918St",
			"ZTE|N9500|gordon|N9500",
			"ZTE|N9510|warplte|N9510",
			"ZTE|N9515|warp4|N9515",
			"ZTE|N9516|eridani|N9516",
			"ZTE|N9520|stormer|N9520",
			"ZTE|N958St|N958St|N958St",
			"ZTE|N9810|quantum|N9810",
			"ZTE|N986D|P188F10|N986+",
			"ZTE|N986D|P188F10|N986D",
			"ZTE|NE501J|NE501J|NE501J",
			"ZTE|NE501J|NE501J|V5",
			"ZTE|NE501J|NE501J|ZTE_V5",
			"ZTE|NX402|NX40X|NX402",
			"ZTE|NX403A|NX403A|NX403A",
			"ZTE|NX404H|NX404H|NX404H",
			"ZTE|NX405H|NX405H|NX405H",
			"ZTE|NX406E|NX406E|NX406E",
			"ZTE|NX40X|NX402|NX402",
			"ZTE|NX40X_APT|NX40X|NX40X",
			"ZTE|NX40X_APT|NX40X|NX40X_APT",
			"ZTE|NX501|NX501|NX501",
			"ZTE|NX501_APT|NX501|NX501_APT",
			"ZTE|NX503A_Z4|NX503A_Z4|NX503A_Z4",
			"ZTE|NX503J|NX503J|NX503J",
			"ZTE|NX505J|NX505J|NX505J",
			"ZTE|NX506J|NX506J|NX506J",
			"ZTE|NX507J|NX507J|NX507H",
			"ZTE|NX507J|NX507J|NX507J",
			"ZTE|NX601J|NX601J|NX601J",
			"ZTE|Optimus Barcelona|blade2|Optimus Barcelona",
			"ZTE|Orange Hi 4G|Orange_Hi_4G|Orange Hi 4G",
			"ZTE|Orange Hi 4G|msm8226|Orange Hi 4G",
			"ZTE|Orange Monte Carlo|skate|Orange Monte Carlo",
			"ZTE|Orange Novi|oceanus|Orange Novi",
			"ZTE|Orange Reyo|P172D10|Avea inTouch 3 Large",
			"ZTE|Orange Reyo|P172D10|Blade Q Maxi",
			"ZTE|Orange Reyo|P172D10|Orange Reyo",
			"ZTE|Orange Reyo|P172D10|ZTE Blade Q Maxi",
			"ZTE|Orange Rono|Vec4G|Orange Rono",
			"ZTE|Orange Rono|msm8226|Orange Rono",
			"ZTE|Orange Zali|nice|Orange Zali",
			"ZTE|RACER III mini|roamer2|RACER III mini",
			"ZTE|Racer|mooncake|Carl",
			"ZTE|Racer|mooncake|MTC 916",
			"ZTE|Racer|mooncake|MTS-SP100",
			"ZTE|Racer|mooncake|Movistar Link",
			"ZTE|Racer|mooncake|RTK D1",
			"ZTE|Racer|mooncake|Racer",
			"ZTE|Racer|mooncake|TaiWan Mobile T2",
			"ZTE|Racer|mooncake|V8402",
			"ZTE|Racer|mooncake|Vip Droid",
			"ZTE|Racer|mooncake|XCD 28",
			"ZTE|Racer|mooncake|ZTE X850",
			"ZTE|Racer|mooncake|ZTE-C N600",
			"ZTE|Racer|mooncake|ZTE-C N600+",
			"ZTE|Racer|mooncake|ZTE-LINK",
			"ZTE|Racer|mooncake|ZTE-RACER",
			"ZTE|Racer|mooncake|ZTE-U V852",
			"ZTE|Racer|mooncake|ZTE-U X850",
			"ZTE|S8Q|MT8123|S8Q",
			"ZTE|STARNAUTE II|V6600|STARNAUTE II",
			"ZTE|STARTRAIL 4|P172E10|Avea inTouch 3",
			"ZTE|STARTRAIL 4|P172E10|STARTRAIL 4",
			"ZTE|STARTRAIL 4|P172E10|ZTE Blade Q",
			"ZTE|STARXTREM|prindle|STARXTREM",
			"ZTE|Skate  Pro|atlas40|Skate Pro",
			"ZTE|SmartTab10|V11A|SmartTab10",
			"ZTE|SmartTab7|V71A|SmartTab7",
			"ZTE|Soshphone 4G|Vec4G|Soshphone 4G",
			"ZTE|Soshphone 4G|msm8226|Soshphone 4G",
			"ZTE|Spro 2|msm8974|MF97V",
			"ZTE|T-Mobile Vivacity|blade2|T-Mobile Vivacity",
			"ZTE|T3|skate|T3",
			"ZTE|TQ150|QB7211|TQ150",
			"ZTE|TURKCELL T40|TURKCELL-T40|TURKCELL T40",
			"ZTE|TURKCELL T50|msm8226|TURKCELL T50",
			"ZTE|TURKCELL T50|msm8226|TURKCELL TURBO T50",
			"ZTE|Telenor Touch Plus|blade2|Telenor Touch Plus",
			"ZTE|Turkcell Maxi Plus 5|nice|Turkcell Maxi Plus 5",
			"ZTE|U880|U880|ZTE-T U880",
			"ZTE|U9180|U9180|U9180",
			"ZTE|U9180|U9180|ZTE U9180",
			"ZTE|UZTE V790|roamer2|UZTE V790",
			"ZTE|UZTE V817|ZTE-V956|UZTE V817",
			"ZTE|Ultym 5.2|msm8916_32|Ultym 5.2",
			"ZTE|V55|V55|V55",
			"ZTE|V72C|v72c|V72C",
			"ZTE|V72M|eos|V72M",
			"ZTE|V8000_USA_Cricket|V8000|V8000_USA_Cricket",
			"ZTE|V8110|V8110|V8110",
			"ZTE|V8200plus|V8200plus|V8200plus",
			"ZTE|V8285|MT8382|V8285",
			"ZTE|V8403|roamer2|V8403",
			"ZTE|V8514|P821E10|V8514",
			"ZTE|V865M|hct77_jb|V865M",
			"ZTE|V9|v9|A100",
			"ZTE|V9|v9|BASE Tab",
			"ZTE|V9|v9|BLACK 03",
			"ZTE|V9|v9|Beeline M2",
			"ZTE|V9|v9|Light",
			"ZTE|V9|v9|Light Tab",
			"ZTE|V9|v9|MTC 1055",
			"ZTE|V9|v9|One Pad",
			"ZTE|V9|v9|RTK V9",
			"ZTE|V9|v9|TO101",
			"ZTE|V9|v9|TT101",
			"ZTE|V9|v9|V9",
			"ZTE|V9|v9|V9C",
			"ZTE|V9|v9|V9c",
			"ZTE|V9|v9|V9e",
			"ZTE|V9|v9|myPad P2",
			"ZTE|V9180|V9180|V9180",
			"ZTE|V96A|V96A|V96A",
			"ZTE|V9A|v9plus|Light Tab 2",
			"ZTE|V9A|v9plus|V9A",
			"ZTE|V9A|v9plus|ZTE V9A",
			"ZTE|V9A|v9plus|my Pad P3",
			"ZTE|V9A|v9plus|myPad P3",
			"ZTE|V9S|V9S|V9S",
			"ZTE|V9S|V9S|myPadP4",
			"ZTE|Vip Racer III|atlas40|Vip Racer III",
			"ZTE|Vodafone Smart Chat|turies|Vodafone Smart Chat",
			"ZTE|X500|X500|X500",
			"ZTE|X501_USA_Cricket|X501|X501_USA_Cricket",
			"ZTE|X501_USA_Cricket|X501|X501_USA_RS",
			"ZTE|X501_USA_OM|X501|X501_USA_OM",
			"ZTE|X9180|X9180|X9180",
			"ZTE|Z660G|nice|Z660G",
			"ZTE|Z665C|seanplus|Z665C",
			"ZTE|Z667|demi|Z667",
			"ZTE|Z667T|bonnie|Z667T",
			"ZTE|Z730|ada|Z730",
			"ZTE|Z740|metis|Z740",
			"ZTE|Z740G|metis|Z740G",
			"ZTE|Z750C|Z750C|Z750C",
			"ZTE|Z752C|eros|Z752C",
			"ZTE|Z753G|faerie|Z753G",
			"ZTE|Z768G|Z768G|Z768G",
			"ZTE|Z777|betty|Z777",
			"ZTE|Z787|apus|Z787",
			"ZTE|Z792|giant|Z792",
			"ZTE|Z795G|ZTE-Z795G|Z795G",
			"ZTE|Z796C|Z796C|Z796C",
			"ZTE|Z797C|carol|Z797C",
			"ZTE|Z826|herculis|Z826",
			"ZTE|Z830|gruis|Z830",
			"ZTE|Z930L|coeus|Z930L",
			"ZTE|Z933|glaucus|Z933",
			"ZTE|Z936L|gift|Z936L",
			"ZTE|Z970|draconis|Z970",
			"ZTE|Z980L|cygni|Z980L",
			"ZTE|Z987|cygni|Z987",
			"ZTE|Z992|aviva|Z992",
			"ZTE|Z993|aviva|Z993",
			"ZTE|Z995|becky|Z995",
			"ZTE|Z998|coeus|Z998",
			"ZTE|ZTE A880|ZTE-A880|ZTE A880",
			"ZTE|ZTE B790|roamer2|ZTE B790",
			"ZTE|ZTE BLADE III|atlas40|Skate Pro",
			"ZTE|ZTE BLADE III|atlas40|ZTE BLADE III",
			"ZTE|ZTE BLADE III|atlas40|ZTE Blade III",
			"ZTE|ZTE Blade Apex|oceanus|ZTE Blade Apex",
			"ZTE|ZTE Blade Apex2|msm8226|ZTE Blade Apex2",
			"ZTE|ZTE Blade C370|ZTE_Blade_C370|ZTE Blade C370",
			"ZTE|ZTE Blade G|ZTE-Blade-G|ZTE Blade G",
			"ZTE|ZTE Blade G LTE|coeus|ZTE Blade G LTE",
			"ZTE|ZTE Blade G Lux|P172F10|Blade G Lux",
			"ZTE|ZTE Blade G Lux|P172F10|DIGICEL DL800",
			"ZTE|ZTE Blade G Lux|P172F10|MEO Smart A40",
			"ZTE|ZTE Blade G Lux|P172F10|Orange Tado",
			"ZTE|ZTE Blade G Lux|P172F10|ZTE Blade G Lux",
			"ZTE|ZTE Blade G Lux|P172F10|ZTE Kis3 max",
			"ZTE|ZTE Blade G Lux|P172F10|ZTE V830W",
			"ZTE|ZTE Blade HN|ZTE_Blade_HN|ZTE Blade HN",
			"ZTE|ZTE Blade III|atlas40|ZTE Blade III",
			"ZTE|ZTE Blade III Pro|Blade-III-Pro|ZTE Blade III Pro",
			"ZTE|ZTE Blade L2|P182A10|BGH Joy Smart AXS",
			"ZTE|ZTE Blade L2|P182A10|BGH Joy Smart AXS D",
			"ZTE|ZTE Blade L2|P182A10|Blade L2",
			"ZTE|ZTE Blade L2|P182A10|MEO Smart A75",
			"ZTE|ZTE Blade L2|P182A10|ZTE Blade L2",
			"ZTE|ZTE Blade L3|P182A20|Blade L3",
			"ZTE|ZTE Blade L3|P182A20|ZTE Blade L3",
			"ZTE|ZTE Blade Q Lux|P632A10|Beeline Pro",
			"ZTE|ZTE Blade Q Lux|P632A10|ZTE Blade A430",
			"ZTE|ZTE Blade Q Lux|P632A10|ZTE Blade Q Lux",
			"ZTE|ZTE Blade Q Lux|P632A10|ZTE Fit 4G Smart",
			"ZTE|ZTE Blade Q Mini|P172G10_UK_VIRGIN|ZTE Blade Q Mini",
			"ZTE|ZTE Blade V|ZTE-Blade-V|Vodafone Blade V",
			"ZTE|ZTE Blade V|ZTE-Blade-V|ZTE Blade V",
			"ZTE|ZTE Blade V2|msm8916_32|ZTE Blade V2",
			"ZTE|ZTE Blade VEC|P692S20_Q82|ZTE Blade VEC",
			"ZTE|ZTE Blade Vec|P692S20_Q82|Blade Vec",
			"ZTE|ZTE Blade Vec|P692S20_Q82|ZTE Blade Vec",
			"ZTE|ZTE Blade Vec|P692S20_Q82|ZTE Geek 2",
			"ZTE|ZTE Blade Vec 4G|msm8226|ZTE Blade Vec 4G",
			"ZTE|ZTE Blade Vec Pro|P692S20_M92|ZTE Blade Vec Pro",
			"ZTE|ZTE C310|P172D04|ZTE C310",
			"ZTE|ZTE G601U|P172E02|ZTE G601U",
			"ZTE|ZTE G717C|P692N30|ZTE G717C",
			"ZTE|ZTE G718C|ZTE-G718C|ZTE G718C",
			"ZTE|ZTE G719C|ZTE-G719C|ZTE G719C",
			"ZTE|ZTE G720C|ZTE-G720C|ZTE G720C",
			"ZTE|ZTE G720T|P839T30|ZTE G720T",
			"ZTE|ZTE G720T|V_P839T30|ZTE G720T",
			"ZTE|ZTE GEEK II 4G|msm8226|ZTE GEEK II 4G",
			"ZTE|ZTE GEEK II Pro|ztexasp92_wet_jb9|ZTE GEEK II Pro",
			"ZTE|ZTE GEEK II Pro|ztexasp92_wet_jb9|ZTE M1001",
			"ZTE|ZTE Geek|redhookbay|ZTE Geek",
			"ZTE|ZTE Geek 2 LTE|msm8226|ZTE Geek 2 LTE",
			"ZTE|ZTE Geek 2 pro|P692S20_M92|ZTE Geek 2 pro",
			"ZTE|ZTE Grand Era|enterprise_RU|ZTE Grand Era",
			"ZTE|ZTE Grand Memo LTE|Grand-Memo|ZTE Grand Memo",
			"ZTE|ZTE Grand Memo LTE|Grand-Memo|ZTE Grand Memo LTE",
			"ZTE|ZTE Grand S Flex|iris|ZTE Grand S Flex",
			"ZTE|ZTE Grand S II|P541T50|ZTE Grand S II",
			"ZTE|ZTE Grand S II|P541T50|ZTE S221",
			"ZTE|ZTE Grand S II|P897A21|ZTE Grand S II",
			"ZTE|ZTE Grand S II LTE|msm8974|ZTE Grand S II LTE",
			"ZTE|ZTE Grand X 2|P682F06|Amazing A7",
			"ZTE|ZTE Grand X 2|P682F06|ZTE Grand X 2",
			"ZTE|ZTE Grand X 2|P682F06|ZTE V968",
			"ZTE|ZTE Grand X 2|P682F06|ZTE V969",
			"ZTE|ZTE Grand X Quad Lite|P188F07|Amazing A6",
			"ZTE|ZTE Grand X Quad Lite|P188F07|V8602",
			"ZTE|ZTE Grand X Quad Lite|P188F07|ZTE Grand X Quad Lite",
			"ZTE|ZTE Grand X Quad Lite|P188F07|ZTE Grand X2",
			"ZTE|ZTE Grand X Quad Lite|P188F07|ZTE Skate 2",
			"ZTE|ZTE Grand X Quad Lite|P188F07|ZTE V967S",
			"ZTE|ZTE Grand X2 In|redhookbay|Grand X2 In",
			"ZTE|ZTE Grand X2 In|redhookbay|ZTE Grand X2 In",
			"ZTE|ZTE Kis 3|ZTE-P821E10|MEO SMART A16",
			"ZTE|ZTE Kis 3|ZTE-P821E10|MEO Smart A16",
			"ZTE|ZTE Kis 3|ZTE-P821E10|MOCHE SMART A16",
			"ZTE|ZTE Kis 3|ZTE-P821E10|ZTE Kis 3",
			"ZTE|ZTE Kis 3|ZTE-P821E10|ZTE V811",
			"ZTE|ZTE Kis Lite|roamer2|ZTE Kis Lite",
			"ZTE|ZTE Kis Pro|nice|Optimus Zali",
			"ZTE|ZTE Kis Pro|nice|ZTE Kis Pro",
			"ZTE|ZTE Kis Q|P172D02|ZTE Kis Q",
			"ZTE|ZTE LEO M1|V883M|V883M",
			"ZTE|ZTE LEO M1|V883M|ZTE LEO M1",
			"ZTE|ZTE LEO S1|V972M|V972M",
			"ZTE|ZTE LEO S1|V972M|ZTE LEO S1",
			"ZTE|ZTE LEO S2|V982M_Z64|ZTE LEO S2",
			"ZTE|ZTE M901C|P692N60|ZTE M901C",
			"ZTE|ZTE N5L|prmthus|ZTE N5L",
			"ZTE|ZTE N5S|P189F10|ZTE N5S",
			"ZTE|ZTE N795|deepblue|ZTE N795",
			"ZTE|ZTE N798|atlas40|ZTE N798",
			"ZTE|ZTE N798+|ZTE_N798|ZTE N798+",
			"ZTE|ZTE N798+|ZTE_N798P|ZTE N798+",
			"ZTE|ZTE N799D|ZTE-N799D|N799D",
			"ZTE|ZTE N799D|ZTE-N799D|ZTE N799D",
			"ZTE|ZTE N818|ZTE-N818|N818",
			"ZTE|ZTE N818|ZTE-N818|ZTE N818",
			"ZTE|ZTE N880E|atlas40|ZTE N880E",
			"ZTE|ZTE N880G|ZTE-N880G|ZTE N880G",
			"ZTE|ZTE N900|ZTE-N900|ZTE N900",
			"ZTE|ZTE N900D|ZTE-N900D|ZTE N900D",
			"ZTE|ZTE N909|ZTE-N909|ZTE N909",
			"ZTE|ZTE N909D|ZTE-N909D|ZTE N909D",
			"ZTE|ZTE N9120|elden|ZTE N9120",
			"ZTE|ZTE N919|ZTE-N919|ZTE N919",
			"ZTE|ZTE N919D|ZTE-N919D|N919D",
			"ZTE|ZTE N919D|ZTE-N919D|ZTE N919D",
			"ZTE|ZTE N983|P177F03|ZTE N983",
			"ZTE|ZTE N9835|chovar|N9835",
			"ZTE|ZTE N9835|chovar|ZTE N9835",
			"ZTE|ZTE N986|P188F02|N986",
			"ZTE|ZTE N986|P188F02|ZTE N986",
			"ZTE|ZTE Q101T|P810T10|ZTE Q101T",
			"ZTE|ZTE Q201T|P172T24|ZTE Q201T",
			"ZTE|ZTE Q2S-C|P826N50|ZTE Q2S-C",
			"ZTE|ZTE Q2S-T|P826T50|ZTE Q2S-T",
			"ZTE|ZTE Q301C|APT_TW_P810N30|ZTE Q301C",
			"ZTE|ZTE Q301C|P810N30|ZTE Q301C",
			"ZTE|ZTE Q301T|P172T33|ZTE Q301T",
			"ZTE|ZTE Q501T|P172T31|ZTE Q501T",
			"ZTE|ZTE Q501U|P172F04|ZTE Q501U",
			"ZTE|ZTE Q503U|P682V53|ZTE Q503U",
			"ZTE|ZTE Q505T|P826T20|ZTE Q505T",
			"ZTE|ZTE Q507T|P682T20|ZTE Q507T",
			"ZTE|ZTE Q508U|P816V50|ZTE Q508U",
			"ZTE|ZTE Q509T|P632T31|Amazing X5",
			"ZTE|ZTE Q509T|P632T31|ZTE Q509T",
			"ZTE|ZTE Q7|CMCC_P839V50|ZTE Q7",
			"ZTE|ZTE Q7|P839V50|ZTE Q7",
			"ZTE|ZTE Q705U|P682V60|ZTE Q705U",
			"ZTE|ZTE Q801T|P181L30|ZTE Q801T",
			"ZTE|ZTE Q801U|P826V30|ZTE Q801U",
			"ZTE|ZTE Q802C|ZTE-Q802C|ZTE Q802C",
			"ZTE|ZTE Q802D|ZTE-Q802D|ZTE Q802D",
			"ZTE|ZTE Q802T|P826T30|ZTE Q802T",
			"ZTE|ZTE Q805T|P120T55|ZTE Q805T",
			"ZTE|ZTE R83|demeter|ZTE R83",
			"ZTE|ZTE S118|P189S10|Grand S Lite",
			"ZTE|ZTE S118|P189S10|ZTE S118",
			"ZTE|ZTE S2004|msm8974|ZTE S2004",
			"ZTE|ZTE S2005|msm8974|ZTE S2005",
			"ZTE|ZTE S2014|ZTE-S2014|ZTE S2014",
			"ZTE|ZTE STAR|msm8226|ZTE GEEK II Pro 4G",
			"ZTE|ZTE STAR|msm8226|ZTE S2002",
			"ZTE|ZTE STAR|msm8226|ZTE STAR",
			"ZTE|ZTE STAR|msm8226|ZTE Star 1",
			"ZTE|ZTE Skate|skate|ZTE Skate",
			"ZTE|ZTE Switch X1|deepblue|ZTE Switch X1",
			"ZTE|ZTE Switch X1|deepblue|ZTE V796",
			"ZTE|ZTE T12|P173A30T|ZTE T12",
			"ZTE|ZTE T28|bluetick|ZTE T28",
			"ZTE|ZTE T28|bluetick|ZTE T28 Prepaid",
			"ZTE|ZTE T60|turies|ZTE T60",
			"ZTE|ZTE T760|P173A10T|ZTE T760",
			"ZTE|ZTE T790|roamer2|ZTE T790",
			"ZTE|ZTE T792|ZTE-T792|ZTE T792",
			"ZTE|ZTE T80|msm8226|ZTE T80",
			"ZTE|ZTE T81|iliamna|ZTE T81",
			"ZTE|ZTE T82|frosty|ZTE T82",
			"ZTE|ZTE T83|demeter|ZTE T83",
			"ZTE|ZTE T86|msm8226|Amazing X1",
			"ZTE|ZTE T86|msm8226|ZTE T86",
			"ZTE|ZTE T88|msm8226|ZTE T88",
			"ZTE|ZTE U5S|P188T51|ZTE U5S",
			"ZTE|ZTE U809|P172T11|ZTE U809",
			"ZTE|ZTE U816|P988T20|ZTE U816",
			"ZTE|ZTE U817|P117A20|ZTE U817",
			"ZTE|ZTE U818|P172T21|ZTE U818",
			"ZTE|ZTE U879|P172T30|ZTE U879",
			"ZTE|ZTE U889|P183T30|ZTE U889",
			"ZTE|ZTE U968|P682T51|ZTE U968",
			"ZTE|ZTE U969|P682T50|ZTE U969",
			"ZTE|ZTE U9815|P945T20|ZTE U9815",
			"ZTE|ZTE U988S|pluto|ZTE U988S",
			"ZTE|ZTE V5S|P189F13|ZTE Grand Memo lite",
			"ZTE|ZTE V5S|P189F13|ZTE V5S",
			"ZTE|ZTE V70|ZTE-V70|ZTE V70",
			"ZTE|ZTE V7073|V7073|ZTE V7073",
			"ZTE|ZTE V72|V72|MT7A",
			"ZTE|ZTE V72|V72|ZTE V72",
			"ZTE|ZTE V72|V72|myPad P4 Lite",
			"ZTE|ZTE V768|V768|ZTE V768",
			"ZTE|ZTE V769M|V769M|V769M",
			"ZTE|ZTE V769M|V769M|ZTE LEO Q2",
			"ZTE|ZTE V769M|V769M|ZTE V769M",
			"ZTE|ZTE V779M|ZTEV779M|ZTE V779M",
			"ZTE|ZTE V790|roamer2|ZTE V790",
			"ZTE|ZTE V791|P175A40|V791",
			"ZTE|ZTE V791|P175A40|ZTE V791",
			"ZTE|ZTE V792C|P810D01|ZTE V792C",
			"ZTE|ZTE V793|P175A60|Amazing A3",
			"ZTE|ZTE V793|P175A60|Cellcom 4G",
			"ZTE|ZTE V793|P175A60|M9000",
			"ZTE|ZTE V793|P175A60|Telcel T20",
			"ZTE|ZTE V793|P175A60|ZTE KIS Flex",
			"ZTE|ZTE V793|P175A60|ZTE V793",
			"ZTE|ZTE V793|P175A60|tmn smart a6",
			"ZTE|ZTE V795|P172D01|V795(A3S)",
			"ZTE|ZTE V795|P172D01|ZTE B795",
			"ZTE|ZTE V795|P172D01|ZTE KIS II",
			"ZTE|ZTE V795|P172D01|ZTE KIS II PRO",
			"ZTE|ZTE V795|P172D01|ZTE Kis II",
			"ZTE|ZTE V795|P172D01|ZTE V795",
			"ZTE|ZTE V797|P172D03|VIETTEL V8411",
			"ZTE|ZTE V797|P172D03|ZTE V797",
			"ZTE|ZTE V807|P177E01|Amazing A4",
			"ZTE|ZTE V807|P177E01|Beeline E700",
			"ZTE|ZTE V807|P177E01|Leopard MF900",
			"ZTE|ZTE V807|P177E01|UZTE V807",
			"ZTE|ZTE V807|P177E01|V8501",
			"ZTE|ZTE V807|P177E01|ZTE BLADE C",
			"ZTE|ZTE V807|P177E01|ZTE V807",
			"ZTE|ZTE V807|P177E01|ZTE V889S",
			"ZTE|ZTE V808|P172E01|UZTE V808",
			"ZTE|ZTE V808|P172E01|ZTE V808",
			"ZTE|ZTE V809|P172A10|ZTE Blade C2",
			"ZTE|ZTE V809|P172A10|ZTE T809",
			"ZTE|ZTE V809|P172A10|ZTE V809",
			"ZTE|ZTE V809|P172A10|meo smart a12",
			"ZTE|ZTE V811|ZTE-V811|Beeline Smart2",
			"ZTE|ZTE V811|ZTE-V811|ZTE V811",
			"ZTE|ZTE V811C|P810E01|ZTE V811C",
			"ZTE|ZTE V811W|P821E10|ZTE V811",
			"ZTE|ZTE V811W|P821E10|ZTE V811W",
			"ZTE|ZTE V813W|P172B20|ZTE Blade C2 Plus",
			"ZTE|ZTE V813W|P172B20|ZTE V813W",
			"ZTE|ZTE V815W|P172R10|Amazing A4S",
			"ZTE|ZTE V815W|P172R10|B8405",
			"ZTE|ZTE V815W|P172R10|KIS II Max",
			"ZTE|ZTE V815W|P172R10|SMART Start",
			"ZTE|ZTE V815W|P172R10|UZTE GRAND V7",
			"ZTE|ZTE V815W|P172R10|ZTE Blade Buzz",
			"ZTE|ZTE V815W|P172R10|ZTE Kis II Max",
			"ZTE|ZTE V815W|P172R10|ZTE Maxx",
			"ZTE|ZTE V815W|P172R10|ZTE T815",
			"ZTE|ZTE V815W|P172R10|ZTE V815W",
			"ZTE|ZTE V815W|P172R10|ZTE V816W",
			"ZTE|ZTE V817|ZTE-V817|V817",
			"ZTE|ZTE V817|ZTE-V817|ZTE V817",
			"ZTE|ZTE V817|ZTE-V956|ZTE V817",
			"ZTE|ZTE V818|P172F01|ZTE Blade 2",
			"ZTE|ZTE V818|P172F01|ZTE V818",
			"ZTE|ZTE V829|P172A30|Blade G Pro",
			"ZTE|ZTE V829|P172A30|V8507",
			"ZTE|ZTE V829|P172A30|ZTE Blade G Plus",
			"ZTE|ZTE V829|P172A30|ZTE Blade G Pro",
			"ZTE|ZTE V829|P172A30|ZTE V829",
			"ZTE|ZTE V879|P172F02|ZTE V879",
			"ZTE|ZTE V880E|atlas40|ZTE V880E",
			"ZTE|ZTE V880G|ZTE-V880G|ZTE V880G",
			"ZTE|ZTE V887|P177A10|ZTE Blade L",
			"ZTE|ZTE V887|P177A10|ZTE V887",
			"ZTE|ZTE V887|P177A10|tmn smart a20",
			"ZTE|ZTE V889D|atlas40|ZTE V889D",
			"ZTE|ZTE V889M|P175A10|UZTE V889M",
			"ZTE|ZTE V889M|P175A10|ZTE V889M",
			"ZTE|ZTE V891|P986E01|ZTE V891",
			"ZTE|ZTE V956|ZTE-V956|ZTE V956",
			"ZTE|ZTE V965|P188F03|ZTE Blade G2",
			"ZTE|ZTE V965|P188F03|ZTE R880H",
			"ZTE|ZTE V965|P188F03|ZTE V880H",
			"ZTE|ZTE V965|P188F03|ZTE V965",
			"ZTE|ZTE V969|P682F01|ZTE V969",
			"ZTE|ZTE V970|whistler|UZTE V970",
			"ZTE|ZTE V970|whistler|ZTE V970",
			"ZTE|ZTE V975|redhookbay|V975",
			"ZTE|ZTE V975|redhookbay|ZTE Geek",
			"ZTE|ZTE V975|redhookbay|ZTE V975",
			"ZTE|ZTE V9800|kiska|ZTE V9800",
			"ZTE|ZTE V9820|ZTE-V9820|ZTE V9820",
			"ZTE|ZTE V983|P177A30|ZTE V983",
			"ZTE|ZTE V987|P188F04|UZTE GRAND X Quad",
			"ZTE|ZTE V987|P188F04|ZTE Grand X",
			"ZTE|ZTE V987|P188F04|ZTE V987",
			"ZTE|ZTE V993W|ZTE_V993W|ZTE V993W",
			"ZTE|ZTE V9A|v9plus|ZTE V9A",
			"ZTE|ZTE Z932L|warplte|Z932L",
			"ZTE|ZTE Z932L|warplte|ZTE Z932L",
			"ZTE|ZTE-BLADE|blade|ZTE-BLADE",
			"ZTE|ZTE-N910|arthur4g|ZTE-N910",
			"ZTE|ZTE-SKATE|skate|ZTE-SKATE",
			"ZTE|ZTE-U N721|N721|ZTE-U N721",
			"ZTE|ZTE-U V760|ztenj73_gb|MD Smart",
			"ZTE|ZTE-U V760|ztenj73_gb|Telenor Touch Mini",
			"ZTE|ZTE-U V760|ztenj73_gb|ZTE-U V760",
			"ZTE|ZTE-U V760|ztenj73_gb|ZTE-U V856",
			"ZTE|ZTE-U V760|ztenj73_gb|ZTE-U V857",
			"ZTE|ZTE-U V760|ztenj73_gb|moii E598",
			"ZTE|ZTE-V6500|V6500|Etisalat Smartphone",
			"ZTE|ZTE-V6500|V6500|ZTE-V6500",
			"ZTE|ZTE-V856|zte_v856|ZTE-V856",
			"ZTE|ZTE-X500|X500|ZTE-X500",
			"ZTE|ZTE-Z990|roamer|ZTE-Z990",
			"ZTE|ZTE-Z990G|roamer|ZTE-Z990G",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|V765M",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|ZTE LEO Q1",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|ZTE V765M",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|ZTE_CLARO_Q1",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|ZTE_LEO_Q1",
			"ZTE|ZTE_LEO_Q1|hct72_wet_jb3|mobifone M9001",
			"ZTE|ZTE_N9511|hera|ZTE_N9511",
			"ZTE|ZTE_Q701C|ZTE-Q701C|ZTE Q701C",
			"ZTE|ZTE_Q801C|ZTE-Q801C|ZTE Q801C",
			"ZTE|ZTE_Q801L|ZTE-Q801L|ZTE Q801L",
			"ZTE|ZXY-ZTE_N8010|N8010_YM|ZXY-ZTE_N8010",
			"ZTE|ZXY-ZTE_V6700|V6700|ZTE V6700",
			"ZTE|ZXY-ZTE_V6700|V6700|ZXY-ZTE_V6700",
			"ZTE|blade S|blade2|Blade S",
			"ZTE|myPad P5|K78|Amazing_P5",
			"ZTE|myPad P5|K78|myPad P5",
			"ZTE|nubia Z5|NX501|nubia Z5",
			"ZTE|tmn smart a15|atlas40|tmn smart a15",
			"ZTE|tmn smart a60|kiska|tmn smart a60",
			"ZTE|tmn smart a7|sailboat|tmn smart a7",
			"Zeki|TBDG734|TBDG734|TBDG734",
			"Zeki|TBQG1084|TBQG1084|TBQG1084",
			"Zeki|TBQG855|TBQG855|TBQG855",
			"Zeki|Zeki TBDG1073|TBDG1073|TBDG1073",
			"Zeki|Zeki TBDG874 Tablet|TBDG874B|TBDG874",
			"Zeki|Zeki TBQC1063B|TBQC1063B|TBQC1063",
			"Zeki|Zeki TBQG774|TBQG774|TBQG774",
			"Zeki|Zeki TBQG884|TBQG884|TBQG884",
			"Zilo|Zilo|Zilo|Zilo",
			"bq|Aquaris E10|Aquaris_E10|Aquaris E10",
			"bq|Aquaris E10 3G|Aquaris_E10_3G|Aquaris E10 3G",
			"bq|Aquaris E4|Aquaris_E4|Aquaris E4",
			"bq|Aquaris E4.5|Aquaris_E45|Aquaris E4.5",
			"bq|Aquaris E5|Aquaris_E5|Aquaris E5",
			"bq|Aquaris E5 FHD|Aquaris_E5_FHD|Aquaris E5 FHD",
			"bq|Aquaris E5 HD|Aquaris_E5_HD|Aquaris E5 HD",
			"bq|Aquaris E6|Aquaris_E6|Aquaris E6",
			"bq|Edison 3|Edison_3|Edison 3",
			"bq|Edison_3_mini|Edison_3_mini|Edison 3 mini",
			"bq|bq Edison 3 3G|Edison_3_3G|Edison 3 3G",
			"essentielb|Essentielb- Black Diamond|bird77_a_cu_ics2|Essentielb-Black Diamond",
			"essentielb|Pyxis|pyxis_boulanger|Essentielb-Pixis",
			"essentielb|essentielb ST7001 Tablet|ST7001|ST7001",
			"geanee|FXC-5A|FXC-5A|FXC-5A",
			"harris|RF-3590-RT|msm8660_cougar|Harris 12131-1000",
			"hisense|lt971|LT971|LT971",
			"i-Buddie|TA10CA1|TA10CA1|TA10CA1",
			"i-Buddie|TA10CA2|TA10CA2|TA10CA2",
			"i-Buddie|TA10CA3|TA10CA3|TA10CA3",
			"i-Buddie|TA10CA4|TA10CA4|TA10CA4",
			"i-Buddie|TA71CA1|TA71CA1|TA71CA1",
			"i-Buddie|TA71CA5|TA71CA5|TA71CA5",
			"i-Buddie|TA80CA1|TA80CA1|TA80CA1",
			"i-Buddie|TM75A-V2|TM75A-V2|TM75A-V2",
			"i-Buddie|TR10CD1|TR10CD1_2|TR10CD1",
			"i-Buddie|TR10CD1|TR10CD1_4|TR10CD1",
			"i-Buddie|TR10CD1_P|TR10CD1|TR10CD1",
			"i-Buddie|TR10CS1|TR10CS1_12|TR10CS1",
			"i-Buddie|TR10CS1|TR10CS1_2|TR10CS1",
			"i-Buddie|TR10CS1_P|TR10CS1|TR10CS1",
			"i-mobile|IQX KEN|IQX_KEN|IQX KEN",
			"i-mobile|i-mobile i-note WiFi 1.1|M1703|i-mobile M1703",
			"iDea USA|CT8|CT8|CT8",
			"iDeaUSA|iDeaUSA CT10|CT10|CT10",
			"iDeaUSA|iDeaUSA CT7|CT7|CT7",
			"iRiver|DMT580D|m805_892x|DDA800R",
			"iRiver|DMT580D|m805_892x|DMT580D",
			"iRiver|DMT580D|m805_892x|MIT700",
			"iRiver|MM3202|mm3201|MM-3201",
			"iRiver|MM3202|mm3201|MM-3202",
			"iRiver|ULALA|msm7627a_sku3|I-K1",
			"iRiver|WOW Tab+(ITQ1000)|itq1000|ITQ1000",
			"iRiver|WPT005|wikipad|WPT005",
			"iRiver|Wow TAB +|itq701|ITQ701",
			"iRiver|Wow(Window of the world)|itq700|ITQ700",
			"iRiver|Wow(Window of the world)|itq700|YBMK01",
			"iView|776TPCIII|776TPCIII|776TPCIII",
			"iView|iView 778TPC|778TPC|778TPC",
			"iView|iView 788TPCII|iView_788TPCII|788TPCII",
			"iView|iView 920TPC|920TPC|920TPC",
			"iView|iView SupraPad i700|g01i|i700",
			"iView|iView SupraPad i785Q|anchor|i785Q",
			"mobicel|Retro|Retro|Retro",
			"mobistel|Cynus T6|Cynus_T6|Cynus T6",
			"myPhone|CUBE|myphone_cube|myPhone Cube",
			"nabi|nabi 2 Tablet|mt799|NABI2-NV7A",
			"nabi|nabi 2S Tablet|nabi2S|SNB02-NV7A",
			"nabi|\"nabi Big Tab HD\\xe2\\x84\\xa2 20\"\"\"|DMTAB-NV20A|DMTAB-NV20A",
			"nabi|\"nabi Big Tab HD\\xe2\\x84\\xa2 24\"\"\"|DMTAB-NV24A|DMTAB-NV24A",
			"nabi|nabi DreamTab HD8 Tablet|DMTAB|DMTAB-IN08A",
			"nabi|nabi DreamTab HD8 Tablet|t8400n|DMTAB-NV08B",
			"oioo|oioo Model 2|wecct|Model 2",
			"oioo|oioo Model 3|wecct3|Model 3",
			"pendo|PNDPP44QC10|PNDPP44QC10|PNDPP44QC10",
			"pendo|PNDPP44QC7|PNDPP44QC7|PNDPP44QC7",
			"sugar_aums|QPOINT|QPI-1|QPI-1",
			"tecmobile|OmnisOne|OmnisOne|Omnis One",
			"ucall|EASY1|EASY1|EASY1" };

	// @formatter:on

	private static String sDeviceName;

	/** Returns the consumer friendly device name */
	public static String getDeviceName() {
		if (sDeviceName != null) {
			return sDeviceName;
		}
		final String key = Build.DEVICE + "|" + Build.MODEL;
		for (final String str : SUPPORTED_DEVICES) {
			if (str.endsWith(key)) {
				final String[] names = str.split("\\|");
				final String retailBranding = names[0];
				final String marketingName = names[1];
				if (marketingName.startsWith(retailBranding)) {
					sDeviceName = marketingName;
				}
				else {
					sDeviceName = retailBranding + " " + marketingName;
				}
				return sDeviceName;
			}
		}
		final String manufacturer = Build.MANUFACTURER;
		final String model = Build.MODEL;
		if (model.startsWith(manufacturer)) {
			sDeviceName = capitalize(model);
		}
		else if (manufacturer.equalsIgnoreCase("HTC")) {
			sDeviceName = "HTC " + model; // make sure HTC is capitalized
		}
		else {
			sDeviceName = capitalize(manufacturer) + " " + model;
		}
		return sDeviceName;
	}

	/**
	 * Returns the consumer friendly device name based on the value of
	 * {@link Build#DEVICE}
	 */
	public static String getDeviceName(final String device) {
		final String key = "|" + device + "|";
		for (final String str : SUPPORTED_DEVICES) {
			if (str.contains(key)) {
				final String[] names = str.split("\\|");
				final String retailBranding = names[0];
				final String marketingName = names[1];
				if (marketingName.startsWith(retailBranding)
						|| marketingName.toLowerCase(Locale.ENGLISH).startsWith("moto ")) {
					return marketingName;
				}
				return retailBranding + " " + marketingName;
			}
		}
		return null;
	}

	/**
	 * <p>
	 * Capitalizes all the whitespace separated words in a String. Only the first
	 * letter of each word is changed. To convert the rest of each word to
	 * lowercase at the same time, use {@link #capitalizeFully(String)}.
	 * </p>
	 * 
	 * <p>
	 * Whitespace is defined by {@link Character#isWhitespace(char)}.
	 * </p>
	 * 
	 * @param str
	 *          the String to capitalize
	 * @return capitalized The capitalized String
	 */
	private static final String capitalize(final String str) {
		if (TextUtils.isEmpty(str)) {
			return str;
		}
		final char[] arr = str.toCharArray();
		boolean capitalizeNext = true;
		String phrase = "";
		for (final char c : arr) {
			if (capitalizeNext && Character.isLetter(c)) {
				phrase += Character.toUpperCase(c);
				capitalizeNext = false;
				continue;
			}
			else if (Character.isWhitespace(c)) {
				capitalizeNext = true;
			}
			phrase += c;
		}
		return phrase;
	}

	private Devices() {

	}

}
