/* global WebSocket */

module.exports = Socket

var Buffer = require('safe-buffer').Buffer
var debug = require('debug')('simple-websocket')
var inherits = require('inherits')
var randombytes = require('randombytes')
var stream = require('readable-stream')
var ws = require('ws') // websockets in node - will be empty object in browser

var _WebSocket = typeof ws !== 'function' ? WebSocket : ws

var MAX_BUFFERED_AMOUNT = 64 * 1024

inherits(Socket, stream.Duplex)

/**
 * WebSocket. Same API as node core `net.Socket`. Duplex stream.
 * @param {Object} opts
 * @param {string=} opts.url websocket server url
 * @param {string=} opts.socket raw websocket instance to wrap
 */
function Socket (opts) {
  var self = this
  if (!(self instanceof Socket)) return new Socket(opts)
  if (!opts) opts = {}

  // Support simple usage: `new Socket(url)`
  if (typeof opts === 'string') {
    opts = { url: opts }
  }

  if (opts.url == null && opts.socket == null) {
    throw new Error('Missing required `url` or `socket` option')
  }
  if (opts.url != null && opts.socket != null) {
    throw new Error('Must specify either `url` or `socket` option, not both')
  }

  self._id = randombytes(4).toString('hex').slice(0, 7)
  self._debug('new websocket: %o', opts)

  opts = Object.assign({
    allowHalfOpen: false
  }, opts)

  stream.Duplex.call(self, opts)

  self.connected = false
  self.destroyed = false

  self._chunk = null
  self._cb = null
  self._interval = null

  if (opts.socket) {
    self.url = opts.socket.url
    self._ws = opts.socket
  } else {
    self.url = opts.url
    try {
      if (typeof ws === 'function') {
        // `ws` package accepts options
        self._ws = new _WebSocket(opts.url, opts)
      } else {
        self._ws = new _WebSocket(opts.url)
      }
    } catch (err) {
      process.nextTick(function () {
        self._destroy(err)
      })
      return
    }
  }

  self._ws.binaryType = 'arraybuffer'
  self._ws.onopen = function () {
    self._onOpen()
  }
  self._ws.onmessage = function (event) {
    self._onMessage(event)
  }
  self._ws.onclose = function () {
    self._onClose()
  }
  self._ws.onerror = function () {
    self._destroy(new Error('connection error to ' + self.url))
  }

  self._onFinishBound = function () {
    self._onFinish()
  }
  self.once('finish', self._onFinishBound)
}

Socket.WEBSOCKET_SUPPORT = !!_WebSocket

/**
 * Send text/binary data to the WebSocket server.
 * @param {TypedArrayView|ArrayBuffer|Buffer|string|Blob|Object} chunk
 */
Socket.prototype.send = function (chunk) {
  this._ws.send(chunk)
}

Socket.prototype.destroy = function (onclose) {
  this._destroy(null, onclose)
}

Socket.prototype._destroy = function (err, onclose) {
  var self = this
  if (self.destroyed) return
  if (onclose) self.once('close', onclose)

  self._debug('destroy (error: %s)', err && (err.message || err))

  self.readable = self.writable = false
  if (!self._readableState.ended) self.push(null)
  if (!self._writableState.finished) self.end()

  self.connected = false
  self.destroyed = true

  clearInterval(self._interval)
  self._interval = null
  self._chunk = null
  self._cb = null

  if (self._onFinishBound) self.removeListener('finish', self._onFinishBound)
  self._onFinishBound = null

  if (self._ws) {
    var ws = self._ws
    var onClose = function () {
      ws.onclose = null
    }
    if (ws.readyState === _WebSocket.CLOSED) {
      onClose()
    } else {
      try {
        ws.onclose = onClose
        ws.close()
      } catch (err) {
        onClose()
      }
    }

    ws.onopen = null
    ws.onmessage = null
    ws.onerror = function () {}
  }
  self._ws = null

  if (err) self.emit('error', err)
  self.emit('close')
}

Socket.prototype._read = function () {}

Socket.prototype._write = function (chunk, encoding, cb) {
  if (this.destroyed) return cb(new Error('cannot write after socket is destroyed'))

  if (this.connected) {
    try {
      this.send(chunk)
    } catch (err) {
      return this._destroy(err)
    }
    if (typeof ws !== 'function' && this._ws.bufferedAmount > MAX_BUFFERED_AMOUNT) {
      this._debug('start backpressure: bufferedAmount %d', this._ws.bufferedAmount)
      this._cb = cb
    } else {
      cb(null)
    }
  } else {
    this._debug('write before connect')
    this._chunk = chunk
    this._cb = cb
  }
}

// When stream finishes writing, close socket. Half open connections are not
// supported.
Socket.prototype._onFinish = function () {
  var self = this
  if (self.destroyed) return

  if (self.connected) {
    destroySoon()
  } else {
    self.once('connect', destroySoon)
  }

  // Wait a bit before destroying so the socket flushes.
  // TODO: is there a more reliable way to accomplish this?
  function destroySoon () {
    setTimeout(function () {
      self._destroy()
    }, 1000)
  }
}

Socket.prototype._onMessage = function (event) {
  if (this.destroyed) return
  var data = event.data
  if (data instanceof ArrayBuffer) data = Buffer.from(data)
  this.push(data)
}

Socket.prototype._onOpen = function () {
  var self = this
  if (self.connected || self.destroyed) return
  self.connected = true

  if (self._chunk) {
    try {
      self.send(self._chunk)
    } catch (err) {
      return self._destroy(err)
    }
    self._chunk = null
    self._debug('sent chunk from "write before connect"')

    var cb = self._cb
    self._cb = null
    cb(null)
  }

  // Backpressure is not implemented in Node.js. The `ws` module has a buggy
  // `bufferedAmount` property. See: https://github.com/websockets/ws/issues/492
  if (typeof ws !== 'function') {
    self._interval = setInterval(function () {
      self._onInterval()
    }, 150)
    if (self._interval.unref) self._interval.unref()
  }

  self._debug('connect')
  self.emit('connect')
}

Socket.prototype._onInterval = function () {
  if (!this._cb || !this._ws || this._ws.bufferedAmount > MAX_BUFFERED_AMOUNT) {
    return
  }
  this._debug('ending backpressure: bufferedAmount %d', this._ws.bufferedAmount)
  var cb = this._cb
  this._cb = null
  cb(null)
}

Socket.prototype._onClose = function () {
  if (this.destroyed) return
  this._debug('on close')
  this._destroy()
}

Socket.prototype._debug = function () {
  var args = [].slice.call(arguments)
  args[0] = '[' + this._id + '] ' + args[0]
  debug.apply(null, args)
}
