'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; };

var _invariant = require('../utils/invariant');

var _invariant2 = _interopRequireDefault(_invariant);

var _getScreenForRouteName = require('./getScreenForRouteName');

var _getScreenForRouteName2 = _interopRequireDefault(_getScreenForRouteName);

var _createConfigGetter = require('./createConfigGetter');

var _createConfigGetter2 = _interopRequireDefault(_createConfigGetter);

var _NavigationActions = require('../NavigationActions');

var _NavigationActions2 = _interopRequireDefault(_NavigationActions);

var _StackActions = require('./StackActions');

var _StackActions2 = _interopRequireDefault(_StackActions);

var _validateRouteConfigMap = require('./validateRouteConfigMap');

var _validateRouteConfigMap2 = _interopRequireDefault(_validateRouteConfigMap);

var _pathUtils = require('./pathUtils');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

var defaultActionCreators = () => ({});

function childrenUpdateWithoutSwitchingIndex(actionType) {
  return [_NavigationActions2.default.SET_PARAMS,
  // Todo: make SwitchRouter not depend on StackActions..
  _StackActions2.default.COMPLETE_TRANSITION].includes(actionType);
}

exports.default = (routeConfigs, config = {}) => {
  // Fail fast on invalid route definitions
  (0, _validateRouteConfigMap2.default)(routeConfigs);

  var order = config.order || Object.keys(routeConfigs);

  var getCustomActionCreators = config.getCustomActionCreators || defaultActionCreators;

  var initialRouteParams = config.initialRouteParams;
  var initialRouteName = config.initialRouteName || order[0];
  var backBehavior = config.backBehavior || 'none';
  var resetOnBlur = config.hasOwnProperty('resetOnBlur') ? config.resetOnBlur : true;

  var initialRouteIndex = order.indexOf(initialRouteName);
  if (initialRouteIndex === -1) {
    throw new Error(`Invalid initialRouteName '${initialRouteName}'.` + `Should be one of ${order.map(n => `"${n}"`).join(', ')}`);
  }

  var childRouters = {};
  order.forEach(routeName => {
    childRouters[routeName] = null;
    var screen = (0, _getScreenForRouteName2.default)(routeConfigs, routeName);
    if (screen.router) {
      childRouters[routeName] = screen.router;
    }
  });

  function getParamsForRoute(routeName, params) {
    var routeConfig = routeConfigs[routeName];
    if (routeConfig && routeConfig.params) {
      return _extends({}, routeConfig.params, params);
    } else {
      return params;
    }
  }

  var {
    getPathAndParamsForRoute,
    getActionForPathAndParams
  } = (0, _pathUtils.createPathParser)(childRouters, routeConfigs, config);

  function resetChildRoute(routeName) {
    var initialParams = routeName === initialRouteName ? initialRouteParams : undefined;
    // note(brentvatne): merging initialRouteParams *on top* of default params
    // on the route seems incorrect but it's consistent with existing behavior
    // in stackrouter
    var params = getParamsForRoute(routeName, initialParams);
    var childRouter = childRouters[routeName];
    if (childRouter) {
      var childAction = _NavigationActions2.default.init();
      return _extends({}, childRouter.getStateForAction(childAction), {
        key: routeName,
        routeName,
        params
      });
    }
    return {
      key: routeName,
      routeName,
      params
    };
  }

  function getNextState(action, prevState, possibleNextState) {
    function updateNextStateHistory(nextState) {
      if (backBehavior !== 'history') {
        return nextState;
      }
      var nextRouteKeyHistory = prevState.routeKeyHistory;
      if (action.type === _NavigationActions2.default.NAVIGATE) {
        nextRouteKeyHistory = [...prevState.routeKeyHistory]; // copy
        var keyToAdd = nextState.routes[nextState.index].key;
        nextRouteKeyHistory = nextRouteKeyHistory.filter(k => k !== keyToAdd); // dedup
        nextRouteKeyHistory.push(keyToAdd);
      } else if (action.type === _NavigationActions2.default.BACK) {
        nextRouteKeyHistory = [...prevState.routeKeyHistory]; // copy
        nextRouteKeyHistory.pop();
      }
      return _extends({}, nextState, {
        routeKeyHistory: nextRouteKeyHistory
      });
    }

    var nextState = possibleNextState;
    if (prevState && prevState.index !== possibleNextState.index && resetOnBlur) {
      var prevRouteName = prevState.routes[prevState.index].routeName;
      var nextRoutes = [...possibleNextState.routes];
      nextRoutes[prevState.index] = resetChildRoute(prevRouteName);
      nextState = _extends({}, possibleNextState, {
        routes: nextRoutes
      });
    }
    return updateNextStateHistory(nextState);
  }

  function getInitialState() {
    var routes = order.map(resetChildRoute);
    var initialState = {
      routes,
      index: initialRouteIndex,
      isTransitioning: false
    };
    if (backBehavior === 'history') {
      var initialKey = routes[initialRouteIndex].key;
      initialState['routeKeyHistory'] = [initialKey];
    }
    return initialState;
  }

  return {
    childRouters,

    getActionCreators(route, stateKey) {
      return getCustomActionCreators(route, stateKey);
    },

    getStateForAction(action, inputState) {
      var prevState = inputState ? _extends({}, inputState) : inputState;
      var state = inputState || getInitialState();
      var activeChildIndex = state.index;

      if (action.type === _NavigationActions2.default.INIT) {
        // NOTE(brentvatne): this seems weird... why are we merging these
        // params into child routes?
        // ---------------------------------------------------------------
        // Merge any params from the action into all the child routes
        var { params } = action;
        if (params) {
          state.routes = state.routes.map(route => _extends({}, route, {
            params: _extends({}, route.params, params, route.routeName === initialRouteName ? initialRouteParams : null)
          }));
        }
      }

      // Let the current child handle it
      var activeChildLastState = state.routes[state.index];
      var activeChildRouter = childRouters[order[state.index]];
      if (activeChildRouter) {
        var activeChildState = activeChildRouter.getStateForAction(action, activeChildLastState);
        if (!activeChildState && inputState) {
          return null;
        }
        if (activeChildState && activeChildState !== activeChildLastState) {
          var _routes = [...state.routes];
          _routes[state.index] = activeChildState;
          return getNextState(action, prevState, _extends({}, state, {
            routes: _routes
          }));
        }
      }

      // Handle tab changing. Do this after letting the current tab try to
      // handle the action, to allow inner children to change first
      var isBackEligible = action.key == null || action.key === activeChildLastState.key;
      if (action.type === _NavigationActions2.default.BACK) {
        if (isBackEligible && backBehavior === 'initialRoute') {
          activeChildIndex = initialRouteIndex;
        } else if (isBackEligible && backBehavior === 'order') {
          activeChildIndex = Math.max(0, activeChildIndex - 1);
        }
        // The history contains current route, so we can only go back
        // if there is more than one item in the history
        else if (isBackEligible && backBehavior === 'history' && state.routeKeyHistory.length > 1) {
            var routeKey = state.routeKeyHistory[state.routeKeyHistory.length - 2];
            activeChildIndex = order.indexOf(routeKey);
          } else {
            return state;
          }
      }

      var didNavigate = false;
      if (action.type === _NavigationActions2.default.NAVIGATE) {
        didNavigate = !!order.find((childId, i) => {
          if (childId === action.routeName) {
            activeChildIndex = i;
            return true;
          }
          return false;
        });
        if (didNavigate) {
          var childState = state.routes[activeChildIndex];
          var childRouter = childRouters[action.routeName];
          var newChildState = childState;

          if (action.action && childRouter) {
            var childStateUpdate = childRouter.getStateForAction(action.action, childState);
            if (childStateUpdate) {
              newChildState = childStateUpdate;
            }
          }

          if (action.params) {
            newChildState = _extends({}, newChildState, {
              params: _extends({}, newChildState.params || {}, action.params)
            });
          }

          if (newChildState !== childState) {
            var _routes2 = [...state.routes];
            _routes2[activeChildIndex] = newChildState;
            var nextState = _extends({}, state, {
              routes: _routes2,
              index: activeChildIndex
            });
            return getNextState(action, prevState, nextState);
          } else if (newChildState === childState && state.index === activeChildIndex && prevState) {
            return null;
          }
        }
      }

      if (action.type === _NavigationActions2.default.SET_PARAMS) {
        var key = action.key;
        var lastRoute = state.routes.find(route => route.key === key);
        if (lastRoute) {
          var _params = _extends({}, lastRoute.params, action.params);
          var _routes3 = [...state.routes];
          _routes3[state.routes.indexOf(lastRoute)] = _extends({}, lastRoute, {
            params: _params
          });
          return getNextState(action, prevState, _extends({}, state, {
            routes: _routes3
          }));
        }
      }

      if (activeChildIndex !== state.index) {
        return getNextState(action, prevState, _extends({}, state, {
          index: activeChildIndex
        }));
      } else if (didNavigate && !inputState) {
        return state;
      } else if (didNavigate) {
        return _extends({}, state);
      }

      // Let other children handle it and switch to the first child that returns a new state
      var index = state.index;
      var routes = state.routes;
      order.find((childId, i) => {
        var childRouter = childRouters[childId];
        if (i === index) {
          return false;
        }
        var childState = routes[i];
        if (childRouter) {
          childState = childRouter.getStateForAction(action, childState);
        }
        if (!childState) {
          index = i;
          return true;
        }
        if (childState !== routes[i]) {
          routes = [...routes];
          routes[i] = childState;
          index = i;
          return true;
        }
        return false;
      });

      // Nested routers can be updated after switching children with actions such as SET_PARAMS
      // and COMPLETE_TRANSITION.
      // NOTE: This may be problematic with custom routers because we whitelist the actions
      // that can be handled by child routers without automatically changing index.
      if (childrenUpdateWithoutSwitchingIndex(action.type)) {
        index = state.index;
      }

      if (index !== state.index || routes !== state.routes) {
        return getNextState(action, prevState, _extends({}, state, {
          index,
          routes
        }));
      }
      return state;
    },

    getComponentForState(state) {
      var routeName = state.routes[state.index].routeName;
      (0, _invariant2.default)(routeName, `There is no route defined for index ${state.index}. Check that
        that you passed in a navigation state with a valid tab/screen index.`);
      var childRouter = childRouters[routeName];
      if (childRouter) {
        return childRouter.getComponentForState(state.routes[state.index]);
      }
      return (0, _getScreenForRouteName2.default)(routeConfigs, routeName);
    },

    getComponentForRouteName(routeName) {
      return (0, _getScreenForRouteName2.default)(routeConfigs, routeName);
    },

    getPathAndParamsForState(state) {
      var route = state.routes[state.index];
      return getPathAndParamsForRoute(route);
    },

    getActionForPathAndParams(path, params) {
      return getActionForPathAndParams(path, params);
    },

    getScreenOptions: (0, _createConfigGetter2.default)(routeConfigs, config.defaultNavigationOptions)
  };
};