/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 *  strict-local
 * @format
 */

/* eslint-env worker, serviceworker */
"use strict";

function asyncGeneratorStep(gen, resolve, reject, _next, _throw, key, arg) {
  try {
    var info = gen[key](arg);
    var value = info.value;
  } catch (error) {
    reject(error);
    return;
  }
  if (info.done) {
    resolve(value);
  } else {
    Promise.resolve(value).then(_next, _throw);
  }
}

function _asyncToGenerator(fn) {
  return function() {
    var self = this,
      args = arguments;
    return new Promise(function(resolve, reject) {
      var gen = fn.apply(self, args);
      function _next(value) {
        asyncGeneratorStep(gen, resolve, reject, _next, _throw, "next", value);
      }
      function _throw(err) {
        asyncGeneratorStep(gen, resolve, reject, _next, _throw, "throw", err);
      }
      _next(undefined);
    });
  };
}

const WebSocketHMRClient = require("../WebSocketHMRClient");

const bundleCache = require("./bundleCache");

const bundleToString = require("./bundleToString");

const patchBundle = require("./patchBundle");

const stringToBundle = require("./stringToBundle");

const REVISION_ID_HEADER = "X-Metro-Delta-ID";

class BundleNotFoundError extends Error {
  constructor(url) {
    super(
      `Couldn't retrieve a bundle corresponding to ${url} from neither the bundle cache nor the browser cache. ` +
        "This can happen when the browser cache is cleared but the service worker isn't."
    );
  }
}

class UpdateError extends Error {
  constructor(url, message) {
    super(
      `Error retrieving update from the update server for ${url}. Try refreshing the page.\nError message: ${message}`
    );
  }
}

class RevisionIdHeaderNotFoundError extends Error {
  constructor(url) {
    super(
      `The \`${REVISION_ID_HEADER}\` header is missing from Metro server's response to a request for the bundle \`${url}\`. ` +
        "If you're running the Metro server behind a proxy, make sure that you proxy headers as well."
    );
  }
}

function defaultGetHmrServerUrl(bundleReq, prevBundleRes) {
  const bundleUrl = new URL(bundleReq.url);
  const revisionId = prevBundleRes.headers.get(REVISION_ID_HEADER);

  if (revisionId == null) {
    throw new RevisionIdHeaderNotFoundError(bundleReq.url);
  }

  return `${bundleUrl.protocol === "https:" ? "wss" : "ws"}://${
    bundleUrl.host
  }/hot?revisionId=${revisionId}`;
}

function defaultGetDeltaBundle(_x, _x2) {
  return _defaultGetDeltaBundle.apply(this, arguments);
}

function _defaultGetDeltaBundle() {
  _defaultGetDeltaBundle = _asyncToGenerator(function*(
    bundleReq,
    prevBundleRes
  ) {
    const url = new URL(bundleReq.url);
    url.pathname = url.pathname.replace(/\.(bundle|js)$/, ".delta");
    const revisionId = prevBundleRes.headers.get(REVISION_ID_HEADER);

    if (revisionId == null) {
      throw new RevisionIdHeaderNotFoundError(bundleReq.url);
    }

    url.searchParams.append("revisionId", revisionId);
    const res = yield fetch(url.href, {
      includeCredentials: true
    });
    return yield res.json();
  });
  return _defaultGetDeltaBundle.apply(this, arguments);
}

function defaultOnUpdate(clientId, update) {
  clients.get(clientId).then(client => {
    if (client != null) {
      client.postMessage({
        type: "METRO_UPDATE",
        update
      });
    }
  });
}

function defaultOnUpdateStart(clientId) {
  clients.get(clientId).then(client => {
    if (client != null) {
      client.postMessage({
        type: "METRO_UPDATE_START"
      });
    }
  });
}

function defaultOnUpdateError(clientId, error) {
  clients.get(clientId).then(client => {
    if (client != null) {
      client.postMessage({
        type: "METRO_UPDATE_ERROR",
        error
      });
    }
  });
}

function defaultShouldUpdateBundle(bundleReq, bundleRes, delta) {
  return delta.revisionId !== bundleRes.headers.get(REVISION_ID_HEADER);
}

function createStringResponse(contents, revisionId) {
  return new Response(contents, {
    status: 200,
    statusText: "OK",
    headers: new Headers({
      "Cache-Control": "no-cache",
      "Content-Length": String(contents.length),
      "Content-Type": "application/javascript",
      Date: new Date().toUTCString(),
      [REVISION_ID_HEADER]: revisionId
    })
  });
}

function createDeltaClient() {
  let _ref =
      arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {},
    _ref$getDeltaBundle = _ref.getDeltaBundle,
    getDeltaBundle =
      _ref$getDeltaBundle === void 0
        ? defaultGetDeltaBundle
        : _ref$getDeltaBundle,
    _ref$shouldUpdateBund = _ref.shouldUpdateBundle,
    shouldUpdateBundle =
      _ref$shouldUpdateBund === void 0
        ? defaultShouldUpdateBundle
        : _ref$shouldUpdateBund,
    _ref$getHmrServerUrl = _ref.getHmrServerUrl,
    getHmrServerUrl =
      _ref$getHmrServerUrl === void 0
        ? defaultGetHmrServerUrl
        : _ref$getHmrServerUrl,
    _ref$onUpdateStart = _ref.onUpdateStart,
    onUpdateStart =
      _ref$onUpdateStart === void 0 ? defaultOnUpdateStart : _ref$onUpdateStart,
    _ref$onUpdate = _ref.onUpdate,
    onUpdate = _ref$onUpdate === void 0 ? defaultOnUpdate : _ref$onUpdate,
    _ref$onUpdateError = _ref.onUpdateError,
    onUpdateError =
      _ref$onUpdateError === void 0 ? defaultOnUpdateError : _ref$onUpdateError;

  if (!__DEV__) {
    return (
      /*#__PURE__*/
      (function() {
        var _ref2 = _asyncToGenerator(function*(_ref3) {
          let bundleReq = _ref3.request;
          const prevBundleRes = yield bundleCache.getBundleResponse(bundleReq);

          if (prevBundleRes == null) {
            throw new BundleNotFoundError(bundleReq.url);
          }

          const delta = yield getDeltaBundle(bundleReq, prevBundleRes);

          if (!shouldUpdateBundle(bundleReq, prevBundleRes, delta)) {
            return prevBundleRes;
          }

          const prevStringBundle = yield prevBundleRes.text();
          const prevBundle = stringToBundle(prevStringBundle);
          const bundle = patchBundle(prevBundle, delta);
          const stringBundle = bundleToString(bundle, true);
          const bundleRes = createStringResponse(
            stringBundle,
            bundle.revisionId
          );
          bundleCache.setBundleResponse(bundleReq, bundleRes.clone());
          return bundleRes;
        });

        return function(_x3) {
          return _ref2.apply(this, arguments);
        };
      })()
    );
  }

  const clients = new Map();
  return (
    /*#__PURE__*/
    (function() {
      var _ref4 = _asyncToGenerator(function*(_ref5) {
        let bundleReq = _ref5.request,
          clientId = _ref5.clientId;
        let client = clients.get(bundleReq.url);

        if (client != null) {
          // There's already an update client running for this bundle URL.
          client.ids.add(clientId);
        } else {
          let resolveBundleRes;
          let rejectBundleRes;
          const currentClient = {
            ids: new Set([clientId]),
            bundleResPromise: new Promise((resolve, reject) => {
              resolveBundleRes = resolve;
              rejectBundleRes = reject;
            })
          };
          clients.set(bundleReq.url, currentClient);
          client = currentClient;
          const potentialPrevBundleRes = yield bundleCache.getBundleResponse(
            bundleReq
          );

          if (potentialPrevBundleRes == null) {
            throw new BundleNotFoundError(bundleReq.url);
          }

          let prevBundleRes = potentialPrevBundleRes;
          let currentBundlePromise = prevBundleRes
            .clone()
            .text()
            .then(prevStringBundle => stringToBundle(prevStringBundle));
          const updateServerUrl = getHmrServerUrl(bundleReq, prevBundleRes);
          let resolved = false;
          const wsClient = new WebSocketHMRClient(updateServerUrl);
          wsClient.on("connection-error", error => {
            rejectBundleRes(error);
          });
          wsClient.on("close", () => {
            clients.delete(bundleReq.url);
          });
          wsClient.on("error", error => {
            if (!resolved) {
              rejectBundleRes(error);
              return;
            }

            currentClient.ids.forEach(clientId =>
              onUpdateError(clientId, error)
            );
          });
          wsClient.on("update-start", () => {
            currentClient.ids.forEach(clientId => onUpdateStart(clientId));
          });
          wsClient.on(
            "update",
            /*#__PURE__*/
            (function() {
              var _ref6 = _asyncToGenerator(function*(update) {
                if (resolved) {
                  // Only notify clients for later updates.
                  currentClient.ids.forEach(clientId =>
                    onUpdate(clientId, update)
                  );
                }

                const delta = {
                  base: false,
                  revisionId: update.revisionId,
                  modules: update.modules,
                  deleted: update.deleted
                };
                let bundleRes;

                if (!shouldUpdateBundle(bundleReq, prevBundleRes, delta)) {
                  bundleRes = prevBundleRes;
                } else {
                  const currentBundle = patchBundle(
                    yield currentBundlePromise,
                    delta
                  );
                  currentBundlePromise = Promise.resolve(currentBundle);
                  const stringBundle = bundleToString(currentBundle, true);
                  bundleRes = createStringResponse(
                    stringBundle,
                    currentBundle.revisionId
                  );
                  bundleCache.setBundleResponse(bundleReq, bundleRes.clone());
                }

                if (!resolved) {
                  resolved = true;
                  resolveBundleRes(bundleRes);
                } else {
                  currentClient.bundleResPromise = Promise.resolve(bundleRes);
                }

                prevBundleRes = bundleRes;
              });

              return function(_x5) {
                return _ref6.apply(this, arguments);
              };
            })()
          );
          wsClient.enable();
        }

        let bundleRes;

        try {
          bundleRes = (yield client.bundleResPromise).clone();
        } catch (error) {
          throw new UpdateError(bundleReq.url, error.message);
        }

        return bundleRes;
      });

      return function(_x4) {
        return _ref4.apply(this, arguments);
      };
    })()
  );
}

module.exports = createDeltaClient;
