/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.beerstyle;

import android.database.Cursor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.data.BierverkostungContract.BeerStyleEntry;
import de.retujo.bierverkostung.data.CursorReader;
import de.retujo.bierverkostung.data.UriToIdFunction;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * A factory for creating {@link BeerStyle} instances.
 *
 * @since 1.0.0
 */
@Immutable
public final class BeerStyleFactory {

    private BeerStyleFactory() {
        throw new AssertionError();
    }

    /**
     * Returns a new immutable instance of {@link BeerStyle} with the given name.
     *
     * @param beerStyleName the name of the beer style.
     * @return the instance.
     * @throws NullPointerException if {@code beerStyleName} is {@code null}.
     * @throws IllegalArgumentException if {@code beerStyleName} is empty.
     */
    @Nonnull
    public static BeerStyle newBeerStyle(@Nonnull final CharSequence beerStyleName) {
        argumentNotEmpty(beerStyleName, "beer style name");
        return ImmutableBeerStyle.newInstance(UriToIdFunction.NO_ID, beerStyleName);
    }

    /**
     * Returns a new immutable instance of {@link BeerStyle} with the given name.
     *
     * @param beerStyleId the ID of the beer style.
     * @param beerStyleName the name of the beer style.
     * @return the instance.
     * @throws NullPointerException if {@code beerStyleName} is {@code null}.
     * @throws IllegalArgumentException if {@code beerStyleName} is empty.
     */
    @Nonnull
    public static BeerStyle newBeerStyle(final long beerStyleId, @Nonnull final CharSequence beerStyleName) {
        argumentNotEmpty(beerStyleName, "beer style name");
        return ImmutableBeerStyle.newInstance(beerStyleId, beerStyleName);
    }

    /**
     * Returns a new immutable instance of {@link BeerStyle} from the specified cursor.
     *
     * @param cursor the cursor to provide the brewery data.
     * @return the instance or {@code null}.
     * @throws NullPointerException if {@code cursor} is {@code null}.
     * @throws IllegalArgumentException if {@code cursor} did not contain all expected columns.
     */
    @Nullable
    public static BeerStyle newBeerStyle(@Nonnull final Cursor cursor) {
        final CursorReader cr = CursorReader.of(cursor);
        final long id = cr.getLong(BeerStyleEntry.COLUMN_ID);
        final String name = cr.getString(BeerStyleEntry.COLUMN_NAME);

        if (1 > id || null == name) {
            return null;
        }

        return newBeerStyle(id, name);
    }

}
