/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.brewery;

import android.content.ContentValues;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.text.TextUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.country.Country;
import de.retujo.bierverkostung.data.BierverkostungContract.BreweryEntry;
import de.retujo.bierverkostung.data.ParcelUnwrapper;
import de.retujo.bierverkostung.data.ParcelWrapper;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.ObjectUtil;

import static de.retujo.java.util.Conditions.argumentNotEmpty;
import static de.retujo.java.util.ObjectUtil.areEqual;

/**
 * Immutable implementation of {@link Brewery}.
 *
 * @since 1.0.0
 */
@Immutable
final class ImmutableBrewery implements Brewery {

    /**
     * Creator which creates instances of {@code ImmutableBrewery} from a Parcel.
     */
    public static final Parcelable.Creator<Brewery> CREATOR = new ImmutableBreweryCreator();

    private final long id;
    private final String name;
    private final String location;
    private final Country country;

    private ImmutableBrewery(final long theId, final String theName, final String theLocation,
            final Country theCountry) {
        id = theId;
        name = theName;
        location = theLocation;
        country = theCountry;
    }

    /**
     * Returns a new instance of {@code ImmutableBrewery}.
     *
     * @param id the ID of the brewery.
     * @param name the name of the brewery.
     * @param location the location of the brewery.
     * @param country the country of the brewery.
     * @return the instance.
     * @throws NullPointerException if {@code name} is {@code null}.
     * @throws IllegalArgumentException if {@code name} is empty.
     */
    public static ImmutableBrewery newInstance(final long id, @Nonnull final CharSequence name,
            @Nullable final CharSequence location, @Nullable final Country country) {
        argumentNotEmpty(name, "brewery name");
        return new ImmutableBrewery(id, name.toString(), null != location ? location.toString() : null, country);
    }

    @Override
    @NonNull
    public Maybe<Long> getId() {
        if (UriToIdFunction.NO_ID != id) {
            return Maybe.of(id);
        }
        return Maybe.empty();
    }

    @Override
    @NonNull
    public String getName() {
        return name;
    }

    @Override
    @Nonnull
    public Maybe<String> getLocation() {
        return Maybe.ofNullable(location);
    }

    @Override
    @Nonnull
    public Maybe<Country> getCountry() {
        return Maybe.ofNullable(country);
    }

    @Override
    @NonNull
    public Uri getContentUri() {
        return BreweryEntry.CONTENT_URI;
    }

    @Override
    @NonNull
    public ContentValues asContentValues() {
        final ContentValues result = new ContentValues(4);
        if (UriToIdFunction.NO_ID != id) {
            result.put(BreweryEntry.COLUMN_ID.toString(), id);
        }
        result.put(BreweryEntry.COLUMN_NAME.toString(), name);
        if (hasCountryId()) {
            result.put(BreweryEntry.COLUMN_COUNTRY_ID.toString(), country.getId().orElse(UriToIdFunction.NO_ID));
        }
        if (!TextUtils.isEmpty(location)) {
            result.put(BreweryEntry.COLUMN_LOCATION.toString(), location);
        }
        return result;
    }

    private boolean hasCountryId() {
        if (hasCountry()) {
            final Maybe<Long> countryId = country.getId();
            return countryId.isPresent();
        }
        return false;
    }

    private boolean hasCountry() {
        return null != country;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(final Parcel dest, final int flags) {
        ParcelWrapper.of(dest)
                .wrap(id)
                .wrap(name)
                .wrap(location)
                .wrap(country);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final ImmutableBrewery that = (ImmutableBrewery) o;

        return id == that.id
                && areEqual(name, that.name)
                && areEqual(location, that.location)
                && areEqual(country, that.country);
    }

    @Override
    public int hashCode() {
        return ObjectUtil.hashCode(id, name, location, country);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + " {"
                + "id=" + id
                + ", name='" + name + '\''
                + ", location='" + location + '\''
                + ", country=" + country
                + "}";
    }

    /**
     * This class creates instances of {@code ImmutableBrewery} from a Parcel.
     *
     * @since 1.0.0
     */
    @Immutable
    private static final class ImmutableBreweryCreator implements Parcelable.Creator<Brewery> {
        @Override
        public Brewery createFromParcel(final Parcel source) {
            final ParcelUnwrapper unwrapper = ParcelUnwrapper.of(source);
            final long readId = unwrapper.unwrapLong();
            final String readName = unwrapper.unwrapString();
            final String readLocation = unwrapper.unwrapString();
            final Country readCountry = unwrapper.unwrapParcelable();

            return ImmutableBrewery.newInstance(readId, readName, readLocation, readCountry);
        }

        @Override
        public Brewery[] newArray(final int size) {
            return new Brewery[size];
        }
    }

}
