/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.common;

import android.content.ContentResolver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.net.Uri;
import android.support.v7.app.AlertDialog;

import java.text.MessageFormat;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;

import de.retujo.bierverkostung.R;
import de.retujo.bierverkostung.data.RowData;
import de.retujo.java.util.Acceptor;
import de.retujo.java.util.Maybe;

import static de.retujo.java.util.Conditions.isNotNull;

/**
 * This dialogue is shown before an entity is finally deleted. It shows two buttons where the positive button will
 * delete the entity and the negative button just dismisses the dialogue. In either case an optional listener will be
 * notified.
 *
 * @since 1.0.0
 */
@NotThreadSafe
public final class DeleteEntityDialogue {

    private final AlertDialog alertDialog;

    private DeleteEntityDialogue(final AlertDialog alertDialog) {
        this.alertDialog = alertDialog;
    }

    /**
     * Returns a new instance of a builder with a fluent API for a {@code DeleteEntityDialogue}.
     *
     * @param context the context.
     * @param entity the entity to be deleted.
     * @param <T> the type of the entity to be deleted.
     * @return the builder.
     * @throws NullPointerException if any argument is {@code null}.
     * @throws IllegalArgumentException if {@code entity} has no ID.
     */
    public static <T extends RowData> Builder<T> getBuilder(@Nonnull final Context context,
            @Nonnull final T entity) {
        isNotNull(context, "context");
        isNotNull(entity, "entity to be deleted");
        final Maybe<Long> entityId = entity.getId();
        if (entityId.isAbsent()) {
            throw new IllegalArgumentException(MessageFormat.format("<{0}> has no ID!", entity));
        }

        return new Builder<>(context, entity);
    }

    /**
     * Starts the dialogue and shows it on the screen.
     *
     * @see AlertDialog#show()
     */
    public void show() {
        alertDialog.show();
    }

    /**
     * A mutable builder with a fluent API for a {@code DeleteEntityDialogue}.
     *
     * @param <T> the type of the entity to be deleted.
     *
     * @since 1.0.0
     */
    @NotThreadSafe
    public static final class Builder<T extends RowData> {
        private final Context context;
        private final T entity;
        private final AlertDialog.Builder dialogBuilder;
        private Acceptor<T> onEntityDeletedListener;
        private Acceptor<T> onKeepEntityListener;
        private Acceptor<T> onDeleteEntityFailedListener;

        private Builder(final Context context, final T entity) {
            this.context = context;
            this.entity = entity;
            dialogBuilder = new AlertDialog.Builder(context);
            onEntityDeletedListener = null;
            onKeepEntityListener = null;
            onDeleteEntityFailedListener = null;
        }

        /**
         * Sets the title using the specified ID.
         *
         * @param titleId the string resource ID.
         * @return this builder instance to allow method chaining.
         * @see AlertDialog.Builder#setTitle(int)
         */
        public Builder<T> setTitle(final int titleId) {
            dialogBuilder.setTitle(titleId);
            return this;
        }

        /**
         * Sets the specified message to be displayed in the dialogue.
         *
         * @param message the message.
         * @return this builder instance to allow method chaining.
         * @see AlertDialog.Builder#setMessage(CharSequence)
         */
        public Builder<T> setMessage(final CharSequence message) {
            dialogBuilder.setMessage(message);
            return this;
        }

        /**
         * Sets the listener to be notified when the entity was successfully deleted.
         *
         * @param listener the listener.
         * @return this builder instance to allow method chaining.
         */
        public Builder<T> setOnEntityDeletedListener(final Acceptor<T> listener) {
            onEntityDeletedListener = listener;
            return this;
        }

        /**
         * Sets the listener to be notified when the entity should be kept.
         *
         * @param listener the listener.
         * @return this builder instance to allow method chaining.
         */
        public Builder<T> setOnKeepEntityListener(final Acceptor<T> listener) {
            onKeepEntityListener = listener;
            return this;
        }

        /**
         * Sets the listener to be notified when deletion of the entity failed.
         *
         * @param listener the listener.
         * @return this builder instance to allow method chaining.
         */
        public Builder<T> setOnDeleteEntityFailedListener(final Acceptor<T> listener) {
            onDeleteEntityFailedListener = listener;
            return this;
        }

        /**
         * Creates a new instance of {@code DeleteEntityDialogue}.
         *
         * @return the instance.
         */
        public DeleteEntityDialogue build() {
            dialogBuilder.setPositiveButton(R.string.delete_dialog_button_positive_label,
                    new EntityDeleter<>(entity, context, onEntityDeletedListener, onDeleteEntityFailedListener));
            dialogBuilder.setNegativeButton(R.string.delete_dialog_button_negative_label, (dialog, which) -> {
                dialog.dismiss();
                if (null != onKeepEntityListener) {
                    onKeepEntityListener.accept(entity);
                }
            });

            final AlertDialog alertDialog = dialogBuilder.create();
            return new DeleteEntityDialogue(alertDialog);
        }
    }

    /**
     * This class deletes a particular entity and notifies the registered listener if available.
     *
     * @param <T> the type of the entity to be deleted.
     *
     * @since 1.0.0
     */
    @NotThreadSafe
    private static final class EntityDeleter<T extends RowData> implements OnClickListener {
        private final T entity;
        private final Context context;
        private final Acceptor<T> onEntityDeletedListener;
        private final Acceptor<T> onDeleteEntityFailedListener;

        private EntityDeleter(final T entity, final Context context, final Acceptor<T> onEntityDeletedListener,
                final Acceptor<T> onDeleteEntityFailedListener) {
            this.entity = entity;
            this.context = context;
            this.onEntityDeletedListener = onEntityDeletedListener;
            this.onDeleteEntityFailedListener = onDeleteEntityFailedListener;
        }


        @Override
        public void onClick(final DialogInterface dialog, final int which) {
            final String entityId;
            final Maybe<Long> maybeEntityId = entity.getId();
            if (maybeEntityId.isPresent()) {
                entityId = String.valueOf(maybeEntityId.get());
            } else {
                throw new IllegalArgumentException(MessageFormat.format("The entity <{0}> has not ID!", entity));
            }
            final Uri contentUri = entity.getContentUri()
                    .buildUpon()
                    .appendPath(entityId)
                    .build();

            final ContentResolver contentResolver = context.getContentResolver();
            final int rowsDeleted = contentResolver.delete(contentUri, null, null);
            if (0 < rowsDeleted) {
                if (null != onEntityDeletedListener) {
                    onEntityDeletedListener.accept(entity);
                }
            } else if (null != onDeleteEntityFailedListener) {
                onDeleteEntityFailedListener.accept(entity);
            }
        }
    }

}
