/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.country;

import android.database.Cursor;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.data.BierverkostungContract.CountryEntry;
import de.retujo.bierverkostung.data.CursorReader;
import de.retujo.bierverkostung.data.UriToIdFunction;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * A factory for creating {@link Country} instances.
 *
 * @since 1.0.0
 */
@Immutable
public final class CountryFactory {

    private CountryFactory() {
        throw new AssertionError();
    }

    /**
     * Returns a new immutable instance of {@link Country} with the given name.
     *
     * @param countryName the name of the country.
     * @return the instance.
     * @throws NullPointerException if {@code countryName} is {@code null}.
     * @throws IllegalArgumentException if {@code countryName} is empty.
     */
    @Nonnull
    public static Country newCountry(@Nonnull final CharSequence countryName) {
        return newCountry(UriToIdFunction.NO_ID, countryName);
    }

    /**
     * Returns a new immutable instance of {@link Country} with the given name.
     *
     * @param countryId the ID of the country.
     * @param countryName the name of the country.
     * @return the instance.
     * @throws NullPointerException if {@code countryName} is {@code null}.
     * @throws IllegalArgumentException if {@code countryName} is empty.
     */
    @Nonnull
    public static Country newCountry(final long countryId, @Nonnull final CharSequence countryName) {
        argumentNotEmpty(countryName, "country name");
        return ImmutableCountry.newInstance(countryId, countryName.toString());
    }

    /**
     * Returns a new immutable instance of {@link Country} from the specified Cursor.
     *
     * @param cursor the cursor to provide the country data.
     * @return the instance.
     * @throws NullPointerException if cursor is {@code null}.
     * @throws IllegalArgumentException if {@code cursor} did not contain all expected columns.
     */
    @Nullable
    public static Country newCountry(@Nonnull final Cursor cursor) {
        final CursorReader cr = CursorReader.of(cursor);

        final long countryId = cr.getLong(CountryEntry.COLUMN_ID);
        final String countryName = cr.getString(CountryEntry.COLUMN_NAME);

        if (1 > countryId || null == countryName) {
            return null;
        }

        return newCountry(countryId, countryName);
    }

}
