/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.country;

import android.content.ContentValues;
import android.net.Uri;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import de.retujo.bierverkostung.data.BierverkostungContract.CountryEntry;
import de.retujo.bierverkostung.data.UriToIdFunction;
import de.retujo.java.util.Maybe;
import de.retujo.java.util.ObjectUtil;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * Immutable implementation of {@link Country}.
 *
 * @since 1.0.0
 */
@Immutable
final class ImmutableCountry implements Country {

    /**
     * Creator which creates instances of {@code ImmutableCountry} from a Parcel.
     */
    public static final Parcelable.Creator<Country> CREATOR = new ImmutableCountryCreator();

    private final long id;
    private final String name;

    private ImmutableCountry(final long theId, final String theName) {
        id = theId;
        name = theName;
    }

    /**
     * Returns a new instance of {@code ImmutableCountry} with the given ID and name.
     *
     * @param id the ID of the country.
     * @param name the name of the country.
     * @return the instance.
     * @throws NullPointerException if {@code name} is {@code null}.
     * @throws IllegalArgumentException if {@code name} is empty.
     */
    public static ImmutableCountry newInstance(final long id, @Nonnull final String name) {
        argumentNotEmpty(name, "name");
        return new ImmutableCountry(id, name);
    }

    @Override
    @NonNull
    public Maybe<Long> getId() {
        if (UriToIdFunction.NO_ID != id) {
            return Maybe.of(id);
        }
        return Maybe.empty();
    }

    @Override
    @Nonnull
    public String getName() {
        return name;
    }

    @Override
    @NonNull
    public Uri getContentUri() {
        return CountryEntry.CONTENT_URI;
    }

    @Override
    @NonNull
    public ContentValues asContentValues() {
        final ContentValues result = new ContentValues(2);
        if (UriToIdFunction.NO_ID != id) {
            result.put(CountryEntry.COLUMN_ID.toString(), id);
        }
        result.put(CountryEntry.COLUMN_NAME.toString(), name);
        return result;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(final Parcel dest, final int flags) {
        // Both fields are always not null thus renouncing on use of ParcelWrapper.
        dest.writeLong(id);
        dest.writeString(name);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final ImmutableCountry that = (ImmutableCountry) o;

        return id == that.id && ObjectUtil.areEqual(name, that.name);
    }

    @Override
    public int hashCode() {
        return ObjectUtil.hashCode(id, name);
    }

    @Override
    public String toString() {
        return getClass().getSimpleName() + " {" + "id=" + id + ", name='" + name + '\'' + "}";
    }

    /**
     * This class creates instances of {@code ImmutableCountry} from a Parcel.
     *
     * @since 1.0.0
     */
    @Immutable
    private static final class ImmutableCountryCreator implements Parcelable.Creator<Country> {
        @Override
        public Country createFromParcel(final Parcel source) {
            return ImmutableCountry.newInstance(source.readLong(), source.readString());
        }

        @Override
        public Country[] newArray(final int size) {
            return new Country[size];
        }
    }

}
