/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import android.net.Uri;
import android.provider.BaseColumns;

import javax.annotation.concurrent.Immutable;

import static de.retujo.bierverkostung.data.ColumnType.INTEGER;
import static de.retujo.bierverkostung.data.ColumnType.TEXT;

/**
 * This class defines to contract so that clients know how to access the beer data.
 *
 * @since 1.0.0
 */
@Immutable
public final class BierverkostungContract {

    /**
     * The authority, which is how your code knows which Content Provider to access.
     */
    public static final String AUTHORITY = "de.retujo.bierverkostung";

    /**
     * The base content URI = {@code "content://" + <authority>}.
     */
    public static final Uri BASE_CONTENT_URI = Uri.parse("content://" + AUTHORITY);

    /**
     * The path for the countries directory.
     */
    public static final String PATH_COUNTRIES = "countries";

    /**
     * The path for the breweries directory.
     */
    public static final String PATH_BREWERIES = "breweries";

    /**
     * The path for the beer styles directory.
     */
    public static final String PATH_BEER_STYLES = "beer_styles";

    /**
     * Define the possible paths for accessing data in this contract. This is the path for the "beers" directory.
     */
    public static final String PATH_BEERS = "beers";

    /**
     * The path for the tastings directory.
     */
    public static final String PATH_TASTINGS = "tastings";

    /**
     * The path for a sub-directory for accessing the values of a single table column.
     */
    public static final String PATH_SINGLE_COLUMN_VALUES = "column_values";

    /**
     * CountryEntry defines the contents of the countries table.
     */
    @Immutable
    public static final class CountryEntry implements BaseColumns {
        /**
         * The {@code content://} style URI for the countries table, which requests a directory of data rows matching
         * the selection criteria.
         */
        public static final Uri CONTENT_URI = BASE_CONTENT_URI.buildUpon()
                .appendPath(PATH_COUNTRIES)
                .build();

        /**
         * Table which contains the countries.
         */
        public static final Table TABLE = DefaultTable.newInstance("countries");

        /**
         * Column which contains the unique IDs of the countries.
         */
        public static final Column COLUMN_ID = TABLE.createColumn(_ID, INTEGER);

        /**
         * Column which contains the names of the countries.
         */
        public static final Column COLUMN_NAME = TABLE.createColumn("country_name", TEXT);

        private CountryEntry() {
            throw new AssertionError();
        }
    }

    /**
     * BreweryEntry defines the contents of the breweries table.
     */
    @Immutable
    public static final class BreweryEntry implements BaseColumns {
        /**
         * The {@code content://} style URI for the breweries table, which requests a directory of data rows matching
         * the selection criteria.
         */
        public static final Uri CONTENT_URI = BASE_CONTENT_URI.buildUpon()
                .appendPath(PATH_BREWERIES)
                .build();

        /**
         * Table which contains the breweries.
         */
        public static final Table TABLE = DefaultTable.newInstance("breweries");

        /**
         * Column which contains the breweries' unique ID.
         */
        public static final Column COLUMN_ID = TABLE.createColumn(_ID, INTEGER);

        /**
         * Column which contains the breweries' name.
         */
        public static final Column COLUMN_NAME = TABLE.createColumn("brewery_name", TEXT);

        /**
         * Column which contains the breweries' location.
         */
        public static final Column COLUMN_LOCATION = TABLE.createColumn("brewery_location", TEXT);

        /**
         * Column which contains the breweries' country.
         */
        public static final Column COLUMN_COUNTRY_ID = TABLE.createColumn("country_id", INTEGER);

        private BreweryEntry() {
            throw new AssertionError();
        }
    }

    /**
     * BeerStyleEntry defines the contents of the breweries table.
     */
    @Immutable
    public static final class BeerStyleEntry implements BaseColumns {
        /**
         * The {@code content://} style URI for the beer styles table, which requests a directory of data rows matching
         * the selection criteria.
         */
        public static final Uri CONTENT_URI = BASE_CONTENT_URI.buildUpon()
                .appendPath(PATH_BEER_STYLES)
                .build();

        /**
         * Table which contains the beer styles.
         */
        public static final Table TABLE = DefaultTable.newInstance("beer_styles");

        /**
         * Column which contains the unique IDs of the beer styles.
         */
        public static final Column COLUMN_ID = TABLE.createColumn(_ID, INTEGER);

        /**
         * Column which contains the names of the beer styles.
         */
        public static final Column COLUMN_NAME = TABLE.createColumn("beer_style_name", TEXT);

        private BeerStyleEntry() {
            throw new AssertionError();
        }
    }

    /**
     * BeerEntry defines the contents of the beers table.
     */
    @Immutable
    public static final class BeerEntry implements BaseColumns {
        /**
         * The {@code content://} style URI for the beers table, which requests a directory of data rows matching
         * the selection criteria.
         */
        public static final Uri CONTENT_URI = BASE_CONTENT_URI.buildUpon()
                .appendPath(PATH_BEERS)
                .build();

        /**
         * The {@code content://} style URI for an arbitrary column of the "beers" table. It distinctly requests all
         * values of a particular table matching the selection criteria.
         */
        public static final Uri CONTENT_URI_SINGLE_COLUMN = CONTENT_URI.buildUpon()
                .appendPath(PATH_SINGLE_COLUMN_VALUES)
                .build();

        /**
         * Name of the table containing the beer data.
         */
        public static final Table TABLE = DefaultTable.newInstance("beers");

        /**
         * Column which contains the unique IDs of the beers.
         */
        public static final Column COLUMN_ID = TABLE.createColumn(_ID, INTEGER);

        /**
         * Column which contains the names of the beers.
         */
        public static final Column COLUMN_NAME = TABLE.createColumn("beer_name", TEXT);

        /**
         * Column which contains the IDs of the breweries of the beers.
         */
        public static final Column COLUMN_BREWERY_ID = TABLE.createColumn("brewery_id", INTEGER);

        /**
         * Column which contains the IDs of the beer styles of the beers.
         */
        public static final Column COLUMN_STYLE_ID = TABLE.createColumn("style_id", INTEGER);

        /**
         * Column which contains the original wort of the beers.
         */
        public static final Column COLUMN_ORIGINAL_WORT = TABLE.createColumn("original_wort", ColumnType.REAL);

        /**
         * Column which contains the volume alcohol of the beers.
         */
        public static final Column COLUMN_ALCOHOL = TABLE.createColumn("alcohol", ColumnType.REAL);

        /**
         * Column which contains the IBU of the beers.
         */
        public static final Column COLUMN_IBU = TABLE.createColumn("ibu", INTEGER);

        /**
         * Column which contains the ingredients of the beers.
         */
        public static final Column COLUMN_INGREDIENTS = TABLE.createColumn("ingredients", TEXT);

        /**
         * Column which contains the specifics of the beers.
         */
        public static final Column COLUMN_SPECIFICS = TABLE.createColumn("specifics", TEXT);

        /**
         * Column which contains the notes of the beers.
         */
        public static final Column COLUMN_NOTES = TABLE.createColumn("beer_notes", TEXT);

        private BeerEntry() {
            throw new AssertionError();
        }
    }

    /**
     * TastingEntry defines the contents of the beers table.
     */
    @Immutable
    public static final class TastingEntry implements BaseColumns {
        /**
         * The {@code content://} style URI for the tastings table, which requests a directory of data rows matching
         * the selection criteria.
         */
        public static final Uri CONTENT_URI = BASE_CONTENT_URI.buildUpon()
                .appendPath(PATH_TASTINGS)
                .build();

        /**
         * The {@code content://} style URI for an arbitrary column of the "tastings" table. It distinctly requests
         * all values of a particular table matching the selection criteria.
         */
        public static final Uri CONTENT_URI_SINGLE_COLUMN = CONTENT_URI.buildUpon()
                .appendPath(PATH_SINGLE_COLUMN_VALUES)
                .build();

        /**
         * Name of the table containing the tasting data.
         */
        public static final Table TABLE = DefaultTable.newInstance("tastings");

        /**
         * Column which contains the unique IDs of the beers.
         */
        public static final Column COLUMN_ID = TABLE.createColumn(_ID, INTEGER);

        /**
         * Column which contains the date in ISO 8601 when each tasting happened.
         */
        public static final Column COLUMN_DATE = TABLE.createColumn("date", TEXT);

        /**
         * Column which contains the locations where the tastings happened.
         */
        public static final Column COLUMN_LOCATION = TABLE.createColumn("location", TEXT);

        /**
         * Column which contains the ID of the tasted beer.
         */
        public static final Column COLUMN_BEER_ID = TABLE.createColumn("beer_id", INTEGER);

        /**
         * Column which contains the perceived beer colour.
         */
        public static final Column COLUMN_BEER_COLOUR = TABLE.createColumn("beer_colour", TEXT);

        /**
         * Column which contains a description of the perceived beer colour.
         */
        public static final Column COLUMN_BEER_COLOUR_DESCRIPTION = TABLE.createColumn("beer_colour_desc", TEXT);

        /**
         * Column which contains the perceived beer EBC value.
         */
        public static final Column COLUMN_COLOUR_EBC = TABLE.createColumn("colour_ebc", INTEGER);

        /**
         * Column which contains the description of the beer's perceived clarity.
         */
        public static final Column COLUMN_CLARITY_DESCRIPTION = TABLE.createColumn("clarity", TEXT);

        /**
         * Column which contains the colour of the beer's foam.
         */
        public static final Column COLUMN_FOAM_COLOUR = TABLE.createColumn("foam_colour", TEXT);

        /**
         * Column which contains the structure of the beer's foam.
         */
        public static final Column COLUMN_FOAM_STRUCTURE = TABLE.createColumn("foam_structure", TEXT);

        /**
         * Column which contains the stability of the beer's foam.
         */
        public static final Column COLUMN_FOAM_STABILITY = TABLE.createColumn("foam_stability", INTEGER);

        /**
         * Column which contains the description of the beer's perceived fruit flavour (scent).
         */
        public static final Column COLUMN_FRUIT_DESCRIPTION = TABLE.createColumn("fruit_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived fruit flavour (scent).
         */
        public static final Column COLUMN_FRUIT_RATING = TABLE.createColumn("fruit_rating", INTEGER);

        /**
         * Column which contains the description of the beer's perceived flower flavour (scent).
         */
        public static final Column COLUMN_FLOWER_DESCRIPTION = TABLE.createColumn("flower_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived flower flavour (scent).
         */
        public static final Column COLUMN_FLOWER_RATING = TABLE.createColumn("flower_rating", INTEGER);

        /**
         * Column which contains the description of the beer's perceived vegetal flavour (scent).
         */
        public static final Column COLUMN_VEGETAL_DESCRIPTION = TABLE.createColumn("vegetal_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived vegetal flavour (scent).
         */
        public static final Column COLUMN_VEGETAL_RATING = TABLE.createColumn("vegetal_rating", INTEGER);

        /**
         * Column which contains the description of the beer's perceived spicy flavour (scent).
         */
        public static final Column COLUMN_SPICY_DESCRIPTION = TABLE.createColumn("spicy_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived spicy flavour (scent).
         */
        public static final Column COLUMN_SPICY_RATING = TABLE.createColumn("spicy_rating", INTEGER);

        /**
         * Column which contains the description of the beer's perceived warmth-minted flavour (scent).
         */
        public static final Column COLUMN_WARMTH_MINTED_DESCRIPTION = TABLE.createColumn("warmth_minted_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived warmth-minted flavour (scent).
         */
        public static final Column COLUMN_WARMTH_MINTED_RATING = TABLE.createColumn("warmth_minted_rating", INTEGER);

        /**
         * Column which contains the description of the beer's perceived biological flavour (scent).
         */
        public static final Column COLUMN_BIOLOGICAL_DESCRIPTION = TABLE.createColumn("biological_desc", TEXT);

        /**
         * Column which contains the rating of the beer's perceived biological flavour (scent).
         */
        public static final Column COLUMN_BIOLOGICAL_RATING = TABLE.createColumn("biological_rating", INTEGER);

        /**
         * Column which contains the rating for the beer's perceived bitterness (taste).
         */
        public static final Column COLUMN_BITTERNESS_RATING = TABLE.createColumn("bitterness_bitterness", INTEGER);

        /**
         * Column which contains the rating for the beer's perceived sweetness (taste).
         */
        public static final Column COLUMN_SWEETNESS_RATING = TABLE.createColumn("sweetness_rating", INTEGER);

        /**
         * Column which contains the rating for the beer's perceived acidity (taste).
         */
        public static final Column COLUMN_ACIDITY_RATING = TABLE.createColumn("acidity_rating", INTEGER);

        /**
         * Column which contains the description for the beer's perceived mouthfeel (taste).
         */
        public static final Column COLUMN_MOUTHFEEL_DESCRIPTION = TABLE.createColumn("mouth_feel_desc", INTEGER);

        /**
         * Column which contains the rating for the beer's perceived body fullness.
         */
        public static final Column COLUMN_FULL_BODIED_RATING = TABLE.createColumn("full_bodied_rating", INTEGER);

        /**
         * Column which contains the description for the beer's perceived body (taste).
         */
        public static final Column COLUMN_BODY_DESCRIPTION = TABLE.createColumn("body_desc", TEXT);

        /**
         * Column which contains the description for the beer's perceived aftertaste (taste).
         */
        public static final Column COLUMN_AFTERTASTE_DESCRIPTION = TABLE.createColumn("aftertaste_desc", TEXT);

        /**
         * Column which contains the rating for the beer's perceived aftertaste (taste).
         */
        public static final Column COLUMN_AFTERTASTE_RATING = TABLE.createColumn("aftertaste_rating", INTEGER);

        /**
         * Column which contains the food recommendation.
         */
        public static final Column COLUMN_FOOD_RECOMMENDATION = TABLE.createColumn("food_recommendation", TEXT);

        /**
         * Column which contains the description for the beer's total impression.
         */
        public static final Column COLUMN_TOTAL_IMPRESSION_DESCRIPTION = TABLE.createColumn("total_impression_desc",
                TEXT);

        /**
         * Column which contains the rating of the total impression of the beer.
         */
        public static final Column COLUMN_TOTAL_IMPRESSION_RATING = TABLE.createColumn("total_impression_rating",
                INTEGER);

        private TastingEntry() {
            throw new AssertionError();
        }
    }

    private BierverkostungContract() {
        throw new AssertionError();
    }

}
