/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.NotThreadSafe;

import static de.retujo.java.util.Conditions.argumentNotEmpty;

/**
 * The default implementation of {@link Table}.
 *
 * @since 1.0.0
 */
@NotThreadSafe
final class DefaultTable implements Table {

    private final String tableName;
    private final List<Column> columns;

    private DefaultTable(final String theTableName) {
        this.tableName = theTableName;
        columns = new ArrayList<>();
    }

    /**
     * Returns a new instance of {@code DefaultTable}.
     *
     * @param tableName the name of the table.
     * @return the instance.
     * @throws NullPointerException if {@code tableName} is {@code null}.
     * @throws IllegalArgumentException if {@code tableName} is empty.
     */
    public static DefaultTable newInstance(@Nonnull final CharSequence tableName) {
        argumentNotEmpty(tableName, "table name");
        return new DefaultTable(tableName.toString());
    }

    @Nonnull
    @Override
    public Column createColumn(@Nonnull final CharSequence name, @Nonnull final ColumnType type) {
        final ImmutableColumn result = ImmutableColumn.newInstance(this, name, type);
        columns.add(result);
        return result;
    }

    @Nonnull
    @Override
    public List<Column> getColumns() {
        return new ArrayList<>(columns);
    }

    @Override
    public int length() {
        return tableName.length();
    }

    @Override
    public char charAt(final int index) {
        return tableName.charAt(index);
    }

    @Override
    public CharSequence subSequence(final int start, final int end) {
        return tableName.subSequence(start, end);
    }

    @Nonnull
    @Override
    public String toString() {
        return tableName;
    }

}
