/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package de.retujo.bierverkostung.data;

import org.junit.BeforeClass;
import org.junit.Test;

import static org.assertj.core.api.Assertions.assertThat;

/**
 * Unit test for {@link QuerySqlBuilder}.
 *
 * @since 1.0.0
 */
public final class QuerySqlBuilderTest {

    private static final String TABLE_NAME_1 = "facts";
    private static final String TABLE_NAME_2 = "alternative_facts";
    private static final String COL_NAME_1 = "foo";
    private static final String COL_NAME_2 = "bar";
    private static final String COL_NAME_3 = "baz";

    private static Table table1;
    private static Column column1;
    private static Column column2;
    private static Column column3;

    /** */
    @BeforeClass
    public static void setUp() {
        table1 = DefaultTable.newInstance(TABLE_NAME_1);
        column1 = table1.createColumn(COL_NAME_1, ColumnType.TEXT);
        column2 = table1.createColumn(COL_NAME_2, ColumnType.INTEGER);
        column3 = table1.createColumn(COL_NAME_3, ColumnType.TEXT);
    }

    /** */
    @Test
    public void selectCreatesExpectedString() {
        final String[] colNames = new String[] {COL_NAME_1, COL_NAME_2, COL_NAME_3};
        final String expectedString = "SELECT " + colNames[0] + ", " + colNames[1] + ", " + colNames[2] + ";";

//        final String actualString = QuerySqlBuilder.select(colNames).toString();

//        assertThat(actualString).isEqualTo(expectedString);
    }

    /** */
    @Test
    public void fromCreatesExpectedString() {
        final String expectedString = "SELECT " + COL_NAME_1 + " FROM " + TABLE_NAME_1 + ";";

        final String actualString = QuerySqlBuilder.select(column1)
                .from(table1)
                .toString();

        assertThat(actualString).isEqualTo(expectedString);
    }

    /** */
    @Test
    public void leftOuterJoinCreatesExpectedString() {
        final String expectedString = "SELECT " + COL_NAME_1
                + " FROM " + TABLE_NAME_1
                + " LEFT OUTER JOIN " + TABLE_NAME_2 + ";";

        final String actualString = QuerySqlBuilder.select(column1)
                .from(table1)
                .leftOuterJoin(TABLE_NAME_2)
                .toString();

        assertThat(actualString).isEqualTo(expectedString);
    }

    /** */
    @Test
    public void onCreatesExpectedString() {
        final String expectedString = "SELECT " + COL_NAME_1
                + " FROM " + TABLE_NAME_1
                + " LEFT OUTER JOIN " + TABLE_NAME_2
                + " ON " + COL_NAME_1 + "=" + COL_NAME_2 + ";";

        final String actualString = QuerySqlBuilder.select(column1)
                .from(table1)
                .leftOuterJoin(TABLE_NAME_2)
                .on(COL_NAME_1, COL_NAME_2)
                .toString();

        assertThat(actualString).isEqualTo(expectedString);
    }

    /** */
    @Test
    public void orderByCreatesExpectedString() {
        final String sortOrder = COL_NAME_2 + " ASC";

        final String expectedString = "SELECT " + COL_NAME_1
                + " FROM " + TABLE_NAME_1
                + " LEFT OUTER JOIN " + TABLE_NAME_2
                + " ON " + COL_NAME_1 + "=" + COL_NAME_2
                + " ORDER BY " + sortOrder + ";";

        final String actualString = QuerySqlBuilder.select(column1)
                .from(table1)
                .leftOuterJoin(TABLE_NAME_2)
                .on(column1, column2)
                .orderBy(sortOrder)
                .toString();

        assertThat(actualString).isEqualTo(expectedString);
    }

}