/*
 * Copyright 2017 Juergen Fickel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.retujo.bierverkostung.data;

import android.net.Uri;

import org.assertj.core.api.ThrowableAssert;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import java.util.Collections;

import de.retujo.java.util.Maybe;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static org.mutabilitydetector.unittesting.MutabilityAssert.assertInstancesOf;
import static org.mutabilitydetector.unittesting.MutabilityMatchers.areImmutable;

/**
 * Unit test for {@link UriToIdFunction}.
 *
 * @since 1.0.0
 */
public final class UriToIdFunctionTest {

    private UriToIdFunction underTest = null;
    private Uri baseUri = null;

    /** */
    @Before
    public void setUp() {
        underTest = UriToIdFunction.getInstance();

        baseUri = Mockito.mock(Uri.class);
        Mockito.when(baseUri.getAuthority()).thenReturn("de.retujo.bierverkostung");
        Mockito.when(baseUri.getFragment()).thenReturn("content");
        Mockito.when(baseUri.getPathSegments()).thenReturn(Collections.<String>emptyList());
    }

    /** */
    @Test
    public void assertImmutability() {
        assertInstancesOf(UriToIdFunction.class, areImmutable());
    }

    /** */
    @Test
    public void tryToGetIdFromNullUri() {
        assertThatThrownBy(new ThrowableAssert.ThrowingCallable() {
            @Override
            public void call() throws Exception {
                underTest.apply(null);
            }
        }).isInstanceOf(NullPointerException.class)
          .hasMessage("The <URI> must not be null!")
          .hasNoCause();
    }

    /** */
    @Test
    public void getIdFromUriWithoutId() {
        final Maybe<Long> idMaybe = underTest.apply(baseUri);

        assertThat(idMaybe.isAbsent()).isTrue();
    }

    /** */
    @Test
    public void getIdFromUriWithId() {
        final String id = "42";
        Mockito.when(baseUri.getPathSegments()).thenReturn(Collections.singletonList(id));
        final Maybe<Long> idMaybe = underTest.apply(baseUri);

        assertThat(idMaybe.isPresent()).isTrue();
        assertThat(idMaybe.get()).isEqualTo(Long.parseLong(id));
    }


}