/*
 * ====================================================================
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package com.photobucket.api.core;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Date;
import java.util.Vector;

import org.apache.http.entity.mime.content.FileBody;

/**
 *
 * @since 4.0
 */
public class FileBodyWithProgress extends FileBody implements ObservableUpload {
    private static final String TAG = "FileBodyWithProgress";
	
    private int lastProgressUpdate;
    private String filePath;
    private int size;
    private int bytesSent;
    private Date startTime;
    private Log log;
    
    
    public FileBodyWithProgress(final File file, Log log) {
        super(file);
        this.log = log;
        filePath = file.getAbsolutePath();

        // Make sure we don't truncate the file size/
        long sizeLong = file.length();
        
        if (sizeLong > Integer.MAX_VALUE) {
        	size = -1;
        	log.error(TAG, "File size exceeds Integer.MAX_VALUE (" + Integer.MAX_VALUE + ") ");
        } else {
        	size = (int)sizeLong;
        }
    }

    /**
     * Returns the total number of bytes in this upload.
     */
    public int getTotalBytes() {
    	return size;
    }
    
    /**
     * Returns the number of bytes transmitted so far.
     */
    public int getBytesSent() {
    	return bytesSent;
    }
    
    /**
     * Gets the time that the upload first started.  It can be used to measure overall throughput. 
     */
    public Date getStartTime() {
    	return startTime;
    }

    @Override
    public void writeTo(final OutputStream out) throws IOException {
        if (out == null)
            throw new IllegalArgumentException("Output stream may not be null");
        
        bytesSent = 0;
        float sizeFloat = (float)size;
        InputStream in = new FileInputStream(getFile());
        
        try {
            byte[] tmp = new byte[4096];
            int l;
            
            
            while ((l = in.read(tmp)) != -1) {
                out.write(tmp, 0, l);
                bytesSent += l;

                int newPercentComplete = (int)(((float)bytesSent / sizeFloat) * 100f);
            	
                if (newPercentComplete > lastProgressUpdate) {
                	if (log.isInfoEnabled(TAG))
                		log.info(TAG, "Firing progress update, now " + newPercentComplete + "% complete");
                	
                    FileUploadProgressEvent evt = new FileUploadProgressEvent(this, filePath, newPercentComplete);
                    fireFileUploadProgressEvent(evt);
                }
            }
            
            out.flush();
        } catch (IOException ioe) {
        	if (log.isErrorEnabled(TAG))
        		log.error(TAG, "Upload of " + getFilename() + " failed with IOException: " + ioe.getMessage(), ioe);
        } finally {
            in.close();
        }
    }
    
	// Create the listener list
    /** Stores the List of listeners registered for the UploadProgressEvent */
    protected Vector<IFileUploadProgressEventListener> listenerList = new Vector<IFileUploadProgressEventListener>();

    // This methods allows classes to register for FileUploadProgressEvents
    /**
     * Add a listener to the <code>FileUploadProgressEvent</code> 
     * @param listener the <code>IFileUploadProgressEventListner</code> that will be invoked when the event is fired
     */
    public void addFileUploadProgressEventListener(IFileUploadProgressEventListener listener) {
        listenerList.add(listener);
    }

    // This methods allows classes to unregister for FileUploadProgressEvents
    /**
     * Remove a listener from the event listeners 
     * @param listener the <code>IFileUploadProgressEventListner</code> that will be removed
     */
    public void removeFileUploadProgressEventListener(IFileUploadProgressEventListener listener) {
        listenerList.remove(listener);
    }
    
    /**
     * Dispatch the event for the list of listeners that have registered
     * @param evt the <code>FileUploadProgressEvent</code> that will be dispatched
     */
    private void fireFileUploadProgressEvent(FileUploadProgressEvent evt) {
        for(IFileUploadProgressEventListener listener : listenerList) {
        	listener.fileUploadProgressUpdate(evt);
        }
    }
}
