/**
 * <p>
 * PhotobucketAPI Class 
 * <p>
 * The main Photobucket API class
 * <p>
 * The class handles the proper signing of the message to the Photobucket API.  The API class implements a low 
 * level method for initiating a request and returning back the response message in the format specified by the consumer
 * <p>
 * Note:  This class is dependent on a customized version of the OAuthClient library packaged with the Photobucket Client
 * The Photobucket API requires the hash to be calculated from http://api.photobucket.com even if a request is to be sent to the silo 
 * http://api123.photobucket.com.  This class handles the calculation but required a customization to the OAuthClient library
 * to do so.
 * 
 * @author      Photobucket
 * @version     %I%, %G%
 * @see net.oauth.OAuth
 * @see net.oauth.OAuthAccessor
 * @see net.oauth.OAuthConsumer
 * @see net.oauth.OAuthException
 * @see net.oauth.OAuthMessage
 * @see net.oauth.OAuth.Parameter
 * 
 * @see org.apache.commons.httpclient.methods.multipart.Part
 * 
 * 
 */
package com.photobucket.api.core;

import java.io.File;
import java.io.FileInputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;

import oauth.signpost.exception.OAuthException;

import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;

import com.photobucket.api.rest.RESTfulOAuthClient;
import com.photobucket.api.rest.RESTfulOAuthRequest;
import com.photobucket.api.rest.RESTfulRequest;
import com.photobucket.api.rest.RESTfulResponse;


public class PhotobucketAPI {
	
	private String oauthToken = new String();
	private String oauthTokenSecret = new String();
	private String oauthConsumerKey = new String();
	private String oauthConsumerSecret = new String();
	private Long oauthTimestamp;
    private String subdomain = new String("api.photobucket.com");
    private String requestPath = new String();
	private Map<String, String> parameters = new HashMap<String, String>();
	private String method = new String();
	private File uploadFile = null;
	private FileInputStream fileInputStream = null;
	private String format = new String("xml");
	private String uploadFileName = new String();
	
	private HttpUriRequest httpRequest = null;
	private Integer connectionTimeout = null;
	private Integer soTimeout = null;

	private Log log = NullLog.INSTANCE;
	
    /**
     * Empty Constructor
     * 
     */
    public PhotobucketAPI() {}
    /**
     * Constructor
     * 
     * @param url String to send the request method in the form http://api.photobucket.com - may include the silo number
     * if accessing a user's account e.g http://api123.photobucket.com
     * @param method String Http method used for the request - must be one of (get, post, put, delete)
     * @param oauthConsumerKey String Consumer Key assigned to the API consumer ( also referred to as SCID )
     * @param oauthConsumerSecret String Consumer Secret Key assigned to the API consumer ( also referred to as Private Key )
     * @param parameters Additional Parameters that should be set and passed in the message to the API 
     */
	public PhotobucketAPI(String subdomain, String requestPath, String method, String oauthConsumerKey, String oauthConsumerSecret, String parameters) {
		this(subdomain, requestPath, method, oauthConsumerKey, oauthConsumerSecret, null, null, null);
	}
    /**
     * Constructor
     * 
     * @param url String to send the request method in the form http://api.photobucket.com - may include the silo number
     * if accessing a user's account e.g http://api123.photobucket.com
     * @param method String Http method used for the request - must be one of (get, post, put, delete)
     * @param oauthConsumerKey String Consumer Key assigned to the API consumer ( also referred to as SCID )
     * @param oauthConsumerSecret String Consumer Secret Key assigned to the API consumer ( also referred to as Private Key )]
     * @param oauthToken String Key for the User the API client is authorized - returned by API after a successful login
     * @param oauthTokenSecret String Key for the User the API client is authorized - returned by API after a successful login
     * @param parameters Additional Parameters that should be set and passed in the message to the API 
     */
	public PhotobucketAPI(String subdomain, String requestPath, String method, String oauthConsumerKey, String oauthConsumerSecret, String oauthToken, String oauthTokenSecret, Map<String,String> parameters) {
		this.subdomain = subdomain;
		this.requestPath = requestPath;
		this.method = method;
		this.oauthConsumerKey = oauthConsumerKey;
		this.oauthConsumerSecret = oauthConsumerSecret;
		this.oauthToken = oauthToken;
		this.oauthTokenSecret = oauthTokenSecret;
		this.parameters = parameters;
	}

	/**
	 * Retrieve the Oauth Token
	 * 
	 * @return String oauthToken
	 */
	public String getOauthToken() {
		return oauthToken;
	}

	/**
	 * Sets the OauthToken
	 * 
	 * @param oauthToken String token to set for the request
	 */
	public void setOauthToken(String oauthToken) {
		this.oauthToken = oauthToken;
	}

	/**
	 * Retrieve the OauthTokenSecret Key
	 * 
	 * @return String oauthTokenSecret 
	 */
	public String getOauthTokenSecret() {
		return oauthTokenSecret;
	}

	/**
	 * Set the OauthTokenSecret
	 * 
	 * @param oauthTokenSecret String
	 */
	public void setOauthTokenSecret(String oauthTokenSecret) {
		this.oauthTokenSecret = oauthTokenSecret;
	}
	
	/**
	 * Retrieve the OauthConsumerKey
	 * 
	 * @return String oauthConsumerKey
	 */

	public String getOauthConsumerKey() {
		return oauthConsumerKey;
	}
	
	/**
	 * Set the OauthConsumerKey
	 * 
	 * @param oauthConsumerKey String
	 */

	public void setOauthConsumerKey(String oauthConsumerKey) {
		this.oauthConsumerKey = oauthConsumerKey;
	}

	/**
	 * Retrieve list of http parameters 
	 * 
	 * @return Map List of http parameters
	 */
	public Map<String, String> getParameters() {
		return parameters;
	}
	/**
	 * Set the list of http parameters
	 * 
	 * @param parameters Map of http parameters to set for the api request
	 */

	public void setParameters(Map<String, String> parameters) {
		this.parameters = parameters;
	}
	/**
	 * Retrieve the http method for the request
	 * 
	 * @return String http method for the api request
	 */
	public String getMethod() {
		return method;
	}
	
	/**
	 * Set the http method for the request
	 * 
	 * @param method String http method for the request - should be one of (get, post, put, delete)
	 */

	public void setMethod(String method) {
		this.method = method;
	}
	
	private RESTfulOAuthRequest constructOauthRequest(boolean includeFormat) throws Exception {
	    RESTfulOAuthRequest request = new RESTfulOAuthRequest();
	    request.setFollowRedirects(true);
	    request.setMethod(getHttpMethod());
	    request.setParameters(parameters);
	    request.setSubdomain(getSubdomain());
	    request.setRequestPath(getRequestPath());
	    if ( includeFormat ) {
	        request.setFormat(format);
	    }
	    if ( (getUploadFile() != null )  || (getFileInputStream() != null) ){
	    	request.setMultipart(true);
	    	if ( getUploadFile() != null ) {
        	    request.setUploadFile(getUploadFile());
	    	} 
	    	else {
	    		if ( (uploadFileName == null ) || (uploadFileName.equals(""))) {
	    			throw new Exception("Filename is required for FileInputStream upload");
	    		}
	    	    request.setFileInputStreamAndFilename(getFileInputStream(), uploadFileName);
	    	}
        	if ( listenerList.size() > 0 ) {
        		request.setListenerList(listenerList);
        	}
	    }
	    else {
	    	request.setMultipart(false);
	    }
	    request.setOauthConsumerKey(getOauthConsumerKey());
	    request.setOauthConsumerSecret(getOauthConsumerSecret());
	    request.setOauthToken(getOauthToken());
	    request.setOauthTokenSecret(getOauthTokenSecret());
	    
	    return request;
	}

    /**
     * Abort the currently executing request
     * @return true if the request was successfully aborted, false otherwise
     */
    public boolean abort ()
    {
        if ( httpRequest != null )
        {
            try 
            {
                httpRequest.abort ();
                return true;
            } 
            catch ( UnsupportedOperationException ex ) { }
        }
        return false;
    }
    
    
    public void clearHttpRequest ()
    {
        if ( httpRequest != null )
        {
            httpRequest.abort ();
            httpRequest = null;
        }
    }
    
	/**
	 *  Sets up the call the the Photobucket API by setting the correct OAuth parameters and signing the message
	 *  Executes the call to the Photobucket API
	 *
	 * @return RESTfulResponse the response message received from the api
	 * @throws Exception 
	 * @throws OAuthException thrown if an authentication / authorization error occurs
	 */
	
    public RESTfulResponse execute() throws Exception {
        RESTfulOAuthRequest request = constructOauthRequest(true);

        RESTfulOAuthClient client = new RESTfulOAuthClient(log);
        if (connectionTimeout != null) client.setConnectionTimeout(connectionTimeout);
        if (soTimeout != null) client.setSoTimeout(soTimeout);

        httpRequest = client.getSignedRequest(request);
        return client.execute ( httpRequest, request.shouldFollowRedirects () );        
    }

	public HttpRequestBase getSignedRequest() throws Exception {
		
        RESTfulOAuthRequest request = constructOauthRequest(false);
		RESTfulOAuthClient client = new RESTfulOAuthClient(log);
		
		return client.getSignedRequest(request);		
	}

	/**
	 * Retrieve Oauth Consumer Secret Key
	 * 
	 * @return oauthConsumerSecret String
	 */
	public String getOauthConsumerSecret() {
		return oauthConsumerSecret;
	}

	/**
	 * Set OauthConsumerSecret Key
	 * 
	 * @param oauthConsumerSecret String
	 */
	public void setOauthConsumerSecret(String oauthConsumerSecret) {
		this.oauthConsumerSecret = oauthConsumerSecret;
	}
	
	/**
	 * Retrieve Oauth Timestamp
	 * 
	 * @return oauthTimestamp Long
	 */
	public Long getOauthTimestamp() {
		return oauthTimestamp;
	}
	
	/**
	 * Set OauthTimestamp
	 * 
	 * @param oauthTimestamp Long
	 */
	public void setOauthTimestamp(Long oauthTimestamp) {
		this.oauthTimestamp = oauthTimestamp;
	}
	


	/**
	 * Retrieve the File that is to be uploaded
	 * 
	 * @return File The file that will be uploaded
	 */
	public File getUploadFile() {
		return uploadFile;
	}
	
	/**
	 * Set the File that is to be uploaded
	 * 
	 * @param uploadFile The file that will be uploaded
	 */

	public void setUploadFile(File uploadFile) {
		this.uploadFile = uploadFile;
    }

	
	// Create the listener list
	/**
	 * List of Listeners registered for the file upload progress event
	 * 
	 */
    protected Vector<IFileUploadProgressEventListener> listenerList = new Vector<IFileUploadProgressEventListener>();

    // This methods allows classes to register for FileUploadProgressEvents
    /**
     * Add a listener for the <code>FileUploadProgressEvent</code>
     * 
     * @param listener IFileUploadProgressEventListener
     * 
     */
    public void addFileUploadProgressEventListener(IFileUploadProgressEventListener listener) {
        listenerList.add(listener);
    }
    /**
     * Remove a listener for the <code>FileUploadProgressEvent</code>
     * 
     * @param listener IFileUploadProgressEventListener
     * 
     */
    // This methods allows classes to unregister for FileUploadProgressEvents
    public void removeFileUploadProgressEventListener(IFileUploadProgressEventListener listener) {
        listenerList.remove(listener);
    }
    
	private RESTfulRequest.Method getHttpMethod() {
		if ( this.method.equalsIgnoreCase("delete")) {
			return RESTfulRequest.Method.DELETE;
		}
		else if ( this.method.equalsIgnoreCase("post")) {
			return RESTfulRequest.Method.POST;
		}
		else if ( this.method.equalsIgnoreCase("put")) {
			return RESTfulRequest.Method.PUT;
		}
		else if ( this.method.equalsIgnoreCase("get")) {
			return RESTfulRequest.Method.GET;
		}
		else {
			return RESTfulRequest.Method.GET;
		}
	}
	
	public Log getLog() {
		return log;
	}
	
	public void setLog(Log log) {
		this.log = log;
	}
	
	public String getSubdomain() {
		return subdomain;
	}
	public void setSubdomain(String subdomain) {
		this.subdomain = subdomain;
	}
	public String getRequestPath() {
		return requestPath;
	}
	public void setRequestPath(String requestPath) {
		this.requestPath = requestPath;
	}
	public String getFormat() {
		return this.format;
	}
	public void setFormat(String format) {
		this.format = format;
	}
	public FileInputStream getFileInputStream() {
		return fileInputStream;
	}
	public void setFileInputStream(FileInputStream fileInputStream) {
		this.fileInputStream = fileInputStream;
	}
	public String getUploadFileName() {
		return uploadFileName;
	}
	public void setUploadFileName(String uploadFileName) {
		this.uploadFileName = uploadFileName;
	}
    public Integer getConnectionTimeout() {
        return connectionTimeout;
    }
    public void setConnectionTimeout(Integer connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }
    public Integer getSoTimeout() {
        return soTimeout;
    }
    public void setSoTimeout(Integer soTimeout) {
        this.soTimeout = soTimeout;
    }
}
