/**
 * <p>
 * RESTfuleOAuthClient Class 
 * <p>
 * Class that leverages the OAuth Client Library to build correct RESTful messages.  This class supports HTTP GET, PUT, POST, and DELETE.
 * <p>
 * For more infomation on REST see http://en.wikipedia.org/wiki/Representational_State_Transfer
 * 
 * @author      Photobucket
 * @version     %I%, %G%
 */

package com.photobucket.api.rest;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import oauth.signpost.OAuth;
import oauth.signpost.OAuthConsumer;
import oauth.signpost.exception.OAuthCommunicationException;
import oauth.signpost.exception.OAuthExpectationFailedException;
import oauth.signpost.exception.OAuthMessageSignerException;
import oauth.signpost.signature.HmacSha1MessageSigner;

import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.params.ClientPNames;
import org.apache.http.client.utils.URIUtils;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.AbstractContentBody;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.HttpConnectionParams;

import com.photobucket.api.core.FileBodyWithProgress;
import com.photobucket.api.core.IFileUploadProgressEventListener;
import com.photobucket.api.core.InputStreamBodyWithProgress;
import com.photobucket.api.core.Log;
import com.photobucket.api.oauth.PhotobucketHttpOAuthConsumer;

public class RESTfulOAuthClient {
	
	/** Photobucket Base domain - always api.photobucket.com */
	private static final String PHOTOBUCKET_BASE_DOMAIN = "api.photobucket.com";
	
	private static final int ACTUAL_REQUEST = 0;
	private static final int BASE_DOMAIN_REQUEST = 1;
	
	private static final String TAG = "RESTfulOAuthclient";
	
	private int connectionTimeout = 0;
	private int soTimeout = 0;
	private Log log;
	
	/**
	 * Empty Constructor
	 * 
	 */
	public RESTfulOAuthClient(Log log) {
		this.log = log;
	}
	
	public HttpRequestBase getSignedRequest(RESTfulOAuthRequest restfulRequest) throws IOException, URISyntaxException, OAuthMessageSignerException, 
	OAuthExpectationFailedException, OAuthCommunicationException {
		HttpRequestBase[] request = methodFactory(restfulRequest);
    
		if (log.isDebugEnabled(TAG)) {
			log.debug(TAG, "Actual Request: " + request[ACTUAL_REQUEST].getURI());
			log.debug(TAG, "Base Request: " + request[BASE_DOMAIN_REQUEST].getURI());
		}

		// This kind of sucks but we need to override the CommonsHttpConsumer so we can make
		// sure the timestamp is in sync with photobucket servers
		OAuthConsumer consumer = new PhotobucketHttpOAuthConsumer(restfulRequest.getOauthConsumerKey(), 
				restfulRequest.getOauthConsumerSecret(), new HmacSha1MessageSigner(), log);
		consumer.setTokenWithSecret(restfulRequest.getOauthToken(), restfulRequest.getOauthTokenSecret());
		consumer.sign(request[BASE_DOMAIN_REQUEST]);

		// Transfer the OAuth header from the base request to the actual request 
		request[ACTUAL_REQUEST].setHeader(OAuth.HTTP_AUTHORIZATION_HEADER, 
				request[BASE_DOMAIN_REQUEST].getFirstHeader(OAuth.HTTP_AUTHORIZATION_HEADER).getValue());	

		return request[ACTUAL_REQUEST];
	}

    /**
     * Execute a call to the API using the OAuth client library.
     * 
     * @param signedRequest Signed HTTP request
     * @param shouldFollowRedirects If true, the library will automatically follow redirects 
     * @return response object
     * @throws IOException
     * @throws OAuthMessageSignerException
     * @throws OAuthExpectationFailedException
     * @throws URISyntaxException
     */
    public RESTfulResponse execute ( HttpUriRequest signedRequest, boolean shouldFollowRedirects ) 
        throws IOException, OAuthMessageSignerException, OAuthExpectationFailedException, URISyntaxException 
    {
        BasicHttpParams httpParams = new BasicHttpParams();
        httpParams.setBooleanParameter(ClientPNames.HANDLE_REDIRECTS, shouldFollowRedirects);
        
        HttpConnectionParams.setConnectionTimeout(httpParams, connectionTimeout);
        HttpConnectionParams.setSoTimeout(httpParams, soTimeout);
        
		HttpClient client = new DefaultHttpClient(httpParams);
		HttpResponse response = client.execute(signedRequest);

        String responseString = new String();
        responseString = new PhotobucketResponseHandler().handleResponse(response); 

        return new RESTfulResponse(response.getStatusLine().getStatusCode(), responseString);

	}
	
	/**
	 *  Execute a call to the API using the OAuth client library
	 * 
	 * @param request RESTfulOAuthMessage the request message that is to be sent to the api
	 * @return RESTfulResponse Response message received from the api
	 * @throws IOException 
	 * @throws IOException thrown when there is problem communicating with the api
	 * @throws OAuthExpectationFailedException 
	 * @throws OAuthMessageSignerException 
	 * @throws OAuthCommunicationException
	 * @throws URISyntaxException 
	 * @throws CloneNotSupportedException 
	 */
	
    public RESTfulResponse execute(RESTfulOAuthRequest restfulRequest) throws IOException, OAuthMessageSignerException, 
    	OAuthExpectationFailedException, OAuthCommunicationException, URISyntaxException {

        HttpRequestBase request = getSignedRequest(restfulRequest);
        return execute ( request, restfulRequest.shouldFollowRedirects () );        
    }
    
    /**
     * Private methodFactory class that sets up the correct HTTP method for the REST call
     * 
     * @param message OAuthMessage that will be sent to the api
     * @return HttpMethod the method class that implements the correct type e.g GET, PUT, POST, DELETE
     * @throws IOException
     * @throws URISyntaxException 
     */
    private HttpRequestBase[] methodFactory(RESTfulRequest restfulRequest) throws IOException, URISyntaxException {
    	// Capture parameters
        List<NameValuePair> qParams = new ArrayList<NameValuePair>();
        
        for (Map.Entry<String,String> entry : restfulRequest.getParameters().entrySet()) {
            qParams.add(new BasicNameValuePair(entry.getKey(), entry.getValue()));
        }
        
        // The baseDomain request is used for OAuth signature generation, actual request is what get's 
        // sent to the server.
    	HttpRequestBase actualRequest; 
    	HttpRequestBase baseDomainRequest;
    	RESTfulRequest.Method method = restfulRequest.getMethod();
    	
    	if ( (method == RESTfulRequest.Method.GET)  ||
    		(method == RESTfulRequest.Method.PUT)	||
    		(method == RESTfulRequest.Method.DELETE) ) {
    		// Construct the URIs
    		URL url = new URL(restfulRequest.getRequestUrl());
    		String parameters = URLEncodedUtils.format(qParams, "UTF-8"); 
    		URI actualUri = URIUtils.createURI(url.getProtocol(), url.getHost(), url.getPort(), url.getPath(), 
    				parameters, null);
    		URI baseDomainUri = URIUtils.createURI(url.getProtocol(), PHOTOBUCKET_BASE_DOMAIN, url.getPort(), 
    				url.getPath(), parameters, null);

    		// Construct the right kind of HTTP Request
            if (method == RESTfulRequest.Method.GET) {
    	        actualRequest = new HttpGet(actualUri);
    	        baseDomainRequest = new HttpGet(baseDomainUri);
            } else if (method == RESTfulRequest.Method.PUT) {
    	        actualRequest = new HttpPut(actualUri);
    	        baseDomainRequest = new HttpPut(baseDomainUri);
            } else {
                actualRequest = new HttpDelete(actualUri);	
                baseDomainRequest = new HttpDelete(baseDomainUri);	
            }
    	} else {
    	    actualRequest = new HttpPost(restfulRequest.getRequestUrl());
    	    baseDomainRequest = new HttpPost(restfulRequest.getBaseDomainRequestUrl());
    	    
    	    // Does the request include a file? If so, add the filename and size to the parameter list
    	    if (restfulRequest.isFileUpload()) {
    	    	String fileName = restfulRequest.getUploadFileName();
    	    	Integer fileSize = restfulRequest.getUploadFileSize();
    	    	
    	    	// Verify parameters to prevent null pointer exceptions
    	    	if (fileName == null || fileName.length() == 0)
    	    		throw new IllegalArgumentException("A file upload request must supply a file name");
    	    	
    	    	if (fileSize == null)
    	    		throw new IllegalArgumentException("A file upload request must supply a file size");
    	    	
    	    	qParams.add(new BasicNameValuePair(RESTfulRequest.UPLOAD_FILE_NAME_KEY, 
    	    			restfulRequest.getUploadFileName()));
    	    	qParams.add(new BasicNameValuePair(RESTfulRequest.UPLOAD_FILE_LENGTH_KEY, 
    	    			restfulRequest.getUploadFileSize().toString()));
    	    }
    	    	
    	    // Create a simple form body for the base entity, since the OAuth signature doesn't care and doesn't 
    	    // know how to deal with multi-part bodies.
   	    	UrlEncodedFormEntity urlEntity = new UrlEncodedFormEntity(qParams, "UTF-8");
   	    	((HttpEntityEnclosingRequestBase)baseDomainRequest).setEntity(urlEntity);
    	    
   	    	// Is it a multi-part post?
    	    if ((restfulRequest.isMultipart()) && (restfulRequest.getMethod() == RESTfulRequest.Method.POST)) {
        		MultipartEntity reqEntity = new MultipartEntity();

        		// Add each parameter as a separate string body part
        		Charset stringCharset = Charset.forName("UTF-8");
        		for (NameValuePair pair : qParams)
        			reqEntity.addPart(pair.getName(), new StringBody(pair.getValue(), stringCharset));
        		
        		// Do we have a file to add?
        		if (restfulRequest.isFileUpload())
        			reqEntity.addPart(RESTfulRequest.UPLOAD_FILE_KEY, createFilePart(restfulRequest));
        		
        		((HttpPost)actualRequest).setEntity(reqEntity);
            } else {
            	// It's not a multi-part post, so just re-use the same body we created for the base request. 
                ((HttpEntityEnclosingRequestBase)actualRequest).setEntity(urlEntity);
            }
    	}
    	
    	// This construction must respect the defined index constants: ACTUAL_REQUEST and BASE_DOMAIN_REQUEST
    	return new HttpRequestBase[] { actualRequest, baseDomainRequest };
    }
    
    private AbstractContentBody createFilePart(RESTfulRequest restfulRequest) {
    	AbstractContentBody filePart;
    	
    	// Is it a file or input stream?
    	if (restfulRequest.getUploadFile() != null) {
    		// Do we need to track progress?
    		if (restfulRequest.getListenerList().size() > 0) {
    			// Create a progress reporting file body part and attach listeners
    			FileBodyWithProgress ptPart = new FileBodyWithProgress(restfulRequest.getUploadFile(), log);
    			
    			for (IFileUploadProgressEventListener listener : restfulRequest.getListenerList())
    				ptPart.addFileUploadProgressEventListener(listener);
    			
    			filePart = ptPart;
    		} else {
    			// Nobody cares so just use a regular file body.
    			filePart = new FileBody(restfulRequest.getUploadFile());
    		}
		} else if (restfulRequest.getFileInputStream() != null) {
    		// Do we need to track progress?
			if (restfulRequest.getListenerList().size() > 0) {
				// Create a progress reporting input stream body part and attach listeners
				InputStreamBodyWithProgress ptStreamBody = new InputStreamBodyWithProgress(
					restfulRequest.getFileInputStream(), restfulRequest.getUploadFileName(), log);
				
				for (IFileUploadProgressEventListener listener : restfulRequest.getListenerList())
					ptStreamBody.addFileUploadProgressEventListener(listener);
					
				filePart = ptStreamBody;
		    } else {
		    	// Nobody cares so just create a regular input stream body.
		    	filePart = new InputStreamBody(restfulRequest.getFileInputStream(), restfulRequest.getUploadFileName());
		    }
		} else {
			// This is a programming error, since this is only called if request.isFileUpload() returns true,
			// and it should only return true, if there's a file or input stream.
			throw new UnsupportedOperationException("Request does not have a file or input stream.");
		}
			
		return filePart;
    }

    public int getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setConnectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public int getSoTimeout() {
        return soTimeout;
    }

    public void setSoTimeout(int soTimeout) {
        this.soTimeout = soTimeout;
    }

}
