// This software is released into the Public Domain by the author Victor H Olvera
/*
   the job of a parser is to understand input and either create code (ie: a 
   compiler) or a data structure (ie: an interpreter) in order to evaluate it

   primitive syntax is syntax that is readily recognizable without the need to decend further

   ***the rule with handling whitespaces is if a "grab" method handles primitive syntax it 
   also handles the whitespaces that came before it, examples:

      "    34.2" 
      GrabNumber() would handle the "34.2" and all the whitespaces that came before it

     "3 + 2"
      GrabMathExpression() would handle only the whitespaces before the '+' sign
      GrabNumber() would handle the whitespaces before the '2'
*/

package dudeofx.eval;

import android.app.Activity;

import java.util.Hashtable; 
import java.util.HashSet;
import java.util.Dictionary;
import java.util.Enumeration;
import java.lang.Math;
import android.content.Intent;

//=============================================================================
public class tParseTree {
   public static final int tSTRING = 1;
   public static final int tDOUBLE = 2;

   //=============================================================================
   class tNodeCarrier {
      tSyntaxNode node;
      int run;
      //-----------------------------------------------------------------------------
      tNodeCarrier(tSyntaxNode node, int run) {
         this.node = node;
         this.run = run;
      }
      //-----------------------------------------------------------------------------
   }
   //=============================================================================
   class tVarStore {
      int    type;
      String str;
      Double num;
      //-----------------------------------------------------------------------------
      tVarStore(String st) {
         this.type = tSTRING;
         this.str = st;
      }
      //-----------------------------------------------------------------------------
      tVarStore(Double num) {
         this.type = tDOUBLE;
         this.num = num;
      }
      //-----------------------------------------------------------------------------
   }
   //=============================================================================

   tSyntaxNode root;
   Hashtable<String, tVarStore> var_table;
   HashSet<String> fail_safe;
   String previous_entry;
   String current_entry;

   //-----------------------------------------------------------------------------
   tParseTree() {
      this.var_table = new Hashtable<String, tVarStore>();
      this.fail_safe = new HashSet<String>();
      this.root = null;
      previous_entry = "help";
   }
   //-----------------------------------------------------------------------------
   tParseTree(String st) {
      this.var_table = new Hashtable<String, tVarStore>();
      this.fail_safe = new HashSet<String>();
      this.root = null;
      previous_entry = "help";
      Parse(st);
      eval();
   }
   //-----------------------------------------------------------------------------
   int CountWhiteSpaces(String st) {
      int len;
      int i;

      if (st == null) return 0;

      len = st.length();
      for (i = 0; i < len; i++) {
         if (st.charAt(i) > ' ') break;
      }

      return i;
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabIdentifier(String st, boolean filter) {
      String[] keywords = {"lsvars","static","rep","help","clean"};
      tNodeCarrier tmp;
      int len;
      int i;
      int mark;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                                 // our fail return is a (null node, negative run)

      if (st == null) return fail;
      len = st.length();
      if (len == 0) return fail;
      i = CountWhiteSpaces(st);

      if (st.charAt(i) >= '0' && st.charAt(i) <= '9') return fail;      // identifiers can't start with a digit
      mark = i;
      for (; i < len; i++) {                                    // size up the identifier
         int check = 0;
         if (st.charAt(i) >= '0' && st.charAt(i) <= '9') check = 1;
         if (st.charAt(i) >= 'a' && st.charAt(i) <= 'z') check = 1;
         if (st.charAt(i) >= 'A' && st.charAt(i) <= 'Z') check = 1;
         if (st.charAt(i) == '_') check = 1;
         if (check == 0) break;
      }
      if (mark == i) return fail;

      String identifier = st.substring(mark, i);

      if (filter) {
         int count = keywords.length;
         for (int n = 0; n < count; n++) {
             if (identifier.equals(keywords[n])) {
                return new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: " + identifier + " is a reserved keyword!"), -1);
             }
         }
      }

      tmp = new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cIDENTIFIER, identifier), i);
      return tmp;
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabValue(String st) {
      tNodeCarrier tmp;
      int len;
      int i;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                                 // our fail return is a (null node, negative run)

      tmp = GrabNumber(st);
      if (tmp.node != null) return tmp;

      tmp = GrabFunction(st);
      if (tmp.node != null) return tmp;

      tmp = GrabVariable(st);
      if (tmp.node != null) return tmp;

      // handle  '(' expression ')'  as a single value
      len = st.length();
      i = CountWhiteSpaces(st);
      if (i != len) {
         if (st.charAt(i) == '(') {
            i++;
            tmp = GrabCodeExpression(st.substring(i, len));
            if (tmp.run < 0) return tmp;                                    // <-- syntax error most likely
            i += tmp.run;
            i += CountWhiteSpaces(st.substring(i, len));
            if (i == len) return fail;                                      // missing ')' syntax errors
            if (st.charAt(i) != ')') return fail;
            i++;
            return new tNodeCarrier(tmp.node, i);
         }
      }

      return fail;

   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabAssignment(String st) {
      int len;
      int i;
      int mark;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                                 // our fail return is a (null node, negative run)

      tNodeCarrier tmp = GrabIdentifier(st, true);
      if (tmp.run < 0) return tmp;
      String identifier = tmp.node.ident;
      i = tmp.run;
      len = st.length();
      i += CountWhiteSpaces(st.substring(i, len));
      if (i == len) return fail;

      int op = 0;
      int ch = st.charAt(i);
      if (ch == '=') {                                               // regular assignment
         op = ch;
         i++;
      }
      if ( (ch == '+') || (ch == '-') || (ch == '*') || (ch == '/') || (ch == 'i') ) { // modified assignments +=, -=, *=, /=
         op = ch;
         i++;
         if (i == len) return fail;
         if (st.charAt(i) != '=') return fail;
         i++;
      }
      if (op == 0) return fail;

//       i += CountWhiteSpaces(st.substring(i, len)); // not responsible for these whitespaces
      tNodeCarrier exp = GrabCodeExpression(st.substring(i, len));
      if (exp.run < 0) return exp;
      i += exp.run;
      tSyntaxNode var_node;
      var_node = new tSyntaxNode(tSyntaxNode.cVARIABLE, identifier);
      if (op != '=') { // handle modification of assignment
         tSyntaxNode op_node;
         if (op == 'i') {   // i= is the inverse of /=
           op_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, '/');
           op_node.AddChild(exp.node);
           op_node.AddChild(var_node);
         } else {
           op_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, op);
           op_node.AddChild(var_node);
           op_node.AddChild(exp.node);
         }
         exp.node = op_node;
      }
      tSyntaxNode equ_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, '=');
      var_node = new tSyntaxNode(tSyntaxNode.cVARIABLE, identifier);          //<--- apperently we have to make another node of the same variable or we get bugs
      equ_node.AddChild(var_node);
      equ_node.AddChild(exp.node);
      return new tNodeCarrier(equ_node, i);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabMathExpression(String st) {
      int len;
      int i;
      int op;
      int mark;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                   // our fail return is a (null node, negative run)
      tNodeCarrier term = null;
      tSyntaxNode  A_node = null;
      tSyntaxNode  B_node = null;
      tSyntaxNode  op_node = null;

      term = GrabTerm(st);
      if (term.run < 0) return term;
      op_node = A_node = term.node;

      mark = i = term.run;
      len = st.length();

      i += CountWhiteSpaces(st.substring(i, len));
      if (i == len) return term;
      op = st.charAt(i);
      while ((op == '+') || (op == '-')) {
         i++;
         op_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, op);
         term = GrabTerm(st.substring(i, len));
         B_node = term.node;
         if (term.run < 0) return term;                          // <--- I believe it should be a syntax error
         i += term.run;
         op_node.AddChild(A_node);
         op_node.AddChild(B_node);
         mark = i;
         i += CountWhiteSpaces(st.substring(i, len));
         if (i == len) break;
         A_node = op_node;
         op = st.charAt(i);
      }

      return new tNodeCarrier(op_node, mark);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabTerm(String st) {
      int len;
      int i;
      int op;
      int mark;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                   // our fail return is a (null node, negative run)
      tNodeCarrier factor = null;
      tSyntaxNode  A_node = null;
      tSyntaxNode  B_node = null;
      tSyntaxNode  op_node = null;

      factor = GrabFactor(st);
      if (factor.run < 0) return factor;
      op_node = A_node = factor.node;

      mark = i = factor.run;
      len = st.length();

      i += CountWhiteSpaces(st.substring(i, len));
      if (i == len) return factor;
      op = st.charAt(i);
      while ((op == '*') || (op == '/')) {
         i++;
         op_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, op);
         factor = GrabFactor(st.substring(i, len));
         B_node = factor.node;
         if (factor.run < 0) return factor;                          // <--- I believe it should be a syntax error
         i += factor.run;
         op_node.AddChild(A_node);
         op_node.AddChild(B_node);
         mark = i;
         i += CountWhiteSpaces(st.substring(i, len));
         if (i == len) break;
         A_node = op_node;
         op = st.charAt(i);
      }

      return new tNodeCarrier(op_node, mark);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabFactor(String st) {
      int len;
      int i;
      int op;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                   // our fail return is a (null node, negative run)
      tNodeCarrier value = null;
      tSyntaxNode  base = null;
      tSyntaxNode  exponent = null;
      tSyntaxNode  op_node = null;

      value = GrabValue(st);
      if (value.run < 0) return value;
      base = value.node;

      i = value.run;
      len = st.length();

      i += CountWhiteSpaces(st.substring(i, len));
      if (i == len) return value;
      op = st.charAt(i);
      if (op != '^') return value;
      i++;
      value = GrabFactor(st.substring(i, len));
      exponent = value.node;
      if (value.run < 0) return value;
      i += value.run;
      op_node = new tSyntaxNode(tSyntaxNode.cOPERATOR, op);
      op_node.AddChild(base);
      op_node.AddChild(exponent);

      return new tNodeCarrier(op_node, i);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabKeyword(String st) {
      tSyntaxNode tmp = null;
      int len;
      int i;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                                 // our fail return is a (null node, negative run)
      tNodeCarrier syntax_error;

      tNodeCarrier test = GrabIdentifier(st, false);
      if (test.run < 0) return test;
      String keyword = test.node.ident;
      i = test.run;
      len = st.length();

      //..........................................................................
      if (keyword.equals("lsvars")) {
         i += CountWhiteSpaces(st.substring(i, len));
         if (i != len) {
            tmp =  new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: \"lsvars\" is a reserved keyword!");
            return new tNodeCarrier(tmp, -1);
         }

         String msg = "";
         Enumeration<String> e = var_table.keys();
         while (e.hasMoreElements()) {
            String ident = e.nextElement();
            tVarStore var = var_table.get(ident);
            if (var.type == tDOUBLE) {
               msg += String.format("%-10s", ident) + Double.toString(var.num) + "\n";
            } else {
               msg += String.format("%-10s", ident) + var.str + "\n";
            }
         }
         tmp = new tSyntaxNode(tSyntaxNode.cMISC, msg);
         return new tNodeCarrier(tmp, -1);
      }
      //..........................................................................
      if (keyword.equals("static")) {

         String error_msg = "";
         error_msg += "ERROR: keyword \"static\" : syntax error!\n";
         error_msg += "USAGE: \"static\" [identifier] '=' [expression]\n";
         error_msg += "EXAMPLE: static myexp = (3+y)*x\n";
         error_msg += "NOTE: 'x' and 'y' should be initialized before executing 'myexp'";

         syntax_error = new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cERROR, error_msg), -1);

         test = GrabIdentifier(st.substring(i, len), true);
         if (test.run < 0) return syntax_error;
         i += test.run;
         String identifier = test.node.ident;

         i += CountWhiteSpaces(st.substring(i, len));
         if (i == len) return syntax_error;

         if (st.charAt(i) != '=') return syntax_error;
         i++;
         i += CountWhiteSpaces(st.substring(i, len));
         if (i == len) return syntax_error;
         String exp_str = st.substring(i, len);
//         tSyntaxNode exp_node = new tSyntaxNode(tSyntaxNode.cPARSE, identifier, exp_str);
         var_table.put(identifier, new tVarStore(exp_str));
         return new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cMISC, exp_str), -1);
      }
      //..........................................................................
      if (keyword.equals("help")) {
         i += CountWhiteSpaces(st.substring(i, len));
         if (i != len) {
            tmp =  new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: \"help\" is a reserved keyword!");
            return new tNodeCarrier(tmp, -1);
         }

         String msg = "";
         msg += "Built in Functions:\n";
         msg += String.format("%-10s", "lsvars")   + "- list variables\n";
         msg += String.format("%-10s", "static")   + "- declare a static expression\n";
         msg += String.format("%-10s", "rep")      + "- repeat last entry\n";
         msg += String.format("%-10s", "clean")    + "- delete all variables\n";    
         msg += String.format("%-10s", "random()") + "- returns a random number between 0 and 1\n";
         msg += String.format("%-10s", "pi()")     + "- returns pi with 15 digits of precision\n";
         msg += String.format("%-10s", "e()")      + "- returns e with 15 digits of precision\n";
         msg += String.format("%-10s", "abs(x)")   + "- absolute value of x\n";
         msg += String.format("%-10s", "sqrt(x)")  + "- square-root of x\n";
         msg += String.format("%-10s", "cbrt(x)")  + "- cube-root of x\n";
         msg += String.format("%-10s", "exp(x)")   + "- e to the power of x\n"; // exp is a reserved keyword and e()^y works pretty good
         msg += String.format("%-10s", "expm1(x)") + "- (e to the power of x) - 1\n";
         msg += String.format("%-10s", "ln(x)")    + "- natural log of x\n";
         msg += String.format("%-10s", "log(x)")   + "- log base 10 of x\n";
         msg += String.format("%-10s", "round(x)") + "- x rounded\n";
         msg += String.format("%-10s", "floor(x)") + "- floor of x\n";
         msg += String.format("%-10s", "ceil(x)")  + "- ceiling of x\n";
         msg += String.format("%-10s", "cos(x)")   + "- cosine of x\n";
         msg += String.format("%-10s", "sin(x)")   + "- sine of x\n";
         msg += String.format("%-10s", "tan(x)")   + "- tangent of x\n";
         msg += String.format("%-10s", "acos(x)")  + "- arc-cosine of x\n";
         msg += String.format("%-10s", "asin(x)")  + "- arc-sine of x\n";
         msg += String.format("%-10s", "atan(x)")  + "- arc-tangent of x\n\n";
         msg += "Supported Operators:\n";
         msg += "+ - * / basic arithmatic operators\n";
         msg += "^ exponent x^y is x raised to the power of y\n";
         msg += "( ) precedence grouping\n";
         msg += "= assignment\n";
         msg += "+= -= += *= /= special assignment operators x+=y is the same as x = x + y\n\n";
         msg += "pressing [done] with no input repeats the last entry";
         tmp = new tSyntaxNode(tSyntaxNode.cMISC, msg);
         return new tNodeCarrier(tmp, -1);
      }
      //..........................................................................
      if (keyword.equals("rep")) {
         i += CountWhiteSpaces(st.substring(i, len));
         if (i != len) {
            tmp =  new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: \"rep\" is a reserved keyword!");
            return new tNodeCarrier(tmp, -1);
         }

         current_entry = previous_entry;
         tmp = new tSyntaxNode(tSyntaxNode.cPARSE, current_entry);
         return new tNodeCarrier(tmp, i);
      }
      //..........................................................................
      if (keyword.equals("clean")) {
         i += CountWhiteSpaces(st.substring(i, len));
         if (i != len) {
            tmp =  new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: \"rep\" is a reserved keyword!");
            return new tNodeCarrier(tmp, -1);
         }
         var_table.clear();
         tmp = new tSyntaxNode(tSyntaxNode.cMISC, "done!");
         return new tNodeCarrier(tmp, -1);
      }
      //..........................................................................
/*      if (keyword.equals("load")) {
         Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
         intent.setType("file/*");
         startActivityForResult(intent, 1024);
      }
*/
      //..........................................................................

      return fail;
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabCodeExpression(String st) {
      tNodeCarrier test;

      test = GrabAssignment(st);
      if (test.node != null) return test;

      test = GrabMathExpression(st);
      if (test.node != null) return test;

      return new tNodeCarrier(null, -1);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabStatement(String st) {
      tNodeCarrier test;
      tNodeCarrier fail = new tNodeCarrier(null, -1);

      test = GrabKeyword(st);
      if (test.node != null) return test;

      int i = 0;
      int len = st.length();
      int comma = 0;
      tSyntaxNode head = null;
      tSyntaxNode current = null;
      do {
         comma = 0;
         i += CountWhiteSpaces(st.substring(i, len));
         test = GrabCodeExpression(st.substring(i, len));
         if (test.run < 0) return test;

         if (head == null) {
            head = current = test.node;
         } else {
            current.sibling = test.node;
            current = current.sibling;
         }

         i += test.run;
         i += CountWhiteSpaces(st.substring(i, len));
         if (i == len) break;
         if (st.charAt(i) == ',') {
            comma = 1;
            i++;
         }
      } while (comma == 1);
 
      return new tNodeCarrier(head, i);
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabNumber(String st) {
      tSyntaxNode tmp;
      int len;
      int i;
      Long number;
      Long decimal;
      Long fraction;
      Long sign = 1L;

      tNodeCarrier fail  = new tNodeCarrier(null, -1);                                 // our fail return is a (null node, negative run)
      tNodeCarrier error = new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: improper numeric format"), -1);
      tNodeCarrier bounds_error = new tNodeCarrier(new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: number out of bounds"), -1);

      if (st == null) return fail;
      len = st.length();
      if (len == 0) return fail;

      i = CountWhiteSpaces(st);
      if (i == len) return fail;                                       // all we got was whitespaces :(

      if (st.charAt(i) == '-') {                                       // check for a neg sign
         i++;
         sign = -1L;
      }

      if (st.charAt(i) < '0' || st.charAt(i) > '9') return fail;       // we require a digit to start .# not allowed

      number = 0L;
      for (; i < len; i++) {                                           // grab the whole number part
         if (st.charAt(i) < '0' || st.charAt(i) > '9') break;
         number *= 10L;
         number += (st.charAt(i) - '0');
         if (number < 0L) return bounds_error;
      }
      if (i == len) {                                                  // buffer is empty, it was just a whole number, we are done
         number *= sign;
         tmp = new tSyntaxNode(number.doubleValue());
         return new tNodeCarrier(tmp, i);
      }

      if (st.charAt(i) == '.') {                                      // at this point we require a decimal point to go any further
         i++;
 
         if (i == len) return error;                                     // we don't allow #. either
         if (st.charAt(i) < '0' || st.charAt(i) > '9') return error;     // we require a digit after the decimal

         decimal = 1L;
         fraction = 0L;
         int count = 0;
         for (; i < len; i++) {                                           // grab the decimal part as a fraction.
            if (st.charAt(i) < '0' || st.charAt(i) > '9') break;
            if (count < 18) {
               decimal *= 10L;
               fraction *= 10L;
               fraction += (st.charAt(i) - '0');
            }
            count++;
         }
      
         tmp = new tSyntaxNode(sign*(number + (double) fraction / (double) decimal)); // we got a valid floating point number
         return new tNodeCarrier(tmp, i);           
      }

      number *= sign;
      tmp = new tSyntaxNode(number.doubleValue());
      return new tNodeCarrier(tmp, i);                        // we cant go any further but we got a valid whole number, 

   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabFunction(String st) {
      int len;
      int i;

      tNodeCarrier fail = new tNodeCarrier(null, -1);                  // our fail return is a (null node, negative run)

      tNodeCarrier tmp = GrabIdentifier(st, true);
      if (tmp.run < 0) return fail;
      String identifier = tmp.node.ident;
      i = tmp.run;
      len = st.length();

      tSyntaxNode func = new tSyntaxNode(tSyntaxNode.cFUNCTION, identifier);
      if (i == len) return fail;
      if (st.charAt(i) == '(') {
          i++;
          tNodeCarrier param;
          while (true) {
             i += CountWhiteSpaces(st.substring(i, len));
             if (i == len) return fail;
             param = GrabCodeExpression(st.substring(i, len));
             if (param.run < 0) {
                if (st.charAt(i) == ')') {   // empty param list
                   i++;
                   break;
                }
                return fail;
             }
             i += param.run;
             func.AddChild(param.node);
             i += CountWhiteSpaces(st.substring(i, len));
             if (i == len) return fail;
             if (st.charAt(i) == ',') {
                i++;
                continue;
             }
             if (st.charAt(i) != ')') return fail;
             i++;
             break;
          }
         return new tNodeCarrier(func, i);
      }
      
      return fail;
   }
   //-----------------------------------------------------------------------------
   tNodeCarrier GrabVariable(String st) {
      tNodeCarrier fail = new tNodeCarrier(null, -1);                  // our fail return is a (null node, negative run)

      tNodeCarrier tmp = GrabIdentifier(st, true);
      if (tmp.run < 0) return fail;

      if (tmp.node.type == tSyntaxNode.cIDENTIFIER) {
         // a node for a variable is the same as identifier execept for type
         tmp.node.type = tSyntaxNode.cVARIABLE;
      }

      return tmp;
   }
   //-----------------------------------------------------------------------------
   void Parse(String st) {
      tSyntaxNode syntax_error = new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: dangling characters found: ");
      fail_safe.clear();

      current_entry = st;

      tNodeCarrier stmt = GrabStatement(st);

      if (stmt.run < 0) {
         this.root = stmt.node;
      } else if (stmt.run < st.length()) {
         this.root = syntax_error;
         this.root.aux_msg += "\"" + st.substring(stmt.run, st.length()) + "\"";
      } else {
         this.root = stmt.node;
      }

      previous_entry = current_entry;
   }
   //-----------------------------------------------------------------------------
   tSyntaxNode eval() {
      // ......................................
      class tCourier {
         tSyntaxNode head = null;
         tSyntaxNode current = null;
         void add(tSyntaxNode node) {
            if (head == null) {
               head = current = node;
            } else {
               current.sibling = node;
               current = current.sibling;
            }
         }
      }
      // ......................................
      tCourier ret = new tCourier();
      tSyntaxNode tmp;
      tmp = this.root;
      if (tmp == null) return eval(null);
      while (tmp != null) {
         ret.add(eval(tmp));
         tmp = tmp.sibling;
      }
      return ret.head;
   }
   //-----------------------------------------------------------------------------
   tSyntaxNode eval(tSyntaxNode node) {
      tSyntaxNode tmp_node;

      if (node == null) {
         return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: unexpected error: eval(null)");
      }

      switch (node.type) {
         case tSyntaxNode.cVALUE:
            return new tSyntaxNode(node.value);
         case tSyntaxNode.cVARIABLE:
            tVarStore var = var_table.get(node.ident);
            if (var == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: uninitialized identifier: " + node.ident + "");
            if (var.type == tDOUBLE) return new tSyntaxNode(var.num);
            return eval(new tSyntaxNode(tSyntaxNode.cPARSE, node.ident, var.str));
         case tSyntaxNode.cFUNCTION:
            tSyntaxNode too_many_params = new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: too many parameters!");
            tSyntaxNode param = node.child;

            if (node.ident.equals("random")) {
               if (param == null) return new tSyntaxNode(Math.random());
               return too_many_params;
            }
            if (node.ident.equals("pi")) {
               if (param == null) return new tSyntaxNode(3.141592653589793);
               return too_many_params;
            }
            if (node.ident.equals("e")) {
               if (param == null) return new tSyntaxNode(2.718281828459045);
               return too_many_params;
            }

            if (param == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: parameter required!");

            param = eval(param);
            if (param.type != tSyntaxNode.cVALUE) return param; 

            if (node.ident.equals("abs")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.abs(param.value));
               return too_many_params;
            }
            if (node.ident.equals("sqrt")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.sqrt(param.value));
               return too_many_params;
            }
            if (node.ident.equals("cbrt")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.cbrt(param.value));
               return too_many_params;
            }
            if (node.ident.equals("exp")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.exp(param.value));
               return too_many_params;
            }
            if (node.ident.equals("expm1")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.expm1(param.value));
               return too_many_params;
            }
            if (node.ident.equals("ln")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.log(param.value));
               return too_many_params;
            }
            if (node.ident.equals("log")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.log10(param.value));
               return too_many_params;
            }
            if (node.ident.equals("round")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.round(param.value));
               return too_many_params;
            }
            if (node.ident.equals("floor")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.floor(param.value));
               return too_many_params;
            }
            if (node.ident.equals("ceil")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.ceil(param.value));
               return too_many_params;
            }
            if (node.ident.equals("cos")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.cos(param.value));
               return too_many_params;
            }
            if (node.ident.equals("sin")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.sin(param.value));
               return too_many_params;
            }
            if (node.ident.equals("tan")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.tan(param.value));
               return too_many_params;
            }
            if (node.ident.equals("acos")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.acos(param.value));
               return too_many_params;
            }
            if (node.ident.equals("asin")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.asin(param.value));
               return too_many_params;
            }
            if (node.ident.equals("atan")) {
               if (node.child.sibling == null) return new tSyntaxNode(Math.atan(param.value));
               return too_many_params;
            }

            return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: unkown function: " + node.aux_msg + "");
         case tSyntaxNode.cOPERATOR:
            switch (node.token) {
               case '=':
                  tSyntaxNode exp_node = node.child;
                  if (exp_node == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: expression node is null");
                  tSyntaxNode var_node = exp_node.sibling;
                  if (var_node == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: variable node is null");
   
                  tmp_node = eval(exp_node);
                  if (tmp_node.type != tSyntaxNode.cVALUE) return tmp_node;

                  var_table.put(var_node.ident, new tVarStore(tmp_node.value));
                  return tmp_node;
               case '^':
               case '+':
               case '-':
               case '*':
               case '/':
                  tmp_node = new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: unknown arithmatic anomaly");
                  tSyntaxNode node_B = node.child;
                  if (node_B == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: term B is null");

                  tSyntaxNode node_A = node_B.sibling;
                  if (node_A == null) return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: term A is null");
   
                  node_B = eval(node_B);
                  if (node_B.type != tSyntaxNode.cVALUE) return node_B;

                  node_A = eval(node_A);
                  if (node_A.type != tSyntaxNode.cVALUE) return node_A;
   
                  switch (node.token) {
                     case '^':
                        tmp_node = new tSyntaxNode(Math.pow(node_A.value, node_B.value));
                        break;
                     case '+':
                        tmp_node = new tSyntaxNode(node_A.value + node_B.value);
                        break;
                     case '-':
                        tmp_node = new tSyntaxNode(node_A.value - node_B.value);
                        break;
                     case '*':
                        tmp_node = new tSyntaxNode(node_A.value * node_B.value);
                        break;
                     case '/':
                        tmp_node = new tSyntaxNode(node_A.value / node_B.value);
                        break;
                  }
                  return tmp_node;
            }
            return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: can't evaluate unknown operator");
         case tSyntaxNode.cERROR:
         case tSyntaxNode.cMISC:
            return node;
         case tSyntaxNode.cPARSE:
            if (!fail_safe.add(node.ident)) {
               return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: infinite static variable expanse detected!");
            }
 
            tNodeCarrier stmt = GrabStatement(node.aux_msg);
            if (stmt.node == null) return eval(null);
            if (stmt.run < 0) return stmt.node;
            if (stmt.run < node.aux_msg.length()) {
               tSyntaxNode error = new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: dangling characters found: ");
               error.aux_msg += "\"" + node.aux_msg.substring(stmt.run, node.aux_msg.length()) + "\"";
               return error;
            }
            
            tSyntaxNode head = null;
            tSyntaxNode current = null;
            head = eval(stmt.node);
            current = head;
            while (stmt.node.sibling != null) {
               stmt.node = stmt.node.sibling;
               current.sibling = eval(stmt.node);
               current = current.sibling;
            }

            fail_safe.remove(node.ident);
   
            return head;
      }
      return new tSyntaxNode(tSyntaxNode.cERROR, "ERROR: can't evaluate unknown type");
   }
   //-----------------------------------------------------------------------------
}
//=============================================================================

