//
// Copyright 2017 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

public abstract class Engine implements IEngine {
        
    @Override
    public AccessPointInfo[] getAvailableNetworks() {

        Utils.logDebug("getAvailableNetworks():");

        if (!WpaSupplicant.start()) {
            Utils.logError("Failed starting wpa_supplicant");
            return null;
        }

        if (!WpaCli.scanNetworks()) {
            Utils.logError("failed scanning networks");
            return null;
        }

        String scanRes = WpaCli.getScanResults();

        if (scanRes == null) {
            Utils.logError("failed getting scan results");
            return null;
        }

        AccessPointInfo[] a = AccessPointInfo.parseScanResult(scanRes);
        if (a == null) {
            Utils.logError("Unable to parse scan file into AccessPointInfo array");
            return a;
        }

        Utils.logDebug("# of APs found: " + a.length);

        return a;

    }

    @Override
    public abstract boolean connect(AccessPointInfo info);

    @Override
    public boolean disconnect() {

        return WpaCli.disconnect();

    }

    /***
     * returns null if unable to determine connection status for any reason.
     */
    @Override
    public ConnectionStatus getConnectionStatus() {
        return WpaCli.getConnectionStatus();
    }

    @Override
    public boolean isInterfaceAvailable(String ifaceName) throws SocketException {

        Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
        while (interfaces.hasMoreElements()) {
            NetworkInterface nif = interfaces.nextElement();
            String nm = nif.getName();
            if (nm.equals(ifaceName)) {
                return true;
            }
        }
        return false;

    }

    public boolean runDhcpcd() {
        // needs root
        // option -w avoids dhcpcd forking to background,
        // in order to keep control over its exit code, and be able to wait for it.
        // option -A avoids ARP IP checking, we use it to save some seconds in the connection process.
        return RootCommand.executeRootCmd("dhcpcd -w -A " + WpaSupplicant.INTERFACE_NAME);

    }
    
    public boolean runDhcpcd(DhcpSettings dhcpConfig) {
        
        if (dhcpConfig == null || dhcpConfig.useDhcp){
            Utils.logDebug("running dhchpc without dhcp settings, reverting to dhcp");
            return runDhcpcd();
        }
        
        Utils.logDebug("Running dhcpcd with custom ip settings");
        String cmdMask = "dhcpcd -w -A -S ip_address=%s -S routers=%s %s";
        String cmd = String.format(cmdMask,
                                    dhcpConfig.getStaticIPwithMask(),
                                    dhcpConfig.getDefaultGateway(),
                                    WpaSupplicant.INTERFACE_NAME);
        
        return RootCommand.executeRootCmd(cmd);
        
    }

    public static boolean killDhcpcd(){
        return RootCommand.executeRootCmd("killall -SIGINT dhcpcd");
    }
    
    public boolean interfaceUp() {
        // needs root (tested)
        return RootCommand.executeRootCmd("ifconfig " + WpaSupplicant.INTERFACE_NAME + " up");
    }

}
