//
// Copyright 2017 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import org.json.JSONException;
import org.json.JSONObject;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public class AccessPointInfo implements Serializable {

    private static final long serialVersionUID = 2L;
    private static final int MAX_SSID_LENGTH = 32;
    protected static final String SCAN_FILE_HDR = "bssid / frequency / signal level / flags / ssid";

    private static final String JSONKEY_BSSID = "BSSID";
    private static final String JSONKEY_SSID = "SSID";
    private static final String JSONKEY_LASTUSED = "LastUsed";
    private static final String JSONKEY_AUTH = "Auth";
    private static final String JSONKEY_PSK = "PSK";
    private static final String JSONKEY_VPN_PROFILE = "VpnProfile";
    private static final String JSONKEY_DHCPSETS = "DhcpSettings";
    
    private String _ssid;
    private String _bssid;
    private String _auth;
    private String _level;
    private String _freq;
    private String _password;
    private String _vpnProfileName = null;
    private boolean _isHidden = false;
    private long _lastTimeUsed;
    private DhcpSettings _dhcpsets;

    public AccessPointInfo(String ssid, String bssid, String authType, String level, String freq) {

        this._ssid = ssid;
        this._bssid = bssid;
        this._auth = authType;
        this._level = level;
        this._freq = freq;

    }

    private AccessPointInfo(){
        // for inner use;
    }
    
    public void setDhcpConfiguration(DhcpSettings sets) {
        this._dhcpsets = sets;
    }

    public DhcpSettings getDhcpConfiguration() {

        if (this._dhcpsets == null) {
            return DhcpSettings.getDefault();
        } else {
            return this._dhcpsets;
        }
    }

    public String getSsid() {
        return this._ssid;
    }

    public String getSsid(int maxLength) {
        String txt = getSsid();
        if (maxLength > 4 && txt.length() > maxLength) {
            txt = txt.substring(0, maxLength - 3) + "...";
        }
        return txt;
    }

    public void setHidden(boolean hidden) {
        this._isHidden = hidden;
    }

    public boolean isHidden() {
        return this._isHidden;
    }

    public String getBssid() {
        return this._bssid;
    }

    public void setBssid(String bssid) {
        this._bssid = bssid;
    }

    public String getVpnProfileName(){
        if (_vpnProfileName == null){
            return "";
        } else {
            return _vpnProfileName;
        }
    }
    
    public void setVpnProfileName(String profileName){
        _vpnProfileName = profileName;
    }
    
    public String getAuthType() {
        if (_auth == null){
            return "";
        }
        return this._auth;
    }

    public int getSignlalStrength() {
        // return this._level;

        if (this._level == null || this._level.isEmpty()) {
            return 0;
        }

        int retval = 0;

        try {
            retval = Integer.parseInt(this._level);
        } catch (NumberFormatException e) {
            retval = 0;
        }

        return retval;

    }

    public String getFrequency() {
        return this._freq;
    }

    public long getLastTimeUsed() {
        return this._lastTimeUsed;
    }

    public void setLastTimeUsed(long timeStampInMillis) {
        this._lastTimeUsed = timeStampInMillis;
    }

    public boolean isOlderThan(int days) {

        if (this._lastTimeUsed == 0) {
            return false;
        }

        long timeDiff = System.currentTimeMillis() - this._lastTimeUsed;
        long spanMillis = Utils.daysToMilliseconds(days);

        if (timeDiff > spanMillis) {
            return true;
        } else {
            return false;
        }

    }

    public String getPassword() {
        if (_password == null){
            return "";
        }
        return this._password;
    }

    public void setPassword(String password) {
        this._password = password;
    }

    public boolean needsPassword() {

        if ((this._auth == null) || (this._auth.equals(""))) {
            // TODO
            // check if default behavior should be with or without password,
            // when no auth info is available.
            return false;
        }

        if (this._auth.contains("WPA2") || this._auth.contains("WPA")) {
            return true;
        } else {
            return false;
        }

    }

    protected static AccessPointInfo parseLine(String line) {

        try {

            String[] params = line.split("\t");
            if (params.length != 5) {
                return null;
            }

            String bssid = params[0];
            String freq = params[1];
            String level = params[2];
            String auth = params[3];
            String ssid = params[4];

            if (ssid.length() == 0 || ssid.length() > MAX_SSID_LENGTH) {
                // invalid SSID.
                return null;
            }

            AccessPointInfo info = new AccessPointInfo(ssid, bssid, auth, level, freq);
            return info;

        } catch (Exception e) {
            Utils.logError("Error while parsing line: " + line, e);
            return null;
        }

    }

    public static AccessPointInfo[] parseScanResult(String scanResultContent) {

        try {

            if (scanResultContent == null) {
                return null;
            }

            Utils.logDebug("AccesPointInfo trying to parse file scan content:\n"
                            + scanResultContent);

            String[] lines = scanResultContent.split("\n");
            List<AccessPointInfo> nets = new ArrayList<AccessPointInfo>();

            if (lines == null) {
                return null;
            }

            for (String l : lines) {
                if (l.startsWith(SCAN_FILE_HDR)) {
                    // strip off the header
                    continue;
                }

                if (l.trim().equals("")) {
                    // empty line, skip.
                    continue;
                }

                // try to parse line into network info
                AccessPointInfo info = AccessPointInfo.parseLine(l);
                if (info == null) {
                    Utils.logError("Failed to parse line into AccessPointInfo: " + l);
                    continue;
                }

                nets.add(info);

            }

            sortInfosBySignalStrength(nets);

            AccessPointInfo[] a = new AccessPointInfo[nets.size()];
            a = nets.toArray(a);
            return a;

        } catch (Exception e) {
            Utils.logError("Error while parsing scan results in class AccessPointInfo", e);
            return null;
        }

    }

    public JSONObject toJson(){
        
        try {
        
            JSONObject j = new JSONObject();
            
            j.put(JSONKEY_BSSID, getBssid());
            j.put(JSONKEY_SSID, getSsid());
            j.put(JSONKEY_PSK, getPassword());
            j.put(JSONKEY_AUTH, getAuthType());
            j.put(JSONKEY_LASTUSED, getLastTimeUsed());
            j.put(JSONKEY_VPN_PROFILE, getVpnProfileName());
            
            DhcpSettings sets = getDhcpConfiguration();
            if (sets != null){
                JSONObject dhcpj = sets.toJson();
                if (dhcpj != null){
                    j.put(JSONKEY_DHCPSETS, dhcpj);
                }
                
            }
            
            return j;
            
        } catch (JSONException e) {
            Utils.logError("Exception while converting AccessPointInfo to JSON.", e);
            return null;
        }
        
    }
    
    public static AccessPointInfo fromJsonObject(JSONObject json){
        
        if (json == null || json.isNull(JSONKEY_BSSID) || json.isNull(JSONKEY_SSID)){
            return null;
        }
        
        AccessPointInfo info = new AccessPointInfo();

        try {
            info._bssid = json.getString(JSONKEY_BSSID);
            info._ssid = json.getString(JSONKEY_SSID);
            info._auth = json.getString(JSONKEY_AUTH);
            info._lastTimeUsed = json.getLong(JSONKEY_LASTUSED);
            
            if (json.has(JSONKEY_PSK) && ! json.isNull(JSONKEY_PSK)){
                info._password = json.getString(JSONKEY_PSK);
            }

            if ( json.has(JSONKEY_VPN_PROFILE) && ! json.isNull(JSONKEY_VPN_PROFILE)){
                info._vpnProfileName = json.getString(JSONKEY_VPN_PROFILE);
            }

            if (json.has(JSONKEY_DHCPSETS) && ! json.isNull(JSONKEY_DHCPSETS)){
                info._dhcpsets = DhcpSettings.fromJsonObject(json.getJSONObject(JSONKEY_DHCPSETS));
            }
            
            return info;
                            
        } catch (JSONException e) {
            Utils.logError("Exception while parsing json object to AccessPointInfo", e);
            return null;
        }
                        
                        
        
        
    }
    
    private static void sortInfosBySignalStrength(List<AccessPointInfo> toSort) {

        Collections.sort(toSort, new Comparator<AccessPointInfo>() {
            public int compare(AccessPointInfo o1, AccessPointInfo o2) {
                if (o1.getSignlalStrength() == o2.getSignlalStrength())
                    return 0;
                return o1.getSignlalStrength() < o2.getSignlalStrength() ? -1 : 1;
            }
        });

    }

}
