//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.network.Engine;
import fil.libre.repwifiapp.network.WpaSupplicant;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;

public class Utils {

    private static final long MILLIS_IN_DAY = 86400000;

    private static Exception _lastException = null;

    public static Exception getLastException() {
        return _lastException;
    }
    
    public static boolean writeFileLines(String filePath, String[] lines, boolean overwrite) {

        if (lines == null) {
            return false;
        }

        if (lines.length == 0) {
            return true;
        }

        StringBuilder sb = new StringBuilder();

        for (String l : lines) {

            if (l == null) {
                return false;
            }

            sb.append(l + "\n");
        }

        return writeFile(filePath, sb.toString(), overwrite);

    }

    public static boolean writeFile(String filePath, String content, boolean overwrite) {

        if (content == null) {
            return false;
        }

        FileWriter writer = null;

        try {

            writer = new FileWriter(filePath, (!overwrite));
            writer.write(content);

            return true;

        } catch (Exception e) {
            _lastException = e;
            return false;

        } finally {

            if (writer != null) {
                try {
                    writer.close();
                } catch (IOException e) {
                    Logger.logError("error while closing filewriter", e);
                }
            }

        }

    }

    public static String[] readFileLines(String filePath) {

        if (filePath == null) {
            return null;
        }

        File f = new File(filePath);
        if (!f.exists()) {
            Logger.logError("File doesn't exist: " + filePath);
            return null;
        }

        FileReader fr = null;
        BufferedReader bufr = null;

        List<String> lines = new ArrayList<String>();

        try {

            fr = new FileReader(filePath);
            bufr = new BufferedReader(fr);
            String line = "";

            while ((line = bufr.readLine()) != null) {
                lines.add(line);
            }

            String[] ar = new String[lines.size()];

            return lines.toArray(ar);

        } catch (Exception e) {
            Logger.logError("Error while reading file " + filePath, e);
            return null;

        } finally {
            try {
                if (bufr != null) {
                    bufr.close();
                }
            } catch (IOException ex) {
                Logger.logError("error while closing filereader", ex);
            }
            try {
                if (fr != null) {
                    fr.close();
                }
            } catch (IOException exc) {
                Logger.logError("error while closing filereader", exc);
            }
        }

    }

    public static String readFile(String filePath) {

        if (filePath == null) {
            return null;
        }

        File f = new File(filePath);
        if (!f.exists()) {
            Logger.logError("File doesn't exist: " + filePath);
            return null;
        }

        FileReader fr = null;
        BufferedReader bufr = null;

        StringBuilder sb = new StringBuilder();

        try {

            fr = new FileReader(filePath);
            bufr = new BufferedReader(fr);
            String line = "";

            while ((line = bufr.readLine()) != null) {
                sb.append(line);
                sb.append("\n");
            }

            return sb.toString();

        } catch (Exception e) {
            Logger.logError("Error while reading file " + filePath, e);
            return null;

        } finally {
            try {
                if (bufr != null) {
                    bufr.close();
                }
            } catch (IOException ex) {
                Logger.logError("error while closing filereader", ex);
            }
            try {
                if (fr != null) {
                    fr.close();
                }
            } catch (IOException exc) {
                Logger.logError("error while closing filereader", exc);
            }
        }

    }

   /* private static String flagExists = "EXISTS";
    private static String flagNotExists = "NOT-EXISTS";
    public static boolean fileExistsRoot(String filePath) throws Exception{
        
        String cmd = "if [ -e \""+ filePath +"\" ]; then echo \"" + flagExists + "\"; else echo \""+ flagNotExists + "\";fi";
        
        RootCommand su = new RootCommand(cmd);
        if (su.execute() !=0){
            throw new Exception("RepWifi.Utils Failed to check for file existence!");
        }
        
        if (su._cmdOut.trim().equals(flagExists)){
            return true;
        } else if (su._cmdOut.trim().equals(flagNotExists)){
            return false;
        } else {
            throw new Exception("RepWifi.Utils received unknown flag while checking for file existence!");
        }
    }*/
    
    /* public static boolean createDirectoryRoot(String dirPath){
        
        String cmd = "mkdir " + dirPath;
        return RootCommand.executeRootCmd(cmd);        
        
    }
    */
    public static long daysToMilliseconds(int days) {
        return (days * MILLIS_IN_DAY);
    }

    public static long millisecondsToDays(long milliseconds) {
        return (milliseconds / MILLIS_IN_DAY);
    }

    public static String netmaskIntToString(int mask) {

        if (mask < 8 || mask > 32) {
            return null;
        }

        StringBuilder sb = new StringBuilder(32);
        StringBuilder sb2 = new StringBuilder(15);

        for (int i = 0; i < mask; i++) {
            sb.append("1");
        }

        for (int i = 0; i < 32 - mask; i++) {
            sb.append("0");
        }

        for (int i = 0; i < 3; i++) {
            String bitString = sb.substring((i * 8), (i * 8) + 8);
            int ibyte = Integer.parseInt(bitString, 2);
            sb2.append(ibyte);
            sb2.append(".");
        }
        String bitString = sb.substring(24, 32);
        int ibyte = Integer.parseInt(bitString, 2);
        sb2.append(ibyte);

        return sb2.toString();

    }

    public static int netmaskStringToInt(String mask) {

        if (mask == null) {
            Logger.logError("netmaskStringToInt received null mask");
            return -1;
        }

        mask = mask.trim();

        String[] octs = mask.split("\\.");
        if (octs.length != 4) {
            Logger.logError("netmaskStringToInt invalid input string: " + mask);
            return -1;
        }

        int intmask = 0;
        boolean prevIsZero = false;
        for (String o : octs) {

            int intval = 0;

            try {
                intval = Integer.parseInt(o, 10);
            } catch (NumberFormatException e) {
                Logger.logError("netmaskStringToInt", e);
                return -1;
            }

            String b = Integer.toBinaryString(intval);
            if (b.length() != 8 && b.contains("1")) {
                // invalid mask! has ones after a zero
                Logger.logError("netmaskStringToInt invalid mask! has ones after a zero");
                return -1;
            }
            for (int i = 0; i < b.length(); i++) {
                if (b.charAt(i) == '0') {
                    prevIsZero = true;

                } else if (prevIsZero) {
                    // invalid mask
                    Logger.logError("netmaskStringToInt invalid mask");
                    return -1;

                } else {
                    intmask += 1;
                }

            }

        }

        return intmask;

    }

    public static String rawResourceAsString(Context context, int resId) {

        InputStream s = null;
        Scanner scan = null;

        try {

            s = context.getResources().openRawResource(resId);
            scan = new Scanner(s, "UTF-8").useDelimiter("\\A");
            return scan.next();

        } catch (Exception e) {
            Logger.logError("Exception while reading raw resource as string", e);
            return null;

        } finally {

            try {
                if (scan != null) {
                    scan.close();
                }
            } catch (Exception e2) {
            }

            try {
                if (s != null) {
                    s.close();
                }
            } catch (Exception e2) {
            }

        }

    }

    public static void showMessage(String msg, Context context) {

        AlertDialog.Builder dlgAlert = new AlertDialog.Builder(context,
                        R.style.Theme_RepWifiDialogTheme);
        dlgAlert.setMessage(msg);
        dlgAlert.setPositiveButton(context.getString(android.R.string.ok),
                        new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int whichButton) {
                                return;
                            }
                        });

        dlgAlert.setCancelable(false);
        AlertDialog diag = dlgAlert.create();

        diag.show();

    }

    public static void killBackEnd(Context context, boolean silent) {

        if (silent) {

            Engine.killDhcpcd();
            WpaSupplicant.kill();

        } else {

            String msg = context.getString(R.string.confirm_kill_backend);
            AlertDialog.Builder dlgAlert = new AlertDialog.Builder(context,
                            R.style.Theme_RepWifiDialogTheme);
            dlgAlert.setMessage(msg);
            dlgAlert.setPositiveButton(context.getString(android.R.string.ok),
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int whichButton) {
                                    killBackEnd(null, true);
                                    return;
                                }
                            });
            dlgAlert.setNegativeButton(context.getString(android.R.string.cancel),
                            new DialogInterface.OnClickListener() {
                                @Override
                                public void onClick(DialogInterface dialog, int whichButton) {
                                    return;
                                }
                            });

            dlgAlert.setCancelable(false);
            AlertDialog diag = dlgAlert.create();

            diag.show();
            return;

        }

    }

    
}
