//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import fil.libre.repwifiapp.Commons;
import fil.libre.repwifiapp.Utils;
import java.io.DataOutputStream;
import java.io.InputStream;
import java.util.concurrent.TimeoutException;


public class RootCommand extends ShellCommand {

    protected static final String CMD_WRAPPING = "TEMPOUT=\"$(%s)\";ec=$?;echo \"$TEMPOUT\";echo $ec > \"%s\";exit $ec";
    //protected static final String CMD_WRAPPING = "TEMPOUT=\"$(%s)\";ec=$?;echo \"$TEMPOUT\";exit $ec";
    
    public RootCommand(String commandText) {
        super(commandText);
        this._cmdTxt = commandText;
    }
    
    public static boolean executeRootCmd(String cmd){
        // won't return 
        try {
            return executeRootCmd(cmd, -1);
        } catch (TimeoutException e){
            // will never throw timeout exception as it won't use asynchronous wait.
            return false;
        }
    }
    
    public static boolean executeRootCmd(String cmd, long timeoutMillis) throws TimeoutException{
        
        try {

            RootCommand c = new RootCommand(cmd);
            if (c.execute(timeoutMillis) == 0) {
                return true;
            } else {
                return false;
            }

        }
        catch (TimeoutException te){
            throw te;
        }
        catch (Exception e) {
            Logger.logError("Error executing \"" + cmd + "\"", e);
            return false;
        }
    }
    
    @Override
    public int execute() throws Exception {
        return execute(-1);
    }
    
    public int execute(long timeoutMillis) throws Exception{
        
        if (this._cmdTxt == null) {
            return EXITCODE_INVALID_INPUT;
        }
        
        Process su = Runtime.getRuntime().exec("su");

        DataOutputStream stdin = new DataOutputStream(su.getOutputStream());
        InputStream os = su.getInputStream();
        InputStream es = su.getErrorStream();

        Logger.logDebug("SU:EXEC: " + this._cmdTxt);

        String wrappedCmd = String.format(CMD_WRAPPING, this._cmdTxt, Commons.getExitCodeTempFile());
        
        stdin.writeBytes(wrappedCmd + "\n");
        stdin.flush();

        StringBuilder sb = new StringBuilder();

        sb.append(getStringFromStream(es));
        sb.append(getStringFromStream(os));

        int res;
        
        if (timeoutMillis <= MIN_TIMEOUT_MILLIS){
            res = su.waitFor();
            
        } else {
            Logger.logDebug("Executing command with " + timeoutMillis + "ms timeout.");
            ProcessTimeout w = new ProcessTimeout();
            res = w.waitFor(su, timeoutMillis);            
        }

        // re-read the output, in case it was empty when first tried
        sb.append(getStringFromStream(es));
        sb.append(getStringFromStream(os));

        this._cmdOut = sb.toString();

        Logger.logDebug("OUT: " + getOutput());

        if (res == 0){
            // could be su's own exit code hiding the original one:
            res = readLastExitCodeFromFile();
        }
        
        Logger.logDebug("ExitCode: " + res);
        return res;
        
    }
    
    public int testRootAccess() throws Exception {

        Process su = Runtime.getRuntime().exec("su");

        DataOutputStream stdin = new DataOutputStream(su.getOutputStream());

        Logger.logDebug("Testing root access: executing simple \"su\"");
        stdin.writeBytes("exit\n");
        stdin.flush();

        int res = su.waitFor();

        Logger.logDebug("Simple \"su\" exitcode: " + res);
        
        return res;

    }
    
    private int readLastExitCodeFromFile(){

        String strec = Utils.readFile(Commons.getExitCodeTempFile()).trim();
        try{
            return Integer.parseInt(strec);
        }catch(NumberFormatException e){
            Logger.logError("NumberFormatException while parsing contents of ExitCodeTempFile: " + strec);
            return EXITCODE_PARSING_ERROR;
        }
    }
    
}
