//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************

package fil.libre.repwifiapp.helpers;

import java.io.IOException;
import java.io.InputStream;
import java.util.concurrent.TimeoutException;

public class ShellCommand {

    public static final int EXITCODE_INVALID_INPUT = -9;
    public static final int EXITCODE_PARSING_ERROR = 999;
    public static final int MIN_TIMEOUT_MILLIS = 100;

    protected String _cmdOut = "";
    protected String _cmdTxt = "";

    public ShellCommand(String commandText) {
        this._cmdTxt = commandText;
    }

    public int execute() throws Exception {

        if (this._cmdTxt == null) {
            return EXITCODE_INVALID_INPUT;
        }

        Logger.logDebug("EXEC: " + this._cmdTxt);

        Process cmd = Runtime.getRuntime().exec(this._cmdTxt);
        

        InputStream os = cmd.getInputStream();
        InputStream es = cmd.getErrorStream();

        StringBuilder sb = new StringBuilder();

        sb.append(getStringFromStream(es));
        sb.append(getStringFromStream(os));

        int res = cmd.waitFor();

        // re-read the output, in case it was empty when first tried
        sb.append(getStringFromStream(es));
        sb.append(getStringFromStream(os));

        this._cmdOut = sb.toString();

        Logger.logDebug("EXITCODE: " + res);
        Logger.logDebug("OUT: " + getOutput());

        return res;

    }

    protected String getStringFromStream(InputStream s) throws IOException {
        java.util.Scanner sc = new java.util.Scanner(s,"UTF-8").useDelimiter("\\A");
        return sc.hasNext() ? sc.next() : "";
    }

    public String getOutput() {

        return this._cmdOut;

    }
    
    protected class ProcessTimeout{
        
        private Process _proc;
        private int _exitCode;
        private boolean _hasExited = false;
        private InterruptedException e = null;
        
        public int waitFor(Process p, long timeoutMillis) throws InterruptedException, TimeoutException{
            
            // waits for a maximum of timeoutMillis milliseconds
            // for the process to exit.        
            int msWaited = 0;
            this._proc = p;
            startWaitingForProcess();
            while (msWaited < timeoutMillis) {

                if (e != null){
                    // exception from the underlying thread;
                    throw e;
                }
                
                if (! _hasExited){
                    Thread.sleep(MIN_TIMEOUT_MILLIS);
                    msWaited += MIN_TIMEOUT_MILLIS;
                    
                } else {
                    return _exitCode;
                }

            }
            
            throw new TimeoutException("Timeout elapsed while waiting for inner process to finish its execution.");
            
        }
        
        private void startWaitingForProcess(){

            new Thread(new Runnable() {
                public void run(){
                    try {
                        int exitCode = _proc.waitFor();
                        setExitCode(exitCode);
                    } catch (InterruptedException e) {
                        setException(e);
                    }
                }
            }).start();           
            
        }
        
        private void setExitCode(int ec){
            _hasExited = true;
            _exitCode = ec;
        }
        
        private void setException(InterruptedException e){
            this.e = e;
        }
        
    }
    
}

