//
// Copyright 2017, 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************


package fil.libre.repwifiapp.network;

import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.helpers.RootCommand;

public abstract class WpaCli {

    private static final int SCAN_WAIT_INTERVAL = 4;
    private static final String BASE_COMMAND = "wpa_cli -p" + WpaSupplicant.SOCKET_DIR + " -P"
                    + WpaSupplicant.PID_FILE + " -i" + WpaSupplicant.INTERFACE_NAME;

    public static String createNetworkGetId() {

        try {

            RootCommand su = new RootCommand(BASE_COMMAND + " add_network");
            if (su.execute() == 0) {
                String out = su.getOutput();
                if (out == null || out.trim().equals("")) {
                    return null;
                } else {
                    return out.replace("\n", "");
                }
            } else {
                return null;
            }

        } catch (Exception e) {
            Logger.logError("Error while creating network", e);
            return null;
        }

    }

    public static boolean setNetworkSSID(String ssid, String networkID) {

        try {

            // needs root (wpa_cli)
            return executeCmd(BASE_COMMAND + " set_network " + networkID + " ssid '\"" + ssid
                            + "\"'");

        } catch (Exception e) {
            Logger.logError("Error while setting network SSID", e);
            return false;
        }

    }

    public static boolean setNetworkPSK(AccessPointInfo info, String networkID) {

        try {

            // needs root (wpa_cli)

            String cmdSetPass = null;
            if (info.needsPassword()) {
                cmdSetPass = BASE_COMMAND + " set_network " + networkID + " psk '\""
                                + info.getPassword() + "\"'";
            } else {
                cmdSetPass = BASE_COMMAND + " set_network " + networkID + " key_mgmt NONE";
            }

            return executeCmd(cmdSetPass);

        } catch (Exception e) {
            Logger.logError("Error while setting network PSK", e);
            return false;
        }

    }

    public static boolean setNetworkScanSSID(String networkID) {

        try {

            // needs root (wpa_cli)
            return executeCmd(BASE_COMMAND + " set_network " + networkID + " scan_ssid 1");

        } catch (Exception e) {
            Logger.logError("Error while setting network SSID", e);
            return false;
        }
    }

    public static boolean selectNetwork(String networkID) {

        try {

            // needs root (wpa_cli)
            return executeCmd(BASE_COMMAND + " select_network " + networkID);

        } catch (Exception e) {
            Logger.logError("Error while selecting network", e);
            return false;
        }

    }

    public static boolean enableNetwork(String networkID) {

        try {

            // needs root (wpa_cli)
            return executeCmd(BASE_COMMAND + " enable_network " + networkID);

        } catch (Exception e) {
            Logger.logError("Error while enabling network", e);
            return false;
        }

    }

    public static boolean scanNetworks() {

        // needs root (for wpa_supplicant and wpa_cli)
        return RootCommand.executeRootCmd(BASE_COMMAND
                        + " scan; if [ $? -ne 0 ]; then exit 1; fi; sleep " + SCAN_WAIT_INTERVAL);

    }

    public static String getScanResults() {

        try {

            RootCommand su = new RootCommand(BASE_COMMAND
                            + " scan_results; if [ $? -ne 0 ]; then exit 1; fi ");
            if (su.execute() == 0) {
                String out = su.getOutput();
                if (out == null || out.trim().equals("")) {
                    return null;
                } else {
                    return out;
                }
            } else {
                return null;
            }

        } catch (Exception e) {
            Logger.logError("Error while executing wpa_cli status", e);
            return null;
        }

    }

    /***
     * returns null if unable to determine connection status for any reason.
     */
    public static ConnectionStatus getConnectionStatus() {

        Logger.logDebug("called getConnecitonStatus()");
        if (!WpaSupplicant.isRunning()) {
            // wpa_supplicant is not running.
            // unable to determine status.
            Logger.logDebug("wpa not running, cannot get connection status.");
            return null;

        }

        try {

            RootCommand su = new RootCommand(BASE_COMMAND + " status");
            if (su.execute() == 0) {
                String out = su.getOutput();
                if (out == null || out.trim().equals("")) {
                    return null;
                } else {
                    return ConnectionStatus.parseWpaCliOutput(out);
                }
            } else {
                return null;
            }

        } catch (Exception e) {
            Logger.logError("Error while executing wpa_cli status", e);
            return null;
        }

    }

    public static boolean disconnect() {

        // needs root (for wpa_cli)

        if (!WpaSupplicant.isRunning()) {
            return true;
        }

        try {

            return executeCmd(BASE_COMMAND + " disconnect");
         
        } catch (Exception e) {
            Logger.logError("Error while enabling network", e);
            return false;
        }
    }

    public static boolean terminateSupplicant() {

        if (!WpaSupplicant.isRunning()){
            return true;
        }
        try {

            return executeCmd(BASE_COMMAND + " terminate");
         
        } catch (Exception e) {
            Logger.logError("Error while enabling network", e);
            return false;
        }
        
    }

    private static boolean executeCmd(String cmdTxt) throws Exception {

        RootCommand su = new RootCommand(cmdTxt);
        if (su.execute() == 0) {
            String out = su.getOutput();
            if (out != null && out.trim().replace("\n", "").equals("OK")) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }

    }

}
