//
// Copyright 2018 Filippo "Fil" Bergamo <fil.bergamo@riseup.net>
// 
// This file is part of RepWifiApp.
//
// RepWifiApp is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// RepWifiApp is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with RepWifiApp.  If not, see <http://www.gnu.org/licenses/>.
// 
// ********************************************************************
package fil.libre.repwifiapp.service;

import android.os.Handler;
import fil.libre.repwifiapp.helpers.Logger;
import fil.libre.repwifiapp.network.ConnectionStatus;
import fil.libre.repwifiapp.network.IEngine;

public class StatusManager {

    public interface ConnectionStatusChangeListener {

        public void onConnectionStatusChange(ConnectionStatus status);

    }

    private Handler handler;
    private Runnable runnable;
    private boolean isStarted = false;
    private int interval = 30000;
    private ConnectionStatus currentStatus;

    private IEngine eng = null;
    private ConnectionStatusChangeListener listener = null;

    public StatusManager(IEngine engine, ConnectionStatusChangeListener listener) {

        this.listener = listener;
        handler = new Handler();
        currentStatus = ConnectionStatus.getDummyDisconnected();
        eng = engine;

    }

    public ConnectionStatus getConnectionStatus() {
        updateStatus();
        return currentStatus;
    }

    private void updateStatus() {

        synchronized (currentStatus) {
            
            ConnectionStatus nextStatus = eng.getConnectionStatus();

            if (nextStatus == null) {
                nextStatus = ConnectionStatus.getDummyDisconnected();
            }

            if (! currentStatus.equals(nextStatus) && listener != null) {
                listener.onConnectionStatusChange(nextStatus);
            }

            this.currentStatus = nextStatus;

        }

    }

    public void startPolling(int intervalMs) {

        synchronized (this) {

            if (isStarted) {
                Logger.logDebug("Called StatusMonitor.start() but monitor already started.. ignoring..");
                return;
            }
            
            isStarted = true;

            this.interval = intervalMs;

            runnable = new Runnable() {
                @Override
                public void run() {

                    try {
                        
                        Logger.logDebug("Polling enabled, connection status update triggered.");
                        updateStatus();

                    } catch (Exception e) {
                        Logger.logError("Exception while executing recurring network status update..",
                                        e);
                    }

                    handler.postDelayed(this, interval);
                }
            };

            Logger.logDebug("Starting recurring network status update every " + interval + " ms");
            handler.postDelayed(runnable, interval);
        }

    }

    public void stopPolling() {

        synchronized (this) {

            if (!isStarted) {
                Logger.logDebug("Called StatusMonitor.stop() but already stopped.. ignoring..");
                return;
            }

            try {
                handler.removeCallbacks(runnable);
                Logger.logDebug("Stopped recurring network status update.");
            } finally {
                isStarted = false;
            }
        }

    }
    
    public void unsetListener(){
        stopPolling();
        this.listener = null;
    }

}
