/*
 * Copyright 2019 Thibault Seisel
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package fr.nihilus.music.media.repo

import android.Manifest
import android.net.Uri
import fr.nihilus.music.core.context.AppDispatchers
import fr.nihilus.music.core.database.playlists.Playlist
import fr.nihilus.music.core.database.playlists.PlaylistDao
import fr.nihilus.music.core.database.playlists.PlaylistTrack
import fr.nihilus.music.core.database.usage.MediaUsageEvent
import fr.nihilus.music.core.database.usage.TrackUsage
import fr.nihilus.music.core.database.usage.UsageDao
import fr.nihilus.music.core.os.PermissionDeniedException
import fr.nihilus.music.core.test.coroutines.CoroutineTestRule
import fr.nihilus.music.core.test.coroutines.NeverFlow
import fr.nihilus.music.core.test.coroutines.test
import fr.nihilus.music.media.playlists.SAMPLE_PLAYLISTS
import fr.nihilus.music.media.playlists.SAMPLE_PLAYLIST_TRACKS
import fr.nihilus.music.media.playlists.TestPlaylistDao
import fr.nihilus.music.media.provider.*
import io.kotlintest.matchers.collections.*
import io.kotlintest.matchers.types.shouldBeNull
import io.kotlintest.matchers.types.shouldNotBeNull
import io.kotlintest.shouldBe
import io.kotlintest.shouldNotThrowAny
import io.kotlintest.shouldThrow
import io.mockk.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.emptyFlow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.launch
import kotlinx.coroutines.test.TestCoroutineScope
import org.junit.Rule
import org.junit.Test

/**
 * Tests that describe and validate the behavior of
 * [MediaRepository][fr.nihilus.music.media.repo.MediaRepository] implementations
 * and especially [MediaRepositoryImpl].
 */
internal class MediaRepositoryTest {

    @get:Rule
    val test = CoroutineTestRule()

    private val dispatchers = AppDispatchers(test.dispatcher)

    /**
     * Execute the given [block] in the context of a child coroutine scope, then cancel that scope.
     * This simulates the whole lifespan of a coroutine scope, cancelling it when [block] ends.
     */
    private fun TestCoroutineScope.runInScope(block: suspend CoroutineScope.() -> Unit) {
        val job = launch(block = block)
        advanceUntilIdle()
        job.cancel()
    }

    @Test
    fun `When requesting all tracks, then load tracks from Dao`() = test.run {
        val sampleTracks = TestTrackDao(SAMPLE_TRACKS)

        runInScope {
            val repository = MediaRepository(this, mediaDao = sampleTracks)
            val tracks = repository.getTracks()

            tracks shouldContainExactly SAMPLE_TRACKS
        }
    }

    @Test
    fun `When requesting tracks, then always return the latest track list`() = test.run {
        val initialTracks = listOf(SAMPLE_TRACKS[0])
        val updatedTracks = listOf(SAMPLE_TRACKS[1])
        val trackDao = TestTrackDao(initialTracks)

        runInScope {
            val repository = MediaRepository(this, mediaDao = trackDao)

            val initialLoad = repository.getTracks()
            initialLoad shouldBe initialTracks
            val secondLoad = repository.getTracks()
            secondLoad shouldBe initialTracks

            trackDao.update(updatedTracks)

            val loadAfterAfter = repository.getTracks()
            loadAfterAfter shouldBe updatedTracks
            val secondLoadAfterUpdate = repository.getTracks()
            secondLoadAfterUpdate shouldBe updatedTracks
        }
    }

    @Test
    fun `Given denied permission, when requesting tracks then throw PermissionDeniedException`() = test.run {
        runInScope {
            val repository = MediaRepository(this, mediaDao = PermissionDeniedDao)
            val permissionException = shouldThrow<PermissionDeniedException> {
                repository.getTracks()
            }

            permissionException.permission shouldBe Manifest.permission.READ_EXTERNAL_STORAGE
        }
    }

    @Test
    fun `Given denied permission, when requesting tracks after being granted then load the current track list`() = test.run {
        val grantingPermissionDao = PermissionMediaDao()
        grantingPermissionDao.hasPermissions = false

        runInScope {
            val repository = MediaRepository(this, mediaDao = grantingPermissionDao)
            runCatching { repository.getTracks() }

            grantingPermissionDao.hasPermissions = true
            shouldNotThrowAny {
                repository.getTracks()
            }
        }
    }

    @Test
    fun `When requesting albums, then load albums from Dao`() = test.run {
        val albumDao = TestAlbumDao(SAMPLE_ALBUMS)

        runInScope {
            val repository = MediaRepository(this, mediaDao = albumDao)
            val albums = repository.getAlbums()

            albums shouldContainExactly SAMPLE_ALBUMS
        }
    }

    @Test
    fun `When requesting albums, then always return the latest album list`() = test.run {
        val initialAlbums = listOf(SAMPLE_ALBUMS[0])
        val updatedAlbums = listOf(SAMPLE_ALBUMS[1])
        val albumDao = TestAlbumDao(initialAlbums)

        runInScope {
            val repository = MediaRepository(this, mediaDao = albumDao)

            val initialLoad = repository.getAlbums()
            initialLoad shouldBe initialAlbums
            val secondLoad = repository.getAlbums()
            secondLoad shouldBe initialAlbums

            albumDao.update(updatedAlbums)

            val loadAfterAfter = repository.getAlbums()
            loadAfterAfter shouldBe updatedAlbums
            val secondLoadAfterUpdate = repository.getAlbums()
            secondLoadAfterUpdate shouldBe updatedAlbums
        }
    }

    @Test
    fun `Given denied permission, when requesting albums then throw PermissionDeniedException`() = test.run {
        val failingAlbumDao = TestAlbumDao()
        failingAlbumDao.failWith(PermissionDeniedException(Manifest.permission.READ_EXTERNAL_STORAGE))

        runInScope {
            val repository = MediaRepository(this, mediaDao = failingAlbumDao)
            val permissionException = shouldThrow<PermissionDeniedException> {
                repository.getAlbums()
            }

            permissionException.permission shouldBe Manifest.permission.READ_EXTERNAL_STORAGE
        }
    }

    @Test
    fun `Given denied permission, when requesting albums after being granted then load the current album list`() = test.run {
        val grantingPermissionDao = PermissionMediaDao()
        grantingPermissionDao.hasPermissions = false

        runInScope {
            val repository = MediaRepository(this, mediaDao = grantingPermissionDao)
            runCatching { repository.getAlbums() }

            grantingPermissionDao.hasPermissions = true
            shouldNotThrowAny {
                repository.getAlbums()
            }
        }
    }

    @Test
    fun `When requesting artists, then load artists from Dao`() = test.run {
        val artistDao = TestArtistDao(SAMPLE_ARTISTS)

        runInScope {
            val repository = MediaRepository(this, mediaDao = artistDao)
            val artists = repository.getAlbums()

            artists shouldContainExactly SAMPLE_ARTISTS
        }
    }

    @Test
    fun `When requesting artists, then always return the latest artist list`() = test.run {
        val initialArtists = listOf(SAMPLE_ARTISTS[0])
        val updatedArtists = listOf(SAMPLE_ARTISTS[1])
        val artistDao = TestArtistDao(initialArtists)

        runInScope {
            val repository = MediaRepository(this, mediaDao = artistDao)

            val initialLoad = repository.getArtists()
            initialLoad shouldBe initialArtists
            val secondLoad = repository.getArtists()
            secondLoad shouldBe initialArtists

            artistDao.update(updatedArtists)

            val loadAfterAfter = repository.getArtists()
            loadAfterAfter shouldBe updatedArtists
            val secondLoadAfterUpdate = repository.getArtists()
            secondLoadAfterUpdate shouldBe updatedArtists
        }
    }

    @Test
    fun `Given denied permission, when requesting artists then throw PermissionDeniedException`() = test.run {
        val failingArtistDao = TestArtistDao()
        failingArtistDao.failWith(PermissionDeniedException(Manifest.permission.READ_EXTERNAL_STORAGE))

        runInScope {
            val repository = MediaRepository(this, mediaDao = failingArtistDao)
            val permissionException = shouldThrow<PermissionDeniedException> {
                repository.getArtists()
            }

            permissionException.permission shouldBe Manifest.permission.READ_EXTERNAL_STORAGE
        }
    }

    @Test
    fun `Given denied permission, when requesting artists after being granted then load the current artist list`() = test.run {
        val grantingPermissionDao = PermissionMediaDao()
        grantingPermissionDao.hasPermissions = false

        runInScope {
            val repository = MediaRepository(this, mediaDao = grantingPermissionDao)
            runCatching { repository.getArtists() }

            grantingPermissionDao.hasPermissions = true
            shouldNotThrowAny {
                repository.getArtists()
            }
        }
    }

    @Test
    fun `When requesting playlists, then load playlist from Dao`() = test.run {
        val dao = TestPlaylistDao(SAMPLE_PLAYLISTS, SAMPLE_PLAYLIST_TRACKS)

        runInScope {
            val repository = MediaRepository(this, playlistDao = dao)
            val playlists = repository.getPlaylists()

            playlists shouldContainExactly SAMPLE_PLAYLISTS
        }
    }

    @Test
    fun `When loading playlists, then always return the latest playlist set`() = test.run {
        val original = listOf(SAMPLE_PLAYLISTS[0])
        val updated = listOf(SAMPLE_PLAYLISTS[1])
        val dao = TestPlaylistDao(original, emptyList())

        runInScope {
            val repository = MediaRepository(this, playlistDao = dao)

            val initialLoad = repository.getPlaylists()
            initialLoad shouldBe original
            val secondLoad = repository.getPlaylists()
            secondLoad shouldBe original

            dao.update(updated)

            val loadAfterUpdated = repository.getPlaylists()
            loadAfterUpdated shouldBe updated
            val secondLoadAfterUpdated = repository.getPlaylists()
            secondLoadAfterUpdated shouldBe updated
        }
    }

    @Test
    fun `Given the id of a playlist that does not exists, when requesting its tracks then return null`() = test.run {
        val mediaDao = TestTrackDao(SAMPLE_TRACKS)
        val playlistDao = TestPlaylistDao(SAMPLE_PLAYLISTS, SAMPLE_PLAYLIST_TRACKS)

        runInScope {
            val repository = MediaRepository(this, mediaDao, playlistDao)
            val playlistTracks = repository.getPlaylistTracks(42L)

            playlistTracks.shouldBeNull()
        }
    }

    @Test
    fun `Given an empty existing playlist, when requesting its tracks then return an empty list`() = test.run {
        val mediaDao = TestTrackDao(SAMPLE_TRACKS)
        val playlistDao = TestPlaylistDao(SAMPLE_PLAYLISTS, emptyList())

        runInScope {
            val repository = MediaRepository(this, mediaDao, playlistDao)
            val playlistTracks = repository.getPlaylistTracks(1L)

            playlistTracks.shouldNotBeNull()
            playlistTracks.shouldBeEmpty()
        }
    }

    @Test
    fun `Given an existing playlist, when requesting its tracks then load them from both Dao`() = test.run {
        val mediaDao = TestTrackDao(SAMPLE_TRACKS)
        val playlistDao = TestPlaylistDao(SAMPLE_PLAYLISTS, SAMPLE_PLAYLIST_TRACKS)

        runInScope {
            val repository = MediaRepository(this, mediaDao, playlistDao)
            val playlistTracks = repository.getPlaylistTracks(1L)

            playlistTracks.shouldNotBeNull()
            playlistTracks.shouldContainExactly(SAMPLE_TRACKS[1])
        }
    }

    @Test
    fun `When creating a playlist, then delegate to PlaylistDao`() = test.run {
        val newPlaylist =
            Playlist("My Favorites", Uri.EMPTY)
        val playlistTracks = longArrayOf(481L, 75L)

        val mockDao = mockk<PlaylistDao> {
            coEvery { playlists } returns NeverFlow
            coEvery { createPlaylist(any(), any()) } just Runs
        }

        runInScope {
            val repository = MediaRepository(this, playlistDao = mockDao)
            repository.createPlaylist(newPlaylist, playlistTracks)
        }

        coVerify(exactly = 1) { mockDao.createPlaylist(newPlaylist, playlistTracks) }
    }

    @Test
    fun `When deleting a playlist, then delegate to PlaylistDao`() = test.run {
        val dao = mockk<PlaylistDao> {
            coEvery { playlists } returns NeverFlow
            coEvery { deletePlaylist(any()) } just Runs
        }

        runInScope {
            val repository = MediaRepository(this, playlistDao = dao)
            repository.deletePlaylist(3L)
        }

        coVerify(exactly = 1) { dao.deletePlaylist(3L) }
    }

    @Test
    fun `When deleting tracks, then delegate to MediaDao`() = test.run {
        val deletedTrackIds = longArrayOf(481L, 75L)
        val dao = mockk<MediaDao> {
            coEvery { tracks } returns NeverFlow
            coEvery { albums } returns NeverFlow
            coEvery { artists } returns NeverFlow
            coEvery { deleteTracks(any()) } answers { firstArg<LongArray>().size }
        }

        runInScope {
            val repository = MediaRepository(this, mediaDao = dao)
            val deleteCount = repository.deleteTracks(deletedTrackIds)
            deleteCount shouldBe 2
        }

        coVerify(exactly = 1) { dao.deleteTracks(deletedTrackIds) }
    }

    @Test
    fun `When track list changed for the first time, then do not dispatch change notifications`() = test.run {
        val mediaDao = TestTrackDao(initialTrackList = listOf(SAMPLE_TRACKS[0], SAMPLE_TRACKS[1]))

        runInScope {
            val repository = MediaRepository(this, mediaDao)

            repository.changeNotifications.test {
                repository.getTracks()
                expectNone()
            }
        }
    }

    @Test
    fun `When receiving a track list update, then notify a change of all tracks`() = test.run {
        val initial = listOf(SAMPLE_TRACKS[0], SAMPLE_TRACKS[1])
        val updated = listOf(SAMPLE_TRACKS[1], SAMPLE_TRACKS[2])

        // Given a dao with the specified initial tracks...
        val mediaDao = TestTrackDao(initialTrackList = initial)

        runInScope {
            // and a repository that started caching tracks...
            val repository = MediaRepository(this, mediaDao)
            repository.getTracks()

            // and we are listening to media change notifications...
            repository.changeNotifications.test {
                mediaDao.update(updated)
                expectAtLeast(1)

                values[0] shouldBe ChangeNotification.AllTracks
            }
        }
    }

    @Test
    fun `When receiving a track list update, then notify for the album of each modified track`() = test.run {
        val initial = listOf(SAMPLE_TRACKS[0], SAMPLE_TRACKS[1])
        val updated = listOf(SAMPLE_TRACKS[1], SAMPLE_TRACKS[2])

        // Given a dao with the specified initial tracks...
        val mediaDao = TestTrackDao(initialTrackList = initial)

        runInScope {
            // and a repository that started caching tracks...
            val repository = MediaRepository(this, mediaDao)
            repository.getTracks()

            // and we are listening to media change notifications...
            repository.changeNotifications.test {
                mediaDao.update(updated)

                expectAtLeast(2)
                values.shouldContainAll(
                    ChangeNotification.Album(65),
                    ChangeNotification.Album(102)
                )
            }
        }
    }

    @Test
    fun `When receiving a track list update, then notify for the artist of each modified track`() = test.run {
        val initial = listOf(SAMPLE_TRACKS[0], SAMPLE_TRACKS[1])
        val updated = listOf(SAMPLE_TRACKS[1], SAMPLE_TRACKS[2])

        // Given a dao with the specified initial tracks...
        val mediaDao = TestTrackDao(initialTrackList = initial)

        runInScope {
            // and a repository that started caching tracks...
            val repository = MediaRepository(this, mediaDao)
            repository.getTracks()

            // and we are listening to media change notifications...
            val subscriber = repository.changeNotifications.test {
                // when receiving a new track list
                mediaDao.update(updated)

                // Then receive the expected notifications.
                expectAtLeast(2)
                values.shouldContainAll(
                    ChangeNotification.Artist(26),
                    ChangeNotification.Artist(13)
                )
            }
        }
    }

    @Test
    fun `When receiving an album list update, then notify a change of all albums`() = test.run {
        val initial = listOf(SAMPLE_ALBUMS[0], SAMPLE_ALBUMS[1])
        val updated = listOf(SAMPLE_ALBUMS[1], SAMPLE_ALBUMS[2])

        // Given a dao with the specified initial albums...
        val mediaDao = TestAlbumDao(initialAlbumList = initial)

        runInScope {
            // and a repository that started caching albums...
            val repository = MediaRepository(this, mediaDao)
            repository.getAlbums()

            // and we are listening to media change notifications...
            repository.changeNotifications.test {
                // when receiving a new album list
                mediaDao.update(updated)

                expectAtLeast(1)
                values shouldContain ChangeNotification.AllAlbums
            }
        }
    }

    @Test
    fun `When receiving an album list update, then notify for the artist of each modified album`() = test.run {
        val initial = listOf(SAMPLE_ALBUMS[0], SAMPLE_ALBUMS[1])
        val updated = listOf(SAMPLE_ALBUMS[1], SAMPLE_ALBUMS[2])

        // Given a dao with the specified initial albums...
        val mediaDao = TestAlbumDao(initialAlbumList = initial)

        runInScope {
            // and a repository that started caching albums...
            val repository = MediaRepository(this, mediaDao)
            repository.getAlbums()

            // and we are listening to media change notifications...
            repository.changeNotifications.test {
                // when receiving a new album list
                mediaDao.update(updated)

                expectAtLeast(2)
                values.shouldContainAll(
                    ChangeNotification.Artist(18),
                    ChangeNotification.Artist(13)
                )
            }
        }
    }

    @Test
    fun `When receiving playlists update, then notify a change of all playlists`() = test.run {
        val initial = listOf(SAMPLE_PLAYLISTS[0], SAMPLE_PLAYLISTS[1])
        val updated = listOf(SAMPLE_PLAYLISTS[1], SAMPLE_PLAYLISTS[2])
        val playlistDao = TestPlaylistDao(initialPlaylists = initial)

        runInScope {
            val repository = MediaRepository(this, playlistDao = playlistDao)
            repository.getPlaylists()

            repository.changeNotifications.test {
                playlistDao.update(updated)

                expectAtLeast(1)
                values shouldContain ChangeNotification.AllPlaylists
            }
        }
    }

    @Test
    fun `When receiving an artists update, then notify a change of all artists`() = test.run {
        val initial = listOf(SAMPLE_ARTISTS[0], SAMPLE_ARTISTS[1])
        val updated = listOf(SAMPLE_ARTISTS[1], SAMPLE_ARTISTS[2])
        val mediaDao = TestArtistDao(initialArtistList = initial)

        runInScope {
            val repository = MediaRepository(this, mediaDao)
            repository.getArtists()

            repository.changeNotifications.test {
                mediaDao.update(updated)
                expectAtLeast(1)
                values shouldContain ChangeNotification.AllArtists
            }
        }
    }

    /**
     * Convenience function for creating the [MediaRepository] under test.
     *
     * @param scope The scope in which coroutines run by this repository will be executed.
     * @param mediaDao The source of tracks, artists and albums. Defaults to a dummy implementation.
     * @param playlistDao The source of playlists. Defaults to a dummy implementation.
     * @param usageDao The source of usage statistics. Defaults to a dummy implementation.
     */
    @Suppress("TestFunctionName")
    private fun MediaRepository(
        scope: CoroutineScope,
        mediaDao: MediaDao = DummyMediaDao,
        playlistDao: PlaylistDao = DummyPlaylistDao,
        usageDao: UsageDao = DummyUsageDao
    ) = MediaRepositoryImpl(scope, mediaDao, playlistDao, usageDao)

    private object DummyPlaylistDao : PlaylistDao() {
        override val playlists: Flow<List<Playlist>> get() = NeverFlow
        override suspend fun getPlaylistTracks(playlistId: Long): List<PlaylistTrack> = emptyList()
        override suspend fun getPlaylistsHavingTracks(trackIds: LongArray): LongArray = LongArray(0)
        override suspend fun savePlaylist(playlist: Playlist): Long = 0L
        override suspend fun addTracks(tracks: List<PlaylistTrack>) = Unit
        override suspend fun deletePlaylist(playlistId: Long) = Unit
        override suspend fun deletePlaylistTracks(trackIds: LongArray) = Unit
    }

    private object DummyMediaDao : MediaDao {
        override val tracks: Flow<List<Track>> get() = emptyFlow()
        override val albums: Flow<List<Album>> get() = emptyFlow()
        override val artists: Flow<List<Artist>> get() = emptyFlow()
        override suspend fun deleteTracks(trackIds: LongArray) = 0
    }

    private object DummyUsageDao : UsageDao {
        override suspend fun recordEvent(usageEvent: MediaUsageEvent) = Unit
        override suspend fun getTracksUsage(since: Long): List<TrackUsage> = emptyList()
        override suspend fun deleteEventsForTracks(trackIds: LongArray) = Unit
    }

    private object PermissionDeniedDao : MediaDao {
        override val tracks: Flow<List<Track>> = permissionDeniedFlow()
        override val albums: Flow<List<Album>> = permissionDeniedFlow()
        override val artists: Flow<List<Artist>> = permissionDeniedFlow()

        override suspend fun deleteTracks(trackIds: LongArray): Int {
            throw PermissionDeniedException(Manifest.permission.WRITE_EXTERNAL_STORAGE)
        }

        private fun permissionDeniedFlow() = flow<Nothing> {
            throw PermissionDeniedException(Manifest.permission.READ_EXTERNAL_STORAGE)
        }
    }

    /**
     * A [MediaDao] that returns different results when permission is granted or denied.
     * When permission is granted, all flows emit an empty list an never completes.
     * When permission is denied, all flows emit a [PermissionDeniedException].
     */
    private class PermissionMediaDao : MediaDao {
        /**
         * Whether permission should be granted.
         */
        var hasPermissions = false

        override val tracks: Flow<List<Track>>
            get() = if (hasPermissions) mediaUpdates() else permissionFailure()

        override val albums: Flow<List<Album>>
            get() = if (hasPermissions) mediaUpdates() else permissionFailure()

        override val artists: Flow<List<Artist>>
            get() = if (hasPermissions) mediaUpdates() else permissionFailure()

        override suspend fun deleteTracks(trackIds: LongArray): Int =
            throw PermissionDeniedException(Manifest.permission.WRITE_EXTERNAL_STORAGE)

        private fun <T> mediaUpdates(): Flow<List<T>> = flow {
            emit(emptyList())
            delay(Long.MAX_VALUE)
        }

        private fun permissionFailure() = flow<Nothing> {
            throw PermissionDeniedException(Manifest.permission.READ_EXTERNAL_STORAGE)
        }
    }
}