/*
 * bandcampDirect
 * Copyright (C) 2019 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.activity;

import android.net.Uri;
import android.os.Handler;
import android.os.Message;
import android.util.Log;
import android.view.View;
import android.webkit.CookieManager;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.TextView;
import android.widget.Toast;
import androidx.appcompat.app.AppCompatActivity;
import android.os.Bundle;
import godau.fynn.bandcampdirect.Bandcamp;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.SharedPreferences;
import godau.fynn.bandcampdirect.discover.Database;
import godau.fynn.bandcampdirect.model.Album;
import godau.fynn.bandcampdirect.model.User;
import org.json.JSONException;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import java.util.ArrayList;

public class DiscoverActivity extends AppCompatActivity {

    private static final String DISCOVER_URL = "https://bandcamp.com/";
    public static final String EXTRA_URL = "open";

    // Blacklist: don't discover these pages
    private static final String[] BLACKLIST = {
            // Commonly linked to pages that are not bandcamp
            "facebook.com",
            "instagram.com",
            "twitter.com",
            "soundcloud.com",
            // Not a link
            "#",
            // Bandcamp but not album or track view
            "bcbits.com",
            "bandcamp.com/redeem",
            "bandcamp.com/partner",
            "bandcamp.com/?from=menubar_logo_logged_in",
            "bandcamp.help",
            "bandcamp.com/redeem",
            "bandcamp.com?from=logo",
            "bandcamp.com/terms_of_use",
            "bandcamp.com/privacy",
            "bandcamp.com/copyright",
            "bandcamp.com/help_contact",
            "/help/downloading?from=tralbum_downloading",
            "/contact?b=",
            "bandcamp.com/login?from=menubar",
            "bandcamp.com/settings?from=menubar",
            // Don't try to load every fan's page
            "?from=fanthanks",
            // Usually a duplicate
            "?action=buy",
            "?action=gift",
            // Don't accidentally log out!
            "bandcamp.com/logout"
    };

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_discover);

        setTitle("discover | bandcampDirect");

        SharedPreferences sharedPreferences = new SharedPreferences(DiscoverActivity.this);
        User user = new User(sharedPreferences.getString(SharedPreferences.IDENTITY_TOKEN, null));

        Bandcamp bandcamp = new Bandcamp(user);

        Database database = Database.build(DiscoverActivity.this);

        WebView webView = findViewById(R.id.webview);

        webView.getSettings().setJavaScriptEnabled(true);

        webView.setWebViewClient(new WebViewClient());

        String url;
        if (getIntent().hasExtra(EXTRA_URL)) {
            url = getIntent().getStringExtra(EXTRA_URL);
        } else {
            url = DISCOVER_URL;
        }


        CookieManager cookieMan = CookieManager.getInstance();
        cookieMan.setCookie("bandcamp.com", user.getIdentityCookie(), value -> {
            webView.loadUrl(url);
        });


        findViewById(R.id.discover).setOnClickListener((view) -> {

            findViewById(R.id.webview).setVisibility(View.INVISIBLE);
            findViewById(R.id.discover).setVisibility(View.GONE);

            TextView status = findViewById(R.id.text);

            status.setText("Counting");

            // Get current URL
            String cUrl = webView.getUrl();
            bandcamp.request(cUrl, new Handler((message) -> {
                String html = Bandcamp.getHtmlFromMessage(message);


                ArrayList<String> discoverUrls = new ArrayList<>();
                discoverUrls.add(cUrl);

                // Find urls
                Document d = Jsoup.parse(html);

                Elements as = d.getElementsByTag("a");

                as:
                for (Element a :
                        as) {
                    // Get discover URL
                    String dUrl = a.attr("href");

                    // Test against blacklist
                    blacklist:
                    for (String black :
                            BLACKLIST) {
                        if (dUrl.contains(black)) {
                            continue as;
                        }
                    }

                    // https everywhere
                    dUrl = dUrl.replaceAll("http://", "https://");


                    if (!dUrl.contains("https://")) {
                        // Link not absolute

                        // What kind of relative is it?
                        if (dUrl.startsWith("/")) {
                            dUrl = "https://" + Uri.parse(cUrl).getHost() + dUrl;
                        } else {
                            dUrl = cUrl + dUrl;
                            dUrl = dUrl.replaceAll("//(?!.*bandcamp)", "/");
                        }
                    }

                    // Test for duplicate
                    if (discoverUrls.contains(dUrl)) {
                        continue as;
                    }

                    discoverUrls.add(dUrl);


                    Log.d("DISCOVERA", dUrl);
                }


                // Download each discoverUrl

                final int[] discoveredItems = {0};

                for (String discoverUrl :
                        discoverUrls) {
                    bandcamp.request(discoverUrl, new Handler((discoverMessage) -> {
                        String discoveredHtml = Bandcamp.getHtmlFromMessage(discoverMessage);
                        try {
                            Album album = new Album(discoveredHtml);


                            if (album.getTracks().size() > 1) {
                                AlbumActivity.discoverTracks(album, bandcamp, database, new Handler((msg -> {
                                    // Tracks will be written to database by function

                                    discoveredItems[0]++;
                                    updateStatus(discoveredItems[0], discoverUrls);

                                    return false;
                                })), new Handler(msg -> {
                                    // Fail
                                    discoveredItems[0]++;
                                    updateStatus(discoveredItems[0], discoverUrls);
                                    return false;
                                }));
                            } else {

                                database.writeAlbumTracksAsync(album);
                                discoveredItems[0]++;
                                updateStatus(discoveredItems[0], discoverUrls);

                            }




                        } catch (JSONException | ArrayIndexOutOfBoundsException e) {
                            // Apparently, this item can't be discovered, but let's not spam logcat
                            discoveredItems[0]++;
                            updateStatus(discoveredItems[0], discoverUrls);
                        }
                        return false;
                    }), new Handler((discoverMessage) -> {
                        discoveredItems[0]++;

                        updateStatus(discoveredItems[0], discoverUrls);


                        return false;
                    }));
                }


                return false;
            }), new Handler((message) -> {
                Toast.makeText(this, "An error occured", Toast.LENGTH_SHORT).show();
                return false;
            }));

        });
    }

    private void updateStatus(int discovered, ArrayList toDiscover) {

        TextView status = findViewById(R.id.text);


        if (discovered >= toDiscover.size()) {
            // This was the last item
            status.setText("Done");
            setResult(RESULT_OK);
            finish();
        } else {
            status.setText("Discovering " + (toDiscover.size() - discovered) + " items");
        }
    }

}
