/*
 * bandcampDirect
 * Copyright (C) 2019 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.activity;

import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.os.Bundle;
import android.os.Handler;
import android.text.InputType;
import android.view.*;
import android.widget.EditText;
import android.widget.LinearLayout;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AlertDialog;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.widget.AppCompatImageView;
import androidx.appcompat.widget.Toolbar;
import androidx.core.view.WindowCompat;
import com.android.volley.Response;
import com.android.volley.toolbox.Volley;
import godau.fynn.bandcampdirect.Bandcamp;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.SharedPreferences;
import godau.fynn.bandcampdirect.discover.Database;
import godau.fynn.bandcampdirect.model.Album;
import godau.fynn.bandcampdirect.model.Track;
import godau.fynn.bandcampdirect.view.AlbumRowView;
import godau.fynn.bandcampdirect.view.RowView;
import godau.fynn.bandcampdirect.view.TrackRowView;

import java.util.ArrayList;
import java.util.List;

public class FavoritesActivity extends AppCompatActivity {

    private static final int REQUEST_DISCOVER = 1;
    private static final int REQUEST_SEARCH = 2;
    private static final int REQUEST_LOGIN = 3;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_favorites);

        Toolbar toolbar = findViewById(R.id.toolbar);
        toolbar.setBackgroundColor(getResources().getColor(R.color.colorPrimary));
        setSupportActionBar(toolbar);

        // Apply correct colors
        getWindow().setStatusBarColor(getResources().getColor(R.color.colorPrimaryDark));

        setTitle("discover database | bandcampDirect");

        SharedPreferences sharedPreferences = new SharedPreferences(FavoritesActivity.this);
        Database database = Database.build(FavoritesActivity.this);


        LinearLayout trackLayout = findViewById(R.id.tracks);

        new Thread(() -> {

            List<Track> trackList = database.trackDao().getAll();
            List<Album> albums = database.albumDao().getAll();

            // Assign each album to a track
            categorizing:
            for (Track t :
                    trackList) {
                // Find a matching album

                searching:
                for (Album a :
                        albums) {
                    if (a.getId() == t.getAlbumId()) {
                        // match
                        a.getTracks().add(t);
                        continue categorizing;
                    } else {
                        continue searching;
                    }
                }
            }

            // Order every album
            for (Album a :
                    albums) {
                a.orderTracks();
            }

            // Tracks have been loaded, display them in UI
            runOnUiThread(() -> {


                for (Album a :
                        albums) {

                    Handler onLongClick = new Handler((m) -> {
                        new Thread(() -> { // Drop album, then recreate
                            database.drop(a);
                            runOnUiThread(this::recreate);
                        }).start();

                        return false;
                    });

                    RowView view;
                    if (a.getTracks().size() == 1) {
                        view = new TrackRowView(FavoritesActivity.this, trackLayout, a, 0, true, onLongClick);
                    } else {
                        view = new AlbumRowView(FavoritesActivity.this, trackLayout, a, true, onLongClick);
                    }
                    downloadCover(FavoritesActivity.this, view, a.getCover());
                }
            });

        }).start();


        View discover = findViewById(R.id.discover);

        discover.setOnLongClickListener((view) -> {
            new AlertDialog.Builder(FavoritesActivity.this)
                    .setTitle("Flush")
                    .setMessage("Would you like to flush your discover database?")
                    .setPositiveButton("Yes", (dialog, which) -> {
                        new Thread(() -> {
                            database.flush();
                            runOnUiThread(this::recreate);
                        }).start();
                    })
                    .setNegativeButton("No", null)
                    .show();

            return true;
        });

        discover.setOnClickListener((view) -> startActivityForResult(new Intent(
                        FavoritesActivity.this, DiscoverActivity.class),
                REQUEST_DISCOVER)
        );

        View search = findViewById(R.id.search);
        search.setOnClickListener((view) -> {

            EditText queryText = new EditText(FavoritesActivity.this);
            queryText.setInputType(InputType.TYPE_CLASS_TEXT);

            new AlertDialog.Builder(FavoritesActivity.this)
                    .setTitle("Search")
                    .setView(queryText)
                    .setPositiveButton("OK", (dialog, which) -> {
                        Intent intent = new Intent(FavoritesActivity.this, SearchActivity.class);
                        intent.putExtra(SearchActivity.EXTRA_QUERY, queryText.getText().toString());
                        startActivityForResult(intent, REQUEST_SEARCH);
                    })
                    .setNegativeButton("Cancel", null)
                    .show();


        });

        View login = findViewById(R.id.login);
        login.setOnClickListener(v -> startActivityForResult(
                new Intent(FavoritesActivity.this, LoginActivity.class), REQUEST_LOGIN));
    }

    public static void downloadCover(Context context, RowView rowView, String cover) {
        Volley.newRequestQueue(context).add(Bandcamp.makeCoverArtRequest(
                cover, new Response.Listener<Bitmap>() {
                    @Override
                    public void onResponse(Bitmap response) {
                        rowView.addCoverArt(response);
                    }
                }, null));
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        getMenuInflater().inflate(R.menu.main, menu);
        setOverflowButtonColor(this, getResources().getColor(R.color.white));
        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.action_about:
                startActivity(new Intent(FavoritesActivity.this, AboutLibrariesActivity.class));
        }
        return super.onOptionsItemSelected(item);
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, @Nullable Intent data) {

        if (resultCode == RESULT_OK) {
            recreate();
        }

        super.onActivityResult(requestCode, resultCode, data);
    }

    public static void setOverflowButtonColor(final AppCompatActivity activity, final int color) {
        // from https://stackoverflow.com/a/36278375 by https://stackoverflow.com/users/5064289/barun-kumar
        final String overflowDescription = activity.getString(R.string.abc_action_menu_overflow_description);
        final ViewGroup decorView = (ViewGroup) activity.getWindow().getDecorView();
        final ViewTreeObserver viewTreeObserver = decorView.getViewTreeObserver();
        viewTreeObserver.addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                final ArrayList<View> outViews = new ArrayList<View>();
                decorView.findViewsWithText(outViews, overflowDescription, View.FIND_VIEWS_WITH_CONTENT_DESCRIPTION);
                if (outViews.isEmpty()) {
                    return;
                }
                AppCompatImageView overflow = (AppCompatImageView) outViews.get(0);
                overflow.setColorFilter(color);
            }
        });
    }

}
