/*
 * bandcampDirect
 * Copyright (C) 2019 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.activity;

import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.view.View;
import android.webkit.CookieManager;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Button;
import android.widget.TextView;
import androidx.appcompat.app.AppCompatActivity;
import godau.fynn.bandcampdirect.Bandcamp;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.SharedPreferences;
import godau.fynn.bandcampdirect.model.User;

public class LoginActivity extends AppCompatActivity {

    public static final String LOGIN_URL = "https://bandcamp.com/login";

    /**
     * URL to be queried when testing login. It <i>needs</i> to be either a track or an album.
     */
    public static final String LOGIN_TEST_URL = "https://zachbenson.bandcamp.com/track/sunset";

    public static final String USER_PROFILE_URL_PREFIX = "https://bandcamp.com/";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_login);

        SharedPreferences sharedPreferences = new SharedPreferences(LoginActivity.this);
        boolean loggedIn = sharedPreferences.has(SharedPreferences.IDENTITY_TOKEN);

        TextView status = findViewById(R.id.status);
        Button login = findViewById(R.id.login);

        if (!loggedIn) {
            status.setText("You are not logged in");

            login.setText("Log in");
            login.setOnClickListener(v -> initLoginFlow());
        } else {
            status.setText("You are logged in as …");

            login.setText("Delete login cookie");
            login.setOnClickListener(v -> {
                sharedPreferences.edit().remove(SharedPreferences.IDENTITY_TOKEN).apply();
                recreate();
            });

            User user = sharedPreferences.createUser();
            Bandcamp bandcamp = new Bandcamp(user);
            bandcamp.request(LOGIN_TEST_URL, new Handler((message) -> {

                if (user.getUsername() != null) {

                    status.setText("You are logged in as " + user.getUsername());

                } else {

                    status.setText("Your login is broken, please delete its cookie");
                }

                View discover = findViewById(R.id.discover);
                discover.setVisibility(View.VISIBLE);
                discover.setOnClickListener(v -> {
                    Intent intent = new Intent(LoginActivity.this, DiscoverActivity.class);
                    intent.putExtra(DiscoverActivity.EXTRA_URL, USER_PROFILE_URL_PREFIX + user.getUsername());
                    startActivity(intent);
                    setResult(RESULT_OK);
                });



                return false;

            }), new Handler((message) -> {
                status.setText("You are logged in, but not connected to the internet");
                return false;
            }));
        }
    }

    private void initLoginFlow() {
        WebView webView = findViewById(R.id.webview);

        webView.setVisibility(View.VISIBLE);
        findViewById(R.id.root).setVisibility(View.GONE);

        WebSettings webSettings = webView.getSettings();
        webSettings.setJavaScriptEnabled(true);

        CookieManager cookieMan = CookieManager.getInstance();
        cookieMan.removeAllCookies(aBoolean -> {
            webView.loadUrl(LOGIN_URL);
        });

        webView.setWebViewClient(new WebViewClient() {

            @Override
            public void onLoadResource(WebView view, String url) {

                testIdentityCookie();

                super.onLoadResource(view, url);
            }

            @Override
            public void onPageFinished(WebView view, String url) {

                testIdentityCookie();

                super.onPageFinished(view, url);
            }

            private void testIdentityCookie() {
                String cookies = CookieManager.getInstance().getCookie("https://bandcamp.com");

                if (cookies != null && cookies.contains("identity")) {
                    String[] cookiesSeparated = cookies.split("; ");
                    for (String cookie :
                            cookiesSeparated) {
                        String[] cookieData = cookie.split("=");
                        if (cookieData[0].equals("identity")) {

                            // Identity was found
                            String identity = cookieData[1];
                            new SharedPreferences(LoginActivity.this).edit()
                                    .putString(SharedPreferences.IDENTITY_TOKEN, identity).apply();

                            setResult(RESULT_OK);
                            recreate();

                        }
                    }
                }
            }

        });
    }
}
