/*
 * bandcampDirect
 * Copyright (C) 2019 Fynn Godau
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Not affiliated with bandcamp, Incorporated.
 */

package godau.fynn.bandcampdirect.activity;

import android.os.Bundle;
import android.os.Handler;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.TextView;
import androidx.appcompat.app.AppCompatActivity;
import godau.fynn.bandcampdirect.Bandcamp;
import godau.fynn.bandcampdirect.R;
import godau.fynn.bandcampdirect.SharedPreferences;
import godau.fynn.bandcampdirect.model.*;
import godau.fynn.bandcampdirect.view.RowView;
import godau.fynn.bandcampdirect.view.SearchResultRowView;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import java.util.ArrayList;

public class SearchActivity extends AppCompatActivity {

    public static final String EXTRA_QUERY = "query";
    private static final String SEARCH_URL = "https://bandcamp.com/search?q=";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_search);

        SharedPreferences sharedPreferences = new SharedPreferences(SearchActivity.this);
        Bandcamp bandcamp = new Bandcamp(sharedPreferences.createUser());

        String query = getIntent().getStringExtra(EXTRA_QUERY);

        setTitle("searching " + query + " | bandcampDirect");

        TextView status = findViewById(R.id.status);

        status.setText("Searching");


        bandcamp.request(makeRequestUrl(query), new Handler((message) -> {

            String html = Bandcamp.getHtmlFromMessage(message);

            // Parse search results

            ArrayList<SearchResult> searchResults = new ArrayList<>();

            try {

                Document d = Jsoup.parse(html);

                Elements searchResultsElements = d.getElementsByClass("searchresult");

                for (Element searchResult :
                        searchResultsElements) {

                    Element resultInfo = searchResult.getElementsByClass("result-info").first();

                    String type = resultInfo
                            .getElementsByClass("itemtype").first().text();

                    String image = null;
                    Element img = searchResult.getElementsByClass("art").first()
                            .getElementsByTag("img").first();
                    if (img != null) {
                        image = img.attr("src");
                    }

                    String heading = resultInfo.getElementsByClass("heading").text();

                    String subhead = resultInfo.getElementsByClass("subhead").text();

                    String url = resultInfo.getElementsByClass("itemurl").text();

                    switch (type) {
                        default:
                            continue;
                        case "FAN":
                            searchResults.add(new Fan(heading, url, image));
                            break;

                        case "ARTIST":
                            String id = resultInfo.getElementsByClass("itemurl").first()
                                    .getElementsByTag("a").first()
                                    .attr("href") // the link contains the id
                                    .split("search_item_id=")
                                    [1] // the number is behind its name
                                    .split("&") // there is another attribute behind the name
                                    [0]; // get the number

                            searchResults.add(new Artist(heading, Long.parseLong(id), image, subhead));
                            break;

                        case "ALBUM":
                            String artist = subhead.split(" by")[0];
                            searchResults.add(new Album(heading, artist, url, image));
                            break;

                        case "TRACK":
                            String album = subhead.split("from ")[0].split(" by")[0];

                            String[] splitBy = subhead.split(" by");
                            String artist1 = null;
                            if (splitBy.length > 1) {
                                artist1 = subhead.split(" by")[1];
                            }
                            searchResults.add(new Track(heading, url, image, artist1, album));
                            break;
                    }

                }

                status.setVisibility(View.GONE);
            } catch (Exception e) {
                e.printStackTrace();
                status.setVisibility(View.VISIBLE);
                status.setText("bandcamp broke search");
            }

            // Display results

            LinearLayout parent = findViewById(R.id.results);


            for (SearchResult result :
                    searchResults) {

                RowView rowView = new SearchResultRowView(SearchActivity.this, parent, result);

                FavoritesActivity.downloadCover(SearchActivity.this, rowView, result.getCover());


            }

            // User might click something
            setResult(RESULT_OK);


            return false;
        }), new Handler(msg -> {
            status.setText("Networking issue");
            return false;
        }));
    }

    private String makeRequestUrl(String query) {

        query = query.replaceAll("%", "%25");
        query = query.replaceAll(" ", "%20");
        query = query.replaceAll("#", "%23");
        query = query.replaceAll("\\$", "%24");
        query = query.replaceAll("&", "%26");
        query = query.replaceAll("'", "%27");

        return SEARCH_URL + query;
    }
}
